import random

from pendulum import Duration

DEFAULT_JITTER_MIN = 0.5  # 50% of calculated delay (minimum)
DEFAULT_JITTER_MAX = 1.0  # 100% of calculated delay (maximum)


def exponential_backoff(
    attempt: int,
    min_delay: Duration,
    max_delay: Duration,
    jitter: bool = True,
    jitter_min: float = DEFAULT_JITTER_MIN,
    jitter_max: float = DEFAULT_JITTER_MAX,
) -> float:
    """Calculate exponential backoff delay with configurable jitter.

    Implements exponential backoff algorithm where delay doubles with each
    attempt, capped at max_delay. Optional jitter reduces thundering herd
    effects by adding randomization within specified bounds.

    Args:
        attempt: Current attempt number (0-based).
        min_delay: Minimum delay duration.
        max_delay: Maximum delay duration (cap).
        jitter: Whether to apply random jitter multiplier.
        jitter_min: Minimum jitter multiplier (default: DEFAULT_JITTER_MIN).
        jitter_max: Maximum jitter multiplier (default: DEFAULT_JITTER_MAX).

    Returns:
        Delay in seconds to wait before next attempt.

    Raises:
        ValueError: If jitter_min >= jitter_max or values are negative.
    """
    # Min jitter cannot be greater than max jitter
    if jitter and jitter_min >= jitter_max:
        raise ValueError("jitter_min must be less than jitter_max")

    # Jitter bounds cannot be negative
    if jitter and (jitter_min < 0 or jitter_max < 0):
        raise ValueError("jitter bounds must be non-negative")

    base_delay: float = min_delay.total_seconds() * (2**attempt)
    capped_delay: float = min(base_delay, max_delay.total_seconds())

    if jitter:
        # Apply jitter: multiply by random value in [jitter_min, jitter_max]
        jitter_range: float = jitter_max - jitter_min
        jitter_factor: float = jitter_min + (random.random() * jitter_range)
        capped_delay *= jitter_factor

    return capped_delay
