"""Async signals for cancellation and pause/resume control."""

from asyncio import Event


class BaseSignal:
    """Base class for async signal primitives."""

    def __init__(self) -> None:
        self._event: Event = Event()

    async def wait(self) -> None:
        """Block until signal is set."""
        await self._event.wait()

    def set(self) -> None:
        """Set the signal."""
        self._event.set()

    def clear(self) -> None:
        """Clear the signal."""
        self._event.clear()

    @property
    def is_set(self) -> bool:
        """True if signal is set."""
        return self._event.is_set()


class CancelSignal(BaseSignal):
    """Signal for cancellation control."""

    def cancel(self) -> None:
        """Mark as cancelled."""
        self.set()

    @property
    def is_cancelled(self) -> bool:
        """True if cancelled."""
        return self.is_set

    def __repr__(self) -> str:
        """Object representation."""
        return f"<CancelSignal is_cancelled={self.is_cancelled}>"


class PauseResumeEvent(BaseSignal):
    """Signal for pause/resume control."""

    def pause(self) -> None:
        """Mark as paused."""
        self.set()

    def resume(self) -> None:
        """Clear pause state."""
        self.clear()

    @property
    def is_paused(self) -> bool:
        """True if paused."""
        return self.is_set

    def __repr__(self) -> str:
        """Object representation."""
        return f"<PauseResumeEvent is_paused={self.is_paused}>"
