from pydantic import BaseModel, Field
from typing import Annotated, Generic, Literal, TypeVar, overload
from uuid import UUID
from nexo.enums.status import (
    ListOfDataStatuses,
    FULL_DATA_STATUSES,
)
from nexo.schemas.mixins.filter import convert as convert_filter
from nexo.schemas.mixins.identity import (
    IdentifierMixin,
    Ids,
    UUIDs,
    UUIDOrganizationId,
    UUIDOrganizationIds,
    UUIDPatientId,
    UUIDPatientIds,
    UUIDUserId,
    UUIDUserIds,
)
from nexo.schemas.mixins.sort import convert as convert_sort
from nexo.schemas.operation.enums import ResourceOperationStatusUpdateType
from nexo.schemas.parameter import (
    ReadSingleParameter as BaseReadSingleParameter,
    ReadPaginatedMultipleParameter,
    StatusUpdateParameter as BaseStatusUpdateParameter,
    DeleteSingleParameter as BaseDeleteSingleParameter,
)
from nexo.types.dict import StrToAnyDict
from nexo.types.integer import OptListOfInts
from nexo.types.uuid import OptListOfUUIDs
from ..enums.checkup import (
    IdentifierType,
    CheckupStatus as CheckupStatusEnum,
    OptCheckupStatus,
)
from ..mixins.common import IncludeURL, SessionId, SessionIds
from ..mixins.checkup import (
    CheckupStatus,
    CheckupStatuses,
    CheckupIdentifier,
)
from ..types.checkup import IdentifierValueType
from .document import DocumentMixin, GroupCSVDocument, PDFDocument


class CreateParameter(
    IncludeURL,
    DocumentMixin[GroupCSVDocument | PDFDocument],
    UUIDPatientId[UUID],
    SessionId[UUID],
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
):
    pass


class ReadMultipleParameter(
    IncludeURL,
    ReadPaginatedMultipleParameter,
    UUIDPatientIds[OptListOfUUIDs],
    SessionIds[OptListOfUUIDs],
    CheckupStatuses,
    UUIDOrganizationIds[OptListOfUUIDs],
    UUIDUserIds[OptListOfUUIDs],
    UUIDs[OptListOfUUIDs],
    Ids[OptListOfInts],
):
    ids: Annotated[OptListOfInts, Field(None, description="Ids")] = None
    uuids: Annotated[OptListOfUUIDs, Field(None, description="UUIDs")] = None
    user_ids: Annotated[OptListOfUUIDs, Field(None, description="User's IDs")] = None
    organization_ids: Annotated[
        OptListOfUUIDs, Field(None, description="Organization's IDs")
    ] = None
    session_ids: Annotated[OptListOfUUIDs, Field(None, description="Session's IDs")] = (
        None
    )
    patient_ids: Annotated[OptListOfUUIDs, Field(None, description="Patient's IDs")] = (
        None
    )

    @property
    def _query_param_fields(self) -> set[str]:
        return {
            "ids",
            "uuids",
            "statuses",
            "user_ids",
            "organization_ids",
            "checkup_statuses",
            "session_ids",
            "patient_ids",
            "search",
            "page",
            "limit",
            "use_cache",
            "include_url",
        }

    def to_query_params(self) -> StrToAnyDict:
        params = self.model_dump(
            mode="json", include=self._query_param_fields, exclude_none=True
        )
        params["filters"] = convert_filter(self.date_filters)
        params["sorts"] = convert_sort(self.sort_columns)
        params = {k: v for k, v in params.items()}
        return params


class ReadSingleParameter(IncludeURL, BaseReadSingleParameter[CheckupIdentifier]):
    @classmethod
    def from_identifier(
        cls,
        identifier: CheckupIdentifier,
        statuses: ListOfDataStatuses = FULL_DATA_STATUSES,
        use_cache: bool = True,
        include_url: bool = False,
    ) -> "ReadSingleParameter":
        return cls(
            identifier=identifier,
            statuses=statuses,
            use_cache=use_cache,
            include_url=include_url,
        )

    @overload
    @classmethod
    def new(
        cls,
        identifier_type: Literal[IdentifierType.ID],
        identifier_value: int,
        statuses: ListOfDataStatuses = FULL_DATA_STATUSES,
        use_cache: bool = True,
        include_url: bool = False,
    ) -> "ReadSingleParameter": ...
    @overload
    @classmethod
    def new(
        cls,
        identifier_type: Literal[IdentifierType.UUID],
        identifier_value: UUID,
        statuses: ListOfDataStatuses = FULL_DATA_STATUSES,
        use_cache: bool = True,
        include_url: bool = False,
    ) -> "ReadSingleParameter": ...
    @overload
    @classmethod
    def new(
        cls,
        identifier_type: IdentifierType,
        identifier_value: IdentifierValueType,
        statuses: ListOfDataStatuses = FULL_DATA_STATUSES,
        use_cache: bool = True,
        include_url: bool = False,
    ) -> "ReadSingleParameter": ...
    @classmethod
    def new(
        cls,
        identifier_type: IdentifierType,
        identifier_value: IdentifierValueType,
        statuses: ListOfDataStatuses = FULL_DATA_STATUSES,
        use_cache: bool = True,
        include_url: bool = False,
    ) -> "ReadSingleParameter":
        return cls(
            identifier=CheckupIdentifier(type=identifier_type, value=identifier_value),
            statuses=statuses,
            use_cache=use_cache,
            include_url=include_url,
        )

    def to_query_params(self) -> StrToAnyDict:
        return self.model_dump(
            mode="json",
            include={"statuses", "use_cache", "include_url"},
            exclude_none=True,
        )


class TransitionExamination(BaseModel):
    parameter_id: Annotated[UUID, Field(..., description="Parameter's ID")]
    value: Annotated[bool | float | int | str | None, Field(..., description="Value")]


ListOfTransitionExaminations = list[TransitionExamination]
OptListOfTransitionExaminations = ListOfTransitionExaminations | None


class TransitionData(BaseModel):
    examinations: Annotated[
        OptListOfTransitionExaminations,
        Field(None, description="Transition examination"),
    ]


class FullUpdateData(CheckupStatus[CheckupStatusEnum]):
    pass


class PartialUpdateData(CheckupStatus[OptCheckupStatus]):
    checkup_status: Annotated[
        OptCheckupStatus, Field(None, description="Checkup's status")
    ] = None


UpdateDataT = TypeVar("UpdateDataT", FullUpdateData, PartialUpdateData)


class UpdateDataMixin(BaseModel, Generic[UpdateDataT]):
    data: UpdateDataT = Field(..., description="Update data")


class UpdateParameter(
    IncludeURL,
    UpdateDataMixin[UpdateDataT],
    IdentifierMixin[CheckupIdentifier],
    Generic[UpdateDataT],
):
    @overload
    @classmethod
    def new(
        cls,
        identifier_type: Literal[IdentifierType.ID],
        identifier_value: int,
        data: UpdateDataT,
        include_url: bool = False,
    ) -> "UpdateParameter": ...
    @overload
    @classmethod
    def new(
        cls,
        identifier_type: Literal[IdentifierType.UUID],
        identifier_value: UUID,
        data: UpdateDataT,
        include_url: bool = False,
    ) -> "UpdateParameter": ...
    @overload
    @classmethod
    def new(
        cls,
        identifier_type: IdentifierType,
        identifier_value: IdentifierValueType,
        data: UpdateDataT,
        include_url: bool = False,
    ) -> "UpdateParameter": ...
    @classmethod
    def new(
        cls,
        identifier_type: IdentifierType,
        identifier_value: IdentifierValueType,
        data: UpdateDataT,
        include_url: bool = False,
    ) -> "UpdateParameter":
        return cls(
            identifier=CheckupIdentifier(type=identifier_type, value=identifier_value),
            data=data,
            include_url=include_url,
        )


class StatusUpdateParameter(
    IncludeURL,
    BaseStatusUpdateParameter[CheckupIdentifier],
):
    @overload
    @classmethod
    def new(
        cls,
        identifier_type: Literal[IdentifierType.ID],
        identifier_value: int,
        type: ResourceOperationStatusUpdateType,
        include_url: bool = False,
    ) -> "StatusUpdateParameter": ...
    @overload
    @classmethod
    def new(
        cls,
        identifier_type: Literal[IdentifierType.UUID],
        identifier_value: UUID,
        type: ResourceOperationStatusUpdateType,
        include_url: bool = False,
    ) -> "StatusUpdateParameter": ...
    @overload
    @classmethod
    def new(
        cls,
        identifier_type: IdentifierType,
        identifier_value: IdentifierValueType,
        type: ResourceOperationStatusUpdateType,
        include_url: bool = False,
    ) -> "StatusUpdateParameter": ...
    @classmethod
    def new(
        cls,
        identifier_type: IdentifierType,
        identifier_value: IdentifierValueType,
        type: ResourceOperationStatusUpdateType,
        include_url: bool = False,
    ) -> "StatusUpdateParameter":
        return cls(
            identifier=CheckupIdentifier(type=identifier_type, value=identifier_value),
            type=type,
            include_url=include_url,
        )


class DeleteSingleParameter(BaseDeleteSingleParameter[CheckupIdentifier]):
    @overload
    @classmethod
    def new(
        cls, identifier_type: Literal[IdentifierType.ID], identifier_value: int
    ) -> "DeleteSingleParameter": ...
    @overload
    @classmethod
    def new(
        cls, identifier_type: Literal[IdentifierType.UUID], identifier_value: UUID
    ) -> "DeleteSingleParameter": ...
    @overload
    @classmethod
    def new(
        cls, identifier_type: IdentifierType, identifier_value: IdentifierValueType
    ) -> "DeleteSingleParameter": ...
    @classmethod
    def new(
        cls, identifier_type: IdentifierType, identifier_value: IdentifierValueType
    ) -> "DeleteSingleParameter":
        return cls(
            identifier=CheckupIdentifier(type=identifier_type, value=identifier_value)
        )
