from datetime import date
from pydantic import BaseModel, Field, model_validator
from typing import Annotated, Generic, Self, Type, TypeVar
from uuid import UUID
from nexo.enums.identity import (
    OptRhesus,
    RhesusMixin,
    OptBloodType,
    BloodTypeMixin,
    Gender,
    GenderMixin,
)
from nexo.enums.status import (
    DataStatus,
    SimpleDataStatusMixin,
)
from nexo.schemas.mixins.identity import (
    DataIdentifier,
    UUIDUserId,
    UUIDOrganizationId,
    DateOfBirth,
)
from nexo.schemas.mixins.timestamp import LifecycleTimestamp
from nexo.types.string import OptStr, OptListOfStrs
from nexo.types.uuid import ListOfUUIDs
from ..enums.checkup import CheckupStatus as CheckupStatusEnum
from ..enums.examination import (
    ExaminationStatus as ExaminationStatusEnum,
    OptExaminationStatus,
)
from ..enums.parameter import (
    ParameterGroup,
    ValueType as ValueTypeEnum,
)
from ..enums.session import SessionType as SessionTypeEnum
from ..mixins.common import ParameterIds
from ..mixins.checkup import CheckupStatus
from ..mixins.client import Name as ClientName
from ..mixins.examination import ExaminationStatus
from ..mixins.parameter import (
    _validate_value_type_and_options,
    Group,
    IsMandatory,
    Name as ParameterName,
    Aliases,
    ValueType,
    Options,
    IsNullable,
    Unit,
)
from ..mixins.patient import IdCard, FullName, PlaceOfBirth
from ..mixins.rule import RuleData
from ..mixins.session import SessionType, Name as SessionName
from .document import DocumentName, DocumentURL


class ClientSchema(
    ClientName[str],
    UUIDOrganizationId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


OptClientSchema = ClientSchema | None
OptClientSchemaT = TypeVar("OptClientSchemaT", bound=OptClientSchema)


class ClientSchemaMixin(BaseModel, Generic[OptClientSchemaT]):
    client: Annotated[OptClientSchemaT, Field(..., description="Client")]


class PatientSchema(
    RhesusMixin[OptRhesus],
    BloodTypeMixin[OptBloodType],
    GenderMixin[Gender],
    DateOfBirth[date],
    PlaceOfBirth[OptStr],
    FullName[str],
    IdCard[str],
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):

    pass


class PatientSchemaMixin(BaseModel):
    patient: Annotated[PatientSchema, Field(..., description="Patient")]


class StandardParameterSchema(
    Unit[OptStr],
    IsNullable[bool],
    Options[OptListOfStrs],
    ValueType[ValueTypeEnum],
    Aliases[OptListOfStrs],
    ParameterName[str],
    Group[ParameterGroup],
    IsMandatory[bool],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    @model_validator(mode="after")
    def validate_value_type_and_options(self) -> Self:
        _validate_value_type_and_options(self.value_type, self.options)
        return self


class StandardParameterSchemaMixin(BaseModel):
    parameter: Annotated[StandardParameterSchema, Field(..., description="Parameter")]


class StandardRuleSchema(
    RuleData,
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


OptStandardRuleSchema = StandardRuleSchema | None
ListOfStandardRuleSchemas = list[StandardRuleSchema]


class StandardRuleSchemasMixin(BaseModel):
    rules: Annotated[
        ListOfStandardRuleSchemas,
        Field(ListOfStandardRuleSchemas(), description="Rules"),
    ] = ListOfStandardRuleSchemas()


class FullParameterSchema(
    StandardRuleSchemasMixin,
    StandardParameterSchema,
):
    pass


AnyParameterSchemaType = Type[StandardParameterSchema] | Type[FullParameterSchema]
AnyParameterSchema = StandardParameterSchema | FullParameterSchema
AnyParameterSchemaT = TypeVar("AnyParameterSchemaT", bound=AnyParameterSchema)
OptAnyParameterSchema = AnyParameterSchema | None
OptAnyParameterSchemaT = TypeVar("OptAnyParameterSchemaT", bound=OptAnyParameterSchema)


class ParameterSchemaMixin(BaseModel, Generic[OptAnyParameterSchemaT]):
    parameter: Annotated[OptAnyParameterSchemaT, Field(..., description="Parameter")]


class FullRuleSchema(
    RuleData,
    StandardParameterSchemaMixin,
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


AnyRuleSchemaType = Type[StandardRuleSchema] | Type[FullRuleSchema]
AnyRuleSchema = StandardRuleSchema | FullRuleSchema
AnyRuleSchemaT = TypeVar("AnyRuleSchemaT", bound=AnyRuleSchema)
OptAnyRuleSchema = AnyRuleSchema | None
OptAnyRuleSchemaT = TypeVar("OptAnyRuleSchemaT", bound=OptAnyRuleSchema)


class RuleSchemaMixin(BaseModel, Generic[OptAnyRuleSchemaT]):
    rule: Annotated[OptAnyRuleSchemaT, Field(..., description="Rule")]


class BaseExaminationSchema(ParameterSchemaMixin[StandardParameterSchema]):
    value: Annotated[
        bool | float | int | str | None, Field(None, description="Value")
    ] = None


class ExaminationSchema(
    RuleSchemaMixin[OptStandardRuleSchema],
    ExaminationStatus[OptExaminationStatus],
    BaseExaminationSchema,
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    def classify(self, age: float, rules: list[StandardRuleSchema]):
        parameter = self.parameter

        if parameter.group is ParameterGroup.ANAMNESIS:
            self.examination_status = None
            return

        if not rules or len(rules) == 0:
            self.examination_status = None
            return

        if len(rules) <= 0:
            self.examination_status = ExaminationStatusEnum.NO_MATCHING_RULE
        else:
            for rule in rules:
                if not rule.genders:
                    continue
                if rule.age.min > age > rule.age.max:
                    continue


ListOfExaminationSchemas = list[ExaminationSchema]


class ExaminationSchemasMixin(BaseModel):
    examinations: Annotated[
        ListOfExaminationSchemas,
        Field(ListOfExaminationSchemas(), description="Examinations"),
    ] = ListOfExaminationSchemas()


class StandardCheckupSchema(
    ExaminationSchemasMixin,
    PatientSchemaMixin,
    DocumentURL,
    DocumentName,
    CheckupStatus[CheckupStatusEnum],
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


ListOfStandardCheckupSchemas = list[StandardCheckupSchema]


class StandardCheckupSchemasMixin(BaseModel):
    checkups: Annotated[
        ListOfStandardCheckupSchemas,
        Field(ListOfStandardCheckupSchemas(), description="Checkups"),
    ] = ListOfStandardCheckupSchemas()


class StandardSessionSchema(
    ParameterIds[ListOfUUIDs],
    SessionName[str],
    ClientSchemaMixin[OptClientSchema],
    SessionType[SessionTypeEnum],
    UUIDOrganizationId[UUID],
    UUIDUserId[UUID],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    @model_validator(mode="after")
    def validate_type_client(self) -> Self:
        if self.type is SessionTypeEnum.GROUP:
            if self.client is None:
                raise ValueError("Client can not be None for group MCU")
        elif self.type is SessionTypeEnum.INDIVIDUAL:
            if self.client is not None:
                raise ValueError("Client must be None for individual MCU")
        return self


class StandardSessionSchemaMixin(BaseModel):
    session: Annotated[StandardSessionSchema, Field(..., description="Session")]


class FullCheckupSchema(StandardSessionSchemaMixin, StandardCheckupSchema):
    pass


AnyCheckupSchemaType = Type[StandardCheckupSchema] | Type[FullCheckupSchema]
AnyCheckupSchema = StandardCheckupSchema | FullCheckupSchema


class FullSessionSchema(StandardCheckupSchemasMixin, StandardSessionSchema):
    pass


AnySessionSchemaType = Type[StandardSessionSchema] | Type[FullSessionSchema]
AnySessionSchema = StandardSessionSchema | FullSessionSchema
