"""
Productivity Dashboard Web Server.

FastAPI-based web server for Context Cleaner dashboard interface.
"""

from datetime import datetime
from typing import Optional

from fastapi import FastAPI, HTTPException
from fastapi.responses import HTMLResponse, JSONResponse
import uvicorn

from ..config.settings import ContextCleanerConfig
from ..analytics.productivity_analyzer import ProductivityAnalyzer


class ProductivityDashboard:
    """Web-based productivity dashboard for Context Cleaner."""

    def __init__(self, config: Optional[ContextCleanerConfig] = None):
        self.config = config or ContextCleanerConfig.default()
        self.analyzer = ProductivityAnalyzer(self.config)
        self.app = FastAPI(
            title="Context Cleaner Dashboard",
            description="Advanced productivity tracking and context optimization",
            version="0.1.0",
        )
        self._setup_routes()

    def _setup_routes(self):
        """Setup FastAPI routes."""

        @self.app.get("/", response_class=HTMLResponse)
        async def dashboard_home():
            """Serve main dashboard page."""
            return self._generate_dashboard_html()

        @self.app.get("/api/health")
        async def health_check():
            """API health check endpoint."""
            return {"status": "healthy", "timestamp": datetime.now().isoformat()}

        @self.app.get("/api/productivity-summary")
        async def get_productivity_summary(days: int = 7):
            """Get productivity summary for specified number of days."""
            try:
                # This would typically load real data
                summary = {
                    "period_days": days,
                    "avg_productivity_score": 85.3,
                    "total_sessions": 23,
                    "optimization_events": 12,
                    "health_trend": "improving",
                    "recommendations": [
                        "Context health is excellent - keep up the good work!",
                        "Consider periodic cleanup to maintain performance",
                        "Your afternoon sessions show highest productivity",
                    ],
                    "last_updated": datetime.now().isoformat(),
                }
                return JSONResponse(content=summary)
            except Exception as e:
                raise HTTPException(status_code=500, detail=str(e))

        @self.app.get("/api/session-analytics")
        async def get_session_analytics():
            """Get detailed session analytics."""
            try:
                analytics = {
                    "session_types": {
                        "productive_coding": 45,
                        "debugging_session": 25,
                        "optimization_session": 20,
                        "exploration": 10,
                    },
                    "hourly_productivity": {
                        "09": 78,
                        "10": 85,
                        "11": 92,
                        "12": 88,
                        "13": 75,
                        "14": 88,
                        "15": 95,
                        "16": 90,
                        "17": 85,
                        "18": 70,
                    },
                    "weekly_trends": {
                        "Monday": 82,
                        "Tuesday": 88,
                        "Wednesday": 85,
                        "Thursday": 90,
                        "Friday": 78,
                    },
                    "optimization_impact": {
                        "avg_improvement": 15.3,
                        "success_rate": 78.5,
                    },
                }
                return JSONResponse(content=analytics)
            except Exception as e:
                raise HTTPException(status_code=500, detail=str(e))

        @self.app.get("/api/recommendations")
        async def get_recommendations():
            """Get current optimization recommendations."""
            try:
                recommendations = [
                    {
                        "type": "optimization",
                        "priority": "high",
                        "title": "Context Cleanup Recommended",
                        "description": "Your context size has grown to 45K tokens. Consider cleanup.",
                        "action": "context-cleaner optimize",
                    },
                    {
                        "type": "productivity",
                        "priority": "medium",
                        "title": "Peak Performance Window",
                        "description": "Your productivity peaks at 3-4 PM. Schedule complex tasks then.",
                        "action": "Schedule important work in the afternoon",
                    },
                    {
                        "type": "health",
                        "priority": "low",
                        "title": "Excellent Context Health",
                        "description": "Your current context health is 87/100 - keep up the good work!",
                        "action": "Continue current practices",
                    },
                ]
                return JSONResponse(content=recommendations)
            except Exception as e:
                raise HTTPException(status_code=500, detail=str(e))

        @self.app.post("/api/privacy/export-data")
        async def export_user_data():
            """Export all user data for privacy compliance."""
            try:
                data = {
                    "export_timestamp": datetime.now().isoformat(),
                    "version": "0.1.0",
                    "sessions": [],  # Would load actual session data
                    "privacy_notice": "All data is processed locally on your machine",
                }
                return JSONResponse(content=data)
            except Exception as e:
                raise HTTPException(status_code=500, detail=str(e))

        @self.app.delete("/api/privacy/delete-data")
        async def delete_all_data():
            """Delete all collected data for privacy compliance."""
            try:
                # This would delete actual data files
                return {
                    "message": "All data deleted successfully",
                    "timestamp": datetime.now().isoformat(),
                }
            except Exception as e:
                raise HTTPException(status_code=500, detail=str(e))

    def _generate_dashboard_html(self) -> str:
        """Generate HTML for the dashboard interface."""
        return """
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Context Cleaner Dashboard</title>
    <script src="https://cdn.plot.ly/plotly-latest.min.js"></script>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .metric-card { border-left: 4px solid #007bff; }
        .chart-container { height: 400px; margin: 20px 0; }
        .navbar-brand { font-weight: bold; }
        .privacy-notice { background: #f8f9fa; border-left: 4px solid #28a745; padding: 15px; margin: 20px 0; }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <span class="navbar-brand">📊 Context Cleaner Dashboard</span>
            <span class="navbar-text">Advanced Productivity Tracking</span>
        </div>
    </nav>

    <div class="container mt-4">
        <div class="privacy-notice">
            <h6>🔒 Privacy Notice</h6>
            <p class="mb-0">All data is processed locally on your machine. Nothing is sent to external servers.</p>
        </div>

        <!-- Metrics Overview -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card metric-card">
                    <div class="card-body">
                        <h5 class="card-title">Productivity Score</h5>
                        <h2 class="text-primary" id="productivity-score">85.3</h2>
                        <small class="text-muted">Last 7 days average</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card metric-card">
                    <div class="card-body">
                        <h5 class="card-title">Total Sessions</h5>
                        <h2 class="text-success" id="total-sessions">23</h2>
                        <small class="text-muted">Development sessions</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card metric-card">
                    <div class="card-body">
                        <h5 class="card-title">Optimizations</h5>
                        <h2 class="text-warning" id="optimizations">12</h2>
                        <small class="text-muted">Context improvements</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card metric-card">
                    <div class="card-body">
                        <h5 class="card-title">Health Trend</h5>
                        <h2 class="text-info" id="health-trend">📈</h2>
                        <small class="text-muted">Improving</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Charts -->
        <div class="row">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5>📈 Productivity Trend</h5>
                    </div>
                    <div class="card-body">
                        <div id="productivity-chart" class="chart-container"></div>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5>🕐 Hourly Performance</h5>
                    </div>
                    <div class="card-body">
                        <div id="hourly-chart" class="chart-container"></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recommendations -->
        <div class="row mt-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5>💡 Optimization Recommendations</h5>
                    </div>
                    <div class="card-body">
                        <div id="recommendations-list">
                            <div class="d-flex justify-content-center">
                                <div class="spinner-border" role="status">
                                    <span class="visually-hidden">Loading...</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Privacy Controls -->
        <div class="row mt-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h5>🔒 Privacy Controls</h5>
                    </div>
                    <div class="card-body">
                        <button class="btn btn-outline-primary me-2" onclick="exportData()">📦 Export My Data</button>
                        <button class="btn btn-outline-danger" onclick="deleteData()">🗑️ Delete All Data</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Load dashboard data
        async function loadDashboard() {
            try {
                // Load productivity summary
                const summaryResponse = await fetch('/api/productivity-summary');
                const summary = await summaryResponse.json();
                
                // Update metrics
                document.getElementById('productivity-score').textContent = summary.avg_productivity_score;
                document.getElementById('total-sessions').textContent = summary.total_sessions;
                document.getElementById('optimizations').textContent = summary.optimization_events;
                
                // Load analytics
                const analyticsResponse = await fetch('/api/session-analytics');
                const analytics = await analyticsResponse.json();
                
                // Create productivity trend chart
                const productivityData = [{
                    x: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri'],
                    y: Object.values(analytics.weekly_trends),
                    type: 'scatter',
                    mode: 'lines+markers',
                    name: 'Productivity Score',
                    line: { color: '#007bff' }
                }];
                
                Plotly.newPlot('productivity-chart', productivityData, {
                    title: 'Weekly Productivity Trend',
                    xaxis: { title: 'Day of Week' },
                    yaxis: { title: 'Productivity Score' }
                });
                
                // Create hourly performance chart
                const hourlyData = [{
                    x: Object.keys(analytics.hourly_productivity),
                    y: Object.values(analytics.hourly_productivity),
                    type: 'bar',
                    marker: { color: '#28a745' }
                }];
                
                Plotly.newPlot('hourly-chart', hourlyData, {
                    title: 'Hourly Performance Pattern',
                    xaxis: { title: 'Hour of Day' },
                    yaxis: { title: 'Average Score' }
                });
                
                // Load recommendations
                const recsResponse = await fetch('/api/recommendations');
                const recommendations = await recsResponse.json();
                
                const recsHtml = recommendations.map(rec => `
                    <div class="alert alert-${rec.priority === 'high' ? 'warning' : rec.priority === 'medium' ? 'info' : 'success'} d-flex justify-content-between align-items-center">
                        <div>
                            <strong>${rec.title}</strong><br>
                            <small>${rec.description}</small>
                        </div>
                        <span class="badge bg-${rec.priority === 'high' ? 'warning' : rec.priority === 'medium' ? 'info' : 'success'}">${rec.priority.toUpperCase()}</span>
                    </div>
                `).join('');
                
                document.getElementById('recommendations-list').innerHTML = recsHtml;
                
            } catch (error) {
                console.error('Failed to load dashboard data:', error);
            }
        }
        
        async function exportData() {
            try {
                const response = await fetch('/api/privacy/export-data', { method: 'POST' });
                const data = await response.json();
                
                const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = 'context-cleaner-data.json';
                a.click();
                URL.revokeObjectURL(url);
                
                alert('✅ Data exported successfully!');
            } catch (error) {
                alert('❌ Export failed: ' + error.message);
            }
        }
        
        async function deleteData() {
            if (confirm('⚠️ This will permanently delete ALL your productivity data. Continue?')) {
                try {
                    await fetch('/api/privacy/delete-data', { method: 'DELETE' });
                    alert('✅ All data deleted successfully!');
                    location.reload();
                } catch (error) {
                    alert('❌ Deletion failed: ' + error.message);
                }
            }
        }
        
        // Auto-refresh every 30 seconds
        setInterval(loadDashboard, 30000);
        
        // Initial load
        loadDashboard();
    </script>
</body>
</html>
        """

    def start_server(self, host: str = "localhost", port: int = 8548):
        """Start the dashboard server."""
        uvicorn.run(self.app, host=host, port=port, log_level="info")
