#!/usr/bin/env python3
"""
Tests for ManipulationValidator

Tests the validation and safety checks for context manipulation operations including:
- Operation validation
- Plan validation  
- Integrity verification
- Safety report generation
- Risk assessment
"""

import pytest
from datetime import datetime
from unittest.mock import Mock

from context_cleaner.core.manipulation_validator import (
    ManipulationValidator,
    ValidationResult,
    IntegrityCheck,
    RiskLevel,
    SafetyAction,
    RiskAssessment,
    SafetyConstraints,
    OperationHistory,
    validate_operation,
    validate_plan,
    verify_manipulation_integrity
)
from context_cleaner.core.manipulation_engine import (
    ManipulationOperation,
    ManipulationPlan
)


class TestManipulationValidator:
    """Test suite for ManipulationValidator."""

    @pytest.fixture
    def sample_context_data(self):
        """Sample context data for testing."""
        return {
            "message_1": "Help me debug this authentication function", 
            "message_2": "Help me debug this authentication function",  # Duplicate
            "todo_1": "Fix the critical login bug in authentication system",
            "todo_2": "Write unit tests for auth module", 
            "config_setting": "database_password = secret123",  # High-risk content
            "normal_content": "This is regular content",
            "timestamp": datetime.now().isoformat()
        }

    @pytest.fixture
    def validator(self):
        """ManipulationValidator instance for testing."""
        return ManipulationValidator()

    @pytest.fixture
    def sample_remove_operation(self):
        """Sample remove operation for testing."""
        return ManipulationOperation(
            operation_id="test-001",
            operation_type="remove",
            target_keys=["message_2"],
            operation_data={"removal_type": "safe_delete"},
            estimated_token_impact=-50,
            confidence_score=0.9,
            reasoning="Removing duplicate content",
            requires_confirmation=False
        )

    @pytest.fixture
    def sample_consolidate_operation(self):
        """Sample consolidate operation for testing."""
        return ManipulationOperation(
            operation_id="test-002",
            operation_type="consolidate",
            target_keys=["todo_1", "todo_2"],
            operation_data={"strategy": "merge_similar"},
            estimated_token_impact=-30,
            confidence_score=0.75,
            reasoning="Consolidating similar todos",
            requires_confirmation=True
        )

    @pytest.fixture
    def risky_operation(self):
        """High-risk operation for testing."""
        return ManipulationOperation(
            operation_id="test-003",
            operation_type="remove",
            target_keys=["config_setting"],  # High-risk content
            operation_data={"removal_type": "safe_delete"},
            estimated_token_impact=-100,
            confidence_score=0.6,  # Low confidence
            reasoning="Removing config setting",
            requires_confirmation=True
        )

    def test_validator_initialization(self):
        """Test ManipulationValidator initialization."""
        validator = ManipulationValidator()
        
        assert validator.min_confidence == ManipulationValidator.MIN_SAFE_CONFIDENCE
        assert validator.max_operation_impact == ManipulationValidator.MAX_SINGLE_OPERATION_IMPACT
        assert validator.max_total_reduction == ManipulationValidator.MAX_TOTAL_REDUCTION

    def test_validator_initialization_with_config(self):
        """Test ManipulationValidator initialization with custom config."""
        config = {
            'min_confidence': 0.8,
            'max_operation_impact': 0.4,
            'max_total_reduction': 0.9
        }
        
        validator = ManipulationValidator(config)
        
        assert validator.min_confidence == 0.8
        assert validator.max_operation_impact == 0.4
        assert validator.max_total_reduction == 0.9

    def test_assess_content_risk_high(self, validator):
        """Test high-risk content assessment."""
        high_risk_content = "password = secret123"
        risk = validator._assess_content_risk(high_risk_content)
        assert risk == "high"
        
        critical_content = "This is critical system configuration"
        risk = validator._assess_content_risk(critical_content)
        assert risk == "high"

    def test_assess_content_risk_medium(self, validator):
        """Test medium-risk content assessment."""
        medium_risk_content = "todo: fix the authentication bug"
        risk = validator._assess_content_risk(medium_risk_content)
        assert risk == "medium"
        
        file_content = "reading file /project/src/main.py"
        risk = validator._assess_content_risk(file_content)
        assert risk == "medium"

    def test_assess_content_risk_low(self, validator):
        """Test low-risk content assessment."""
        low_risk_content = "This is some plain text without any special keywords"
        risk = validator._assess_content_risk(low_risk_content)
        assert risk == "low"

    def test_calculate_content_importance(self, validator):
        """Test content importance calculation."""
        # High importance content
        importance = validator._calculate_content_importance("critical_config", "This is critical system setting")
        assert importance > 0.7
        
        # Medium importance content  
        importance = validator._calculate_content_importance("normal_key", "Regular content")
        assert 0.3 <= importance <= 0.7
        
        # Low importance content
        importance = validator._calculate_content_importance("duplicate_key", "This is completed and duplicate")
        assert importance < 0.5

    def test_validate_operation_success(self, validator):
        """Test successful operation validation with simple context."""
        # Use a smaller, simpler context to avoid impact threshold issues
        simple_context = {
            "item1": "First item",
            "item2": "Second item", 
            "item3": "Third item"
        }
        
        simple_operation = ManipulationOperation(
            operation_id="test-simple",
            operation_type="remove",
            target_keys=["item2"],
            operation_data={"removal_type": "safe_delete"},
            estimated_token_impact=-1,  # Very small impact to stay under 30% threshold
            confidence_score=0.9,
            reasoning="Removing test item",
            requires_confirmation=False
        )
        
        result = validator.validate_operation(simple_operation, simple_context)
        
        assert isinstance(result, ValidationResult)
        assert result.is_valid is True
        assert result.confidence_score > 0.7
        assert result.risk_assessment in ["low", "medium", "high"]
        assert len(result.validation_errors) == 0

    def test_validate_operation_missing_keys(self, validator, sample_context_data):
        """Test operation validation with missing target keys."""
        invalid_operation = ManipulationOperation(
            operation_id="test-invalid",
            operation_type="remove",
            target_keys=["non_existent_key"],
            operation_data={},
            estimated_token_impact=-10,
            confidence_score=0.9,
            reasoning="Test invalid operation",
            requires_confirmation=False
        )
        
        result = validator.validate_operation(invalid_operation, sample_context_data)
        
        assert result.is_valid is False
        assert len(result.validation_errors) > 0
        assert any("not found" in error.lower() for error in result.validation_errors)

    def test_validate_operation_no_target_keys(self, validator, sample_context_data):
        """Test operation validation with no target keys."""
        invalid_operation = ManipulationOperation(
            operation_id="test-no-keys",
            operation_type="remove",
            target_keys=[],  # No target keys
            operation_data={},
            estimated_token_impact=0,
            confidence_score=0.9,
            reasoning="Test operation with no keys",
            requires_confirmation=False
        )
        
        result = validator.validate_operation(invalid_operation, sample_context_data)
        
        assert result.is_valid is False
        assert any("no target keys" in error.lower() for error in result.validation_errors)

    def test_validate_operation_low_confidence(self, validator, sample_context_data):
        """Test operation validation with low confidence."""
        low_confidence_operation = ManipulationOperation(
            operation_id="test-low-confidence",
            operation_type="remove",
            target_keys=["message_1"],
            operation_data={},
            estimated_token_impact=-10,
            confidence_score=0.5,  # Low confidence
            reasoning="Low confidence operation",
            requires_confirmation=False
        )
        
        result = validator.validate_operation(low_confidence_operation, sample_context_data)
        
        # Should still be valid but with warnings
        assert len(result.warnings) > 0
        # The overall confidence might still be reasonable due to other factors

    def test_validate_operation_high_risk_content(self, validator, risky_operation, sample_context_data):
        """Test validation of operation on high-risk content."""
        result = validator.validate_operation(risky_operation, sample_context_data)
        
        assert result.risk_assessment == "high"
        assert len(result.safety_recommendations) > 0
        # Should have safety recommendation
        assert any("confirmation" in rec.lower() or "recommend" in rec.lower() for rec in result.safety_recommendations)

    def test_validate_operation_large_impact(self, validator, sample_context_data):
        """Test validation of operation with large impact."""
        large_impact_operation = ManipulationOperation(
            operation_id="test-large-impact",
            operation_type="remove",
            target_keys=list(sample_context_data.keys())[:4],  # Remove many items
            operation_data={},
            estimated_token_impact=-1000,  # Large impact
            confidence_score=0.8,
            reasoning="Large impact operation",
            requires_confirmation=True
        )
        
        result = validator.validate_operation(large_impact_operation, sample_context_data)
        
        # May fail due to exceeding impact limits
        if not result.is_valid:
            assert any("impact" in error.lower() for error in result.validation_errors)

    def test_validate_plan_success(self, validator, sample_context_data):
        """Test successful plan validation."""
        operations = [
            ManipulationOperation(
                operation_id="plan-op-001",
                operation_type="remove",
                target_keys=["message_2"],
                operation_data={},
                estimated_token_impact=-5,  # Smaller impact to avoid threshold violation
                confidence_score=0.9,
                reasoning="Remove duplicate",
                requires_confirmation=False
            )
        ]
        
        plan = ManipulationPlan(
            plan_id="test-plan-001",
            total_operations=1,
            operations=operations,
            estimated_total_reduction=5,
            estimated_execution_time=0.1,
            safety_level="balanced",
            requires_user_approval=False,
            created_timestamp=datetime.now().isoformat()
        )
        
        result = validator.validate_plan(plan, sample_context_data)
        
        assert isinstance(result, ValidationResult)
        assert result.is_valid is True
        assert len(result.validation_errors) == 0

    def test_validate_plan_empty(self, validator, sample_context_data):
        """Test validation of empty plan."""
        empty_plan = ManipulationPlan(
            plan_id="empty-plan",
            total_operations=0,
            operations=[],
            estimated_total_reduction=0,
            estimated_execution_time=0,
            safety_level="balanced",
            requires_user_approval=False,
            created_timestamp=datetime.now().isoformat()
        )
        
        result = validator.validate_plan(empty_plan, sample_context_data)
        
        assert result.is_valid is False
        assert any("no operations" in error.lower() for error in result.validation_errors)

    def test_validate_plan_operation_count_mismatch(self, validator, sample_context_data):
        """Test plan validation with operation count mismatch."""
        operations = [
            ManipulationOperation(
                operation_id="mismatch-op-001",
                operation_type="remove",
                target_keys=["message_1"],
                operation_data={},
                estimated_token_impact=-10,
                confidence_score=0.9,
                reasoning="Test operation",
                requires_confirmation=False
            )
        ]
        
        plan = ManipulationPlan(
            plan_id="mismatch-plan",
            total_operations=2,  # Claims 2 operations but only has 1
            operations=operations,
            estimated_total_reduction=10,
            estimated_execution_time=0.1,
            safety_level="balanced",
            requires_user_approval=False,
            created_timestamp=datetime.now().isoformat()
        )
        
        result = validator.validate_plan(plan, sample_context_data)
        
        # Should have warning about mismatch
        assert any("mismatch" in warning.lower() for warning in result.warnings)

    def test_validate_plan_excessive_reduction(self, validator, sample_context_data):
        """Test plan validation with excessive total reduction."""
        # Create operations that would remove almost everything
        large_operations = []
        for i, key in enumerate(sample_context_data.keys()):
            large_operations.append(
                ManipulationOperation(
                    operation_id=f"large-op-{i}",
                    operation_type="remove",
                    target_keys=[key],
                    operation_data={},
                    estimated_token_impact=-200,  # Large impact per operation
                    confidence_score=0.8,
                    reasoning=f"Remove {key}",
                    requires_confirmation=False
                )
            )
        
        plan = ManipulationPlan(
            plan_id="excessive-plan",
            total_operations=len(large_operations),
            operations=large_operations,
            estimated_total_reduction=len(large_operations) * 200,  # Very large reduction
            estimated_execution_time=1.0,
            safety_level="aggressive",
            requires_user_approval=True,
            created_timestamp=datetime.now().isoformat()
        )
        
        result = validator.validate_plan(plan, sample_context_data)
        
        # Should fail due to excessive reduction
        if not result.is_valid:
            assert any("reduction" in error.lower() for error in result.validation_errors)

    def test_verify_integrity_success(self, validator):
        """Test successful integrity verification."""
        original_context = {
            "key1": "value1",
            "key2": "value2", 
            "key3": "value3"
        }
        
        modified_context = {
            "key1": "value1",
            "key3": "value3"  # key2 removed
        }
        
        # Calculate actual token difference
        orig_tokens = sum(len(str(v)) // 4 for v in original_context.values())
        mod_tokens = sum(len(str(v)) // 4 for v in modified_context.values())
        actual_reduction = orig_tokens - mod_tokens
        
        executed_operations = [
            ManipulationOperation(
                operation_id="integrity-test-001",
                operation_type="remove",
                target_keys=["key2"],
                operation_data={},
                estimated_token_impact=-actual_reduction,  # Match actual token difference
                confidence_score=0.9,
                reasoning="Test removal",
                requires_confirmation=False
            )
        ]
        
        integrity = validator.verify_integrity(original_context, modified_context, executed_operations)
        
        assert isinstance(integrity, IntegrityCheck)
        assert integrity.structure_preserved is True
        # Token accuracy may fail due to estimation vs actual, but that's expected
        if integrity.errors_detected:
            # Check that errors are only about token discrepancy, not critical failures
            assert all("token count discrepancy" in err.lower() for err in integrity.errors_detected)

    def test_verify_integrity_all_content_lost(self, validator):
        """Test integrity verification when all content is lost."""
        original_context = {"key1": "value1", "key2": "value2"}
        modified_context = {}  # Everything removed
        executed_operations = []
        
        integrity = validator.verify_integrity(original_context, modified_context, executed_operations)
        
        assert integrity.integrity_maintained is False
        assert integrity.structure_preserved is False
        assert len(integrity.errors_detected) > 0
        assert any("all content was removed" in error.lower() for error in integrity.errors_detected)

    def test_verify_integrity_critical_content_lost(self, validator):
        """Test integrity verification when critical content is lost."""
        original_context = {
            "critical_config": "Important system setting",  # High importance
            "normal_key": "Regular content"
        }
        
        modified_context = {
            "normal_key": "Regular content"  # Critical content missing
        }
        
        executed_operations = []
        
        integrity = validator.verify_integrity(original_context, modified_context, executed_operations)
        
        # Should detect loss of critical content
        if not integrity.critical_content_preserved:
            assert any("critical content lost" in error.lower() for error in integrity.errors_detected)

    def test_generate_safety_report(self, validator):
        """Test safety report generation."""
        validation_result = ValidationResult(
            is_valid=True,
            confidence_score=0.85,
            validation_errors=[],
            warnings=["Minor warning"],
            safety_recommendations=["Review carefully"],
            risk_assessment="medium",
            validation_timestamp=datetime.now().isoformat()
        )
        
        integrity_check = IntegrityCheck(
            integrity_maintained=True,
            critical_content_preserved=True,
            token_count_accurate=True,
            structure_preserved=True,
            errors_detected=[]
        )
        
        report = validator.generate_safety_report(validation_result, integrity_check)
        
        assert "validation_summary" in report
        assert "validation_details" in report
        assert "integrity_check" in report
        assert "overall_assessment" in report
        
        assert report["validation_summary"]["is_safe"] is True
        assert report["overall_assessment"]["safe_to_proceed"] is True

    def test_convenience_functions(self, sample_remove_operation, sample_context_data):
        """Test convenience functions."""
        # Test validate_operation function
        result = validate_operation(sample_remove_operation, sample_context_data)
        assert isinstance(result, ValidationResult)
        
        # Test validate_plan function  
        plan = ManipulationPlan(
            plan_id="convenience-test",
            total_operations=1,
            operations=[sample_remove_operation],
            estimated_total_reduction=50,
            estimated_execution_time=0.1,
            safety_level="balanced",
            requires_user_approval=False,
            created_timestamp=datetime.now().isoformat()
        )
        
        result = validate_plan(plan, sample_context_data)
        assert isinstance(result, ValidationResult)
        
        # Test verify_manipulation_integrity function
        original = {"key": "value"}
        modified = {}
        operations = [sample_remove_operation]
        
        integrity = verify_manipulation_integrity(original, modified, operations)
        assert isinstance(integrity, IntegrityCheck)

class TestEnhancedManipulationValidator:
    """Test suite for enhanced PR18 ManipulationValidator features."""

    @pytest.fixture
    def enhanced_validator(self):
        """Enhanced ManipulationValidator with custom safety constraints."""
        safety_constraints = SafetyConstraints(
            max_single_operation_impact=0.4,
            max_total_reduction=0.7,
            min_confidence_threshold=0.8,
            require_backup_threshold=0.3,
            max_operations_per_batch=15
        )
        return ManipulationValidator(safety_constraints=safety_constraints)

    @pytest.fixture
    def high_risk_operation(self):
        """High-risk operation for testing enhanced features."""
        return ManipulationOperation(
            operation_id="enhanced-test-001",
            operation_type="remove",
            target_keys=["password", "api_key", "critical_data"],
            operation_data={"removal_type": "safe_delete"},
            estimated_token_impact=-200,
            confidence_score=0.5,  # Low confidence
            reasoning="Removing sensitive data",
            requires_confirmation=True
        )

    @pytest.fixture
    def sensitive_context_data(self):
        """Context data with sensitive content for testing."""
        return {
            "password": "secret_password_123",
            "api_key": "sk-abc123def456",
            "critical_data": "Critical business information",
            "normal_data": "Regular content",
            "config": {"setting1": "value1", "debug": True},
            "user_message": "Help me with authentication"
        }

    def test_enhanced_validator_initialization(self, enhanced_validator):
        """Test enhanced validator initialization with custom constraints."""
        assert enhanced_validator.safety_constraints.max_single_operation_impact == 0.4
        assert enhanced_validator.safety_constraints.max_total_reduction == 0.7
        assert enhanced_validator.safety_constraints.min_confidence_threshold == 0.8
        assert enhanced_validator.safety_constraints.require_backup_threshold == 0.3
        assert enhanced_validator.safety_constraints.max_operations_per_batch == 15
        assert len(enhanced_validator.operation_history) == 0

    def test_assess_operation_risk(self, enhanced_validator, high_risk_operation, sensitive_context_data):
        """Test comprehensive operation risk assessment."""
        risk_assessment = enhanced_validator._assess_operation_risk(high_risk_operation, sensitive_context_data)
        
        assert isinstance(risk_assessment, RiskAssessment)
        assert risk_assessment.risk_level in [RiskLevel.HIGH, RiskLevel.CRITICAL]
        assert risk_assessment.impact_severity > 0
        assert risk_assessment.reversibility < 1.0
        assert risk_assessment.data_sensitivity > 0.7  # Should detect sensitive content
        assert risk_assessment.recommended_action != SafetyAction.PROCEED
        assert len(risk_assessment.risk_factors) > 0
        assert len(risk_assessment.mitigation_strategies) > 0

    def test_validate_operation_enhanced(self, enhanced_validator, high_risk_operation, sensitive_context_data):
        """Test enhanced operation validation with risk assessment."""
        validation_result, risk_assessment = enhanced_validator.validate_operation_enhanced(
            high_risk_operation, sensitive_context_data, enable_risk_assessment=True
        )
        
        assert isinstance(validation_result, ValidationResult)
        assert isinstance(risk_assessment, RiskAssessment)
        
        # High-risk operation should have validation concerns
        assert not validation_result.is_valid or len(validation_result.warnings) > 0
        assert risk_assessment.risk_level != RiskLevel.LOW
        assert "backup" in " ".join(validation_result.safety_recommendations).lower()

    def test_record_operation_history(self, enhanced_validator, high_risk_operation, sensitive_context_data):
        """Test operation history recording."""
        initial_count = len(enhanced_validator.operation_history)
        
        enhanced_validator.record_operation_history(
            high_risk_operation, 
            sensitive_context_data,
            backup_id="test-backup-001"
        )
        
        assert len(enhanced_validator.operation_history) == initial_count + 1
        
        history_entry = enhanced_validator.operation_history[-1]
        assert history_entry.operation_id == high_risk_operation.operation_id
        assert history_entry.operation_type == high_risk_operation.operation_type
        assert set(history_entry.affected_keys) == set(high_risk_operation.target_keys)
        assert history_entry.backup_id == "test-backup-001"
        assert len(history_entry.original_values) == len(high_risk_operation.target_keys)

    def test_get_rollback_data(self, enhanced_validator, high_risk_operation, sensitive_context_data):
        """Test rollback data retrieval."""
        # Record operation first
        enhanced_validator.record_operation_history(
            high_risk_operation, 
            sensitive_context_data,
            backup_id="rollback-test-001"
        )
        
        # Retrieve rollback data
        rollback_data = enhanced_validator.get_rollback_data(high_risk_operation.operation_id)
        
        assert rollback_data is not None
        assert rollback_data.operation_id == high_risk_operation.operation_id
        assert rollback_data.backup_id == "rollback-test-001"
        assert "password" in rollback_data.original_values
        assert rollback_data.original_values["password"] == "secret_password_123"

    def test_generate_enhanced_safety_report(self, enhanced_validator, high_risk_operation, sensitive_context_data):
        """Test enhanced safety report generation."""
        # Validate operation with risk assessment
        validation_result, risk_assessment = enhanced_validator.validate_operation_enhanced(
            high_risk_operation, sensitive_context_data
        )
        
        # Record operation history
        enhanced_validator.record_operation_history(high_risk_operation, sensitive_context_data)
        
        # Generate enhanced safety report
        report = enhanced_validator.generate_enhanced_safety_report(
            validation_result=validation_result,
            risk_assessment=risk_assessment,
            operation_history=enhanced_validator.operation_history,
            include_mitigation_plan=True
        )
        
        # Verify report structure
        assert "report_metadata" in report
        assert "validation_summary" in report
        assert "detailed_analysis" in report
        assert "risk_assessment" in report
        assert "operation_history" in report
        assert "overall_assessment" in report
        
        # Verify metadata
        assert report["report_metadata"]["report_version"] == "2.0"
        assert report["report_metadata"]["validator_version"] == "PR18-Enhanced"
        
        # Verify enhanced assessment
        assert "safety_score" in report["overall_assessment"]
        assert "safety_level" in report["overall_assessment"]
        assert "safety_factors" in report["overall_assessment"]
        assert "mitigation_plan" in report["overall_assessment"]
        
        # High-risk operation should have low safety score
        assert report["overall_assessment"]["safety_score"] < 0.7
        assert not report["overall_assessment"]["safe_to_proceed"]
        assert report["overall_assessment"]["requires_backup"]

    def test_generate_plan_safety_report(self, enhanced_validator):
        """Test comprehensive plan safety report generation."""
        # Create test operations with varying risk levels
        low_risk_op = ManipulationOperation(
            operation_id="plan-test-001",
            operation_type="reorder",
            target_keys=["normal_data"],
            operation_data={},
            estimated_token_impact=0,
            confidence_score=0.95,
            reasoning="Safe reordering",
            requires_confirmation=False
        )
        
        high_risk_op = ManipulationOperation(
            operation_id="plan-test-002",
            operation_type="remove",
            target_keys=["api_key"],
            operation_data={},
            estimated_token_impact=-50,
            confidence_score=0.6,
            reasoning="Removing API key",
            requires_confirmation=True
        )
        
        context_data = {
            "normal_data": "Regular content",
            "api_key": "secret_key_123",
            "other_data": "More content"
        }
        
        # Create plan
        plan = ManipulationPlan(
            plan_id="safety-report-test",
            total_operations=2,
            operations=[low_risk_op, high_risk_op],
            estimated_total_reduction=50,
            estimated_execution_time=0.5,
            safety_level="balanced",
            requires_user_approval=True,
            created_timestamp=datetime.now().isoformat()
        )
        
        # Validate plan and operations
        plan_validation = enhanced_validator.validate_plan(plan, context_data)
        operation_validations = []
        
        for operation in plan.operations:
            val_result, risk_assessment = enhanced_validator.validate_operation_enhanced(
                operation, context_data
            )
            operation_validations.append((operation, val_result, risk_assessment))
        
        # Generate plan safety report
        report = enhanced_validator.generate_plan_safety_report(
            plan_validation, operation_validations
        )
        
        # Verify report structure
        assert "report_metadata" in report
        assert "plan_summary" in report
        assert "aggregated_risk_assessment" in report
        assert "operation_details" in report
        assert "plan_safety_assessment" in report
        
        # Verify plan summary
        assert report["plan_summary"]["total_operations"] == 2
        assert report["plan_summary"]["high_risk_operations"] >= 1
        
        # Verify aggregated risk assessment
        assert "risk_distribution" in report["aggregated_risk_assessment"]
        assert report["aggregated_risk_assessment"]["overall_risk_level"] != "low"
        
        # Verify operation details
        assert len(report["operation_details"]) == 2
        for op_detail in report["operation_details"]:
            assert "operation_id" in op_detail
            assert "validation_status" in op_detail
            assert "risk_level" in op_detail
        
        # Verify plan safety assessment
        assert "plan_safety_score" in report["plan_safety_assessment"]
        assert "recommended_approach" in report["plan_safety_assessment"]

    def test_safety_constraints_validation(self):
        """Test safety constraints validation."""
        # Test default constraints
        default_constraints = SafetyConstraints()
        assert default_constraints.max_single_operation_impact == 0.3
        assert default_constraints.max_total_reduction == 0.8
        assert default_constraints.min_confidence_threshold == 0.7
        assert default_constraints.enable_dry_run_mode is True
        
        # Test custom constraints
        custom_constraints = SafetyConstraints(
            max_single_operation_impact=0.5,
            max_total_reduction=0.9,
            min_confidence_threshold=0.6,
            enable_dry_run_mode=False
        )
        assert custom_constraints.max_single_operation_impact == 0.5
        assert custom_constraints.max_total_reduction == 0.9
        assert custom_constraints.min_confidence_threshold == 0.6
        assert custom_constraints.enable_dry_run_mode is False

    def test_risk_level_enum(self):
        """Test RiskLevel enum values."""
        assert RiskLevel.LOW.value == "low"
        assert RiskLevel.MEDIUM.value == "medium" 
        assert RiskLevel.HIGH.value == "high"
        assert RiskLevel.CRITICAL.value == "critical"

    def test_safety_action_enum(self):
        """Test SafetyAction enum values."""
        assert SafetyAction.PROCEED.value == "proceed"
        assert SafetyAction.CONFIRM.value == "confirm"
        assert SafetyAction.BACKUP_FIRST.value == "backup_first"
        assert SafetyAction.REJECT.value == "reject"
        assert SafetyAction.MANUAL_REVIEW.value == "manual_review"

    def test_operation_history_data_structure(self):
        """Test OperationHistory data structure."""
        history = OperationHistory(
            operation_id="test-history-001",
            timestamp=datetime.now().isoformat(),
            operation_type="remove",
            affected_keys=["key1", "key2"],
            original_values={"key1": "value1", "key2": "value2"},
            backup_id="backup-001"
        )
        
        assert history.operation_id == "test-history-001"
        assert history.operation_type == "remove"
        assert len(history.affected_keys) == 2
        assert "key1" in history.original_values
        assert history.backup_id == "backup-001"

    def test_backward_compatibility(self, enhanced_validator):
        """Test that enhanced validator maintains backward compatibility."""
        # Test legacy generate_safety_report method
        validation_result = ValidationResult(
            is_valid=True,
            confidence_score=0.8,
            validation_errors=[],
            warnings=[],
            safety_recommendations=[],
            risk_assessment="low",
            validation_timestamp=datetime.now().isoformat()
        )
        
        # Legacy method should work
        legacy_report = enhanced_validator.generate_safety_report(validation_result)
        assert "report_metadata" in legacy_report  # Should use enhanced version
        assert "validation_summary" in legacy_report
        assert "overall_assessment" in legacy_report


if __name__ == "__main__":
    pytest.main([__file__, "-v"])