'''
* Copyright 2015-2025 European Atomic Energy Community (EURATOM)
*
* Licensed under the EUPL, Version 1.1 or - as soon they
  will be approved by the European Commission - subsequent
  versions of the EUPL (the "Licence");
* You may not use this work except in compliance with the
  Licence.
* You may obtain a copy of the Licence at:
*
* https://joinup.ec.europa.eu/software/page/eupl
*
* Unless required by applicable law or agreed to in
  writing, software distributed under the Licence is
  distributed on an "AS IS" basis,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
  express or implied.
* See the Licence for the specific language governing
  permissions and limitations under the Licence.
'''


import vtk
import numpy as np
import json
import os
import atexit
import warnings
from .config import CalcamConfig
from .io import ZipSaveFile
from .render import get_rz_contour_actor



# A little function to use for status printing if no
# user callback is specified.
def print_status(status):
    if status is not None:
        print(status)



class CADModel():
    '''
    Class for representing a CAD model in Calcam.

    Parameters:
        model_name (str)            : Either the machine name of a CAD model in Calcam's model search path, \
                                      or filename of a .ccm file to load the model from.
        model_variant (str)         : Name of the model variant to load. If not specified, the default variant \
                                      specified in the CAD model is loaded.
        status_callback (callable)  : Function to call with status messages. If given, this function will be called \
                                      with a string describing what this class is currently doing (e.g. loading files etc) \
                                      or `None` which signifies clearing the previous status. This is mainly used for integration \
                                      with the Calcam GUI.
        verbose (bool)              : Whether to issue status updates to status_callback (default) or be completely silent if set to False.
    '''
    def __init__(self,model_name=None,model_variant=None,status_callback=print_status,verbose=True):

        self.verbose = verbose
        self.set_status_callback(status_callback)

        if model_name is not None:
            # -------------------------------Loading model definition-------------------------------------
            
            if not os.path.isfile(model_name):
                # Check whether we know what model definition file to use
                model_defs = CalcamConfig().get_cadmodels()
                if model_name not in model_defs.keys():
                    raise ValueError('Unknown machine model "{:s}". Available models are: {:s}.'.format(model_name,', '.join(model_defs.keys())))
                else:
                    definition_filename = model_defs[model_name][0]

                # If not specified, choose whatever model variant is specified in the metadata
                if model_variant is None:
                    self.model_variant = model_defs[model_name][2]
                else:
                    self.model_variant = model_variant

            
            else:
                definition_filename = model_name
                self.model_variant = model_variant
            

            self.status_callback('Extracting CAD model...')


            # Open the definition file (ZIP file)
            try:
                self.def_file = ZipSaveFile(definition_filename,'rw')
            except:
                self.def_file = ZipSaveFile(definition_filename,'r')

            self.status_callback(None)


            # Load the model definition and grab some properties from it
            with self.def_file.open_file( 'model.json','r' ) as f:
                model_def = json.load(f)

            self.model_def = model_def
            self.machine_name = model_def['machine_name']
            self.views = model_def['views']
            self.initial_view = model_def['initial_view']
            self.linewidth = 1
            self.mesh_path_roots = model_def['mesh_path_roots']
            self.slice_params = (None,0,None)

            if self.model_variant is None:
                self.model_variant = model_def['default_variant']

            self.variants = [str(x) for x in model_def['features'].keys()]

            # Validate the model variant input
            if self.model_variant not in self.variants:
                raise ValueError('Unknown model variant for {:s}: {:s}.'.format(self.machine_name,self.model_variant))

            # Check if the mesh files are from the CAD definition file itself, in which case
            # we need to point the mesh loader in the direction of out temporary extracted path
            if model_def['mesh_path_roots'][self.model_variant].startswith('.large'):
                self.mesh_path_root = os.path.join(self.def_file.get_temp_path(),model_def['mesh_path_roots'][self.model_variant])
            else:
                self.mesh_path_root = model_def['mesh_path_roots'][self.model_variant]


            # Load the wall contour, if present
            if 'wall_contour.txt' in self.def_file.list_contents():
                with self.def_file.open_file('wall_contour.txt','r') as cf:
                    self.wall_contour = np.loadtxt(cf)
            else:
                self.wall_contour = None

            
            # See if we have a user-written coordinate formatter, and if
            # we do, load it over the standard format_coord method
            self.usermodule = None
            usermodule = self.def_file.get_usercode()
            if usermodule is not None:
                if callable(usermodule.format_coord):
                    # Check the user function returns a string as expected, and use
                    # it only if it does.
                    try:
                        test_out = usermodule.format_coord( (0.1,0.1,0.1) )
                    except Exception as e:
                        self.def_file.close()
                        raise

                    if type(test_out) in [str,bytes]:
                        self.usermodule = usermodule
                    else:
                        self.def_file.close()
                        raise Exception('CAD model user function format_coord() did not return a string as required.')


            # Create the features!
            self.features = {}
            self.groups = {}

            for feature_name,feature_def in model_def['features'][self.model_variant].items():

                # Get the feature's group, if any
                if len(feature_name.split('/')) > 1:
                    group = feature_name.split('/')[0]
                    if group not in self.groups.keys():
                        self.groups[group] = [feature_name]
                    else:
                        self.groups[group].append(feature_name)

                # Actually make the feature object
                self.features[feature_name] = ModelFeature(self,feature_def)
            
            # ----------------------------------------------------------------------------------------------

        else:
            self.variants = []
            self.model_def = {}
            self.groups = {}
            self.features = {}
            self.usermodule = None
            self.wall_contour = None
            self.def_file = None
            self.mesh_path_root = ''
            self.machine_name = ''
            self.model_variant = ''
            self.views = {}
            self.initial_view = None
            self.mesh_path_roots = {}

        self.renderers = []
        self.slice_planes = None
        self.flat_shading = False
        self.edges = False
        self.cell_locator = None
        self.discard_changes = False

        atexit.register(self.unload)


    def set_status_callback(self,status_callback):
        '''
        Set the status callback function.

        The given function will be called with a string when the CAD model
        object does something, and is called with None when the operatio is
        finished. If set to None, the object will provide no output when it is
        busy doing things.

        Parameters:

            status_callback (fun): Status callback function.
        '''
        self._status_callback = status_callback


    def get_status_callback(self):
        '''
        Get the current callback function

        Returns:

            func or NoneType : Current status callback function, if present.
        '''
        return self._status_callback


    def status_callback(self,msg):

        if self._status_callback is not None and self.verbose:
            self._status_callback(msg)


    def add_to_renderer(self,renderer):
        '''
        Add the CAD model to a VTK renderer.

        Parameters:

            renderer (vtk.vtkRenderer) : Renderer to add the model to.
        '''
        if renderer in self.renderers:
            return

        else:
            for feature in self.features.values():
                actors = feature.get_vtk_actors()
                for actor in actors:
                    renderer.AddActor(actor)

            self.renderers.append(renderer)


    def set_slicing(self,slice_phi=None,slice_r=0,slice_z=None):
        """
        Apply slicing / cross-sectioning to the model.

        This will affect the geometry, octree etc returned from other methods, and will update
        any renderers that the model is currently in to show the sliced model.

        Calling without any arguments (i.e. with default slice_phi=None and slice_z = None) will reset
        the model to not be sliced.

        Parameters:

            slice_phi        : Either None, a single float or sequence of 2 floats: toroidal angles in degrees.  \
                               If None, no toroidal slicing is done. \
                               If a single number, slices the model with a plane whose normal is at toroidal angle phi, possibly \
                               with a radial offset from machine centre (see slice_r).
                               If a sequence of 2 numbers, the part of the machine between the two given toroidal angles is kept.

            slice_r (float)  : If slicing at a single toroidal angle, this will offset the slicing plane so it goes through this major radius \
                               at its tangency point.

            slice_z          : Either None or a sequence of 2 floats. \
                               If None, no slicing by horizontal planes is done.
                               If a 2 element sequence, parts of the model between these 2 Z values are kept. \
                               -np.inf or np.inf can be used for the Z values to include all the way to the top or bottom of the model.
        """

        slice_params = (slice_phi,slice_r,slice_z)

        outside = True

        if slice_r < 0:
            raise ValueError('slice_r is a major radius so cannot be < 0 (passed value of slice_r: {:.3f}m)'.format(slice_r))

        if slice_phi is not None:
            slice_phi = np.array(slice_phi)
            if slice_phi.size == 2:

                if slice_phi[0] > slice_phi[1]:
                    slice_phi[1] = slice_phi[1] + 360

                if slice_phi[1] - slice_phi[0] > 180:
                    outside = False

                elif slice_phi[1] - slice_phi[0] == 180:
                    slice_phi = np.mean(slice_phi)

                if slice_r > 0 and slice_phi.size == 2:
                    # If we have an r > 0 slice, the above check should have made slice_phi a single value here
                    raise ValueError('Cannot use slice_r > 0 when slice_phi angles differ by other than 180 degrees!')

            elif slice_phi.size != 1:
                raise ValueError('Slice angle must be a single value or sequence of 2 values specifying the toroidal angle range to show')

            slice_phi = slice_phi / 180 * np.pi

        if slice_z is not None:
            if len(slice_z) == 2:
                slice_z = np.sort(slice_z)
            else:
                raise ValueError('Slice height must be a sequence of 2 values specifying the Z range to show')


        if slice_z is None and slice_phi is None:
            if self.slice_planes is None:
                # No need to update renderers etc; nothing has changed so just return
                return
            else:
                # Set planes to None and skips to the end to update renderers etc
                self.slice_planes = None

        else:
            # VTK objects to define the slicing
            planes = vtk.vtkPlanes()
            points = vtk.vtkPoints()
            norms = vtk.vtkFloatArray()
            norms.SetNumberOfComponents(3)

            npt = 0

            if slice_phi is not None:

                if slice_phi.size == 1:
                    # Single slice at possibly offset R
                    norm_x = np.cos(slice_phi)
                    norm_y = np.sin(slice_phi)
                    points.InsertPoint(0, slice_r*norm_x, slice_r*norm_y, 0.)
                    norms.InsertTuple3(0, norm_x, norm_y, 0.)
                    npt = 1
                else:
                    # Cake slice from origin
                    if outside:
                        normal_sign = 1
                    else:
                        normal_sign = -1
                    points.InsertPoint(0, 0.,0.,0.)
                    points.InsertPoint(1, 0., 0., 0.)
                    norms.InsertTuple3(0,normal_sign*np.sin(slice_phi[0]),-normal_sign*np.cos(slice_phi[0]),0.)
                    norms.InsertTuple3(1, -normal_sign*np.sin(slice_phi[1]), normal_sign*np.cos(slice_phi[1]), 0.)
                    npt = 2

            if slice_z is not None:

                if slice_z[0] > -np.inf:
                    points.InsertPoint(npt, 0.,0.,slice_z[0])
                    norms.InsertTuple3(npt, 0.,0., -1. if outside else 1)
                    npt += 1
                if slice_z[1] < np.inf:
                    points.InsertPoint(npt, 0.,0.,slice_z[1])
                    norms.InsertTuple3(npt, 0.,0., 1. if outside else -1)

            planes.SetPoints(points)
            planes.SetNormals(norms)
            self.slice_planes = (planes,outside)

        # If we are attached to any renderers, this will update the actual display
        if len(self.renderers) > 0:
            enable_features = self.get_enabled_features()
            for feature in self.features:
                self.features[feature].set_enabled(False)
                self.features[feature].solid_actor = None
                self.features[feature].edge_actor = None

            for feature in enable_features:
                self.features[feature].set_enabled(True)

        self.slice_params = slice_params

        if self.cell_locator is not None and self.cell_locator[1]:
            self.cell_locator = None



    def remove_from_renderer(self,renderer):
        '''
        Remove the CAD model from the given VTK renderer.

        Parameters:

            renderer (vtk.vtkRenderer) : Renderer to remove the model from.
        '''
        if renderer not in self.renderers:
            return

        else:
            for feature in self.features.values():
                actors = feature.get_vtk_actors()
                for actor in actors:
                    renderer.RemoveActor(actor)

            self.renderers.remove(renderer)



    def set_features_enabled(self,enable,features=None):
        '''
        Enable or disable parts of the CAD model.

        Parameters:

            enable (bool)                   : Whether to set the relevant features \
                                              as enabeled (True) or disabled (False).
            features (str or list of str)   : Name(s) of the feature(s) and/or group(s) \
                                              of features of  to enable or disable. \
                                              If not specified, applies to all features \
                                              in the model.
        '''
        if features is None:
            features = self.features.keys()
        elif type(features) is not list:
            features = [features]


        for requested in features:

            if requested in self.groups.keys():
                for fname in self.groups[requested]:
                    self.features[fname].set_enabled(enable)
            elif requested in self.features.keys():
                self.features[requested].set_enabled(enable)
            else:
                raise ValueError('Unknown feature "{:s}"!'.format(requested))

        self.cell_locator = None



    def enable_only(self,features):
        '''
        Disable all model parts except those specified.

        Parameters:

            features (str or list of str) : Name(s) of the feature(s) and/or \
                                            group(s) to have enabled.
        '''
        if type(features) is not list:
            features = [features]

        self.set_features_enabled(False)
        for requested in features:
            if requested in self.groups.keys():
                for fname in self.groups[requested]:
                    self.features[fname].set_enabled(True)
            elif requested in self.features.keys():
                self.features[requested].set_enabled(True)
            else:
                raise ValueError('Unknown feature "{:s}"!'.format(requested))

        self.cell_locator = None



    def get_enabled_features(self):
        '''
        Get a list of the currently enabled features.

        Returns:

            list of str : Names of the currently enabled features.
        '''
        flist = []
        for fname,fobj in self.features.items():
            if fobj.enabled:
                flist.append(fname)

        return sorted(flist)



    def get_group_enable_state(self,group=None):
        '''
        Check the enable status of a named group of features.

        Parameters:

            group (str) : Name of the group to check. If not given, \
                          the entire model is taken to be the group.

        Returns:

            int         : 0 if no features in the group are anebled; \
                          1 if some features in the group are enabled; \
                          2 if all features in the group are enabled.
        '''
        if group is None:
            flist = self.features.keys()
        else:
            flist = self.groups[group]

        enable_state = 0
        for fname in flist:
            enable_state = enable_state + self.features[fname].enabled

        if enable_state == len(flist):
            enable_state = 2
        elif enable_state > 0:
            enable_state = 1

        return enable_state




    def format_coord(self,coords):
        '''
        Return a pretty string giving information about a specified 3D position.

        Parameters:

            coords (array-like) : 3 element array-like specifying a point in 3D \
                                  as X,Y,Z in metres.

        Returns:

            str                 : String containing information about the given point.
        '''

        if self.usermodule is not None:
            return self.usermodule.format_coord(coords)

        else:

            phi = np.arctan2(coords[1],coords[0])
            if phi < 0.:
                phi = phi + 2*3.14159
            phi = phi / 3.14159 * 180
            
            formatted_coord = 'X,Y,Z: ( {:.3f} m , {:.3f} m , {:.3f} m )'.format(coords[0],coords[1],coords[2])
            formatted_coord = formatted_coord + u'\nR,Z,\u03d5: ( {:.3f} m , {:.3f}m , {:.1f}\xb0 )'.format(np.sqrt(coords[0]**2 + coords[1]**2),coords[2],phi)

            return  formatted_coord
    



    def set_flat_shading(self,flat_shading):
        '''
        Set flat shading (no lighting applied to model rendering)
        enabled or disabled.

        Parameters:

            flat_shading (bool) : Whether to enable or disable flat shading.
        '''
        self.flat_shading = flat_shading
        if flat_shading != self.flat_shading:

            # Just running through each feature like this will force it to
            # update the colour & lighting settings
            for feature in self.features.values():
                feature.get_vtk_actors()




    def reset_colour(self,features=None):
        '''
        Reset the colour of part or all of the CAD model to the default(s).

        Parameters:

            features (list of str) : List of features for which to reset the colours. \
                                     If not given, all features will have their colours reset.
        '''
        if features is None:
            features = self.get_feature_list()

        for feature in features:
            if feature in self.groups.keys():
                for fname in self.groups[feature]:
                    self.features[fname].set_colour(self.features[fname].default_colour)
            elif feature in self.features.keys():
                self.features[feature].set_colour(self.features[feature].default_colour)
            else:
                raise ValueError('Unknown feature "{:s}"!'.format(feature))



    def set_colour(self,colour,features=None):
        '''
        Set the colour of part or all of the CAD model.

        Parameters:

            colour (tuple)          : 3-element tuple specifying a colour in (R,G,B) \
                                      where the values are in the range 0 to 1.
            features (list of str)  : List of names of the features to set this colour. \
                                      If not specified, applies to the whole model.
        '''
        if features is None:
            features = self.get_feature_list()

        try:
            0. + colour[0]
            colour = [colour] * len(features)
        except:
            if len(colour) != len(features):
                raise ValueError('The same number of colours and features must be provided!')

        for i,requested in enumerate(features):

            if requested in self.groups.keys():
                for fname in self.groups[requested]:
                    self.features[fname].set_colour(colour[i])
            elif requested in self.features.keys():
                self.features[requested].set_colour(colour[i])
            else:
                raise ValueError('Unknown feature "{:s}"!'.format(requested))



    def get_colour(self,features = None):
        '''
        Get the current colour of part or all of the CAD model.

        Parameters:

            features (list of str)  : List of names of the features to get the colour for. \
                                      If not specified, all feature colours are returned.

        Returns:

            List                    : List of 3 element tuples specifying the colours (R,G,B) \
                                      of the given features, where R, G and B range from 0 to 1.
        '''
        clist = []
        if features is None:
            features = self.get_feature_list()

        for feature in features:
            if feature in self.groups.keys():
                for fname in self.groups[feature]:
                    clist.append( self.features[fname].colour )
            elif feature in self.features.keys():
                clist.append( self.features[feature].colour )
            else:
                raise ValueError('Unknown feature "{:s}"!'.format(feature))
            
        return clist


    def get_linewidth(self,features=None):
        '''
        Get the line width used for rendering the model as wireframe.

        Parameters:

            features (list of str)  : List of names of the features to get the line width for. \
                                      If not specified, all feature line widths are returned.

        Returns:

            list of float           : List of the line widths.
        '''
        wlist = []
        if features is None:
            features = self.get_feature_list()

        for feature in features:
            if feature in self.groups.keys():
                for fname in self.groups[feature]:
                    wlist.append( self.features[fname].linewidth )
            elif feature in self.features.keys():
                wlist.append( self.features[feature].linewidth )
            else:
                raise ValueError('Unknown feature "{:s}"!'.format(feature))
            
        return wlist       



    def set_linewidth(self,linewidth,features=None):
        '''
        Set the line width used when rendering the CAD model as wireframe.

        Parameters:

            linewidth (float)       : Line width.
            features (list of str)  : List of names of the features to set the line \
                                      width for. If not specified, applies to the whole model.
        '''
        if features is None:
            features = self.get_feature_list()

        try:
            0 + linewidth
            linewidth = [linewidth] * len(features)
        except:
            if len(linewidth) != len(features):
                raise ValueError('The same number of line widths and features must be provided!')

        for i,requested in enumerate(features):

            if requested in self.groups.keys():
                for fname in self.groups[requested]:
                    self.features[fname].set_linewidth(linewidth[i])
            elif requested in self.features.keys():
                self.features[requested].set_linewidth(linewidth[i])
            else:
                raise ValueError('Unknown feature "{:s}"!'.format(requested))


    def get_vtk_cell_locator(self,apply_slicing=False):
        '''
        Create a vtkCellLocator object used for testing
        the intersection of the CAD model with line segments.
        '''

        # Don't return anything if we have no enabled geometry
        if len(self.get_enabled_features()) == 0:
            return

        if self.cell_locator is None or self.cell_locator[1] != apply_slicing:

            appender = vtk.vtkAppendPolyData()

            if not apply_slicing:
                slice_planes = self.slice_planes
                self.slice_planes = None

            for fname in self.get_enabled_features():
                appender.AddInputData(self.features[fname].get_polydata())

            appender.Update()
            if vtk.vtkVersion().GetVTKMajorVersion() > 8:
                self.cell_locator = [vtk.vtkStaticCellLocator(),apply_slicing]
            else:
                self.cell_locator = [vtk.vtkCellLocator(),apply_slicing]

            self.cell_locator[0].SetTolerance(1e-6)
            self.cell_locator[0].SetDataSet(appender.GetOutput())
            self.cell_locator[0].BuildLocator()

            if not apply_slicing:
                self.slice_planes = slice_planes

            # Initialise some faffy input variables for c-like interface of cellLocator's IntersectWithLine()
            # Keep these as properties so we only have to bother once
            self.raycast_args = (vtk.mutable(0), np.zeros(3), np.zeros(3), vtk.mutable(0), vtk.mutable(0), vtk.vtkGenericCell())

        return self.cell_locator[0]


    def intersect_with_line(self,line_start,line_end,surface_normal=False,apply_slicing=False):
        """
        Find the first intersection of a straight line segment with the CAD geometry, if one
        occurs ("first" meaning first when moving from the start to the end of the line segment).
        Optionally also calculates the surface normal vector of the CAD model at the intersection point.

        Parameters:

            line_start (sequence) : 3-element sequence x,y,z of the line segment start coordinates (in metres)
            line_end   (sequence) : 3-element sequence x,y,z of the line segment end coordinates (in metres)
            surface_normal (bool) : Whether or not to calculate the surface normal vector of the CAD model at the intersection.

        Returns:

            Multiple return values:
                - bool              : Whether or not the line segment intersects the CAD geometry
                - np.array          : 3-element NumPy array with x,y,z position of the intersection. If there is \
                                      no intersection, `line_end` is returned.
                - np.array or None  : Only returned if surface_normal = True; the surface normal at the intersection. \
                                      If there is no intersection, returns `None`.

        """
        line_start = np.array(line_start)
        line_end = np.array(line_end)
        if len(self.get_enabled_features()) == 0:
            # Don't return anything if we have no enabled geometry
            intersects = False
            position = line_end
            n = None

        else:
            # Make sure we have an octree
            cell_locator = self.get_vtk_cell_locator(apply_slicing=apply_slicing)

            # Actually do the intersection using VTK
            result = cell_locator.IntersectWithLine(line_start, line_end, 1.e-6, *self.raycast_args)

            if abs(result) > 0:
                intersects = True
                position = self.raycast_args[1].copy()
                if surface_normal:
                    v0 = np.array(self.raycast_args[5].GetPoints().GetPoint(2)) - np.array(self.raycast_args[5].GetPoints().GetPoint(0))
                    v1 = np.array(self.raycast_args[5].GetPoints().GetPoint(2)) - np.array(self.raycast_args[5].GetPoints().GetPoint(1))
                    n = np.cross(v0,v1)
                    n = n / np.sqrt(np.sum(n**2))
                    if np.dot(line_end - line_start,n) > 0:
                        n = -n
            else:
                intersects = False
                position = line_end
                n = None

        if surface_normal:
            return intersects,position,n
        else:
            return intersects, position



    def set_wireframe(self,wireframe):
        '''
        Enable or disable rendering the model as wireframe.

        Parameters:

            wireframe (bool) : Whether to render as wireframe.
        '''
        enable_features = self.get_enabled_features()

        for feature in enable_features:
            self.features[feature].set_enabled(False)

        self.edges = wireframe

        for feature in enable_features:
            self.features[feature].set_enabled(True)



    def get_feature_list(self):
        '''
        Get a list of the names of all features
        which constitute the model.

        Returns:

            list of str: List of feature names.
        '''
        return sorted(self.features.keys())



    def get_extent(self):
        '''
        Get the extent of the model in 3D space.

        Returns:

            np.array : 6 element array specifying the extent of the model in metres: \
                       [x_min, x_max, y_min, y_max, z_min, z_max]
        '''
        model_extent = np.zeros(6)

        for fname in self.get_enabled_features():
            feature_extent = self.features[fname].get_polydata().GetBounds()
            model_extent[::2] = np.minimum(model_extent[::2],feature_extent[::2])
            model_extent[1::2] = np.maximum(model_extent[1::2],feature_extent[1::2])

        return model_extent



    def get_view_names(self):
        '''
        Get a list of the views configured in the model.

        Returns:
            list of str: Names of the views configured in the model.
        '''
        return sorted(self.views.keys())


    def get_wall_contour_actor(self,actor_type,phi=None,toroidal_res=128):
        """
        Get vtkActor for the wall contour
        """
        actor = get_rz_contour_actor(self.wall_contour,actor_type,phi=phi,toroidal_res=toroidal_res,closed_contour=True)

        if self.slice_planes is not None:
            slice_filter = vtk.vtkExtractPolyDataGeometry()
            slice_filter.SetImplicitFunction(self.slice_planes[0])
            slice_filter.SetExtractInside(self.slice_planes[1])
            slice_filter.SetInputData(actor.GetMapper().GetInput())
            slice_filter.Update()

            mapper = vtk.vtkPolyDataMapper()
            mapper.SetInputData(slice_filter.GetOutput())
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)

        return actor


    def get_view(self,view_name):
        '''
        Get a dictionary describing one of the 
        views configured in the model.

        Parameters:

            view_name (str) : Name of the view to retrieve.

        Returns:

            dict            : Dictionary describing the view geometry.
        '''
        return self.views[view_name.replace('*','')]



    def __str__(self):
        '''
        Make print() do something useful for CAD model objects.

        Returns:

            str : String specifying the model machine name and model file path.
        '''
        return 'Calcam CAD model: "{:s}" / "{:s}" from {:s}'.format(self.machine_name,self.model_variant,self.definition_filename)



    def add_view(self,viewname,campos,camtar,fov,roll,projection,slicing=None,xsection=None):
        '''
        Add a specified camera view to the model's pre-defined views.

        Parameters:
    
            viewname (str)          : A name for the added view.
            campos (array-like)     : 3-element array-like specifying the camera position (X,Y,Z) in metres.
            camtar (array-like)     : 3-element array-like specifying a 3D point at which the camera is pointing.
            fov (float)             : Vertical field-of-view of the camera.
            roll (float)            : Camera roll in degrees. This is the angle between the model's +Z direction \
                                      and the camera view up direction. Positie values indicate an anti-clockwise \
                                      roll of the camera.
            projection (str)        : Either ``perspective`` or ``orthographic``, what camera projection to use.
            slicing (3 el. sequence): Slicing parameters slice_phi,slice_r,slice_z ("new" style cross-sectioning)
            xsection (array-like)   : A 3D point through which the model will be cross-sectioned ("old style" cross-section; kept \
                                      only to allow backwards-compatibility of CAD def files - should not be used).
        '''
        self.views[viewname] = {'cam_pos':campos,'target':camtar,'y_fov':fov,'xsection':xsection,'roll':roll,'projection':projection,'slicing':slicing}
        self.model_def['views'] = self.views
        if not self.discard_changes:
            self.update_definition_file()


    def set_default_colour(self,colour,features=None):
        '''
        Set the default colour of some or all of the model.

        Parameters:

            colour (tuple)          : 3-element tuple specifying an (R,G,B) colour where \
                                      R, G and B range from 0 to 1.
            features (list of str)  : List of feature names to which to apply the colour. \
                                      If not given, applies to the entire model.
        '''
        if features is None:
            features = self.get_feature_list()

        try:
            0. + colour[0]
            colour = [colour] * len(features)
        except:
            if len(colour) != len(features):
                raise ValueError('The same number of colours and features must be provided!')

        for i,requested in enumerate(features):

            if requested in self.groups.keys():
                for fname in self.groups[requested]:
                    self.features[fname].default_colour = colour[i]
                    self.model_def['features'][self.model_variant][fname]['colour'] = colour[i]
            elif requested in self.features.keys():
                self.features[requested].default_colour = colour[i]
                self.model_def['features'][self.model_variant][requested]['colour'] = colour[i]
            else:
                raise ValueError('Unknown feature "{:s}"!'.format(requested))

        self.update_definition_file()


    def unload(self):
        '''
        Unloads the CAD model object.
        '''

        self.status_callback('Closing model definition {:s}/{:s}...'.format(self.machine_name,self.model_variant))

        if self.def_file is not None:
            temp_dir = self.def_file.get_temp_path()
            try:
                self.def_file.close(discard_changes=self.discard_changes)
            except:
                warnings.warn('CAD model definition {:s}/{:s} could not be closed cleanly. There may be temporary files left in {:s}.'.format(self.machine_name,self.model_variant,temp_dir))

        self.status_callback(None)


    def update_definition_file(self):
        '''
        Update the CAD definition on disk with any
        changes to views or colours.
        '''
        try:

            with self.def_file.open_file( 'model.json','w' ) as f:
                json.dump(self.model_def,f,indent=4,sort_keys=True)
        
        except Exception as e:
            raise UserWarning('Cannot write changesto the model dfinition file ({:s}). The changes will only persist until this CAD model instance is unloaded.'.format(str(e)))        


# Class to represent a single CAD model feature.
# Does various grunt work and keeps the code nice and modular.
class ModelFeature():

    # Initialise with the parent CAD mdel and a dictionary
    # defining the feature
    def __init__(self,parent,definition_dict,abs_path=False):

        self.parent = parent

        if abs_path:
            self.filename = definition_dict['mesh_file']
        else:
            self.filename = os.path.join(self.parent.mesh_path_root,definition_dict['mesh_file'])

        if not os.path.isfile(self.filename):
            raise IOError('CAD mesh file {:s} not found.'.format(self.filename))

        self.filetype = self.filename.split('.')[-1].lower()

        self.enabled = definition_dict['default_enable']

        self.scale = definition_dict['mesh_scale']

        try:
            self.mesh_up = definition_dict['mesh_up_direction']
        except KeyError:
            self.mesh_up = '+Z'

        try:
            self.toroidal_rotation = definition_dict['rotate_toroidal']
        except KeyError:
            self.toroidal_rotation = 0.

        try:
            self.coord_handedness = definition_dict['coord_handedness']
        except KeyError:
            self.coord_handedness = 'right'

        self.polydata = None
        self.solid_actor = None
        self.edge_actor = None

        self.default_colour = definition_dict['colour']
        self.colour = self.default_colour
        self.linewidth = 1

    # Get a vtkPolyData object for this
    def get_polydata(self):

        if not self.enabled:
            return None

        if self.polydata is None:

            self.parent.status_callback('Loading mesh file: {:s}...'.format(os.path.split(self.filename)[1]))

            if self.filetype == 'stl':
                reader = vtk.vtkSTLReader()
            elif self.filetype == 'obj':
                reader = vtk.vtkOBJReader()

            reader.SetFileName(self.filename)
            reader.Update()

            transformer = vtk.vtkTransformPolyDataFilter()

            transform = vtk.vtkTransform()
            transform.PostMultiply()

            if self.coord_handedness == 'left':
                transform.Scale(self.scale,self.scale,-self.scale)
            elif self.coord_handedness == 'right':
                transform.Scale(self.scale, self.scale, self.scale)

            if self.mesh_up == '+X':
                transform.RotateY(-90)
            elif self.mesh_up == '-X':
                transform.RotateY(90)
            elif self.mesh_up == '+Y':
                transform.RotateX(90)
            elif self.mesh_up == '-Y':
                transform.RotateX(-90)
            elif self.mesh_up == '-Z' and self.coord_handedness == 'right':
                transform.RotateX(180)
            elif self.mesh_up == '+Z' and self.coord_handedness == 'left':
                transform.RotateX(180)

            transform.RotateZ(self.toroidal_rotation)
            transformer.SetInputData(reader.GetOutput())
            transformer.SetTransform(transform)
            transformer.Update()

            if self.coord_handedness == 'left':
                reverser = vtk.vtkReverseSense()
                reverser.ReverseNormalsOff()
                reverser.ReverseCellsOn()
                reverser.SetInputData(transformer.GetOutput())
                reverser.Update()
                self.polydata = reverser.GetOutput()
                transformer.SetInputData(reverser.GetOutput())
            elif self.coord_handedness == 'right':
                self.polydata = transformer.GetOutput()

            # Remove all the lines from the PolyData. As far as I can tell for "normal" mesh files this shouldn't
            # remove anything visually important, but it avoids running in to issues with vtkFeatureEdges trying to allocate
            # way too much memory in VTK 9.1+.
            self.polydata.SetLines(vtk.vtkCellArray())

            self.parent.status_callback(None)

        if self.parent.slice_planes is None:
            return self.polydata
        else:
            slice_filter = vtk.vtkExtractPolyDataGeometry()
            slice_filter.SetImplicitFunction(self.parent.slice_planes[0])
            slice_filter.SetExtractInside(self.parent.slice_planes[1])
            slice_filter.SetInputData(self.polydata)
            slice_filter.Update()
            return slice_filter.GetOutput()



    # Enable or disable the feature
    def set_enabled(self,enable):

        if enable and not self.enabled:

            self.enabled = True

            for renderer in self.parent.renderers:
                for actor in self.get_vtk_actors():
                    renderer.AddActor(actor)

        elif self.enabled and not enable:
            for renderer in self.parent.renderers:
                for actor in self.get_vtk_actors():
                    renderer.RemoveActor(actor)

            self.enabled = False       


    # Get vtkActor object(s) for this feature
    def get_vtk_actors(self):

        if not self.enabled:

            return []
        
        else:
            
            if self.solid_actor is None:

                mapper =  vtk.vtkPolyDataMapper()
                mapper.SetInputData( self.get_polydata() )

                self.solid_actor = vtk.vtkActor()
                self.solid_actor.SetMapper(mapper)


            # Make the edge actor if it doesn't already exist and is needed
            if self.parent.edges and self.edge_actor is None:

                self.parent.status_callback('Detecting mesh edges...')

                edge_finder = vtk.vtkFeatureEdges()

                edge_finder.SetInputData( self.get_polydata() )

                edge_finder.ManifoldEdgesOff()
                edge_finder.BoundaryEdgesOff()
                edge_finder.NonManifoldEdgesOff()
                edge_finder.SetFeatureAngle(20)
                edge_finder.ColoringOff()
                edge_finder.Update()

                data = edge_finder.GetOutput()
                n_edges = data.GetNumberOfCells()
                angle = 20
                while n_edges < 1 and angle > 5:
                    angle = angle - 5
                    edge_finder.SetFeatureAngle(angle)
                    edge_finder.Update()
                    data = edge_finder.GetOutput()
                    n_edges = data.GetNumberOfCells()

                if n_edges == 0:
                    edge_finder.BoundaryEdgesOn()
                    edge_finder.Update()
                    data = edge_finder.GetOutput()

                mapper = vtk.vtkPolyDataMapper()
                mapper.SetInputData(data)

                self.edge_actor = vtk.vtkActor()
                self.edge_actor.SetMapper(mapper)
                
                self.edge_actor.GetProperty().SetLineWidth(self.linewidth)

                self.parent.status_callback(None)

            # Make sure the colour and lighing are set appropriately
            if self.parent.edges:
                self.solid_actor.GetProperty().SetColor((0,0,0))
                self.edge_actor.GetProperty().SetColor(self.colour)
            else:
                self.solid_actor.GetProperty().SetColor(self.colour)

                if self.parent.flat_shading:
                   self.solid_actor.GetProperty().LightingOff()

            
            if self.parent.edges:
                return [self.solid_actor,self.edge_actor]
            else:
                return [self.solid_actor]


    def set_linewidth(self,linewidth):
        self.linewidth = linewidth
        if self.edge_actor is not None:
            self.edge_actor.GetProperty().SetLineWidth(linewidth)


    # Set the colour of the feature
    def set_colour(self,colour):

        self.colour = colour
        if self.parent.edges:
            if self.edge_actor is not None:
                self.edge_actor.GetProperty().SetColor(colour)
        else:
            if self.solid_actor is not None:
                self.solid_actor.GetProperty().SetColor(colour)