"""CLI-Einstiegspunkt für MongoDB → Solr Sync."""

from __future__ import annotations

import sys

import click
from prefect import flow

from .sync_flow import mongodb_to_solr_sync


@click.group()
def cli():
    """MongoDB → Solr Sync Pipeline."""
    pass


@cli.command()
@click.option(
    "--config",
    required=True,
    help="Name der Prefect Variable mit Konfiguration",
)
@click.option(
    "--mode",
    type=click.Choice(["upsert", "delete-before-import", "delete-only"]),
    help="Modus-Override (überschreibt Config)",
)
@click.option(
    "--explain-indexes",
    is_flag=True,
    help="Führt explain() aus und prüft, ob Indexe genutzt werden",
)
@flow(log_prints=True)
def sync(
    config: str,
    mode: str | None,
    explain_indexes: bool,
) -> None:
    """Synchronisation ausführen.

    Beispiele:

      # Upsert-Modus (aus Config)
      python -m mongodbsolr.flows.cli sync --config mongodbsolr-config

      # Delete-Before-Import Modus
      python -m mongodbsolr.flows.cli sync --config mongodbsolr-config --mode delete-before-import

      # Explain aktivieren
      python -m mongodbsolr.flows.cli sync --config mongodbsolr-config --explain-indexes
    """
    mongodb_to_solr_sync(
        config_var=config,
        mode_override=mode,
        explain_indexes=explain_indexes,
    )


@cli.command()
@click.option(
    "--config",
    required=True,
    help="Name der Prefect Variable mit Konfiguration",
)
@flow(log_prints=True)
def test_config(config: str) -> None:
    """Konfiguration laden und anzeigen."""
    _test_config_impl(config)


def _test_config_impl(config_var: str) -> None:
    """Implementierung des Config-Tests."""
    import asyncio
    import json

    from ..utils.config import load_config_from_variable, validate_config

    print(f"\n=== Lade Konfiguration: {config_var} ===\n")

    try:
        config = asyncio.run(load_config_from_variable(config_var))
        print("✓ Konfiguration erfolgreich geladen\n")

        print("=== Konfiguration (ohne Secrets) ===")
        # Secrets ausblenden
        safe_config = config.copy()
        if "mongodb" in safe_config and "uri" in safe_config["mongodb"]:
            safe_config["mongodb"]["uri"] = "***HIDDEN***"
        if "solr" in safe_config and "url" in safe_config["solr"]:
            safe_config["solr"]["url"] = "***HIDDEN***"

        print(json.dumps(safe_config, indent=2))
        print()

        print("=== Validierung ===")
        validate_config(config)
        print("✓ Konfiguration ist gültig\n")

    except Exception as e:
        print(f"✗ Fehler: {e}\n")
        sys.exit(1)


if __name__ == "__main__":
    cli()
