"""Solr-Tasks für Prefect Flows."""

from __future__ import annotations

from typing import Any, Dict, List

from prefect import task, get_run_logger
from prefect.concurrency.sync import concurrency
from prefect.tasks import exponential_backoff
from prefect.client.orchestration import get_client
import anyio
import pysolr

CONCURRENCY_LIMIT_NAME = "solr-upsert"


def _log_concurrency_state(logger) -> None:
    """Loggt den aktuellen Zustand des globalen Prefect-Concurrency-Limits für Solr."""

    async def _read() -> None:
        async with get_client() as client:
            try:
                cl = await client.read_concurrency_limit_v2(CONCURRENCY_LIMIT_NAME)
            except Exception as exc:
                logger.debug(
                    f"Lesen des Concurrency-Limits '{CONCURRENCY_LIMIT_NAME}' fehlgeschlagen: {exc}"
                )
                return

            available = max(cl.limit - cl.active_slots, 0)
            logger.info(
                f"Concurrency-Limit '{CONCURRENCY_LIMIT_NAME}': "
                f"limit={cl.limit}, active_slots={cl.active_slots}, "
                f"available_slots={available}"
            )

    anyio.run(_read)


@task(
    name="Solr Delete by Query",
    tags=["solr", "delete"],
    retries=3,
    retry_delay_seconds=5,
    persist_result=False,
)
def delete_by_query(
    solr_url: str,
    core: str,
    query: str,
    commit: bool = True,
) -> int:
    """Löscht Dokumente in Solr anhand einer Query.

    Args:
        solr_url: Solr-Base-URL
        core: Solr-Core-Name
        query: Solr-Query-String (z.B. "source:mongodb")
        commit: Ob Commit ausgeführt werden soll

    Returns:
        Anzahl gelöschter Dokumente (ermittelt durch Count vor Delete)

    Raises:
        Exception: Bei Solr-Fehlern
    """
    logger = get_run_logger()

    # Solr-URL mit Core
    full_url = f"{solr_url.rstrip('/')}/{core}"
    logger.info(f"Solr Delete-by-Query: URL={full_url}, Query={query}")

    try:
        solr = pysolr.Solr(full_url, always_commit=commit, timeout=30)

        # Zähle Dokumente vor dem Löschen
        count_before = solr.search(query, rows=0).hits
        logger.info(f"Dokumente gefunden: {count_before}")

        if count_before == 0:
            logger.info("Keine Dokumente zum Löschen gefunden")
            return 0

        # Delete ausführen, begrenzt durch globales Concurrency-Limit
        with concurrency(CONCURRENCY_LIMIT_NAME, occupy=1):
            _log_concurrency_state(logger)
            solr.delete(q=query)

        logger.info(f"Solr Delete-by-Query erfolgreich: {count_before} Dokumente gelöscht")
        return count_before

    except Exception as e:
        logger.error(f"Fehler bei Solr Delete-by-Query: {e}")
        raise


@task(
    name="Solr Upsert Batch",
    tags=["solr", "upsert"],
    retries=5,
    retry_delay_seconds=exponential_backoff(backoff_factor=5),
    persist_result=False,
)
def upsert_batch(
    solr_url: str,
    core: str,
    documents: List[Dict[str, Any]],
    commit_within: int = 60000,
) -> int:
    """Fügt Dokumente in Solr ein oder aktualisiert sie.

    Args:
        solr_url: Solr-Base-URL
        core: Solr-Core-Name
        documents: Liste von Dokumenten
        commit_within: Commit-Timeout in Millisekunden (default: 60000ms = 60s für Bulk-Imports)

    Returns:
        Anzahl eingefügter/aktualisierter Dokumente

    Raises:
        Exception: Bei Solr-Fehlern
    """
    logger = get_run_logger()

    if not documents:
        logger.warning("Keine Dokumente zum Upsert")
        return 0

    # Solr-URL mit Core
    full_url = f"{solr_url.rstrip('/')}/{core}"
    logger.info(
        f"Solr Upsert: URL={full_url}, Dokumente={len(documents)}, commit_within={commit_within}ms"
    )

    try:
        solr = pysolr.Solr(full_url, always_commit=False, timeout=60)

        # Upsert ausführen, begrenzt durch globales Concurrency-Limit
        with concurrency(CONCURRENCY_LIMIT_NAME, occupy=1):
            _log_concurrency_state(logger)
            solr.add(documents, commit=False, commitWithin=commit_within)

        logger.info(f"Solr Upsert erfolgreich: {len(documents)} Dokumente")
        return len(documents)

    except Exception as e:
        logger.error(f"Fehler bei Solr Upsert: {e}")
        raise


@task(
    name="Solr Delete by IDs",
    tags=["solr", "delete"],
    retries=3,
    retry_delay_seconds=5,
    persist_result=False,
)
def delete_by_ids(
    solr_url: str,
    core: str,
    ids: List[str],
    id_field: str = "id",
    batch_size: int = 100,
    commit: bool = True,
) -> int:
    """Löscht Dokumente in Solr anhand von IDs.

    Args:
        solr_url: Solr-Base-URL
        core: Solr-Core-Name
        ids: Liste von IDs
        id_field: Name des ID-Feldes in Solr (default: "id")
        batch_size: Anzahl IDs pro Batch
        commit: Ob Commit ausgeführt werden soll

    Returns:
        Anzahl gelöschter Dokumente

    Raises:
        Exception: Bei Solr-Fehlern
    """
    logger = get_run_logger()

    if not ids:
        logger.warning("Keine IDs zum Löschen")
        return 0

    # Solr-URL mit Core
    full_url = f"{solr_url.rstrip('/')}/{core}"
    logger.info(
        f"Solr Delete-by-IDs: URL={full_url}, IDs={len(ids)}, "
        f"id_field={id_field}, batch_size={batch_size}"
    )

    try:
        solr = pysolr.Solr(full_url, always_commit=commit, timeout=60)

        total_deleted = 0

        # IDs in Batches verarbeiten
        for i in range(0, len(ids), batch_size):
            batch_ids = ids[i : i + batch_size]

            # Delete-Query für Batch erstellen
            # Format: id_field:(id1 OR id2 OR id3 ...)
            id_values = " OR ".join(f'"{id_}"' for id_ in batch_ids)
            query = f"{id_field}:({id_values})"

            logger.debug(f"Lösche Batch {i // batch_size + 1}: {len(batch_ids)} IDs")

            # Delete ausführen, begrenzt durch globales Concurrency-Limit
            with concurrency(CONCURRENCY_LIMIT_NAME, occupy=1):
                _log_concurrency_state(logger)
                solr.delete(q=query)
            total_deleted += len(batch_ids)

        logger.info(f"Solr Delete-by-IDs erfolgreich: {total_deleted} Dokumente")
        return total_deleted

    except Exception as e:
        logger.error(f"Fehler bei Solr Delete-by-IDs: {e}")
        raise


@task(
    name="Solr Commit",
    tags=["solr", "commit"],
    retries=2,
    retry_delay_seconds=3,
    persist_result=False,
)
def commit_solr(solr_url: str, core: str) -> None:
    """Führt expliziten Commit in Solr aus.

    Args:
        solr_url: Solr-Base-URL
        core: Solr-Core-Name

    Raises:
        Exception: Bei Solr-Fehlern
    """
    logger = get_run_logger()

    full_url = f"{solr_url.rstrip('/')}/{core}"
    logger.info(f"Solr Commit: URL={full_url}")

    try:
        solr = pysolr.Solr(full_url, timeout=30)
        solr.commit()
        logger.info("Solr Commit erfolgreich")

    except Exception as e:
        logger.error(f"Fehler bei Solr Commit: {e}")
        raise
