"""Config-Loading aus Prefect Variables und Secret Blocks."""

from __future__ import annotations

import json
from typing import Any, Dict

from prefect import get_run_logger


async def load_config_from_variable(variable_name: str) -> Dict[str, Any]:
    """Lädt Konfiguration aus einer Prefect Variable.

    Args:
        variable_name: Name der Prefect Variable

    Returns:
        Dict mit Konfiguration

    Raises:
        ValueError: Wenn Variable nicht gefunden oder ungültig
    """
    from prefect.client.orchestration import get_client

    logger = get_run_logger()

    async with get_client() as client:
        try:
            var = await client.read_variable_by_name(variable_name)
            if var and var.value is not None:
                config = var.value if isinstance(var.value, dict) else json.loads(str(var.value))
                logger.info(f"Konfiguration aus Variable '{variable_name}' geladen")
                return config
            else:
                raise ValueError(f"Variable '{variable_name}' existiert nicht oder ist leer")
        except json.JSONDecodeError as e:
            raise ValueError(f"Variable '{variable_name}' enthält ungültiges JSON: {e}") from e
        except Exception as e:
            raise ValueError(f"Fehler beim Laden der Variable '{variable_name}': {e}") from e


async def load_secret_from_block(block_name: str) -> str:
    """Lädt Secret aus einem Prefect Secret Block (async).

    Args:
        block_name: Name des Secret Blocks

    Returns:
        Secret-Wert als String

    Raises:
        ValueError: Wenn Block nicht gefunden

    Note:
        Verwendet Secret.aload() für async Kontext in Prefect 3.
    """
    from prefect.blocks.system import Secret

    logger = get_run_logger()

    try:
        # In Prefect 3: aload() für async, load() für sync
        secret_block = await Secret.aload(block_name)
        logger.debug(f"Secret aus Block '{block_name}' geladen")
        return secret_block.get()
    except Exception as e:
        raise ValueError(f"Fehler beim Laden des Secret Blocks '{block_name}': {e}") from e


def validate_config(config: Dict[str, Any]) -> None:
    """Validiert die Konfiguration.

    Args:
        config: Konfiguration als Dict

    Raises:
        ValueError: Wenn Konfiguration ungültig
    """
    # MongoDB-Konfiguration prüfen
    if "mongodb" not in config:
        raise ValueError("Konfiguration fehlt: 'mongodb'")

    mongodb_cfg = config["mongodb"]
    required_mongodb = ["uri_block", "database", "collection", "query"]
    for key in required_mongodb:
        if key not in mongodb_cfg:
            raise ValueError(f"MongoDB-Konfiguration fehlt: '{key}'")

    # Solr-Konfiguration prüfen
    if "solr" not in config:
        raise ValueError("Konfiguration fehlt: 'solr'")

    solr_cfg = config["solr"]
    required_solr = ["url_block", "core"]
    for key in required_solr:
        if key not in solr_cfg:
            raise ValueError(f"Solr-Konfiguration fehlt: '{key}'")

    # Optional: maximale Anzahl paralleler Upserts validieren
    max_concurrent_upserts = solr_cfg.get("max_concurrent_upserts")
    if max_concurrent_upserts is not None:
        if not isinstance(max_concurrent_upserts, int) or max_concurrent_upserts <= 0:
            raise ValueError(
                "Solr-Konfiguration: 'max_concurrent_upserts' muss eine positive ganze Zahl sein"
            )

    # Optional: finales Commit-Flag validieren
    finalcommit = solr_cfg.get("finalcommit")
    if finalcommit is not None and not isinstance(finalcommit, bool):
        raise ValueError(
            "Solr-Konfiguration: 'finalcommit' muss ein boolescher Wert (true/false) sein"
        )

    # Mapping-Konfiguration prüfen
    if "mapping" not in config:
        raise ValueError("Konfiguration fehlt: 'mapping'")

    mapping_cfg = config["mapping"]
    required_mapping = ["mongodb_id_field", "solr_id_field"]
    for key in required_mapping:
        if key not in mapping_cfg:
            raise ValueError(f"Mapping-Konfiguration fehlt: '{key}'")

    # Modus prüfen
    mode = config.get("mode", "upsert")
    valid_modes = ["upsert", "delete-before-import", "delete-only"]
    if mode not in valid_modes:
        raise ValueError(f"Ungültiger Modus: '{mode}'. Erlaubt: {', '.join(valid_modes)}")

    # Delete-Query prüfen (nur für delete-before-import)
    if mode == "delete-before-import" and not config.get("delete_query"):
        raise ValueError(
            "Modus 'delete-before-import' erfordert 'delete_query' in der Konfiguration"
        )


async def resolve_config(config: Dict[str, Any]) -> Dict[str, Any]:
    """Löst Secret Blocks in der Konfiguration auf (async).

    Args:
        config: Konfiguration mit Block-Referenzen

    Returns:
        Konfiguration mit aufgelösten Secrets

    Note:
        Verwendet Secret.aload() für async Kontext in Prefect 3.
    """
    resolved = config.copy()

    # MongoDB-URI auflösen
    mongodb_uri_block = config["mongodb"].get("uri_block")
    if mongodb_uri_block:
        resolved["mongodb"]["uri"] = await load_secret_from_block(mongodb_uri_block)

    # Solr-URL auflösen
    solr_url_block = config["solr"].get("url_block")
    if solr_url_block:
        resolved["solr"]["url"] = await load_secret_from_block(solr_url_block)

    return resolved
