"""Feld-Validierung und Filterung für Dokumente."""

from __future__ import annotations

from typing import Any, Dict, List, Optional, Tuple


def validate_document(
    doc: Dict[str, Any], required_fields: List[str]
) -> Tuple[bool, Optional[List[str]]]:
    """Prüft, ob ein Dokument alle Pflichtfelder enthält.

    Args:
        doc: Dokument als Dict
        required_fields: Liste von Pflichtfeldern

    Returns:
        Tuple (is_valid, missing_fields)
        - is_valid: True wenn alle Pflichtfelder vorhanden
        - missing_fields: Liste fehlender Felder (None wenn valid)
    """
    if not required_fields:
        return True, None

    missing = [field for field in required_fields if field not in doc or doc[field] is None]

    if missing:
        return False, missing

    return True, None


def filter_fields(doc: Dict[str, Any], exclude_fields: List[str]) -> Dict[str, Any]:
    """Entfernt Felder aus einem Dokument.

    Args:
        doc: Dokument als Dict
        exclude_fields: Liste von Feldnamen, die entfernt werden sollen

    Returns:
        Gefiltertes Dokument (neue Kopie)
    """
    if not exclude_fields:
        return doc.copy()

    return {k: v for k, v in doc.items() if k not in exclude_fields}


def apply_id_mapping(
    doc: Dict[str, Any], mongodb_id_field: str, solr_id_field: str
) -> Dict[str, Any]:
    """Wendet ID-Mapping auf ein Dokument an.

    Kopiert den Wert von mongodb_id_field nach solr_id_field.
    Wenn mongodb_id_field == solr_id_field, wird nichts geändert.

    Args:
        doc: Dokument als Dict
        mongodb_id_field: Feldname der ID in MongoDB
        solr_id_field: Feldname der ID in Solr

    Returns:
        Dokument mit gemappter ID (neue Kopie)

    Raises:
        ValueError: Wenn mongodb_id_field nicht im Dokument vorhanden
    """
    if mongodb_id_field not in doc:
        raise ValueError(f"ID-Feld '{mongodb_id_field}' nicht im Dokument vorhanden")

    result = doc.copy()

    # Nur mappen wenn Felder unterschiedlich sind
    if mongodb_id_field != solr_id_field:
        result[solr_id_field] = doc[mongodb_id_field]

    return result


def process_document(
    doc: Dict[str, Any],
    required_fields: List[str],
    exclude_fields: List[str],
    mongodb_id_field: str,
    solr_id_field: str,
) -> Tuple[Optional[Dict[str, Any]], Optional[str], Optional[List[str]]]:
    """Verarbeitet ein Dokument: Validierung, ID-Mapping, Filterung.

    Args:
        doc: Dokument als Dict
        required_fields: Liste von Pflichtfeldern
        exclude_fields: Liste von auszublendenden Feldern
        mongodb_id_field: Feldname der ID in MongoDB
        solr_id_field: Feldname der ID in Solr

    Returns:
        Tuple (processed_doc, skip_reason, missing_fields)
        - processed_doc: Verarbeitetes Dokument (None wenn übersprungen)
        - skip_reason: Grund für Überspringen (None wenn valid)
        - missing_fields: Liste fehlender Pflichtfelder (None wenn valid)
    """
    # 1. Validierung
    is_valid, missing = validate_document(doc, required_fields)
    if not is_valid:
        return None, "missing_required_fields", missing

    # 2. ID-Mapping
    try:
        mapped = apply_id_mapping(doc, mongodb_id_field, solr_id_field)
    except ValueError as e:
        return None, f"id_mapping_error: {e}", None

    # 3. Filterung
    filtered = filter_fields(mapped, exclude_fields)

    return filtered, None, None
