"""Base class for deduplication methodologies."""

import warnings
from abc import ABC, abstractmethod

import polars as pl
from pydantic import BaseModel, Field, ValidationInfo, field_validator


class DeduperSettings(BaseModel):
    """A data class to enforce basic settings dictionary shapes."""

    id: str = Field(description="A unique ID field in the data to dedupe")

    @field_validator("id")
    @classmethod
    def _id_for_cmf(cls, v: str, info: ValidationInfo) -> str:
        enforce = "id"
        if v != enforce:
            warnings.warn(
                f"For offline deduplication, {info.field_name} can be any field. \n\n"
                "When deduplicating to write back to the Company Matching "
                f"Framework database, the ID must be '{enforce}', generated by "
                "retrieving data with matchbox.query().",
                stacklevel=3,
            )
        return v


class Deduper(BaseModel, ABC):
    """A base class for dedupers."""

    settings: DeduperSettings

    @classmethod
    @abstractmethod
    def from_settings(cls) -> "Deduper":
        """Create a Deduper from a settings dictionary."""
        raise NotImplementedError(
            """\
            Must implement method to instantiate from settings \
            -- consider creating a pydantic model to enforce shape.
        """
        )

    @abstractmethod
    def prepare(self, data: pl.DataFrame) -> None:
        """Prepare the deduper for deduplication."""
        return

    @abstractmethod
    def dedupe(self, data: pl.DataFrame) -> pl.DataFrame:
        """Deduplicate the dataframe."""
        return
