<a id="user-guide"></a>
A python wrapper for UO-NetDot's RESTful API.

* [Changelog](#changelog)
* [API Documentation](#netdot-python-api-generated-documentation)
* [ENV VARs Documentation](#netdot-python-api-environment-variables)

> ℹ This documentation is targeted for Network Engineers (who want to do more work using Python).
> There are ***Examples** below that can help you get started **(copy-paste friendly)!***
>
> No particular Python knowledge is required (though there is plenty of "Python tip"s and tricks to be learned here)!
> 
> **⚠ NOTE:** From 0.2.0 onward, this API wrapper has not been tested on the [de facto Open Source version of NetDot (GitHub)](https://github.com/cvicente/Netdot).

[![PyPI version](https://badge.fury.io/py/netdot.svg)](https://pypi.org/project/netdot/)

<!-- These need to be uploaded somewhere visible to the internet for us to use them on pypi.org... Time for a github repository I'm thinking!

![Code Coverage](assets/coverage.svg)
![Test Suite](assets/tests.svg)
-->

# Install 

This package is deployed to pypi.org.
Download and install it with `pip`:

```sh
pip install netdot
```


# Interactive Usage (Python interpreter)

Before getting into building a massive integration/tool, you might jump in and get some experience.
Thankfully, we have the [Python Interpreter (external)](https://docs.python.org/3/tutorial/interpreter.html) where we can jump in and do some testing!

```sh
# Enter the Python interpreter by running just "python" in your shell
$ python
Python 3.8.10 (default, May 26 2023, 14:05:08)     
... omitted for brevity...
>>> import netdot
>>>
```


With the netdot package imported, you can proceed with setting up a connecting and downloading some data!

> ℹ Most of the [Python Netdot API](#netdot-python-api-generated-documentation) is actually runtime-generated code.
>
> Use **tab completion** to quickly learn what methods are available.

> ℹ The Python interpreter is often referred to as 'a REPL' (Read-Eval-Print-Loop).
> For more info, see ["Using the REPL (in VSCode)" documentation (external)](https://www.learnpython.dev/01-introduction/02-requirements/05-vs-code/04-the-repl-in-vscode/).


<a id=connecting-in-the-interpreter-netdotconnect></a>

## Connecting in the interpreter: `netdot.connect()`

We have enabled interpreter-usage as a first-class feature.
In particular, you will want to use the `connect` function like the following.

```python
>>> import netdot
>>> nd_repo = netdot.connect()
What is the URL of the NetDot server? [https://nsdb.uoregon.edu]: ('enter' to use default)
NetDot username: myusername
NetDot password: ********** (using getpass module, to securely collect password)
>>> 
```

We now have a `netdot.Repository` named `nd_repo` connected to Netdot!

> ℹ `netdot.connect()` returns a `netdot.Repository` instance with `propose_changes` enabled by default (AKA 'dry run' feature).

### Example 1: Get Location of an IP Address (by Subnet)

If Netdot has well maintained SiteSubnet relationships, then you can use this feature to quickly look up which Site (Building) holds an IP Address.


1. Get the IPBlock for the IP Address:
```python
>>> ipblock = nd_repo.get_ipblock_by_address('128.123.25.41')
```
2. Get the Subnet IPBlock via `load_parent`:
> ℹ Similar to `load_parent`, there is also a `load_children` method.
```python
>>> subnet = ipblock.load_parent()
```
3. Call `load_sites` on the subnet to load any/all sites that are related to this subnet (via SiteSubnet relationships):
> ℹ If your Netdot Subnet stretches across many Netdot Sites, this will return all of those Sites.
```python
>>> subnet.load_sites()
[Site(id=42, name='Oregon Hall'... )]
```

#### Example 1: Code Sample (copy-paste friendly)

```python
ipblock = nd_repo.get_ipblock_by_address('128.123.25.41')
subnet = ipblock.load_parent()
subnet.load_sites()
```

### Example 2: Lookup DNS Record by IP Address

You can use this API to lookup (and modify) the DNS Resource Record (RR) associated to some IP Address.

```python
>>> dns_RR_record = nd_repo.get_rr_by_address('10.243.14.32')
```

The RR contains several pieces of information that may be useful to review!


1. You might wanna see what the Fully Qualified Domain Name (FQDN) is for this RR.
```python
>>> dns_RR_record.infer_FQDN()
'foobar.uoregon.edu'
```
2. You can look at the RR's info (AKA 'comment'):
```python
>>> dns_RR_record.info
'LOC: 123 FooBar Hall CON: Jenny J, 867-5309'
```
3. You can check when this RR was created, and last modified:
```python
>>> dns_RR_record.created
datetime.datetime(2020, 1, 16, 12, 7, 50)
>>> dns_RR_record.modified
datetime.datetime(2020, 1, 16, 12, 7, 50)
```
4. You can propose an update to the name if you like:
> ℹ You can make updates *any* of the fields of `dns_RR_record`.
```python
>>> dns_RR_record.name = 'bazzle'
Will UPDATE RR: RR(id=265246, active=True, auto_update=False, expiration='', info='', name='bazzle',...
```
5. And save those the name change to Netdot by calling `save_changes()`
```python
>>> nd_repo.save_changes()
```

#### Example 2: Code Sample (copy-paste friendly)

```
dns_RR_record = nd_repo.get_rr_by_address('10.243.14.32')
dns_RR_record.infer_FQDN()
dns_RR_record.info
dns_RR_record.created
dns_RR_record.modified
dns_RR_record.name = 'bazzle'
# And save the name change using:
# nd_repo.save_changes()
```

### Example 3: Pretty Printing DNS Records

*Continuing from the prior example,* assume we just want to review all the details for the DNS Records we just retrieved.

6. Import `pprint` from the "pprint" module (part of Python stdlib)
```python
>>> from pprint import pprint
```
7. Use the `pprint` function to print **any** returned Netdot objects
```python
>>> pprint(dns_RR_record)
RR(id=54482,
   active=True,
   auto_update=False,
   expiration='',
   info='LOC: 215A Oregon Hall CON: Chris LeBlanc, 6-2931 ',
   name='metadata2',
   zone='uoregon.edu',
   zone_xlink=1,
   created=datetime.datetime(2020, 1, 16, 12, 7, 50),
   modified=datetime.datetime(2020, 1, 16, 12, 7, 50))
>>> dns_RRADDRs = dns_RR_record.load_rraddr()
>>> pprint(dns_RRADDRs)
[RRADDR(id=16759,
        ipblock='128.223.37.93',
        ipblock_xlink=72430287,
        rr='metadata2.uoregon.edu',
        rr_xlink=54482,
        ttl='86400')]
```

#### Example 3: Code Sample (copy-paste friendly)

```
from pprint import pprint
pprint(dns_RR_record)
dns_RRADDRs = dns_RR_record.load_rraddr()
pprint(dns_RRADDRs)
```

### Example 4: Lookup Edge Port for MAC Address in NetDot

> ℹ Tip: This is useful for tracking down the physical location of some MAC Address.
>
>> **⚠ WARNING**: "find_edge_port" takes a LONG time, requires a LOT of HTTP requests, and includes assumptions that can lead to inconsistent and inaccurate results:
>>
>>Particularly, **if more than one forwarding table contains the MAC Address**, then NetDot will select the one whose forwarding table had the least entries.
>> 
>> This can be inaccurate especially if a 'forwarding table scan' is happening while you are trying to use `find_edge_port`.
>>
>> ℹ This assumption is present when looking up an edge port using NetDot's frontend as well. This method does require additional latency of HTTP requests, which is not required when using Netdot's frontend (frontend may have more accurate/consistent results).

You can use this API to lookup the Edge Port associated to some MAC Address.

```python
>>> interface = nd_repo.find_edge_port('8C3BADDA9EF1')
```

Once the interface lookup is complete (may take more than 60 seconds), it is very easy to check if there is any "`jack`" (location information) associated to this Interface!

```python
>>> interface.jack
'136A246B'
```

To load full details about the jack, instead call `load_jack()`

```python
>>> interface.load_jack()
HorizontalCable(id=6143, account='', closet='ASDF 101 Oregon Hall', ...)
```

#### Example 4: Code Sample (copy-paste friendly)

```
interface = nd_repo.find_edge_port('8C3BADDA9EF1')
interface.jack
interface.load_jack()
```

### Example 5: Check Devices Last ARP for a Site

Want to see the 'last ARP' time for all the devices that are located within the Site named "Death Star?"

1. First lookup the site:
```python
>>> site_name = 'Death Star'
>>> sites = nd_repo.get_sites_where(name=site_name)
>>> assert len(sites) == 1, f"Expected exactly one site with name {site_name}, found: {sites}"
>>> site = sites[0]
```
2. Then, simply call "load_devices" on that site:
```python
>>> devices = site.load_devices()
```
3. Print out the "last ARP" time for these devices (sorted). 
```python
>>> devices = sorted(devices, key=lambda device: device.last_arp)
>>> for device in devices:
>>>     print(f"{device.name} last ARPed at {device.last_arp}")
foo.uoregon.edu last ARPed at 1970-01-01 00:00:00
... trimmed for brevity...
bar.uoregon.edu last ARPed at 2023-10-20 15:00:03
baz.uoregon.edu last ARPed at 2023-11-07 20:00:04
foobar.uoregon.edu last ARPed at 2023-11-10 08:00:04
foobaz.uoregon.edu last ARPed at 2023-11-10 08:00:04
```

#### Example 5: Code Sample (copy-paste friendly)

```
site_name = 'Death Star'
sites = nd_repo.get_sites_where(name=site_name)
assert len(sites) == 1, f"Expected exactly one site with name {site_name}, found: {sites}"
site = sites[0]
devices = site.load_devices()
devices = sorted(devices, key=lambda device: device.last_arp)
for device in devices:
    print(f"{device.name} last ARPed at {device.last_arp}")
```

### Example 6: Delete All Devices for a Site

*Continuing from the last example,* imagine 'Death Star' has been fully removed from your campus (pesky rebels).

4. You need to delete all the devices associated to this site:
```python
>>> for device in devices:
>>>     device.delete()
```
5. The work has been prepared! Now, take a look at what changes will occur using either `show_changes` or `show_changes_as_tables` on the Repository object.
    - `show_changes` provides a dry-run of exactly what will occur, step by step.
    ```python
    >>> nd_repo.show_changes()
    1. Will DELETE Device: Device(id=123, site_xlink=...
    2. Will DELETE Device: Device(id=9000, site_xlink=...
    ```
    - `show_changes_as_tables` provides an aggregated overview of what is going to happen.
    ```python
    >>> nd_repo.show_changes_as_tables(terse=True)
        
    ## Device Changes

    action    site_xlink    asset_id_xlink    monitorstatus_xlink  
    --------  ------------  ----------------  ---------------------
    DELETE    137           None              None                 
    DELETE    137           None              None                 
    ```
6. You just remembered -- your supervisor asked you to append "`DONE`" to the Site name to  once the devices are deleted! Lets make that change as well:
```python
>>> site.name = f'{site.name} DONE'
```
7. Re-use `show_changes_as_tables` to see all the proposed changes:
```python
>>> nd_repo.show_changes_as_tables(terse=True, select_cols=['name'])
## Device Changes

action    name           
--------  ---------------
DELETE    foo.uoregon.edu
DELETE    bar.uoregon.edu

## Site Changes

action    name            
--------  ----------------
UPDATE    [-Death Star-]+D
          eath Star DONE+
```
8. *If the changes all look good*, then go ahead and commit them using `save_changes`:
```python
>>> nd_repo.save_changes()
100%|████████████████████████████████████████| 3/3 [00:00<00:00,  9.26it/s]
```

> ℹ Tip: For exceptionally small screens, adjust `TERSE` settings via [API Environment Variables](#netdot-python-api-environment-variables). 
> 
> Example: You can use the following settings, to *print one line per entry truncated to 16 characters*:
> ```sh
> export NETDOT_CLI_TERSE=True
> export NETDOT_CLI_TERSE_COL_WIDTH=16
> export NETDOT_CLI_TERSE_MAX_CHARS=16
> ```

#### Example 6: Code Sample (copy-paste friendly)

```python
for device in devices:
    device.delete()
nd_repo.show_changes()
nd_repo.show_changes_as_tables(terse=True)
site.name = f'{site.name} DONE'
nd_repo.show_changes_as_tables(terse=True, select_cols=['name'])
# And save all changes using:
# nd_repo.save_changes()
```

### Example 7: Show All Changes

*Continuing from the prior example,* you can use `show_all_changes` to see a report of **all** actions (including completed and failed actions):
> `show_all_changes` includes each of:
> * completed tasks,
> * planned tasks, and
> * *If there was a failure:* the latest failed task.
```python
>>> nd_repo.show_all_changes()

Completed Actions:

1. Finished DELETE Device: Device(id=123, site_xlink=...
2. Finished DELETE Device: Device(id=9000, site_xlink=...
3. Finished UPDATE Site: Site(id=137, site_xlink=...

Remaining Actions:

None, yet...

```

#### Example 7: Code Sample (copy-paste friendly)

```python
nd_repo.show_all_changes()
```

### Example 8: Plan and Create a new Netdot Site
<a id='example-8-plan-and-create-a-new-netdot-site'></a> 

Imagine you want to add a new site in Netdot, with rooms and all. Let's create a 'Test Site,' with 3 rooms and 1 closet, like the following:

- Site: 'Test Site'
    - Floor: 'Test Floor'
        - Room: 'Test Room 1'
        - Room: 'Test Room 2'
            - Closet: 'Test Closet 1'
        - Room: 'Test Room 3'

Let's create all these objects!

1. First, within a dry run, we will propose the new site:
```python
>>> site =  nd_repo.create_new(netdot.Site(name='Test Site'))
Will CREATE Site: Site(id=None, name='Test Site', aliases=None, availab...
```
2. Then, using that returned `site` Python object, we will call `site.add_floor`:
```python
>>> floor = site.add_floor(netdot.Floor(level='Test Floor'))
Will CREATE Floor: Floor(id=None, info=None, level='Test Floor', site=S...
```
3. Next, using that returned `floor` Python object, we will call `floor.add_room`:
```python
>>> room1 = floor.add_room(netdot.Room(name='Test Room 1'))
Will CREATE Room: Room(id=None, floor=Floor(id=None, info=None, level='...
>>> room2 = floor.add_room(netdot.Room(name='Test Room 2'))
Will CREATE Room: Room(id=None, floor=Floor(id=None, info=None, level='...
>>> room3 = floor.add_room(netdot.Room(name='Test Room 3'))
Will CREATE Room: Room(id=None, floor=Floor(id=None, info=None, level='...
```
4. Finally, we can call `room.add_closet` on any of our `room[123]` objects.
```python
>>> closet = room3.add_closet(netdot.Closet(name='Test Closet 1'))
Will CREATE Closet: Closet(id=None, access_key_type=None, asbestos_tile...
```
5. Ensure that the proposed changes look good via `show_changes`:
```
>>> nd_repo.show_changes()
1. Will CREATE Site: Site(id=None, name='Test Site', aliases=None, availabi...
2. Will CREATE Floor: Floor(id=None, info=None, level='Test Floor', site=Si...
3. Will CREATE Room: Room(id=None, floor=Floor(id=None, info=None, level='T...
4. Will CREATE Room: Room(id=None, floor=Floor(id=None, info=None, level='T...
5. Will CREATE Room: Room(id=None, floor=Floor(id=None, info=None, level='T...
6. Will CREATE Closet: Closet(id=None, access_key_type=None, asbestos_tiles...
```
1. Uh oh! Upon reviewing you see that "Test Closet 1" is in "Test Room 3" -- it is supposed to be in Room 2! Update the room for the closet using basic assignment operator:
```python
closet.room = room2
```
1. To ensure the change has occurred, use the `show_as_tables(select_cols=['name', 'room'])` to make sure that the closet has the right room associated to it:
> ℹ Wanna see the full "Room" object in the "Closet Changes" table-output? Use the keyword argument `display_full_objects=True`.
>> Alternately, set up your [`NETDOT_CLI_DISPLAY_FULL_OBJECTS` environment variable](#netdot-python-api-environment-variables)
```python
>>> repository.show_changes_as_tables(select_cols=['name', 'room'])
## Closet Changes

action    name           room
--------  -------------  -------------------------------
CREATE    Test Closet 1  Room(id=None, name=Test Room 2)

... trimmed for brevity... 
```
8. Looks good! So, finally, save the changes to Netdot using `save_changes()`
```python
>>> nd_repo.save_changes()
100%|████████████████████████████████████████| 6/6 [00:00<00:00,  9.26it/s]
```

> ℹ The created objects will have their `id` attribute populated by Netdot during `save_changes`.

#### Example 8: Code Sample (copy-paste friendly)

```python
site =  nd_repo.create_new(netdot.Site(name='Test Site'))
floor = site.add_floor(netdot.Floor(level='Test Floor'))
room1 = floor.add_room(netdot.Room(name='Test Room 1'))
room2 = floor.add_room(netdot.Room(name='Test Room 2'))
room3 = floor.add_room(netdot.Room(name='Test Room 3'))
closet = room3.add_closet(netdot.Closet(name='Test Closet 1'))
nd_repo.show_changes()
nd_repo.save_changes()  # This is a totally non-destructive change -- let it rip!
# Cleanup
# site.delete()
# nd_repo.save_changes()
```

# Scripted Usage

Once you have a feel for the full [Python Netdot API](#netdot-python-api-generated-documentation), you can get to writing some scripts or tools to make your daily work in Netdot automated and simple.

## Connecting in a Script (via Environment Variables): `netdot.Repository()`

For starters, we need to set up a `Repository` to interact with NetDot.
Environment variables can be a good method for providing credentials to your script, as suggested here:

> ℹ This pairs nicely with Continuous Integration! 
> It also works well enough for local development using [python-dotenv](https://pypi.org/project/python-dotenv/) or similar.

```python
import os
import netdot

nd_repo = netdot.Repository(
    netdot_url=os.getenv('NETDOT_URL'), 
    user=os.getenv('NETDOT_USERNAME'), 
    password=os.getenv('NETDOT_PASSWORD'),
    dry_run=True,
    auto_update=True,
    print_changes=False,
)
```

### Example 9: Get all Rooms fed by a Switch

Given some Netdot Switch (Device), you can easily determine which all Rooms it feeds. 

> Assumption: Your Netdot Cable Plant has to have its HorizontalCables well-documented.

1. Lookup the Switch by name.
```python
name = 'foobar-sw09.example.com'
devices = nd_repo.get_devices_where(name=name)
assert len(devices) == 1, f"Found 0/more-than-one one device with name '{name}': {devices}"
switch = devices[0]
```
2. Collect a "Set" containing all of the rooms.
> ℹ Python Tip: Python's `set()` is a handy data structure -- it will automatically ensure that we only collect 'distinct' rooms (duplicates will be ignored)
```python
rooms = set()
for interface in switch.load_interfaces():
    jack = interface.load_jack()
    if jack:
        rooms.add(jack.room)
```
3. Print a nice report back out for reviewing.
> ℹ Python Tip: Python's `sorted` and `filter` functions can 'remove empty strings' and sort output for you!
```python
print(f"Rooms fed by {switch.name}:")
print('\n'.join(sorted(filter(None, rooms))))
```

#### Example 9: Code Sample (copy-paste friendly)

```python
name = 'foobar-sw09.example.com'
devices = nd_repo.get_devices_where(name=name)
assert len(devices) == 1, f"Found 0/more-than-one one device with name '{name}': {devices}"
switch = devices[0]
rooms = set()
for interface in switch.load_interfaces():
    jack = interface.load_jack()
    if jack:
        rooms.add(jack.room)
print(f"Rooms fed by {switch.name}:")
print('\n'.join(sorted(filter(None, rooms))))
```

### Example 10: Get all Jacks fed by a Switch

*Continuing from the prior example,* let's assume you want to know which jacks are actually fed by the switch within each room. 

4. Collect a "Dictionary" that maps from 'Room' to 'List of HorizontalCables'.
> ℹ Python Tip: Python's `defaultdict()` is very similar to `dict()`, but it will call a 'default factory function' if a value is missing.
```python
from collections import defaultdict
rooms = defaultdict(list)
for interface in switch.load_interfaces():
    jack = interface.load_jack()
    if jack:
        rooms[jack.room].append(jack)
```
5. Print a nice report back out for reviewing.
```python
for room, jacks in rooms.items():
    jack_list = '\n'.join([jack.jackid for jack in jacks])
    print(f'{room}\n{jack_list}\n')
```

#### Example 10: Code Sample (copy-paste friendly)

```python
# Get the Switch
name = 'foobar-sw09.example.com'
devices = nd_repo.get_devices_where(name=name)
assert len(devices) == 1, f"Found 0/more-than-one one device with name '{name}': {devices}"
switch = devices[0]

# Get all the Jacks
rooms = defaultdict(list)
for interface in switch.load_interfaces():
    jack = interface.load_jack()
    if jack:
        rooms[jack.room].append(jack)

# Print a report
for room, jacks in rooms.items():
    jack_list = '\n'.join([jack.jackid for jack in jacks])
    print(f'{room}\n{jack_list}\n')
```


### Example 11: Update 'aliases' of several Sites in NetDot

As a simple script example, imagine we want to update the 'aliases' with the string "` (odd layout)`" for some set of sites in NetDot.
In this example, we will write a script to do just that.

1. Now, we are given a list of `SITE_IDS` to which we want to update the 'alias' with the string "(odd layout)".
```python
SITE_IDS = [98, 124, 512, 123, 111]
```
2. We can use Python list comprehensions to download the sites, and make the updates locally.
```python
sites = [ nd_repo.get_site(id) for id in SITE_IDS ]
updated_sites = [ site.aliases=f'{site.aliases} (odd layout)' for site in sites ]
```
3. Then, it is time to apply the updates locally to the repository's "proposed changes"
```python
for updated_site in updated_sites:
    updated_site.update()
```
4. [Optionally] Add a step in your script to review all the proposed changes using `show_changes`
```python
nd_repo.show_changes()
```
5. Finally, save the changes back to Netdot using `save_changes`.
```python
nd_repo.save_changes()
```

#### Example 11: Code Sample (copy-paste friendly)

```python
# Get all the sites
SITE_IDS = [98, 124, 512, 123, 111]
sites = [ nd_repo.get_site(id) for id in SITE_IDS ]

# Update the sites
for site in sites:
    site.aliases=f'{site.aliases} (odd layout)'

# Save changes to Netdot
nd_repo.save_changes()
```

### Example 12: Create Host

You can create new DNS `A record` named 'tinker' using next available address in given subnet:

```python
host = nd_repo.create_host('tinker.example.com', '10.0.0.1/24')
```

This will create a new DNS A Record named 'tinker' in the 'example.com' zone, with the next available IP address in the '10.0.0.1/24' subnet.

### Example 13: Get Host DNS Info

You can get most the DNS info for a given host by its name:

```python
host = nd_repo.get_host_by_name('tinker.example.com')
print(host.names)  # List of all names associated with the host
print(host.addresses)  # List of all IP addresses associated with the host
```

Depending on your use case, it might be wise to wrap this with some additional logic, in case the given name is a CNAME record (which does not return the addresses as you might expect):

```python
host = repository.get_host_by_name('my-cname.uoregon.edu')
if len(host.addresses) == 0:  # Implies this could be a CNAME record
    rrcnames = repository.get_rrcname_by_rr(host.RRs[0])
    host = repository.get_host_by_name(rrcnames[0].cname)
```

# Advanced Usage

Some features in this Netdot Python package are not key to normal use.
Those features are documented here.

## Proposed Changes Pickle File
<a id="proposed-changes-pickle-file"> </a>

If you have been working with a Netdot Repository in dry run mode, you might want to *save a snapshot of proposed changes, or a record of completed changes*.
This can be done with `save_as_pickle()`.

> ℹ By default, the file is saved with a version marker and timestamp -- this helps ensure future recoverability of the file's contents.

```python
nd_repo.proposed_changes.save_as_pickle()
'netdot-cli-0.4.0-proposed_changes-2023-11-08_14-51.pickle'
```

### Recovering Proposed Changes from Pickle File

> ℹ By default, a [Proposed Changes Pickle File](#proposed-changes-pickle-file) will be created in your current working directory *if **any exception** occurs during `save_changes()`.*

If an error occurs when you call `nd_repo.save_changes()`, you will likely want to be able to look at which actions completed and which did not.
When you have a pickle file, there are two routes forward for recovering these changes: Online (connected to Netdot), or Offline.


#### Online Recovery

To recover while connected to Netdot, load the pickle file directly into `nd_repo.proposed_changes` then continue using the `nd_repo` like normal: 

```python
nd_repo = netdot.connect()

# 1. Load the changes
nd_repo.proposed_changes = netdot.load('netdot-cli-0.4.0-proposed_changes-2023-11-08_14-51.pickle')

# 2. Review completed, planned, and failed actions:
nd_repo.show_all_changes()      # As a verbose report
nd_repo.show_failed_changes()   # Only showing failures
nd_repo.show_changes()          # Only showing planned actions
nd_repo.show_changes_as_tables(select_cols=['name', 'address'])

# 3. If planned action look good, try to call `save_changes` (again) to complete those remaining planned actions
nd_repo.save_changes()
```

#### Offline Recovery

If you would rather work with the file offline, the `netdot.UnitOfWork` does expose all of the same information (with slightly more access to the underlying Python objects):

```python
# 1. Load the changes
proposed_changes = netdot.load('netdot-cli-0.4.0-proposed_changes-2023-11-08_14-51.pickle')
# 2. Review them as Human Readable Reports
proposed_changes.status_report()
proposed_changes.failed_actions_msgs()
proposed_changes.dry_run()
# 3. Review them as Lists of Python objects
proposed_changes.failed_actions()
proposed_changes.as_list()
proposed_changes.completed_as_list()
```

## Multithreading for Parallelizing HTTP GET Requests

The `netdot.Repository` class can multithread *certain* HTTP requests.

To enable this, set the `NETDOT_CLI_THREADS` Environment Variable before running your python code.

```bash
export NETDOT_CLI_THREADS=4
```

You can override this number by passing the `threads` keyword argument to the Repository constructor.

```python
>>> repo = netdot.Repository(..., threads=4)
```

> This `threads` keyword argument can be used in the [interactive interface (discussed above)](#connecting-in-the-interpreter-netdotconnect) as well.
> ```python
> >>> repo = netdot.connect(threads=4)
> ```


# Changelog
<a id="changelog"></a>
All notable changes to this project will be documented in this file.

The format is based on [Keep a Changelog](https://keepachangelog.com/en/1.0.0/),
and this project adheres to [Semantic Versioning](https://semver.org/spec/v2.0.0.html).

> Notice: Major version zero (0.y.z) is for initial development. Anything MAY change at any time.
> This public API should **not** be considered stable.

> **⚠ NOTE:** From 0.2.0 onward, this API wrapper does not ensure support for the [de facto Open Source version of NetDot (GitHub)](https://github.com/cvicente/Netdot) (today, we hope it will work for you with limited warnings).

## [Unreleased]
<a id="unreleased"></a>

### Plan to Add

* There are still 2 Python Dataclasses that just need to be implemented:
    - [ ] ARPCache
    - [ ] ARPCacheEntry 
* Retrieve/update various data types that contain **binary blobs** via REST API:
    - [ ] DataCaches
    - [ ] ClosetPictures
    - [ ] SitePictures
    - [ ] FloorPictures
* Provide some mechanism for "retry_failed_actions"
    * Maybe just `failed_actions_as_newly_planned_actions`
* More mechanisms for saving UnitOfWork (aka proposed_changes) to a file:
    * `save_as_excel` would be great to save all proposed changes as a workbook.
    * `save_as_CSVs` would be great to save all proposed changes as a set of CSV files.
* Ensure compatibility with the [de facto Open Source version of NetDot (GitHub)](https://github.com/cvicente/Netdot).


### Plan to Change


* More clarity in `show_changes_as_tables` output, when making updates to Netdot objects.
    > Simplest option: just make `DISPLAY_FULL_OBJECTS` default behavior
    ```
    # Today
    UPDATE    015A050A   [-009 Volcanology (015) (015)-]+008+

    # Cleanest option
    UPDATE    015A050A   [-009-]+008+

    # Simplest option
    UPDATE    015A050A   [-009 Volcanology (015) (015)-]+netdot.Room(name='008',...)+
    ```
* Continuous Integration: Generating 'badges' for 'code coverage' and 'tests passing' would be nice to have.
* Enable parallelizing HTTP requests for: 
    * The `save_changes` method (as an *option*, i.e. only for when *all your changes are independent*)
    * All `load_*` methods that make "N+1 HTTP Requests."
* Wire up `inconsistent_and_ignorable_fields` via `netdot.config` module.
* Generate a prettier document for ["Feature Flags" Environment Variables](#netdot-python-api-environment-variables).
    * Today, we [ab]use [configargparse](https://pypi.org/project/ConfigArgParse/) to generate that documentation.
* Public GitHub.com Repository for this project.
    * Consider if there is anything that should be anonymized before mirroring to github.com.

## [0.5.0]

### Added

- `Repository.create_host(hostname, subnet)`: 
    - Discussed further in [Example 12 in the User Guide](#user-guide#example-12-create-host).
- `Repository.get_host_by_name(hostname)`: Lookup DNS records for the given hostname.
    - Discussed further in [Example 13 in the User Guide](#user-guide#example-13-get-host-dns-info).


## [0.4.5]

### Changed

ArpCache and ArpCacheEntry dataclasses are now implemented.

## [0.4.4]

### Changed

* Internal-only change: Update CICD workflow to use [`ntsjenkins` shared library](https://confluence.uoregon.edu/x/TQRaGw)


## [0.4.3]

### Changed

* A call to `save_changes` will now print, "No changes need to be saved," if that is the case. 
    * BEFORE: `save_changes` would print "Saved changes!", even if there was no changes.
* FIX: auto-update when updating relationships between Netdot objects:
    * Observed: `update` never happens after updating the "room xlink field" for a `netdot.HorizontalCable` via: `horizontal_cable.room = new_room_object`
    * 
* When using a netdot.Repository in dry-run mode with `auto_update=True`: run `update` when setting "room" 

## [0.4.2]

### Changed

* Fix the `circuit` attribute for `netdot.Interface`. This fixes the following `WARNING`s: 
    1. `Received unknown field(s): {'circuit_xlink': '...`
    2. `Unable to parse 'circuit' for 'Interface'...`


## [0.4.1]

### Changed

* FIX: Support for Python 3.6
    > Our Continuous Integration solution does not include support for Python 3.6
    * Some of the ForwardReference shenanigans we do is based on python implementation details.
    * The 'trace' feature was using `asyncio.run`, which was introduced in Python 3.**7**.
        * Instead, we use the "run_until_complete" method, which works well for Python 3.6 onward.
* Default Connection Timeout changed from 20 down to **3 seconds**.
    * This way, it doesn't take 20 seconds to notice that your VPN connection is down.


## [0.4.0]

### Added

* Python 3.6 compatibility
* ["Feature Flags" Environment Variables](#netdot-python-api-environment-variables)
> ℹ Use `netdot.config.help()` to see full info anytime.
* [API documentation](#netdot-python-api-generated-documentation) for this Python Netdot API Wrapper.
* Implemented Python Dataclasses for *most* all of the Netdot Database schema.
    * All except two ArpCache data types and the four 'binary-blob-containing data types' (all listed in [[Unreleased]](#unreleased)).
* Basic '**C**R**UD**' methods on each NetdotAPIDataclass:
    * `create_or_update`: Create or update this object in Netdot.
    * `create`: *Thin wrapper around `create_or_update`.*
    * `update`: *Thin wrapper around `create_or_update`.*
    * `delete`: Delete this object in Netdot.
        * Asks for user confirmation (unless you call `delete(confirm=False)`)
* Generate methods on each NetdotAPIDataclass:
    * `add_*` methods for 1-to-many relationships.
        * Examples: `site.add_device` or `site.add_sitelink_as_farend`
    * `add_*` methods for many-to-many relationships.
        * Examples: `site.add_subnet` (and the more obscure `site.add_sitesubnet`)
    * †`load_*` methods for 1-to-many relationships.
        * Examples: `site.load_devices`
    * †`load_*` methods for many-to-many relationships.
        * Examples: `site.load_subnets`
    > †: `load_*` methods will catch any HTTP 404 errors and instead return Empty List or None.
    > (Use `ignore_404=False` to raise HTTP 404 errors instead)
* Support for 'Dry Runs' when making updates (i.e. create, update, delete).
    > See ["Example 8: Plan and Create a New Netdot Site" in the User Guide](#example-8-plan-and-create-a-new-netdot-site)
    1. Automatically set up a UnitOfWork when calling `netdot.connect`
    > ℹ Bypass 'dry runs' feature via `netdot.connect(propose_changes=False)` or after the fact using `disable_propose_changes()`.
    >
    > ℹ To selectively enable the 'dry runs' features on a Repository, use `enable_propose_changes(print_changes=..., auto_update=...)`
    2. Make changes just like normal.
    3. Show any proposed changes using `show_changes` or `show_changes_as_tables`.
    > ℹ You can select which columns render via `show_changes_as_tables(select_cols=[...])` 
    >
    > ℹ You can render the full nested objects via `show_changes_as_tables(display_full_objects=True)` 
    >
    >```python
    >>>> nd_repo.show_changes_as_tables(select_cols=['name', 'label', 'address'])`
    >
    >## Room Changes
    >
    >action    name
    >--------  -----------
    >CREATE    Test Room 1
    >
    >## Audit Changes
    >
    >action    label
    >--------  -----
    >CREATE    None
    >```
    >
    > ℹ You can control the CLI output (for smaller screens) in these using  `show_changes(terse=True)` or `show_changes_as_tables(terse=True, select_cols=[...])`. 
    >> ℹ To make your 'terse' setting persistent, see `NETDOT_CLI_TERSE` and related environment variables in the ["Feature Flags" Environment Variables](#netdot-python-api-environment-variables)
    4. Save those changes using `save_changes`
    ```python
    >>> nd_repo.save_changes()
    100%|████████████████████████████████████████| 3/3 [00:00<00:00,  9.26it/s]
    ```
    > ℹ If any DELETEs are planned: `save_changes` will asks the user for confirmation (unless you call `save_changes(confirm=False)`)
* After calling `save_changes`, use `show_all_changes` to see a report of **all** actions.
    * includes completed tasks,
    * includes planned tasks, and
    * *if there was a failure*, includes the latest failed task.
* The `UnitOfWork` supports [`save_as_pickle` to save any proposed changes to a file](#proposed-changes-pickle-file).
    * It can later be loaded back up using `netdot.load`.
* [UNTESTED] Log a warning whenever 'Carp::croak' is returned in HTTP response.


### Changed

* Simplify the 'MAC Address parser'
* The `netdot.Repository.get_all_*` methods have each been replaced with an equivalent `netdot.Repository.get_*_where`.
    * Ex. `get_all_sites` => `get_sites_where`
    * Ex. `get_all_devices` => `get_devices_where`
* `netdot.Device.baseMAC` attribute is gone, in favor of the `infer_base_MAC()` method.
    * We saw a lot of WARNING logs when the 'infer base MAC' logic was integrated into the parsing of Netdot Data Transfer Objects.
* `web_url` property tested for ***all** Netdot objects.*
    * web_url is: "A URL to view this object in Netdot's web interface."
    * Now using the simple "/generic/view.html" endpoint (instead of trying to use a more a 'more applicable webpage' for each data type).
* Fix pluralization for various methods in [public API](#netdot-python-api-generated-documentation).
* Simplified NetdotAPIDataclass property setters (by overwriting `__setattr__` directly for NetdotAPIDataclass).
    * ❌ Old way:
    ```
    netdot.Floor(...).with_site(site)
    ```
    * ✅ New way, via `__init__`:
    ```
    floor = netdot.Floor(..., site=site, ...)
    ```
  * ✅ New way, via assignment operator (`=`):
    ```
    floor = netdot.Floor(...)
    floor.site = site
    ```
* DownloadTracer is now optional and more resilient:
    * DownloadTracer is now an opt-in feature, configurable via ["Feature Flags" Environment Variables](#netdot-python-api-environment-variables), and
    * DownloadTracer now does all its logging via asyncio (to ensure that logging doesn't interrupt your downloads)

### Removed

* Removed `Repository.get_all_*` methods (See more discussion in "Changes" above)
* No longer generate the `with_*` and `set_*` methods for NetdotAPIDataclass.
* Do not log a message when 'info' or 'ttl' are absent from HTTP Response (they are VERY optional)
    * Search for `inconsistent_and_ignorable_fields` in source code to learn more.
* Removed old `netdot.Connect` class entirely.

## [0.3.2]

* Enable looking up a DNS Resource Record (RR) by address, using `repo.get_rr_by_address()`

## [0.3.1]

* Speed up `find_edge_port`.
  * HTTP requests are parallelized via multithreading where possible.

## [0.3.0]

> ⚠ Breaking Backwards Compatibility: Several `netdot.Repository` methods are renamed, as discussed below.

* Add `Repository.find_edge_port(mac_address)` method.
  * This requires a lot of HTTP requests since we do not have the ability to run arbitrary database queries (SQL JOIN behavior is unavailable via RESTful interface).
* Wired up the following netdot.dataclasses:
  * `ForwardingTable`
  * `ForwardingTableEntry`
  * `PhysAddr`
* Renamed several generated methods to end in "ies" instead of "ys" when pluralized.
* Dropping Python 3.6 and 3.7 compatibility (required to use [hatch](https://github.com/pypa/hatch))

## [0.2.6]

* Fix typo in `MACAddress:format` method argument: "delimiter" becomes "delimiter"
  * Additionally, force keyword arguments for the `format`using Python 3 feature.

## [0.2.5]

* In `netdot.Client` the base `delete(..., id)` method can now accept an `int`.
  * Before, it only accepted `str`.

## [0.2.4]

* Gracefully handle response from HTTP Delete requests when possible.
  * Delete seems to return 'empty' (a couple of newlines actually) on success.

## [0.2.3]

* Enable a `replace` function for all `netdot.dataclasses`
  * This makes it easier to do 'update' like operations using this library.

## [0.2.2]

* Fix for REGRESSION: The `post` method of `netdot.Client` does not work.
  * Debugged using a simple automated test (captured by a PyVCR Cassette for reproducibility)

## [0.2.1]

> 🐛 REGRESSION: The `post` method of `netdot.Client` does not work!

* Fix for REGRESSION: The `netdot.Client.Connection` class is missing!
  * Re-added `Connection` directly to client.py for now.
  * Aliased `netdot.client` module to also be available as it was formerly named, `netdot.Client` (pep8 suggests lowercase module names instead of CamelCase).
    * Using `__all__` in "netdot/\_\_init\_\_.py"

## [0.2.0]

> 🐛 REGRESSION: The `netdot.Client.Connection` class is MISSING!

> ⚠ We have not ensured support for the [de facto Open Source version of NetDot (GitHub)](https://github.com/cvicente/Netdot).

* Introducing a new layer of abstraction -- a Repository and many Python dataclasses.
  * See more info in the [User Guide](#user-guide)
* Provide technical documentation in "docs/" directory (following NTS's standards).

## [0.1.0]

* Provide Python Netdot Client, as originally authored by Francisco Gray.




# Netdot Python API Generated Documentation

<a id="netdot-python-api-generated-documentation"></a>
> Version 0.5.0 documentation generated on Nov 06, 2025 at 03:48PM 
>
> Netdot Python API contains many generated methods.
> This documentation is intended to help you understand what is available.



## class `netdot.Repository`

<a id="class-netdotrepository"></a>

### Methods

#### Repository.__init__

> ```__init__(self, netdot_url, user, password, dry_run=True, auto_update=True, print_changes=False, threads=None, trace_downloads=None, trace_threshold_bytes=None, **kwargs)```

```
Work with Netdot API using Python objects.

Args:
    netdot_url (str): The URL of the Netdot server, e.g. "https://nsdb.uoregon.edu"
    user (str): The Netdot username to use for authentication.
    password (str): The Netdot password to use for authentication.
    dry_run (bool, optional): Only **'propose changes'** until user calls `save_changes`. Defaults to True.
    print_changes (bool, optional): (When dry_run=True) Print any 'proposed changes' as they occur. Defaults to False.
    auto_update (bool, optional): (When dry_run=True) Automatically 'propose changes' on Netdot objects. Defaults to True. (If an attribute is updated on any Netdot object, that update will be immediately reflected in this repository's 'proposed changes')
    threads (int, optional): How many threads can be used when making GET requests? Defaults to config.THREADS.
```

#### Repository.clear_proposed_changes

> ```clear_proposed_changes(self)```

```
Reset the proposed changes (dry run) for this Netdot Repository.
```

#### Repository.create_host

> ```create_host(self, subnet: str, hostname: str) -> 'netdot.RR'```

```
Create new A record named 'hostname' using next available address in given 'subnet'

NOTE: This function does not work when this Repository is in `dry_run` mode.
NOTE: A subsequent call to `get_host_by_name` is needed to retrieve the created IP Address.
NOTE: This is part of "The special resource '/rest/host' provides a simplified interface for manipulating DNS and DHCP records."

Args:
    subnet (str): The subnet to create the host in.
    hostname (str): The hostname for the new host.

Returns:
    netdot.RR: The DNS Resource Record (RR) object that was created for the new host.

Example:
        >> repo.create_host('10.0.0.1/24', 'tinker.example.com')
        RR(id=1234, name='tinker', zone='example.com', type='A', ...)

    To get the create IP Address, you'll need to lookup the host again:

        >> repo.get_host_by_name('tinker.example.com')
        >> repo.addresses
        ['10.0.0.123']
```

#### Repository.create_new

> ```create_new(self, new_data: netdot.dataclasses.base.NetdotAPIDataclass) -> netdot.dataclasses.base.NetdotAPIDataclass```

```
Create some new object in Netdot.

> NOTE: Upon creation, the `id` field of new_data will be populated.

Args:
    new_data (netdot.NetdotAPIDataclass): The new data to use for the new Netdot object.

Returns:
    netdot.NetdotAPIDataclass: The new object (with `id` populated).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Repository.delete

> ```delete(self, data: netdot.dataclasses.base.NetdotAPIDataclass, confirm=True, ignore_404=True)```

```
Delete an existing object from Netdot.

> ⚠ WARNING: This is irreversible. Use with caution.

Args:
    data (NetdotAPIDataclass): The object to be deleted.
    confirm (bool): Assume interactive and ask user 'Proceed? ...' before deleting. Default True.
    ignore_404 (bool): Default True. Ignore HTTP 404 errors. (Why? no need to delete an object if it doesn't exist!)

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: Could occur for various reasons. (error details can be found in Netdot's apache server logs)
    NetdotDeleteError: If the object cannot be deleted for some other reason.
    TypeError: If data is not a subclass of NetdotAPIDataclass.
```

#### Repository.disable_propose_changes

> ```disable_propose_changes(self)```

```
Disable the 'propose changes' or DRY RUN feature on this Netdot Repository.

After this, all changes will be applied immediately to Netdot.
```

#### Repository.disable_trace_downloads

> ```disable_trace_downloads(self)```

```
Disable download tracing feature.
```

#### Repository.disconnect

> ```disconnect(self)```

```
Disconnect from Netdot.
```

#### Repository.enable_propose_changes

> ```enable_propose_changes(self, print_changes=True, auto_update=True)```

```
Enable the 'propose changes' or DRY RUN feature on this Netdot Repository.

After this, all changes will be queued, only to be applied when `save_changes` is called.
```

#### Repository.enable_trace_downloads

> ```enable_trace_downloads(self, threshold_bytes: int = None)```

```
Enable (and reset) the DownloadTracer object associated to this Netdot Client. (resetting it back to '0 bytes downloaded')

Args:
    threshold_bytes (bool, optional): How many bytes to wait between log messages. Defaults to config.TRACE_THRESHOLD.
```

#### Repository.find_edge_port

> ```find_edge_port(self, mac_address: str) -> 'netdot.Interface'```

```
Get the Edge Port (Interface) associated to some MAC Address.

> NOTE: This will make `N+M` HTTP Requests (where N ).

The idea is to get all device ports whose latest forwarding tables included this address.
If we get more than one, select the one whose forwarding table had the least entries.

Args:
    mac_address (str): A MAC Address to lookup.

Returns:
    netdot.Interface: The edge port associated to the provided MAC Address.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: If the MAC Address cannot be found, or if any subsequent HTTP requests fail. (error details can be found in Netdot's apache server logs)
```

#### Repository.get_accessright

> ```get_accessright(self, id: int) -> netdot.dataclasses.users.AccessRight```

```
Get info about a AccessRight from Netdot.

Args:
    id (int): The ID of the AccessRight to retrieve.

Returns:
    netdot.AccessRight: The AccessRight with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the AccessRight cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_accessrights_where

> ```get_accessrights_where(self, *args, **url_params) -> List[netdot.dataclasses.users.AccessRight]```

```
Get info about AccessRights from Netdot.

> NOTE: This will return ALL AccessRights from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.AccessRight]: AccessRights from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_arpcache

> ```get_arpcache(self, id: int) -> netdot.dataclasses.arpcache.ArpCache```

```
Get info about a ArpCache from Netdot.

Args:
    id (int): The ID of the ArpCache to retrieve.

Returns:
    netdot.ArpCache: The ArpCache with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the ArpCache cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_arpcacheentries_by_arpcache

> ```get_arpcacheentries_by_arpcache(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of ArpCacheEntries associated to a particular ArpCache.

Args:
    other (int,NetdotAPIDataclass): The particular ArpCache or its `id`.

Returns:
    List[netdot.ArpCacheEntry]: The list of ArpCacheEntries associated to the ArpCache.
```

#### Repository.get_arpcacheentries_by_interface

> ```get_arpcacheentries_by_interface(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of ArpCacheEntries associated to a particular Interface.

Args:
    other (int,NetdotAPIDataclass): The particular Interface or its `id`.

Returns:
    List[netdot.ArpCacheEntry]: The list of ArpCacheEntries associated to the Interface.
```

#### Repository.get_arpcacheentries_by_ipaddr

> ```get_arpcacheentries_by_ipaddr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of ArpCacheEntries associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.ArpCacheEntry]: The list of ArpCacheEntries associated to the IPBlock.
```

#### Repository.get_arpcacheentries_by_physaddr

> ```get_arpcacheentries_by_physaddr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of ArpCacheEntries associated to a particular PhysAddr.

Args:
    other (int,NetdotAPIDataclass): The particular PhysAddr or its `id`.

Returns:
    List[netdot.ArpCacheEntry]: The list of ArpCacheEntries associated to the PhysAddr.
```

#### Repository.get_arpcacheentries_where

> ```get_arpcacheentries_where(self, *args, **url_params) -> List[netdot.dataclasses.arpcache.ArpCacheEntry]```

```
Get info about ArpCacheEntries from Netdot.

> NOTE: This will return ALL ArpCacheEntries from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.ArpCacheEntry]: ArpCacheEntries from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_arpcacheentry

> ```get_arpcacheentry(self, id: int) -> netdot.dataclasses.arpcache.ArpCacheEntry```

```
Get info about a ArpCacheEntry from Netdot.

Args:
    id (int): The ID of the ArpCacheEntry to retrieve.

Returns:
    netdot.ArpCacheEntry: The ArpCacheEntry with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the ArpCacheEntry cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_arpcaches_by_device

> ```get_arpcaches_by_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of ArpCaches associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.ArpCache]: The list of ArpCaches associated to the Device.
```

#### Repository.get_arpcaches_where

> ```get_arpcaches_where(self, *args, **url_params) -> List[netdot.dataclasses.arpcache.ArpCache]```

```
Get info about ArpCaches from Netdot.

> NOTE: This will return ALL ArpCaches from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.ArpCache]: ArpCaches from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_asn

> ```get_asn(self, id: int) -> netdot.dataclasses.bgp.ASN```

```
Get info about a ASN from Netdot.

Args:
    id (int): The ID of the ASN to retrieve.

Returns:
    netdot.ASN: The ASN with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the ASN cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_asns_where

> ```get_asns_where(self, *args, **url_params) -> List[netdot.dataclasses.bgp.ASN]```

```
Get info about ASNs from Netdot.

> NOTE: This will return ALL ASNs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.ASN]: ASNs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_asset

> ```get_asset(self, id: int) -> netdot.dataclasses.asset.Asset```

```
Get info about a Asset from Netdot.

Args:
    id (int): The ID of the Asset to retrieve.

Returns:
    netdot.Asset: The Asset with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Asset cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_assets_by_maint_contract

> ```get_assets_by_maint_contract(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Assets associated to a particular MaintContract.

Args:
    other (int,NetdotAPIDataclass): The particular MaintContract or its `id`.

Returns:
    List[netdot.Asset]: The list of Assets associated to the MaintContract.
```

#### Repository.get_assets_by_physaddr

> ```get_assets_by_physaddr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Assets associated to a particular PhysAddr.

Args:
    other (int,NetdotAPIDataclass): The particular PhysAddr or its `id`.

Returns:
    List[netdot.Asset]: The list of Assets associated to the PhysAddr.
```

#### Repository.get_assets_by_product_id

> ```get_assets_by_product_id(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Assets associated to a particular Product.

Args:
    other (int,NetdotAPIDataclass): The particular Product or its `id`.

Returns:
    List[netdot.Asset]: The list of Assets associated to the Product.
```

#### Repository.get_assets_where

> ```get_assets_where(self, *args, **url_params) -> List[netdot.dataclasses.asset.Asset]```

```
Get info about Assets from Netdot.

> NOTE: This will return ALL Assets from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Asset]: Assets from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_audit

> ```get_audit(self, id: int) -> netdot.dataclasses.users.Audit```

```
Get info about a Audit from Netdot.

Args:
    id (int): The ID of the Audit to retrieve.

Returns:
    netdot.Audit: The Audit with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Audit cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_audits_where

> ```get_audits_where(self, *args, **url_params) -> List[netdot.dataclasses.users.Audit]```

```
Get info about Audits from Netdot.

> NOTE: This will return ALL Audits from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Audit]: Audits from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_availabilities_where

> ```get_availabilities_where(self, *args, **url_params) -> List[netdot.dataclasses.misc.Availability]```

```
Get info about Availabilities from Netdot.

> NOTE: This will return ALL Availabilities from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Availability]: Availabilities from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_availability

> ```get_availability(self, id: int) -> netdot.dataclasses.misc.Availability```

```
Get info about a Availability from Netdot.

Args:
    id (int): The ID of the Availability to retrieve.

Returns:
    netdot.Availability: The Availability with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Availability cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_backbonecable

> ```get_backbonecable(self, id: int) -> netdot.dataclasses.cables.BackboneCable```

```
Get info about a BackboneCable from Netdot.

Args:
    id (int): The ID of the BackboneCable to retrieve.

Returns:
    netdot.BackboneCable: The BackboneCable with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the BackboneCable cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_backbonecables_by_end_closet

> ```get_backbonecables_by_end_closet(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of BackboneCables associated to a particular Closet.

Args:
    other (int,NetdotAPIDataclass): The particular Closet or its `id`.

Returns:
    List[netdot.BackboneCable]: The list of BackboneCables associated to the Closet.
```

#### Repository.get_backbonecables_by_owner

> ```get_backbonecables_by_owner(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of BackboneCables associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.BackboneCable]: The list of BackboneCables associated to the Entity.
```

#### Repository.get_backbonecables_by_start_closet

> ```get_backbonecables_by_start_closet(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of BackboneCables associated to a particular Closet.

Args:
    other (int,NetdotAPIDataclass): The particular Closet or its `id`.

Returns:
    List[netdot.BackboneCable]: The list of BackboneCables associated to the Closet.
```

#### Repository.get_backbonecables_by_type

> ```get_backbonecables_by_type(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of BackboneCables associated to a particular CableType.

Args:
    other (int,NetdotAPIDataclass): The particular CableType or its `id`.

Returns:
    List[netdot.BackboneCable]: The list of BackboneCables associated to the CableType.
```

#### Repository.get_backbonecables_where

> ```get_backbonecables_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.BackboneCable]```

```
Get info about BackboneCables from Netdot.

> NOTE: This will return ALL BackboneCables from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.BackboneCable]: BackboneCables from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_bgppeering

> ```get_bgppeering(self, id: int) -> netdot.dataclasses.bgp.BGPPeering```

```
Get info about a BGPPeering from Netdot.

Args:
    id (int): The ID of the BGPPeering to retrieve.

Returns:
    netdot.BGPPeering: The BGPPeering with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the BGPPeering cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_bgppeerings_by_contactlist

> ```get_bgppeerings_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of BGPPeerings associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.BGPPeering]: The list of BGPPeerings associated to the ContactList.
```

#### Repository.get_bgppeerings_by_device

> ```get_bgppeerings_by_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of BGPPeerings associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.BGPPeering]: The list of BGPPeerings associated to the Device.
```

#### Repository.get_bgppeerings_by_entity

> ```get_bgppeerings_by_entity(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of BGPPeerings associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.BGPPeering]: The list of BGPPeerings associated to the Entity.
```

#### Repository.get_bgppeerings_where

> ```get_bgppeerings_where(self, *args, **url_params) -> List[netdot.dataclasses.bgp.BGPPeering]```

```
Get info about BGPPeerings from Netdot.

> NOTE: This will return ALL BGPPeerings from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.BGPPeering]: BGPPeerings from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_cablestrand

> ```get_cablestrand(self, id: int) -> netdot.dataclasses.cables.CableStrand```

```
Get info about a CableStrand from Netdot.

Args:
    id (int): The ID of the CableStrand to retrieve.

Returns:
    netdot.CableStrand: The CableStrand with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the CableStrand cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_cablestrands_by_cable

> ```get_cablestrands_by_cable(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of CableStrands associated to a particular BackboneCable.

Args:
    other (int,NetdotAPIDataclass): The particular BackboneCable or its `id`.

Returns:
    List[netdot.CableStrand]: The list of CableStrands associated to the BackboneCable.
```

#### Repository.get_cablestrands_by_circuit_id

> ```get_cablestrands_by_circuit_id(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of CableStrands associated to a particular Circuit.

Args:
    other (int,NetdotAPIDataclass): The particular Circuit or its `id`.

Returns:
    List[netdot.CableStrand]: The list of CableStrands associated to the Circuit.
```

#### Repository.get_cablestrands_by_fiber_type

> ```get_cablestrands_by_fiber_type(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of CableStrands associated to a particular FiberType.

Args:
    other (int,NetdotAPIDataclass): The particular FiberType or its `id`.

Returns:
    List[netdot.CableStrand]: The list of CableStrands associated to the FiberType.
```

#### Repository.get_cablestrands_by_status

> ```get_cablestrands_by_status(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of CableStrands associated to a particular StrandStatus.

Args:
    other (int,NetdotAPIDataclass): The particular StrandStatus or its `id`.

Returns:
    List[netdot.CableStrand]: The list of CableStrands associated to the StrandStatus.
```

#### Repository.get_cablestrands_where

> ```get_cablestrands_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.CableStrand]```

```
Get info about CableStrands from Netdot.

> NOTE: This will return ALL CableStrands from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.CableStrand]: CableStrands from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_cabletype

> ```get_cabletype(self, id: int) -> netdot.dataclasses.cables.CableType```

```
Get info about a CableType from Netdot.

Args:
    id (int): The ID of the CableType to retrieve.

Returns:
    netdot.CableType: The CableType with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the CableType cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_cabletypes_where

> ```get_cabletypes_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.CableType]```

```
Get info about CableTypes from Netdot.

> NOTE: This will return ALL CableTypes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.CableType]: CableTypes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_circuit

> ```get_circuit(self, id: int) -> netdot.dataclasses.cables.Circuit```

```
Get info about a Circuit from Netdot.

Args:
    id (int): The ID of the Circuit to retrieve.

Returns:
    netdot.Circuit: The Circuit with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Circuit cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_circuits_by_linkid

> ```get_circuits_by_linkid(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Circuits associated to a particular SiteLink.

Args:
    other (int,NetdotAPIDataclass): The particular SiteLink or its `id`.

Returns:
    List[netdot.Circuit]: The list of Circuits associated to the SiteLink.
```

#### Repository.get_circuits_by_status

> ```get_circuits_by_status(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Circuits associated to a particular CircuitStatus.

Args:
    other (int,NetdotAPIDataclass): The particular CircuitStatus or its `id`.

Returns:
    List[netdot.Circuit]: The list of Circuits associated to the CircuitStatus.
```

#### Repository.get_circuits_by_type

> ```get_circuits_by_type(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Circuits associated to a particular CircuitType.

Args:
    other (int,NetdotAPIDataclass): The particular CircuitType or its `id`.

Returns:
    List[netdot.Circuit]: The list of Circuits associated to the CircuitType.
```

#### Repository.get_circuits_by_vendor

> ```get_circuits_by_vendor(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Circuits associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.Circuit]: The list of Circuits associated to the Entity.
```

#### Repository.get_circuits_where

> ```get_circuits_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.Circuit]```

```
Get info about Circuits from Netdot.

> NOTE: This will return ALL Circuits from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Circuit]: Circuits from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_circuitstatus

> ```get_circuitstatus(self, id: int) -> netdot.dataclasses.cables.CircuitStatus```

```
Get info about a CircuitStatus from Netdot.

Args:
    id (int): The ID of the CircuitStatus to retrieve.

Returns:
    netdot.CircuitStatus: The CircuitStatus with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the CircuitStatus cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_circuitstatuses_where

> ```get_circuitstatuses_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.CircuitStatus]```

```
Get info about CircuitStatuses from Netdot.

> NOTE: This will return ALL CircuitStatuses from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.CircuitStatus]: CircuitStatuses from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_circuittype

> ```get_circuittype(self, id: int) -> netdot.dataclasses.cables.CircuitType```

```
Get info about a CircuitType from Netdot.

Args:
    id (int): The ID of the CircuitType to retrieve.

Returns:
    netdot.CircuitType: The CircuitType with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the CircuitType cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_circuittypes_where

> ```get_circuittypes_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.CircuitType]```

```
Get info about CircuitTypes from Netdot.

> NOTE: This will return ALL CircuitTypes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.CircuitType]: CircuitTypes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_closet

> ```get_closet(self, id: int) -> netdot.dataclasses.site.Closet```

```
Get info about a Closet from Netdot.

Args:
    id (int): The ID of the Closet to retrieve.

Returns:
    netdot.Closet: The Closet with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Closet cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_closets_by_room

> ```get_closets_by_room(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Closets associated to a particular Room.

Args:
    other (int,NetdotAPIDataclass): The particular Room or its `id`.

Returns:
    List[netdot.Closet]: The list of Closets associated to the Room.
```

#### Repository.get_closets_where

> ```get_closets_where(self, *args, **url_params) -> List[netdot.dataclasses.site.Closet]```

```
Get info about Closets from Netdot.

> NOTE: This will return ALL Closets from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Closet]: Closets from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_contact

> ```get_contact(self, id: int) -> netdot.dataclasses.users.Contact```

```
Get info about a Contact from Netdot.

Args:
    id (int): The ID of the Contact to retrieve.

Returns:
    netdot.Contact: The Contact with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Contact cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_contactlist

> ```get_contactlist(self, id: int) -> netdot.dataclasses.users.ContactList```

```
Get info about a ContactList from Netdot.

Args:
    id (int): The ID of the ContactList to retrieve.

Returns:
    netdot.ContactList: The ContactList with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the ContactList cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_contactlists_where

> ```get_contactlists_where(self, *args, **url_params) -> List[netdot.dataclasses.users.ContactList]```

```
Get info about ContactLists from Netdot.

> NOTE: This will return ALL ContactLists from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.ContactList]: ContactLists from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_contacts_by_contactlist

> ```get_contacts_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Contacts associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.Contact]: The list of Contacts associated to the ContactList.
```

#### Repository.get_contacts_by_contacttype

> ```get_contacts_by_contacttype(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Contacts associated to a particular ContactType.

Args:
    other (int,NetdotAPIDataclass): The particular ContactType or its `id`.

Returns:
    List[netdot.Contact]: The list of Contacts associated to the ContactType.
```

#### Repository.get_contacts_by_notify_email

> ```get_contacts_by_notify_email(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Contacts associated to a particular Availability.

Args:
    other (int,NetdotAPIDataclass): The particular Availability or its `id`.

Returns:
    List[netdot.Contact]: The list of Contacts associated to the Availability.
```

#### Repository.get_contacts_by_notify_pager

> ```get_contacts_by_notify_pager(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Contacts associated to a particular Availability.

Args:
    other (int,NetdotAPIDataclass): The particular Availability or its `id`.

Returns:
    List[netdot.Contact]: The list of Contacts associated to the Availability.
```

#### Repository.get_contacts_by_notify_voice

> ```get_contacts_by_notify_voice(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Contacts associated to a particular Availability.

Args:
    other (int,NetdotAPIDataclass): The particular Availability or its `id`.

Returns:
    List[netdot.Contact]: The list of Contacts associated to the Availability.
```

#### Repository.get_contacts_by_person

> ```get_contacts_by_person(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Contacts associated to a particular Person.

Args:
    other (int,NetdotAPIDataclass): The particular Person or its `id`.

Returns:
    List[netdot.Contact]: The list of Contacts associated to the Person.
```

#### Repository.get_contacts_where

> ```get_contacts_where(self, *args, **url_params) -> List[netdot.dataclasses.users.Contact]```

```
Get info about Contacts from Netdot.

> NOTE: This will return ALL Contacts from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Contact]: Contacts from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_contacttype

> ```get_contacttype(self, id: int) -> netdot.dataclasses.users.ContactType```

```
Get info about a ContactType from Netdot.

Args:
    id (int): The ID of the ContactType to retrieve.

Returns:
    netdot.ContactType: The ContactType with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the ContactType cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_contacttypes_where

> ```get_contacttypes_where(self, *args, **url_params) -> List[netdot.dataclasses.users.ContactType]```

```
Get info about ContactTypes from Netdot.

> NOTE: This will return ALL ContactTypes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.ContactType]: ContactTypes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_device

> ```get_device(self, id: int) -> netdot.dataclasses.device.Device```

```
Get info about a Device from Netdot.

Args:
    id (int): The ID of the Device to retrieve.

Returns:
    netdot.Device: The Device with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Device cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_deviceattr

> ```get_deviceattr(self, id: int) -> netdot.dataclasses.device.DeviceAttr```

```
Get info about a DeviceAttr from Netdot.

Args:
    id (int): The ID of the DeviceAttr to retrieve.

Returns:
    netdot.DeviceAttr: The DeviceAttr with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DeviceAttr cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_deviceattrname

> ```get_deviceattrname(self, id: int) -> netdot.dataclasses.device.DeviceAttrName```

```
Get info about a DeviceAttrName from Netdot.

Args:
    id (int): The ID of the DeviceAttrName to retrieve.

Returns:
    netdot.DeviceAttrName: The DeviceAttrName with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DeviceAttrName cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_deviceattrnames_where

> ```get_deviceattrnames_where(self, *args, **url_params) -> List[netdot.dataclasses.device.DeviceAttrName]```

```
Get info about DeviceAttrNames from Netdot.

> NOTE: This will return ALL DeviceAttrNames from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DeviceAttrName]: DeviceAttrNames from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_deviceattrs_by_device

> ```get_deviceattrs_by_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DeviceAttrs associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.DeviceAttr]: The list of DeviceAttrs associated to the Device.
```

#### Repository.get_deviceattrs_by_name

> ```get_deviceattrs_by_name(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DeviceAttrs associated to a particular DeviceAttrName.

Args:
    other (int,NetdotAPIDataclass): The particular DeviceAttrName or its `id`.

Returns:
    List[netdot.DeviceAttr]: The list of DeviceAttrs associated to the DeviceAttrName.
```

#### Repository.get_deviceattrs_where

> ```get_deviceattrs_where(self, *args, **url_params) -> List[netdot.dataclasses.device.DeviceAttr]```

```
Get info about DeviceAttrs from Netdot.

> NOTE: This will return ALL DeviceAttrs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DeviceAttr]: DeviceAttrs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_devicecontacts

> ```get_devicecontacts(self, id: int) -> netdot.dataclasses.device.DeviceContacts```

```
Get info about a DeviceContacts from Netdot.

Args:
    id (int): The ID of the DeviceContacts to retrieve.

Returns:
    netdot.DeviceContacts: The DeviceContacts with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DeviceContacts cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_devicecontacts_by_contactlist

> ```get_devicecontacts_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DeviceContacts associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.DeviceContacts]: The list of DeviceContacts associated to the ContactList.
```

#### Repository.get_devicecontacts_by_device

> ```get_devicecontacts_by_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DeviceContacts associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.DeviceContacts]: The list of DeviceContacts associated to the Device.
```

#### Repository.get_devicecontacts_where

> ```get_devicecontacts_where(self, *args, **url_params) -> List[netdot.dataclasses.device.DeviceContacts]```

```
Get info about DeviceContacts from Netdot.

> NOTE: This will return ALL DeviceContacts from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DeviceContacts]: DeviceContacts from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_devicemodule

> ```get_devicemodule(self, id: int) -> netdot.dataclasses.device.DeviceModule```

```
Get info about a DeviceModule from Netdot.

Args:
    id (int): The ID of the DeviceModule to retrieve.

Returns:
    netdot.DeviceModule: The DeviceModule with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DeviceModule cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_devicemodules_by_asset_id

> ```get_devicemodules_by_asset_id(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DeviceModules associated to a particular Asset.

Args:
    other (int,NetdotAPIDataclass): The particular Asset or its `id`.

Returns:
    List[netdot.DeviceModule]: The list of DeviceModules associated to the Asset.
```

#### Repository.get_devicemodules_by_device

> ```get_devicemodules_by_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DeviceModules associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.DeviceModule]: The list of DeviceModules associated to the Device.
```

#### Repository.get_devicemodules_where

> ```get_devicemodules_where(self, *args, **url_params) -> List[netdot.dataclasses.device.DeviceModule]```

```
Get info about DeviceModules from Netdot.

> NOTE: This will return ALL DeviceModules from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DeviceModule]: DeviceModules from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_devices_by_asset_id

> ```get_devices_by_asset_id(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular Asset.

Args:
    other (int,NetdotAPIDataclass): The particular Asset or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the Asset.
```

#### Repository.get_devices_by_bgplocalas

> ```get_devices_by_bgplocalas(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular ASN.

Args:
    other (int,NetdotAPIDataclass): The particular ASN or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the ASN.
```

#### Repository.get_devices_by_host_device

> ```get_devices_by_host_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the Device.
```

#### Repository.get_devices_by_monitorstatus

> ```get_devices_by_monitorstatus(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular MonitorStatus.

Args:
    other (int,NetdotAPIDataclass): The particular MonitorStatus or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the MonitorStatus.
```

#### Repository.get_devices_by_name

> ```get_devices_by_name(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the RR.
```

#### Repository.get_devices_by_owner

> ```get_devices_by_owner(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the Entity.
```

#### Repository.get_devices_by_room

> ```get_devices_by_room(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular Room.

Args:
    other (int,NetdotAPIDataclass): The particular Room or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the Room.
```

#### Repository.get_devices_by_site

> ```get_devices_by_site(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular Site.

Args:
    other (int,NetdotAPIDataclass): The particular Site or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the Site.
```

#### Repository.get_devices_by_snmp_target

> ```get_devices_by_snmp_target(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the IPBlock.
```

#### Repository.get_devices_by_used_by

> ```get_devices_by_used_by(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Devices associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.Device]: The list of Devices associated to the Entity.
```

#### Repository.get_devices_where

> ```get_devices_where(self, *args, **url_params) -> List[netdot.dataclasses.device.Device]```

```
Get info about Devices from Netdot.

> NOTE: This will return ALL Devices from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Device]: Devices from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpattr

> ```get_dhcpattr(self, id: int) -> netdot.dataclasses.dhcp.DHCPAttr```

```
Get info about a DHCPAttr from Netdot.

Args:
    id (int): The ID of the DHCPAttr to retrieve.

Returns:
    netdot.DHCPAttr: The DHCPAttr with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DHCPAttr cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpattrname

> ```get_dhcpattrname(self, id: int) -> netdot.dataclasses.dhcp.DHCPAttrName```

```
Get info about a DHCPAttrName from Netdot.

Args:
    id (int): The ID of the DHCPAttrName to retrieve.

Returns:
    netdot.DHCPAttrName: The DHCPAttrName with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DHCPAttrName cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpattrnames_where

> ```get_dhcpattrnames_where(self, *args, **url_params) -> List[netdot.dataclasses.dhcp.DHCPAttrName]```

```
Get info about DHCPAttrNames from Netdot.

> NOTE: This will return ALL DHCPAttrNames from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DHCPAttrName]: DHCPAttrNames from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpattrs_by_name

> ```get_dhcpattrs_by_name(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DHCPAttrs associated to a particular DHCPAttrName.

Args:
    other (int,NetdotAPIDataclass): The particular DHCPAttrName or its `id`.

Returns:
    List[netdot.DHCPAttr]: The list of DHCPAttrs associated to the DHCPAttrName.
```

#### Repository.get_dhcpattrs_by_scope

> ```get_dhcpattrs_by_scope(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DHCPAttrs associated to a particular DHCPScope.

Args:
    other (int,NetdotAPIDataclass): The particular DHCPScope or its `id`.

Returns:
    List[netdot.DHCPAttr]: The list of DHCPAttrs associated to the DHCPScope.
```

#### Repository.get_dhcpattrs_where

> ```get_dhcpattrs_where(self, *args, **url_params) -> List[netdot.dataclasses.dhcp.DHCPAttr]```

```
Get info about DHCPAttrs from Netdot.

> NOTE: This will return ALL DHCPAttrs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DHCPAttr]: DHCPAttrs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpscope

> ```get_dhcpscope(self, id: int) -> netdot.dataclasses.dhcp.DHCPScope```

```
Get info about a DHCPScope from Netdot.

Args:
    id (int): The ID of the DHCPScope to retrieve.

Returns:
    netdot.DHCPScope: The DHCPScope with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DHCPScope cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpscopes_by_container

> ```get_dhcpscopes_by_container(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DHCPScopes associated to a particular DHCPScope.

Args:
    other (int,NetdotAPIDataclass): The particular DHCPScope or its `id`.

Returns:
    List[netdot.DHCPScope]: The list of DHCPScopes associated to the DHCPScope.
```

#### Repository.get_dhcpscopes_by_ipblock

> ```get_dhcpscopes_by_ipblock(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DHCPScopes associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.DHCPScope]: The list of DHCPScopes associated to the IPBlock.
```

#### Repository.get_dhcpscopes_by_physaddr

> ```get_dhcpscopes_by_physaddr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DHCPScopes associated to a particular PhysAddr.

Args:
    other (int,NetdotAPIDataclass): The particular PhysAddr or its `id`.

Returns:
    List[netdot.DHCPScope]: The list of DHCPScopes associated to the PhysAddr.
```

#### Repository.get_dhcpscopes_by_type

> ```get_dhcpscopes_by_type(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DHCPScopes associated to a particular DHCPScopeType.

Args:
    other (int,NetdotAPIDataclass): The particular DHCPScopeType or its `id`.

Returns:
    List[netdot.DHCPScope]: The list of DHCPScopes associated to the DHCPScopeType.
```

#### Repository.get_dhcpscopes_where

> ```get_dhcpscopes_where(self, *args, **url_params) -> List[netdot.dataclasses.dhcp.DHCPScope]```

```
Get info about DHCPScopes from Netdot.

> NOTE: This will return ALL DHCPScopes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DHCPScope]: DHCPScopes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpscopetype

> ```get_dhcpscopetype(self, id: int) -> netdot.dataclasses.dhcp.DHCPScopeType```

```
Get info about a DHCPScopeType from Netdot.

Args:
    id (int): The ID of the DHCPScopeType to retrieve.

Returns:
    netdot.DHCPScopeType: The DHCPScopeType with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DHCPScopeType cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpscopetypes_where

> ```get_dhcpscopetypes_where(self, *args, **url_params) -> List[netdot.dataclasses.dhcp.DHCPScopeType]```

```
Get info about DHCPScopeTypes from Netdot.

> NOTE: This will return ALL DHCPScopeTypes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DHCPScopeType]: DHCPScopeTypes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpscopeuse

> ```get_dhcpscopeuse(self, id: int) -> netdot.dataclasses.dhcp.DHCPScopeUse```

```
Get info about a DHCPScopeUse from Netdot.

Args:
    id (int): The ID of the DHCPScopeUse to retrieve.

Returns:
    netdot.DHCPScopeUse: The DHCPScopeUse with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the DHCPScopeUse cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_dhcpscopeuses_by_scope

> ```get_dhcpscopeuses_by_scope(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DHCPScopeUses associated to a particular DHCPScope.

Args:
    other (int,NetdotAPIDataclass): The particular DHCPScope or its `id`.

Returns:
    List[netdot.DHCPScopeUse]: The list of DHCPScopeUses associated to the DHCPScope.
```

#### Repository.get_dhcpscopeuses_by_template

> ```get_dhcpscopeuses_by_template(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of DHCPScopeUses associated to a particular DHCPScope.

Args:
    other (int,NetdotAPIDataclass): The particular DHCPScope or its `id`.

Returns:
    List[netdot.DHCPScopeUse]: The list of DHCPScopeUses associated to the DHCPScope.
```

#### Repository.get_dhcpscopeuses_where

> ```get_dhcpscopeuses_where(self, *args, **url_params) -> List[netdot.dataclasses.dhcp.DHCPScopeUse]```

```
Get info about DHCPScopeUses from Netdot.

> NOTE: This will return ALL DHCPScopeUses from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.DHCPScopeUse]: DHCPScopeUses from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_entities_by_availability

> ```get_entities_by_availability(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Entities associated to a particular Availability.

Args:
    other (int,NetdotAPIDataclass): The particular Availability or its `id`.

Returns:
    List[netdot.Entity]: The list of Entities associated to the Availability.
```

#### Repository.get_entities_by_contactlist

> ```get_entities_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Entities associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.Entity]: The list of Entities associated to the ContactList.
```

#### Repository.get_entities_where

> ```get_entities_where(self, *args, **url_params) -> List[netdot.dataclasses.entity.Entity]```

```
Get info about Entities from Netdot.

> NOTE: This will return ALL Entities from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Entity]: Entities from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_entity

> ```get_entity(self, id: int) -> netdot.dataclasses.entity.Entity```

```
Get info about a Entity from Netdot.

Args:
    id (int): The ID of the Entity to retrieve.

Returns:
    netdot.Entity: The Entity with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Entity cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_entityrole

> ```get_entityrole(self, id: int) -> netdot.dataclasses.entity.EntityRole```

```
Get info about a EntityRole from Netdot.

Args:
    id (int): The ID of the EntityRole to retrieve.

Returns:
    netdot.EntityRole: The EntityRole with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the EntityRole cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_entityroles_by_entity

> ```get_entityroles_by_entity(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of EntityRoles associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.EntityRole]: The list of EntityRoles associated to the Entity.
```

#### Repository.get_entityroles_by_type

> ```get_entityroles_by_type(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of EntityRoles associated to a particular EntityType.

Args:
    other (int,NetdotAPIDataclass): The particular EntityType or its `id`.

Returns:
    List[netdot.EntityRole]: The list of EntityRoles associated to the EntityType.
```

#### Repository.get_entityroles_where

> ```get_entityroles_where(self, *args, **url_params) -> List[netdot.dataclasses.entity.EntityRole]```

```
Get info about EntityRoles from Netdot.

> NOTE: This will return ALL EntityRoles from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.EntityRole]: EntityRoles from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_entitysite

> ```get_entitysite(self, id: int) -> netdot.dataclasses.entity.EntitySite```

```
Get info about a EntitySite from Netdot.

Args:
    id (int): The ID of the EntitySite to retrieve.

Returns:
    netdot.EntitySite: The EntitySite with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the EntitySite cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_entitysites_by_entity

> ```get_entitysites_by_entity(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of EntitySites associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.EntitySite]: The list of EntitySites associated to the Entity.
```

#### Repository.get_entitysites_by_site

> ```get_entitysites_by_site(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of EntitySites associated to a particular Site.

Args:
    other (int,NetdotAPIDataclass): The particular Site or its `id`.

Returns:
    List[netdot.EntitySite]: The list of EntitySites associated to the Site.
```

#### Repository.get_entitysites_where

> ```get_entitysites_where(self, *args, **url_params) -> List[netdot.dataclasses.entity.EntitySite]```

```
Get info about EntitySites from Netdot.

> NOTE: This will return ALL EntitySites from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.EntitySite]: EntitySites from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_entitytype

> ```get_entitytype(self, id: int) -> netdot.dataclasses.entity.EntityType```

```
Get info about a EntityType from Netdot.

Args:
    id (int): The ID of the EntityType to retrieve.

Returns:
    netdot.EntityType: The EntityType with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the EntityType cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_entitytypes_where

> ```get_entitytypes_where(self, *args, **url_params) -> List[netdot.dataclasses.entity.EntityType]```

```
Get info about EntityTypes from Netdot.

> NOTE: This will return ALL EntityTypes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.EntityType]: EntityTypes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_fibertype

> ```get_fibertype(self, id: int) -> netdot.dataclasses.cables.FiberType```

```
Get info about a FiberType from Netdot.

Args:
    id (int): The ID of the FiberType to retrieve.

Returns:
    netdot.FiberType: The FiberType with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the FiberType cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_fibertypes_where

> ```get_fibertypes_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.FiberType]```

```
Get info about FiberTypes from Netdot.

> NOTE: This will return ALL FiberTypes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.FiberType]: FiberTypes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_floor

> ```get_floor(self, id: int) -> netdot.dataclasses.site.Floor```

```
Get info about a Floor from Netdot.

Args:
    id (int): The ID of the Floor to retrieve.

Returns:
    netdot.Floor: The Floor with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Floor cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_floors_by_site

> ```get_floors_by_site(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Floors associated to a particular Site.

Args:
    other (int,NetdotAPIDataclass): The particular Site or its `id`.

Returns:
    List[netdot.Floor]: The list of Floors associated to the Site.
```

#### Repository.get_floors_where

> ```get_floors_where(self, *args, **url_params) -> List[netdot.dataclasses.site.Floor]```

```
Get info about Floors from Netdot.

> NOTE: This will return ALL Floors from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Floor]: Floors from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_fwtable

> ```get_fwtable(self, id: int) -> netdot.dataclasses.fwtable.FWTable```

```
Get info about a FWTable from Netdot.

Args:
    id (int): The ID of the FWTable to retrieve.

Returns:
    netdot.FWTable: The FWTable with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the FWTable cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_fwtableentries_by_fwtable

> ```get_fwtableentries_by_fwtable(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of FWTableEntries associated to a particular FWTable.

Args:
    other (int,NetdotAPIDataclass): The particular FWTable or its `id`.

Returns:
    List[netdot.FWTableEntry]: The list of FWTableEntries associated to the FWTable.
```

#### Repository.get_fwtableentries_by_interface

> ```get_fwtableentries_by_interface(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of FWTableEntries associated to a particular Interface.

Args:
    other (int,NetdotAPIDataclass): The particular Interface or its `id`.

Returns:
    List[netdot.FWTableEntry]: The list of FWTableEntries associated to the Interface.
```

#### Repository.get_fwtableentries_by_physaddr

> ```get_fwtableentries_by_physaddr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of FWTableEntries associated to a particular PhysAddr.

Args:
    other (int,NetdotAPIDataclass): The particular PhysAddr or its `id`.

Returns:
    List[netdot.FWTableEntry]: The list of FWTableEntries associated to the PhysAddr.
```

#### Repository.get_fwtableentries_where

> ```get_fwtableentries_where(self, *args, **url_params) -> List[netdot.dataclasses.fwtable.FWTableEntry]```

```
Get info about FWTableEntries from Netdot.

> NOTE: This will return ALL FWTableEntries from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.FWTableEntry]: FWTableEntries from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_fwtableentry

> ```get_fwtableentry(self, id: int) -> netdot.dataclasses.fwtable.FWTableEntry```

```
Get info about a FWTableEntry from Netdot.

Args:
    id (int): The ID of the FWTableEntry to retrieve.

Returns:
    netdot.FWTableEntry: The FWTableEntry with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the FWTableEntry cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_fwtables_by_device

> ```get_fwtables_by_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of FWTables associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.FWTable]: The list of FWTables associated to the Device.
```

#### Repository.get_fwtables_where

> ```get_fwtables_where(self, *args, **url_params) -> List[netdot.dataclasses.fwtable.FWTable]```

```
Get info about FWTables from Netdot.

> NOTE: This will return ALL FWTables from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.FWTable]: FWTables from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_groupright

> ```get_groupright(self, id: int) -> netdot.dataclasses.users.GroupRight```

```
Get info about a GroupRight from Netdot.

Args:
    id (int): The ID of the GroupRight to retrieve.

Returns:
    netdot.GroupRight: The GroupRight with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the GroupRight cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_grouprights_by_accessright

> ```get_grouprights_by_accessright(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of GroupRights associated to a particular AccessRight.

Args:
    other (int,NetdotAPIDataclass): The particular AccessRight or its `id`.

Returns:
    List[netdot.GroupRight]: The list of GroupRights associated to the AccessRight.
```

#### Repository.get_grouprights_by_contactlist

> ```get_grouprights_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of GroupRights associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.GroupRight]: The list of GroupRights associated to the ContactList.
```

#### Repository.get_grouprights_where

> ```get_grouprights_where(self, *args, **url_params) -> List[netdot.dataclasses.users.GroupRight]```

```
Get info about GroupRights from Netdot.

> NOTE: This will return ALL GroupRights from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.GroupRight]: GroupRights from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_horizontalcable

> ```get_horizontalcable(self, id: int) -> netdot.dataclasses.cables.HorizontalCable```

```
Get info about a HorizontalCable from Netdot.

Args:
    id (int): The ID of the HorizontalCable to retrieve.

Returns:
    netdot.HorizontalCable: The HorizontalCable with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the HorizontalCable cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_horizontalcables_by_closet

> ```get_horizontalcables_by_closet(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of HorizontalCables associated to a particular Closet.

Args:
    other (int,NetdotAPIDataclass): The particular Closet or its `id`.

Returns:
    List[netdot.HorizontalCable]: The list of HorizontalCables associated to the Closet.
```

#### Repository.get_horizontalcables_by_contactlist

> ```get_horizontalcables_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of HorizontalCables associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.HorizontalCable]: The list of HorizontalCables associated to the ContactList.
```

#### Repository.get_horizontalcables_by_room

> ```get_horizontalcables_by_room(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of HorizontalCables associated to a particular Room.

Args:
    other (int,NetdotAPIDataclass): The particular Room or its `id`.

Returns:
    List[netdot.HorizontalCable]: The list of HorizontalCables associated to the Room.
```

#### Repository.get_horizontalcables_by_type

> ```get_horizontalcables_by_type(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of HorizontalCables associated to a particular CableType.

Args:
    other (int,NetdotAPIDataclass): The particular CableType or its `id`.

Returns:
    List[netdot.HorizontalCable]: The list of HorizontalCables associated to the CableType.
```

#### Repository.get_horizontalcables_where

> ```get_horizontalcables_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.HorizontalCable]```

```
Get info about HorizontalCables from Netdot.

> NOTE: This will return ALL HorizontalCables from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.HorizontalCable]: HorizontalCables from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_host_by_name

> ```get_host_by_name(self, hostname: str) -> 'netdot.Host'```

```
Retrieve DNS Resource Records with name "hostname", and related IP Block records.

NOTE: This is part of, "The special resource '/rest/host' provides a simplified interface for manipulating DNS and DHCP records."

Args:
    hostname (str): The hostname to lookup, e.g. "myhost.example.com"
Returns:
    netdot.Host: A collection of DNS Resource Records and related IP Blocks for the given hostname.
```

#### Repository.get_hostaudit

> ```get_hostaudit(self, id: int) -> netdot.dataclasses.misc.HostAudit```

```
Get info about a HostAudit from Netdot.

Args:
    id (int): The ID of the HostAudit to retrieve.

Returns:
    netdot.HostAudit: The HostAudit with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the HostAudit cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_hostaudits_where

> ```get_hostaudits_where(self, *args, **url_params) -> List[netdot.dataclasses.misc.HostAudit]```

```
Get info about HostAudits from Netdot.

> NOTE: This will return ALL HostAudits from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.HostAudit]: HostAudits from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_hosts

> ```get_hosts(self, **url_params) -> 'netdot.Host'```

```
Retrieve relevant RR (DNS) objects Hosts from Netdot.

Args:
    **url_params: URL parameters to filter the results. See examples below.

Returns:
    netdot.Host: A collection of DNS Resource Records and related IP Blocks.

Examples:
    >> repo.get_hosts(zone='uoregon.edu')
    >> repo.get_hosts(name='myhost')
    >> repo.get_hosts(rrid=1234)
    >> repo.get_hosts(subnet='10.1.1.1/24')
```

#### Repository.get_hosts_by_subnet

> ```get_hosts_by_subnet(self, subnet: str) -> 'netdot.Host'```

```
Retrieve all Hosts (DNS Resource Records) within a given subnet.

NOTE: This is part of "The special resource '/rest/host' provides a simplified interface for manipulating DNS and DHCP records."

Args:
    subnet (str): The subnet to lookup, e.g. "10.1.1.1/24"

Returns:
    List[netdot.Host]: A list of Hosts (DNS Resource Records) within the given subnet.
```

#### Repository.get_interface

> ```get_interface(self, id: int) -> netdot.dataclasses.interface.Interface```

```
Get info about a Interface from Netdot.

Args:
    id (int): The ID of the Interface to retrieve.

Returns:
    netdot.Interface: The Interface with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Interface cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_interfaces_by_circuit

> ```get_interfaces_by_circuit(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Interfaces associated to a particular Circuit.

Args:
    other (int,NetdotAPIDataclass): The particular Circuit or its `id`.

Returns:
    List[netdot.Interface]: The list of Interfaces associated to the Circuit.
```

#### Repository.get_interfaces_by_contactlist

> ```get_interfaces_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Interfaces associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.Interface]: The list of Interfaces associated to the ContactList.
```

#### Repository.get_interfaces_by_device

> ```get_interfaces_by_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Interfaces associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.Interface]: The list of Interfaces associated to the Device.
```

#### Repository.get_interfaces_by_jack

> ```get_interfaces_by_jack(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Interfaces associated to a particular HorizontalCable.

Args:
    other (int,NetdotAPIDataclass): The particular HorizontalCable or its `id`.

Returns:
    List[netdot.Interface]: The list of Interfaces associated to the HorizontalCable.
```

#### Repository.get_interfaces_by_monitorstatus

> ```get_interfaces_by_monitorstatus(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Interfaces associated to a particular MonitorStatus.

Args:
    other (int,NetdotAPIDataclass): The particular MonitorStatus or its `id`.

Returns:
    List[netdot.Interface]: The list of Interfaces associated to the MonitorStatus.
```

#### Repository.get_interfaces_by_neighbor

> ```get_interfaces_by_neighbor(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Interfaces associated to a particular Interface.

Args:
    other (int,NetdotAPIDataclass): The particular Interface or its `id`.

Returns:
    List[netdot.Interface]: The list of Interfaces associated to the Interface.
```

#### Repository.get_interfaces_by_physaddr

> ```get_interfaces_by_physaddr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Interfaces associated to a particular PhysAddr.

Args:
    other (int,NetdotAPIDataclass): The particular PhysAddr or its `id`.

Returns:
    List[netdot.Interface]: The list of Interfaces associated to the PhysAddr.
```

#### Repository.get_interfaces_where

> ```get_interfaces_where(self, *args, **url_params) -> List[netdot.dataclasses.interface.Interface]```

```
Get info about Interfaces from Netdot.

> NOTE: This will return ALL Interfaces from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Interface]: Interfaces from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_interfacevlan

> ```get_interfacevlan(self, id: int) -> netdot.dataclasses.interface.InterfaceVLAN```

```
Get info about a InterfaceVLAN from Netdot.

Args:
    id (int): The ID of the InterfaceVLAN to retrieve.

Returns:
    netdot.InterfaceVLAN: The InterfaceVLAN with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the InterfaceVLAN cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_interfacevlans_by_interface

> ```get_interfacevlans_by_interface(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of InterfaceVLANs associated to a particular Interface.

Args:
    other (int,NetdotAPIDataclass): The particular Interface or its `id`.

Returns:
    List[netdot.InterfaceVLAN]: The list of InterfaceVLANs associated to the Interface.
```

#### Repository.get_interfacevlans_by_stp_instance

> ```get_interfacevlans_by_stp_instance(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of InterfaceVLANs associated to a particular STPInstance.

Args:
    other (int,NetdotAPIDataclass): The particular STPInstance or its `id`.

Returns:
    List[netdot.InterfaceVLAN]: The list of InterfaceVLANs associated to the STPInstance.
```

#### Repository.get_interfacevlans_by_vlan

> ```get_interfacevlans_by_vlan(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of InterfaceVLANs associated to a particular VLAN.

Args:
    other (int,NetdotAPIDataclass): The particular VLAN or its `id`.

Returns:
    List[netdot.InterfaceVLAN]: The list of InterfaceVLANs associated to the VLAN.
```

#### Repository.get_interfacevlans_where

> ```get_interfacevlans_where(self, *args, **url_params) -> List[netdot.dataclasses.interface.InterfaceVLAN]```

```
Get info about InterfaceVLANs from Netdot.

> NOTE: This will return ALL InterfaceVLANs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.InterfaceVLAN]: InterfaceVLANs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipblock

> ```get_ipblock(self, id: int) -> netdot.dataclasses.ipblock.IPBlock```

```
Get info about a IPBlock from Netdot.

Args:
    id (int): The ID of the IPBlock to retrieve.

Returns:
    netdot.IPBlock: The IPBlock with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the IPBlock cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipblock_by_address

> ```get_ipblock_by_address(self, address: str) -> 'netdot.IPBlock'```

```
Get some IP Block from Netdot Address Space.

Args:
    address (str): The IP Address to lookup, e.g. "10.0.0.0"

Returns:
    IPBlock: The IPBlock object that matched the provided address, or None.
```

#### Repository.get_ipblock_children

> ```get_ipblock_children(self, id: int, **url_params) -> List[ForwardRef('netdot.IPBlock')]```

```
Get the children of some parent IPBlock.

Args:
    id (int): The ID of the parent IPBlock.

Returns:
    List[netdot.IPBlock]: The children IPBlocks

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: If no such IPBlocks are found. (error details can be found in Netdot's apache server logs)
```

#### Repository.get_ipblockattr

> ```get_ipblockattr(self, id: int) -> netdot.dataclasses.ipblock.IPBlockAttr```

```
Get info about a IPBlockAttr from Netdot.

Args:
    id (int): The ID of the IPBlockAttr to retrieve.

Returns:
    netdot.IPBlockAttr: The IPBlockAttr with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the IPBlockAttr cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipblockattrname

> ```get_ipblockattrname(self, id: int) -> netdot.dataclasses.ipblock.IPBlockAttrName```

```
Get info about a IPBlockAttrName from Netdot.

Args:
    id (int): The ID of the IPBlockAttrName to retrieve.

Returns:
    netdot.IPBlockAttrName: The IPBlockAttrName with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the IPBlockAttrName cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipblockattrnames_where

> ```get_ipblockattrnames_where(self, *args, **url_params) -> List[netdot.dataclasses.ipblock.IPBlockAttrName]```

```
Get info about IPBlockAttrNames from Netdot.

> NOTE: This will return ALL IPBlockAttrNames from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.IPBlockAttrName]: IPBlockAttrNames from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipblockattrs_by_ipblock

> ```get_ipblockattrs_by_ipblock(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlockAttrs associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.IPBlockAttr]: The list of IPBlockAttrs associated to the IPBlock.
```

#### Repository.get_ipblockattrs_by_name

> ```get_ipblockattrs_by_name(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlockAttrs associated to a particular IPBlockAttrName.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlockAttrName or its `id`.

Returns:
    List[netdot.IPBlockAttr]: The list of IPBlockAttrs associated to the IPBlockAttrName.
```

#### Repository.get_ipblockattrs_where

> ```get_ipblockattrs_where(self, *args, **url_params) -> List[netdot.dataclasses.ipblock.IPBlockAttr]```

```
Get info about IPBlockAttrs from Netdot.

> NOTE: This will return ALL IPBlockAttrs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.IPBlockAttr]: IPBlockAttrs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipblocks_by_asn

> ```get_ipblocks_by_asn(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlocks associated to a particular ASN.

Args:
    other (int,NetdotAPIDataclass): The particular ASN or its `id`.

Returns:
    List[netdot.IPBlock]: The list of IPBlocks associated to the ASN.
```

#### Repository.get_ipblocks_by_interface

> ```get_ipblocks_by_interface(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlocks associated to a particular Interface.

Args:
    other (int,NetdotAPIDataclass): The particular Interface or its `id`.

Returns:
    List[netdot.IPBlock]: The list of IPBlocks associated to the Interface.
```

#### Repository.get_ipblocks_by_owner

> ```get_ipblocks_by_owner(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlocks associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.IPBlock]: The list of IPBlocks associated to the Entity.
```

#### Repository.get_ipblocks_by_parent

> ```get_ipblocks_by_parent(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlocks associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.IPBlock]: The list of IPBlocks associated to the IPBlock.
```

#### Repository.get_ipblocks_by_status

> ```get_ipblocks_by_status(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlocks associated to a particular IPBlockStatus.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlockStatus or its `id`.

Returns:
    List[netdot.IPBlock]: The list of IPBlocks associated to the IPBlockStatus.
```

#### Repository.get_ipblocks_by_used_by

> ```get_ipblocks_by_used_by(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlocks associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.IPBlock]: The list of IPBlocks associated to the Entity.
```

#### Repository.get_ipblocks_by_vlan

> ```get_ipblocks_by_vlan(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPBlocks associated to a particular VLAN.

Args:
    other (int,NetdotAPIDataclass): The particular VLAN or its `id`.

Returns:
    List[netdot.IPBlock]: The list of IPBlocks associated to the VLAN.
```

#### Repository.get_ipblocks_where

> ```get_ipblocks_where(self, *args, **url_params) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Get info about IPBlocks from Netdot.

> NOTE: This will return ALL IPBlocks from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.IPBlock]: IPBlocks from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipblockstatus

> ```get_ipblockstatus(self, id: int) -> netdot.dataclasses.ipblock.IPBlockStatus```

```
Get info about a IPBlockStatus from Netdot.

Args:
    id (int): The ID of the IPBlockStatus to retrieve.

Returns:
    netdot.IPBlockStatus: The IPBlockStatus with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the IPBlockStatus cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipblockstatuses_where

> ```get_ipblockstatuses_where(self, *args, **url_params) -> List[netdot.dataclasses.ipblock.IPBlockStatus]```

```
Get info about IPBlockStatuses from Netdot.

> NOTE: This will return ALL IPBlockStatuses from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.IPBlockStatus]: IPBlockStatuses from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipservice

> ```get_ipservice(self, id: int) -> netdot.dataclasses.ipblock.IPService```

```
Get info about a IPService from Netdot.

Args:
    id (int): The ID of the IPService to retrieve.

Returns:
    netdot.IPService: The IPService with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the IPService cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ipservices_by_contactlist

> ```get_ipservices_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPServices associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.IPService]: The list of IPServices associated to the ContactList.
```

#### Repository.get_ipservices_by_ip

> ```get_ipservices_by_ip(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPServices associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.IPService]: The list of IPServices associated to the IPBlock.
```

#### Repository.get_ipservices_by_monitorstatus

> ```get_ipservices_by_monitorstatus(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPServices associated to a particular MonitorStatus.

Args:
    other (int,NetdotAPIDataclass): The particular MonitorStatus or its `id`.

Returns:
    List[netdot.IPService]: The list of IPServices associated to the MonitorStatus.
```

#### Repository.get_ipservices_by_service

> ```get_ipservices_by_service(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of IPServices associated to a particular Service.

Args:
    other (int,NetdotAPIDataclass): The particular Service or its `id`.

Returns:
    List[netdot.IPService]: The list of IPServices associated to the Service.
```

#### Repository.get_ipservices_where

> ```get_ipservices_where(self, *args, **url_params) -> List[netdot.dataclasses.ipblock.IPService]```

```
Get info about IPServices from Netdot.

> NOTE: This will return ALL IPServices from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.IPService]: IPServices from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_maintcontract

> ```get_maintcontract(self, id: int) -> netdot.dataclasses.misc.MaintContract```

```
Get info about a MaintContract from Netdot.

Args:
    id (int): The ID of the MaintContract to retrieve.

Returns:
    netdot.MaintContract: The MaintContract with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the MaintContract cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_maintcontracts_by_provider

> ```get_maintcontracts_by_provider(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of MaintContracts associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.MaintContract]: The list of MaintContracts associated to the Entity.
```

#### Repository.get_maintcontracts_where

> ```get_maintcontracts_where(self, *args, **url_params) -> List[netdot.dataclasses.misc.MaintContract]```

```
Get info about MaintContracts from Netdot.

> NOTE: This will return ALL MaintContracts from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.MaintContract]: MaintContracts from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_monitorstatus

> ```get_monitorstatus(self, id: int) -> netdot.dataclasses.misc.MonitorStatus```

```
Get info about a MonitorStatus from Netdot.

Args:
    id (int): The ID of the MonitorStatus to retrieve.

Returns:
    netdot.MonitorStatus: The MonitorStatus with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the MonitorStatus cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_monitorstatuses_where

> ```get_monitorstatuses_where(self, *args, **url_params) -> List[netdot.dataclasses.misc.MonitorStatus]```

```
Get info about MonitorStatuses from Netdot.

> NOTE: This will return ALL MonitorStatuses from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.MonitorStatus]: MonitorStatuses from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_oui

> ```get_oui(self, id: int) -> netdot.dataclasses.device.OUI```

```
Get info about a OUI from Netdot.

Args:
    id (int): The ID of the OUI to retrieve.

Returns:
    netdot.OUI: The OUI with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the OUI cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_ouis_where

> ```get_ouis_where(self, *args, **url_params) -> List[netdot.dataclasses.device.OUI]```

```
Get info about OUIs from Netdot.

> NOTE: This will return ALL OUIs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.OUI]: OUIs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_person

> ```get_person(self, id: int) -> netdot.dataclasses.users.Person```

```
Get info about a Person from Netdot.

Args:
    id (int): The ID of the Person to retrieve.

Returns:
    netdot.Person: The Person with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Person cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_persons_by_entity

> ```get_persons_by_entity(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Persons associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.Person]: The list of Persons associated to the Entity.
```

#### Repository.get_persons_by_location

> ```get_persons_by_location(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Persons associated to a particular Site.

Args:
    other (int,NetdotAPIDataclass): The particular Site or its `id`.

Returns:
    List[netdot.Person]: The list of Persons associated to the Site.
```

#### Repository.get_persons_by_room

> ```get_persons_by_room(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Persons associated to a particular Room.

Args:
    other (int,NetdotAPIDataclass): The particular Room or its `id`.

Returns:
    List[netdot.Person]: The list of Persons associated to the Room.
```

#### Repository.get_persons_by_user_type

> ```get_persons_by_user_type(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Persons associated to a particular UserType.

Args:
    other (int,NetdotAPIDataclass): The particular UserType or its `id`.

Returns:
    List[netdot.Person]: The list of Persons associated to the UserType.
```

#### Repository.get_persons_where

> ```get_persons_where(self, *args, **url_params) -> List[netdot.dataclasses.users.Person]```

```
Get info about Persons from Netdot.

> NOTE: This will return ALL Persons from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Person]: Persons from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_physaddr

> ```get_physaddr(self, id: int) -> netdot.dataclasses.physaddr.PhysAddr```

```
Get info about a PhysAddr from Netdot.

Args:
    id (int): The ID of the PhysAddr to retrieve.

Returns:
    netdot.PhysAddr: The PhysAddr with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the PhysAddr cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_physaddr_by_MACAddress

> ```get_physaddr_by_MACAddress(self, address: str) -> 'netdot.PhysAddr'```

```
Get some Physical Address from Netdot Address Space.

Args:
    address (str): The MAC Address to lookup, e.g. "00:00:00:00:00:00"

Returns:
    netdot.PhysAddr: The Physical Address object that matched the provided address, or None.
```

#### Repository.get_physaddrattr

> ```get_physaddrattr(self, id: int) -> netdot.dataclasses.physaddr.PhysAddrAttr```

```
Get info about a PhysAddrAttr from Netdot.

Args:
    id (int): The ID of the PhysAddrAttr to retrieve.

Returns:
    netdot.PhysAddrAttr: The PhysAddrAttr with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the PhysAddrAttr cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_physaddrattrname

> ```get_physaddrattrname(self, id: int) -> netdot.dataclasses.physaddr.PhysAddrAttrName```

```
Get info about a PhysAddrAttrName from Netdot.

Args:
    id (int): The ID of the PhysAddrAttrName to retrieve.

Returns:
    netdot.PhysAddrAttrName: The PhysAddrAttrName with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the PhysAddrAttrName cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_physaddrattrnames_where

> ```get_physaddrattrnames_where(self, *args, **url_params) -> List[netdot.dataclasses.physaddr.PhysAddrAttrName]```

```
Get info about PhysAddrAttrNames from Netdot.

> NOTE: This will return ALL PhysAddrAttrNames from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.PhysAddrAttrName]: PhysAddrAttrNames from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_physaddrattrs_where

> ```get_physaddrattrs_where(self, *args, **url_params) -> List[netdot.dataclasses.physaddr.PhysAddrAttr]```

```
Get info about PhysAddrAttrs from Netdot.

> NOTE: This will return ALL PhysAddrAttrs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.PhysAddrAttr]: PhysAddrAttrs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_physaddrs_where

> ```get_physaddrs_where(self, *args, **url_params) -> List[netdot.dataclasses.physaddr.PhysAddr]```

```
Get info about PhysAddrs from Netdot.

> NOTE: This will return ALL PhysAddrs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.PhysAddr]: PhysAddrs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_product

> ```get_product(self, id: int) -> netdot.dataclasses.products.Product```

```
Get info about a Product from Netdot.

Args:
    id (int): The ID of the Product to retrieve.

Returns:
    netdot.Product: The Product with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Product cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_products_by_manufacturer

> ```get_products_by_manufacturer(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Products associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.Product]: The list of Products associated to the Entity.
```

#### Repository.get_products_by_type

> ```get_products_by_type(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Products associated to a particular ProductType.

Args:
    other (int,NetdotAPIDataclass): The particular ProductType or its `id`.

Returns:
    List[netdot.Product]: The list of Products associated to the ProductType.
```

#### Repository.get_products_where

> ```get_products_where(self, *args, **url_params) -> List[netdot.dataclasses.products.Product]```

```
Get info about Products from Netdot.

> NOTE: This will return ALL Products from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Product]: Products from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_producttype

> ```get_producttype(self, id: int) -> netdot.dataclasses.products.ProductType```

```
Get info about a ProductType from Netdot.

Args:
    id (int): The ID of the ProductType to retrieve.

Returns:
    netdot.ProductType: The ProductType with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the ProductType cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_producttypes_where

> ```get_producttypes_where(self, *args, **url_params) -> List[netdot.dataclasses.products.ProductType]```

```
Get info about ProductTypes from Netdot.

> NOTE: This will return ALL ProductTypes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.ProductType]: ProductTypes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_room

> ```get_room(self, id: int) -> netdot.dataclasses.site.Room```

```
Get info about a Room from Netdot.

Args:
    id (int): The ID of the Room to retrieve.

Returns:
    netdot.Room: The Room with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Room cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rooms_by_floor

> ```get_rooms_by_floor(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Rooms associated to a particular Floor.

Args:
    other (int,NetdotAPIDataclass): The particular Floor or its `id`.

Returns:
    List[netdot.Room]: The list of Rooms associated to the Floor.
```

#### Repository.get_rooms_where

> ```get_rooms_where(self, *args, **url_params) -> List[netdot.dataclasses.site.Room]```

```
Get info about Rooms from Netdot.

> NOTE: This will return ALL Rooms from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Room]: Rooms from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rr

> ```get_rr(self, id: int) -> netdot.dataclasses.dns.RR```

```
Get info about a RR from Netdot.

Args:
    id (int): The ID of the RR to retrieve.

Returns:
    netdot.RR: The RR with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RR cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rr_by_address

> ```get_rr_by_address(self, address: str) -> 'netdot.RR'```

```
Get a Resource Record from Netdot Address Space, by IP Address.

Args:
    address (str): The IP Address to lookup, e.g. "10.0.0.123"

Returns:
    netdot.RR: The Resource Record that matched the provided address.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: If the Resource Record cannot be found. (error details can be found in Netdot's apache server logs)
```

#### Repository.get_rr_by_zone

> ```get_rr_by_zone(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RR associated to a particular Zone.

Args:
    other (int,NetdotAPIDataclass): The particular Zone or its `id`.

Returns:
    List[netdot.RR]: The list of RR associated to the Zone.
```

#### Repository.get_rr_where

> ```get_rr_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RR]```

```
Get info about RR from Netdot.

> NOTE: This will return ALL RR from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RR]: RR from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rraddr

> ```get_rraddr(self, id: int) -> netdot.dataclasses.dns.RRADDR```

```
Get info about a RRADDR from Netdot.

Args:
    id (int): The ID of the RRADDR to retrieve.

Returns:
    netdot.RRADDR: The RRADDR with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRADDR cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rraddr_by_ipblock

> ```get_rraddr_by_ipblock(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRADDR associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.RRADDR]: The list of RRADDR associated to the IPBlock.
```

#### Repository.get_rraddr_by_rr

> ```get_rraddr_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRADDR associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRADDR]: The list of RRADDR associated to the RR.
```

#### Repository.get_rraddr_where

> ```get_rraddr_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRADDR]```

```
Get info about RRADDR from Netdot.

> NOTE: This will return ALL RRADDR from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRADDR]: RRADDR from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrcname

> ```get_rrcname(self, id: int) -> netdot.dataclasses.dns.RRCNAME```

```
Get info about a RRCNAME from Netdot.

Args:
    id (int): The ID of the RRCNAME to retrieve.

Returns:
    netdot.RRCNAME: The RRCNAME with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRCNAME cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrcname_by_rr

> ```get_rrcname_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRCNAME associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRCNAME]: The list of RRCNAME associated to the RR.
```

#### Repository.get_rrcname_where

> ```get_rrcname_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRCNAME]```

```
Get info about RRCNAME from Netdot.

> NOTE: This will return ALL RRCNAME from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRCNAME]: RRCNAME from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrds

> ```get_rrds(self, id: int) -> netdot.dataclasses.dns.RRDS```

```
Get info about a RRDS from Netdot.

Args:
    id (int): The ID of the RRDS to retrieve.

Returns:
    netdot.RRDS: The RRDS with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRDS cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrds_by_rr

> ```get_rrds_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRDS associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRDS]: The list of RRDS associated to the RR.
```

#### Repository.get_rrds_where

> ```get_rrds_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRDS]```

```
Get info about RRDS from Netdot.

> NOTE: This will return ALL RRDS from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRDS]: RRDS from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrhinfo

> ```get_rrhinfo(self, id: int) -> netdot.dataclasses.dns.RRHINFO```

```
Get info about a RRHINFO from Netdot.

Args:
    id (int): The ID of the RRHINFO to retrieve.

Returns:
    netdot.RRHINFO: The RRHINFO with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRHINFO cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrhinfo_by_rr

> ```get_rrhinfo_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRHINFO associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRHINFO]: The list of RRHINFO associated to the RR.
```

#### Repository.get_rrhinfo_where

> ```get_rrhinfo_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRHINFO]```

```
Get info about RRHINFO from Netdot.

> NOTE: This will return ALL RRHINFO from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRHINFO]: RRHINFO from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrloc

> ```get_rrloc(self, id: int) -> netdot.dataclasses.dns.RRLOC```

```
Get info about a RRLOC from Netdot.

Args:
    id (int): The ID of the RRLOC to retrieve.

Returns:
    netdot.RRLOC: The RRLOC with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRLOC cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrloc_by_rr

> ```get_rrloc_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRLOC associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRLOC]: The list of RRLOC associated to the RR.
```

#### Repository.get_rrloc_where

> ```get_rrloc_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRLOC]```

```
Get info about RRLOC from Netdot.

> NOTE: This will return ALL RRLOC from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRLOC]: RRLOC from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrmx

> ```get_rrmx(self, id: int) -> netdot.dataclasses.dns.RRMX```

```
Get info about a RRMX from Netdot.

Args:
    id (int): The ID of the RRMX to retrieve.

Returns:
    netdot.RRMX: The RRMX with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRMX cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrmx_by_rr

> ```get_rrmx_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRMX associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRMX]: The list of RRMX associated to the RR.
```

#### Repository.get_rrmx_where

> ```get_rrmx_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRMX]```

```
Get info about RRMX from Netdot.

> NOTE: This will return ALL RRMX from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRMX]: RRMX from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrnaptr

> ```get_rrnaptr(self, id: int) -> netdot.dataclasses.dns.RRNAPTR```

```
Get info about a RRNAPTR from Netdot.

Args:
    id (int): The ID of the RRNAPTR to retrieve.

Returns:
    netdot.RRNAPTR: The RRNAPTR with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRNAPTR cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrnaptr_by_rr

> ```get_rrnaptr_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRNAPTR associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRNAPTR]: The list of RRNAPTR associated to the RR.
```

#### Repository.get_rrnaptr_where

> ```get_rrnaptr_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRNAPTR]```

```
Get info about RRNAPTR from Netdot.

> NOTE: This will return ALL RRNAPTR from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRNAPTR]: RRNAPTR from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrns

> ```get_rrns(self, id: int) -> netdot.dataclasses.dns.RRNS```

```
Get info about a RRNS from Netdot.

Args:
    id (int): The ID of the RRNS to retrieve.

Returns:
    netdot.RRNS: The RRNS with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRNS cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrns_by_rr

> ```get_rrns_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRNS associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRNS]: The list of RRNS associated to the RR.
```

#### Repository.get_rrns_where

> ```get_rrns_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRNS]```

```
Get info about RRNS from Netdot.

> NOTE: This will return ALL RRNS from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRNS]: RRNS from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrptr

> ```get_rrptr(self, id: int) -> netdot.dataclasses.dns.RRPTR```

```
Get info about a RRPTR from Netdot.

Args:
    id (int): The ID of the RRPTR to retrieve.

Returns:
    netdot.RRPTR: The RRPTR with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRPTR cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrptr_by_ipblock

> ```get_rrptr_by_ipblock(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRPTR associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.RRPTR]: The list of RRPTR associated to the IPBlock.
```

#### Repository.get_rrptr_by_rr

> ```get_rrptr_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRPTR associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRPTR]: The list of RRPTR associated to the RR.
```

#### Repository.get_rrptr_where

> ```get_rrptr_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRPTR]```

```
Get info about RRPTR from Netdot.

> NOTE: This will return ALL RRPTR from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRPTR]: RRPTR from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrsrv

> ```get_rrsrv(self, id: int) -> netdot.dataclasses.dns.RRSRV```

```
Get info about a RRSRV from Netdot.

Args:
    id (int): The ID of the RRSRV to retrieve.

Returns:
    netdot.RRSRV: The RRSRV with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRSRV cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrsrv_by_rr

> ```get_rrsrv_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRSRV associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRSRV]: The list of RRSRV associated to the RR.
```

#### Repository.get_rrsrv_where

> ```get_rrsrv_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRSRV]```

```
Get info about RRSRV from Netdot.

> NOTE: This will return ALL RRSRV from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRSRV]: RRSRV from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrtxt

> ```get_rrtxt(self, id: int) -> netdot.dataclasses.dns.RRTXT```

```
Get info about a RRTXT from Netdot.

Args:
    id (int): The ID of the RRTXT to retrieve.

Returns:
    netdot.RRTXT: The RRTXT with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the RRTXT cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_rrtxt_by_rr

> ```get_rrtxt_by_rr(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of RRTXT associated to a particular RR.

Args:
    other (int,NetdotAPIDataclass): The particular RR or its `id`.

Returns:
    List[netdot.RRTXT]: The list of RRTXT associated to the RR.
```

#### Repository.get_rrtxt_where

> ```get_rrtxt_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.RRTXT]```

```
Get info about RRTXT from Netdot.

> NOTE: This will return ALL RRTXT from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.RRTXT]: RRTXT from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_savedqueries

> ```get_savedqueries(self, id: int) -> netdot.dataclasses.misc.SavedQueries```

```
Get info about a SavedQueries from Netdot.

Args:
    id (int): The ID of the SavedQueries to retrieve.

Returns:
    netdot.SavedQueries: The SavedQueries with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the SavedQueries cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_savedqueries_where

> ```get_savedqueries_where(self, *args, **url_params) -> List[netdot.dataclasses.misc.SavedQueries]```

```
Get info about SavedQueries from Netdot.

> NOTE: This will return ALL SavedQueries from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.SavedQueries]: SavedQueries from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_schemainfo

> ```get_schemainfo(self, id: int) -> netdot.dataclasses.misc.SchemaInfo```

```
Get info about a SchemaInfo from Netdot.

Args:
    id (int): The ID of the SchemaInfo to retrieve.

Returns:
    netdot.SchemaInfo: The SchemaInfo with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the SchemaInfo cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_schemainfo_where

> ```get_schemainfo_where(self, *args, **url_params) -> List[netdot.dataclasses.misc.SchemaInfo]```

```
Get info about SchemaInfo from Netdot.

> NOTE: This will return ALL SchemaInfo from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.SchemaInfo]: SchemaInfo from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_service

> ```get_service(self, id: int) -> netdot.dataclasses.ipblock.Service```

```
Get info about a Service from Netdot.

Args:
    id (int): The ID of the Service to retrieve.

Returns:
    netdot.Service: The Service with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Service cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_services_where

> ```get_services_where(self, *args, **url_params) -> List[netdot.dataclasses.ipblock.Service]```

```
Get info about Services from Netdot.

> NOTE: This will return ALL Services from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Service]: Services from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_site

> ```get_site(self, id: int) -> netdot.dataclasses.site.Site```

```
Get info about a Site from Netdot.

Args:
    id (int): The ID of the Site to retrieve.

Returns:
    netdot.Site: The Site with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Site cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_sitelink

> ```get_sitelink(self, id: int) -> netdot.dataclasses.site.SiteLink```

```
Get info about a SiteLink from Netdot.

Args:
    id (int): The ID of the SiteLink to retrieve.

Returns:
    netdot.SiteLink: The SiteLink with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the SiteLink cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_sitelinks_by_entity

> ```get_sitelinks_by_entity(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of SiteLinks associated to a particular Entity.

Args:
    other (int,NetdotAPIDataclass): The particular Entity or its `id`.

Returns:
    List[netdot.SiteLink]: The list of SiteLinks associated to the Entity.
```

#### Repository.get_sitelinks_by_farend

> ```get_sitelinks_by_farend(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of SiteLinks associated to a particular Site.

Args:
    other (int,NetdotAPIDataclass): The particular Site or its `id`.

Returns:
    List[netdot.SiteLink]: The list of SiteLinks associated to the Site.
```

#### Repository.get_sitelinks_by_nearend

> ```get_sitelinks_by_nearend(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of SiteLinks associated to a particular Site.

Args:
    other (int,NetdotAPIDataclass): The particular Site or its `id`.

Returns:
    List[netdot.SiteLink]: The list of SiteLinks associated to the Site.
```

#### Repository.get_sitelinks_where

> ```get_sitelinks_where(self, *args, **url_params) -> List[netdot.dataclasses.site.SiteLink]```

```
Get info about SiteLinks from Netdot.

> NOTE: This will return ALL SiteLinks from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.SiteLink]: SiteLinks from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_sites_by_availability

> ```get_sites_by_availability(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Sites associated to a particular Availability.

Args:
    other (int,NetdotAPIDataclass): The particular Availability or its `id`.

Returns:
    List[netdot.Site]: The list of Sites associated to the Availability.
```

#### Repository.get_sites_by_contactlist

> ```get_sites_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Sites associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.Site]: The list of Sites associated to the ContactList.
```

#### Repository.get_sites_where

> ```get_sites_where(self, *args, **url_params) -> List[netdot.dataclasses.site.Site]```

```
Get info about Sites from Netdot.

> NOTE: This will return ALL Sites from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Site]: Sites from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_sitesubnet

> ```get_sitesubnet(self, id: int) -> netdot.dataclasses.site.SiteSubnet```

```
Get info about a SiteSubnet from Netdot.

Args:
    id (int): The ID of the SiteSubnet to retrieve.

Returns:
    netdot.SiteSubnet: The SiteSubnet with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the SiteSubnet cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_sitesubnets_by_site

> ```get_sitesubnets_by_site(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of SiteSubnets associated to a particular Site.

Args:
    other (int,NetdotAPIDataclass): The particular Site or its `id`.

Returns:
    List[netdot.SiteSubnet]: The list of SiteSubnets associated to the Site.
```

#### Repository.get_sitesubnets_by_subnet

> ```get_sitesubnets_by_subnet(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of SiteSubnets associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.SiteSubnet]: The list of SiteSubnets associated to the IPBlock.
```

#### Repository.get_sitesubnets_where

> ```get_sitesubnets_where(self, *args, **url_params) -> List[netdot.dataclasses.site.SiteSubnet]```

```
Get info about SiteSubnets from Netdot.

> NOTE: This will return ALL SiteSubnets from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.SiteSubnet]: SiteSubnets from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_splice

> ```get_splice(self, id: int) -> netdot.dataclasses.cables.Splice```

```
Get info about a Splice from Netdot.

Args:
    id (int): The ID of the Splice to retrieve.

Returns:
    netdot.Splice: The Splice with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Splice cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_splices_by_strand1

> ```get_splices_by_strand1(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Splices associated to a particular CableStrand.

Args:
    other (int,NetdotAPIDataclass): The particular CableStrand or its `id`.

Returns:
    List[netdot.Splice]: The list of Splices associated to the CableStrand.
```

#### Repository.get_splices_by_strand2

> ```get_splices_by_strand2(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Splices associated to a particular CableStrand.

Args:
    other (int,NetdotAPIDataclass): The particular CableStrand or its `id`.

Returns:
    List[netdot.Splice]: The list of Splices associated to the CableStrand.
```

#### Repository.get_splices_where

> ```get_splices_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.Splice]```

```
Get info about Splices from Netdot.

> NOTE: This will return ALL Splices from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Splice]: Splices from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_stpinstance

> ```get_stpinstance(self, id: int) -> netdot.dataclasses.device.STPInstance```

```
Get info about a STPInstance from Netdot.

Args:
    id (int): The ID of the STPInstance to retrieve.

Returns:
    netdot.STPInstance: The STPInstance with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the STPInstance cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_stpinstances_by_device

> ```get_stpinstances_by_device(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of STPInstances associated to a particular Device.

Args:
    other (int,NetdotAPIDataclass): The particular Device or its `id`.

Returns:
    List[netdot.STPInstance]: The list of STPInstances associated to the Device.
```

#### Repository.get_stpinstances_where

> ```get_stpinstances_where(self, *args, **url_params) -> List[netdot.dataclasses.device.STPInstance]```

```
Get info about STPInstances from Netdot.

> NOTE: This will return ALL STPInstances from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.STPInstance]: STPInstances from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_strandstatus

> ```get_strandstatus(self, id: int) -> netdot.dataclasses.cables.StrandStatus```

```
Get info about a StrandStatus from Netdot.

Args:
    id (int): The ID of the StrandStatus to retrieve.

Returns:
    netdot.StrandStatus: The StrandStatus with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the StrandStatus cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_strandstatuses_where

> ```get_strandstatuses_where(self, *args, **url_params) -> List[netdot.dataclasses.cables.StrandStatus]```

```
Get info about StrandStatuses from Netdot.

> NOTE: This will return ALL StrandStatuses from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.StrandStatus]: StrandStatuses from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_subnetzone

> ```get_subnetzone(self, id: int) -> netdot.dataclasses.ipblock.SubnetZone```

```
Get info about a SubnetZone from Netdot.

Args:
    id (int): The ID of the SubnetZone to retrieve.

Returns:
    netdot.SubnetZone: The SubnetZone with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the SubnetZone cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_subnetzones_by_subnet

> ```get_subnetzones_by_subnet(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of SubnetZones associated to a particular IPBlock.

Args:
    other (int,NetdotAPIDataclass): The particular IPBlock or its `id`.

Returns:
    List[netdot.SubnetZone]: The list of SubnetZones associated to the IPBlock.
```

#### Repository.get_subnetzones_by_zone

> ```get_subnetzones_by_zone(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of SubnetZones associated to a particular Zone.

Args:
    other (int,NetdotAPIDataclass): The particular Zone or its `id`.

Returns:
    List[netdot.SubnetZone]: The list of SubnetZones associated to the Zone.
```

#### Repository.get_subnetzones_where

> ```get_subnetzones_where(self, *args, **url_params) -> List[netdot.dataclasses.ipblock.SubnetZone]```

```
Get info about SubnetZones from Netdot.

> NOTE: This will return ALL SubnetZones from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.SubnetZone]: SubnetZones from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_userright

> ```get_userright(self, id: int) -> netdot.dataclasses.users.UserRight```

```
Get info about a UserRight from Netdot.

Args:
    id (int): The ID of the UserRight to retrieve.

Returns:
    netdot.UserRight: The UserRight with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the UserRight cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_userrights_by_accessright

> ```get_userrights_by_accessright(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of UserRights associated to a particular AccessRight.

Args:
    other (int,NetdotAPIDataclass): The particular AccessRight or its `id`.

Returns:
    List[netdot.UserRight]: The list of UserRights associated to the AccessRight.
```

#### Repository.get_userrights_by_person

> ```get_userrights_by_person(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of UserRights associated to a particular Person.

Args:
    other (int,NetdotAPIDataclass): The particular Person or its `id`.

Returns:
    List[netdot.UserRight]: The list of UserRights associated to the Person.
```

#### Repository.get_userrights_where

> ```get_userrights_where(self, *args, **url_params) -> List[netdot.dataclasses.users.UserRight]```

```
Get info about UserRights from Netdot.

> NOTE: This will return ALL UserRights from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.UserRight]: UserRights from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_usertype

> ```get_usertype(self, id: int) -> netdot.dataclasses.users.UserType```

```
Get info about a UserType from Netdot.

Args:
    id (int): The ID of the UserType to retrieve.

Returns:
    netdot.UserType: The UserType with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the UserType cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_usertypes_where

> ```get_usertypes_where(self, *args, **url_params) -> List[netdot.dataclasses.users.UserType]```

```
Get info about UserTypes from Netdot.

> NOTE: This will return ALL UserTypes from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.UserType]: UserTypes from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_vlan

> ```get_vlan(self, id: int) -> netdot.dataclasses.vlan.VLAN```

```
Get info about a VLAN from Netdot.

Args:
    id (int): The ID of the VLAN to retrieve.

Returns:
    netdot.VLAN: The VLAN with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the VLAN cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_vlangroup

> ```get_vlangroup(self, id: int) -> netdot.dataclasses.vlan.VLANGroup```

```
Get info about a VLANGroup from Netdot.

Args:
    id (int): The ID of the VLANGroup to retrieve.

Returns:
    netdot.VLANGroup: The VLANGroup with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the VLANGroup cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_vlangroups_where

> ```get_vlangroups_where(self, *args, **url_params) -> List[netdot.dataclasses.vlan.VLANGroup]```

```
Get info about VLANGroups from Netdot.

> NOTE: This will return ALL VLANGroups from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.VLANGroup]: VLANGroups from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_vlans_by_vlangroup

> ```get_vlans_by_vlangroup(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of VLANs associated to a particular VLANGroup.

Args:
    other (int,NetdotAPIDataclass): The particular VLANGroup or its `id`.

Returns:
    List[netdot.VLAN]: The list of VLANs associated to the VLANGroup.
```

#### Repository.get_vlans_where

> ```get_vlans_where(self, *args, **url_params) -> List[netdot.dataclasses.vlan.VLAN]```

```
Get info about VLANs from Netdot.

> NOTE: This will return ALL VLANs from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.VLAN]: VLANs from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_zone

> ```get_zone(self, id: int) -> netdot.dataclasses.dns.Zone```

```
Get info about a Zone from Netdot.

Args:
    id (int): The ID of the Zone to retrieve.

Returns:
    netdot.Zone: The Zone with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the Zone cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_zonealias

> ```get_zonealias(self, id: int) -> netdot.dataclasses.dns.ZoneAlias```

```
Get info about a ZoneAlias from Netdot.

Args:
    id (int): The ID of the ZoneAlias to retrieve.

Returns:
    netdot.ZoneAlias: The ZoneAlias with `id`. (raises ValueError if `id` is not found)

Raises:
    ValueError: if the ZoneAlias cannot be retrieved for some reason.
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_zonealiases_by_zone

> ```get_zonealiases_by_zone(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of ZoneAliases associated to a particular Zone.

Args:
    other (int,NetdotAPIDataclass): The particular Zone or its `id`.

Returns:
    List[netdot.ZoneAlias]: The list of ZoneAliases associated to the Zone.
```

#### Repository.get_zonealiases_where

> ```get_zonealiases_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.ZoneAlias]```

```
Get info about ZoneAliases from Netdot.

> NOTE: This will return ALL ZoneAliases from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.ZoneAlias]: ZoneAliases from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.get_zones_by_contactlist

> ```get_zones_by_contactlist(self, other: Union[int, netdot.dataclasses.base.NetdotAPIDataclass], **url_params)```

```
Get the list of Zones associated to a particular ContactList.

Args:
    other (int,NetdotAPIDataclass): The particular ContactList or its `id`.

Returns:
    List[netdot.Zone]: The list of Zones associated to the ContactList.
```

#### Repository.get_zones_where

> ```get_zones_where(self, *args, **url_params) -> List[netdot.dataclasses.dns.Zone]```

```
Get info about Zones from Netdot.

> NOTE: This will return ALL Zones from Netdot if no kwargs (URL Parameters) are provided.
> You can provide the special positional argument "all" if you like (for semantic clarity in your scripts).

Args:
    **kwargs: URL Parameters - Any keyword args will be used as URL Parameters. Ex. (id=1) will be translated to `?id=1` in the URL.

Returns:
    List[netdot.Zone]: Zones from Netdot (that match provided URL Parameters).

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors (including HTTP 404 if no matches are found). (error details can be found in Netdot's apache server logs)
    NetdotError: if some internal error happens (in this Python Netdot API wrapper, or on the Netdot Server itself).
```

#### Repository.infer_product

> ```infer_product(self, device_asset_id: str) -> 'netdot.dataclasses.Product'```

```
Infer the Product of some device, based on its `asset_id` string returned from Netdot REST API.

> NOTE: One HTTP Request is made to retrieve all Products from Netdot.
> All subsequent calls to this method will use the cached results.

Args:
    device_asset_id (str): The "asset_id" string returned from Netdot.

Returns:
    netdot.dataclasses.Product: The Product associated to this Device.

Raises:
    NetdotError: If no Product can be inferred from the provided asset_id string.
```

#### Repository.reload_product_index

> ```reload_product_index(self)```

```
Reload all Products from Netdot, and index them by name.
```

#### Repository.save_changes

> ```save_changes(self, confirm=True)```

```
Save all proposed changes to Netdot.

Args:
    confirm (bool, optional): If any delete actions are planned, confirm them with the user first. Defaults to True.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
    RuntimeError: If this Repository's multithreading lock cannot be acquired.
```

#### Repository.show_all_changes

> ```show_all_changes(self, terse=None)```

```
Show a report of all actions that have been completed, and all actions that remain to be completed (including any failed action(s)).

Args:
    terse (bool, optional): Whether to truncate each line according to your terminal width. Defaults to config.TERSE.
```

#### Repository.show_changes

> ```show_changes(self, terse=None)```

```
Show a 'dry run' of all planned changes to be made (but don't actually make the changes).

Args:
    terse (bool, optional): Whether to truncate each line according to your terminal width. Defaults to config.TERSE.
```

#### Repository.show_changes_as_tables

> ```show_changes_as_tables(self, terse=None, select_cols=None)```

```
Show ASCII table(s) representing all of the changes to be made (grouped into tables based on Netdot Data Types).

Args:
    terse (bool, optional): Whether to truncate data in each column. Defaults to config.TERSE
    select_cols (List[str], optional): Which columns to include in the table. Defaults to None (all columns).
```

#### Repository.show_failed_changes

> ```show_failed_changes(self)```

```
Print a message for each action that has failed.

Returns:
    str: A single long message that includes all failed actions and their exceptions.
```

#### Repository.update

> ```update(self, new_data: netdot.dataclasses.base.NetdotAPIDataclass) -> netdot.dataclasses.base.NetdotAPIDataclass```

```
Update an existing object in Netdot.

Args:
    id (int): The ID of the object to update.
    new_data (netdot.NetdotAPIDataclass): The new data to use when updating the object.

Returns:
    netdot.NetdotAPIDataclass: The object provided by Netdot in response to this HTTP POST.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be updated for some reason. (error details can be found in Netdot's apache server logs)
```

## class `netdot.UnitOfWork`

<a id="class-netdotunitofwork"></a>

### Methods

#### UnitOfWork.as_list

> ```as_list(self)```

```
Get a copy of to-be-completed actions in this Unit of Work.

Returns:
    List[NetdotAction]: The list of to-be-completed actions.
```

#### UnitOfWork.changes_as_tables

> ```changes_as_tables(self, terse=None, select_cols=None, display_full_objects=None) -> str```

```
Show ASCII table(s) representing all of the changes to be made (grouped into tables based on Netdot Data Types).

Args:
    terse (bool, optional): Whether to truncate data in each column. Defaults to config.TERSE
    select_cols (List[str], optional): Which columns to include in the table. Defaults to None (all columns).
```

#### UnitOfWork.create

> ```create(self, new_data: netdot.dataclasses.base.NetdotAPIDataclass, print_changes=False, one_line=True)```

```
Create a new object in Netdot.

Args:
    new_data (netdot.NetdotAPIDataclass): The new data to use when creating the object in Netdot.
    truncate (int): Truncate the message to this many characters. Defaults to None.

Raises:
    TypeError: If new_data is not a subclass of NetdotAPIDataclass.
```

#### UnitOfWork.delete

> ```delete(self, old_data: netdot.dataclasses.base.NetdotAPIDataclass, print_changes=False, one_line=True)```

```
Delete an existing object from Netdot.

Args:
    old_data (netdot.NetdotAPIDataclass): The object that will be deleted (must include an 'id').

Raises:
    TypeError: If old_data is not a subclass of NetdotAPIDataclass.
```

#### UnitOfWork.dry_run

> ```dry_run(self, terse=None) -> str```

```
Show a 'dry run' of all planned changes to be made (but don't actually make the changes).

Args:
    terse (bool, optional): Whether to truncate each line according to your terminal width. Defaults to config.TERSE.
```

#### UnitOfWork.dry_run_of_actions

> ```dry_run_of_actions(actions, one_line=True, completed=False, empty_message='None, yet...') -> str```

```
Return a 'dry run' of some 'actions'.

Args:
    actions: The actions to be included in the dry run.
    one_line (bool, optional): Whether to truncate each line according to your terminal width. Defaults to True.
    completed (bool, optional): Whether to show actions as 'COMPLETED'. Defaults to False.

Returns:
    str: Each planned action, printed in a nicely enumerated list.
```

#### UnitOfWork.failed_action

> ```failed_action(self, index=-1) -> netdot.actions.NetdotAction```

```
Get the action that failed when trying to save changes to Netdot.

Args:
    index (int, optional): Index for accessing the list of actions that have failed. Defaults to -1 (latest failed action).

Returns:
    actions.NetdotAction: The action that failed when trying to save changes to Netdot.
```

#### UnitOfWork.failed_action_exception

> ```failed_action_exception(self, index=-1) -> Exception```

```
Get the exception that occurred when trying to save changes to Netdot.

Args:
    index (int, optional): Index for accessing the list of exceptions that have occurred. Defaults to -1 (latest exception).

Returns:
    Exception: The exception that occurred when trying to save changes to Netdot.
```

#### UnitOfWork.failed_action_msg

> ```failed_action_msg(self, index=-1) -> str```

```
If 'save_changes' failed on some action, use this to get info about the failed action.

Returns:
    str: A message indicating what the action would have done, and what exception occurred that prevented it from being completed.
```

#### UnitOfWork.failed_actions_msgs

> ```failed_actions_msgs(self) -> str```

```
Print a message for each action that has failed.

Returns:
    str: A single long message that includes all failed actions and their exceptions.
```

#### UnitOfWork.load

> ```load(filename)```

```
Load a Unit of Work from a pickle file.

Args:
    filename (str): The file to load from.

Returns:
    UnitOfWork: The Unit of Work that was loaded.
```

#### UnitOfWork.save_as_pickle

> ```save_as_pickle(self, filename: str = 'netdot-cli-0.5.0-proposed_changes-2025-11-06_15-48.pickle') -> str```

```
Save this Unit of Work to a file.

To be loaded in the future using :func:`load()`.

Args:
    filename (str, optional): The file to save to. Defaults to a dynamic "defaults.ERROR_PICKLE_FILENAME" (which includes version and timestamp).
```

#### UnitOfWork.save_changes

> ```save_changes(self, netdot_repo: 'netdot.Repository')```

```
Save the changes back to Netdot.

Args:
    netdot_repo (netdot.Repository): The repository to use when saving changes.
```

#### UnitOfWork.status_report

> ```status_report(self, terse=None)```

```
Show a report of all actions that have been completed, and all actions that remain to be completed (including any failed action(s)).

Args:
    terse (bool, optional): Whether to truncate each line according to your terminal width. Defaults to config.TERSE.
```

#### UnitOfWork.update

> ```update(self, old_data: netdot.dataclasses.base.NetdotAPIDataclass, new_data: netdot.dataclasses.base.NetdotAPIDataclass, print_changes=False, one_line=True, deduplicate=True)```

```
Update an existing object in Netdot.

Args:
    new_data (netdot.NetdotAPIDataclass): The new data to use when updating.
    old_data (netdot.NetdotAPIDataclass): The old data that is going to be replaced.
    deduplicate (bool): If True, consolidate duplicate UPDATE actions that are added to this Unit of Work.

Raises:
    TypeError: If new_data and old_data are not the same type (or not a subclass of NetdotAPIDataclass).
```

#### UnitOfWork.with_action_types

> ```with_action_types(self, action_types: List[ForwardRef('actions.SiteActionTypes')])```

```
Get a copy of this Unit of Work containing ONLY the selected actions.

Args:
    action_types (List[actions.NetdotAction]): The types of actions to keep.
```

#### UnitOfWork.with_data_type

> ```with_data_type(self, data_type: netdot.dataclasses.base.NetdotAPIDataclass)```

```
Get a copy of this Unit of Work containing actions of ONLY the selected data type.

Args:
    data_type (NetdotAPIDataclass): The type of data to keep.
```

#### UnitOfWork.without_action_types

> ```without_action_types(self, action_types: List[ForwardRef('actions.ActionTypes')])```

```
Get a copy of this Unit of Work with the selected actions removed.

Args:
    action_types (List[actions.NetdotAction]): The types of actions to be removed.
```

## class `netdot.CSVReport`

<a id="class-netdotcsvreport"></a>

### Methods

#### CSVReport.as_csv

> ```as_csv(self, delim=',', override_header: Iterable[str] = None) -> str```

```
Produce this data structure into a CSV.

Args:
    delim (str, optional): CSV delimiter to be used. Defaults to ','.
    override_header (Iterable[str], optional): Enables overriding the default CSV header (labels only, cannot be used to 'select columns'). (Must be same length as this report's default `table_header`).

Raises:
    ValueError: If override_header is invalid.

Returns:
    str: This report, in CSV format.
```

#### CSVReport.save_as_file

> ```save_as_file(self, filename: str, output_directory: str = './', **kwargs) -> int```

```
Save this CSVReport to a file.

Args:
    output_directory (str): The path where the CSV will be saved.
    filename (str): Will be used as the file name.
    kwargs: Additional arguments are passed to `as_csv()` (e.g. override_header)

Returns:
    int: Amount of bytes written to the file. (See `TextIOBase.write()`)
```

#### CSVReport.sort

> ```sort(self, *sort_by: str)```

```
Sort this report. (in-place)

Args:
    sort_by (Union[str,List[str]], optional): The column(s) to be sorted by. By default sort by the first column indicated by table_header.
```

## class `netdot.ASN`

<a id="class-netdotasn"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| description | str |  |
| info | str |  |
| number | int |  |
| rir | str |  |


### Methods

#### ASN.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this ASN.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ASN.add_ipblock

> ```add_ipblock(self, data: netdot.dataclasses.ipblock.IPBlock) -> netdot.dataclasses.ipblock.IPBlock```

```
Add a IPBlock to this ASN.

Returns:
    netdot.IPBlock: The created IPBlock.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ASN.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### ASN.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### ASN.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### ASN.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this ASN. (Via the `Device.bgplocalas` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this ASN.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ASN.load_ipblocks

> ```load_ipblocks(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Load the IPBlocks associated to this ASN.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlock]: All/Any IPBlocks related to this ASN.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ASN.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### ASN.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.AccessRight`

<a id="class-netdotaccessright"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| access | str |  |
| object_class | str |  |
| object_id | int |  |


### Methods

#### AccessRight.add_contactlist

> ```add_contactlist(self, data: netdot.dataclasses.users.ContactList) -> netdot.dataclasses.users.ContactList```

```
Add a ContactList to this AccessRight (via GroupRight).

Args:
    data (netdot.ContactList): The ContactList to add to this AccessRight.

Returns:
    netdot.GroupRight: The newly created GroupRight.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### AccessRight.add_groupright

> ```add_groupright(self, data: netdot.dataclasses.users.GroupRight) -> netdot.dataclasses.users.GroupRight```

```
Add a GroupRight to this AccessRight.

Returns:
    netdot.GroupRight: The created GroupRight.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### AccessRight.add_person

> ```add_person(self, data: netdot.dataclasses.users.Person) -> netdot.dataclasses.users.Person```

```
Add a Person to this AccessRight (via UserRight).

Args:
    data (netdot.Person): The Person to add to this AccessRight.

Returns:
    netdot.UserRight: The newly created UserRight.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### AccessRight.add_userright

> ```add_userright(self, data: netdot.dataclasses.users.UserRight) -> netdot.dataclasses.users.UserRight```

```
Add a UserRight to this AccessRight.

Returns:
    netdot.UserRight: The created UserRight.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### AccessRight.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### AccessRight.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### AccessRight.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### AccessRight.load_contactlists

> ```load_contactlists(self, ignore_404=True) -> List[netdot.dataclasses.users.ContactList]```

```
Load the contactlists (ContactLists) associated to this AccessRight.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of ContactLists associated to this AccessRight).

You might prefer :func:`load_grouprights` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.ContactList]: Any/All ContactLists related to this AccessRight, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### AccessRight.load_grouprights

> ```load_grouprights(self, ignore_404=True) -> List[netdot.dataclasses.users.GroupRight]```

```
Load the GroupRights associated to this AccessRight.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.GroupRight]: All/Any GroupRights related to this AccessRight.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### AccessRight.load_persons

> ```load_persons(self, ignore_404=True) -> List[netdot.dataclasses.users.Person]```

```
Load the persons (Persons) associated to this AccessRight.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of Persons associated to this AccessRight).

You might prefer :func:`load_userrights` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Person]: Any/All Persons related to this AccessRight, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### AccessRight.load_userrights

> ```load_userrights(self, ignore_404=True) -> List[netdot.dataclasses.users.UserRight]```

```
Load the UserRights associated to this AccessRight.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.UserRight]: All/Any UserRights related to this AccessRight.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### AccessRight.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### AccessRight.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.ArpCache`

<a id="class-netdotarpcache"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| device_xlink | int |  |
| tstamp | DateTime |  |


### Methods

#### ArpCache.add_arpcacheentry

> ```add_arpcacheentry(self, data: netdot.dataclasses.arpcache.ArpCacheEntry) -> netdot.dataclasses.arpcache.ArpCacheEntry```

```
Add a ArpCacheEntry to this ArpCache.

Returns:
    netdot.ArpCacheEntry: The created ArpCacheEntry.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ArpCache.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### ArpCache.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### ArpCache.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### ArpCache.load_arpcacheentries

> ```load_arpcacheentries(self, ignore_404=True) -> List[netdot.dataclasses.arpcache.ArpCacheEntry]```

```
Load the ArpCacheEntries associated to this ArpCache.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.ArpCacheEntry]: All/Any ArpCacheEntries related to this ArpCache.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ArpCache.load_device

> ```load_device(self) -> netdot.dataclasses.device.Device```

```
Load the device (Device) associated to this ArpCache.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### ArpCache.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### ArpCache.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.ArpCacheEntry`

<a id="class-netdotarpcacheentry"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| arpcache_xlink | int |  |
| interface_xlink | int |  |
| physaddr_xlink | int |  |
| ipaddr_xlink | int |  |


### Methods

#### ArpCacheEntry.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### ArpCacheEntry.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### ArpCacheEntry.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### ArpCacheEntry.infer_timestamp

> ```infer_timestamp(self) -> datetime.datetime```

```
Infer the timestamp of this ArpCache (based on the 'arpcache' string returned to us by Netdot API).

Returns:
    datetime: The inferred timestamp.

Raises:
    ValueError: If unable to infer the timestamp.
```

#### ArpCacheEntry.load_arpcache

> ```load_arpcache(self) -> netdot.dataclasses.arpcache.ArpCache```

```
Load the arpcache (ArpCache) associated to this ArpCacheEntry.

Returns:
    netdot.ArpCache: The full ArpCache object if available, else None.
```

#### ArpCacheEntry.load_interface

> ```load_interface(self) -> netdot.dataclasses.interface.Interface```

```
Load the interface (Interface) associated to this ArpCacheEntry.

Returns:
    netdot.Interface: The full Interface object if available, else None.
```

#### ArpCacheEntry.load_ipaddr

> ```load_ipaddr(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the ipaddr (IPBlock) associated to this ArpCacheEntry.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### ArpCacheEntry.load_physaddr

> ```load_physaddr(self) -> netdot.dataclasses.physaddr.PhysAddr```

```
Load the physaddr (PhysAddr) associated to this ArpCacheEntry.

Returns:
    netdot.PhysAddr: The full PhysAddr object if available, else None.
```

#### ArpCacheEntry.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### ArpCacheEntry.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Asset`

<a id="class-netdotasset"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| product_id_xlink | int |  |
| physaddr_xlink | int |  |
| maint_contract_xlink | int |  |
| custom_serial | str |  |
| description | str |  |
| info | str |  |
| inventory_number | str |  |
| maint_from | DateTime |  |
| maint_until | DateTime |  |
| date_purchased | DateTime |  |
| po_number | str |  |
| reserved_for | str |  |
| serial_number | str |  |


### Methods

#### Asset.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this Asset.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Asset.add_devicemodule

> ```add_devicemodule(self, data: netdot.dataclasses.device.DeviceModule) -> netdot.dataclasses.device.DeviceModule```

```
Add a DeviceModule to this Asset.

Returns:
    netdot.DeviceModule: The created DeviceModule.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Asset.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Asset.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Asset.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Asset.load_devicemodules

> ```load_devicemodules(self, ignore_404=True) -> List[netdot.dataclasses.device.DeviceModule]```

```
Load the DeviceModules associated to this Asset. (Via the `DeviceModule.asset_id` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DeviceModule]: All/Any DeviceModules related to this Asset.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Asset.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this Asset. (Via the `Device.asset_id` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this Asset.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Asset.load_maint_contract

> ```load_maint_contract(self) -> netdot.dataclasses.misc.MaintContract```

```
Load the maint_contract (MaintContract) associated to this Asset.

Returns:
    netdot.MaintContract: The full MaintContract object if available, else None.
```

#### Asset.load_physaddr

> ```load_physaddr(self) -> netdot.dataclasses.physaddr.PhysAddr```

```
Load the physaddr (PhysAddr) associated to this Asset.

Returns:
    netdot.PhysAddr: The full PhysAddr object if available, else None.
```

#### Asset.load_product

> ```load_product(self) -> netdot.dataclasses.products.Product```

```
Load the product_id (Product) associated to this Asset.

Returns:
    netdot.Product: The full Product object if available, else None.
```

#### Asset.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Asset.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Audit`

<a id="class-netdotaudit"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| fields | str |  |
| label | str |  |
| object_id | int |  |
| operation | str |  |
| tablename | str |  |
| tstamp | DateTime |  |
| username | str |  |
| vals | str |  |


### Methods

#### Audit.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Audit.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Audit.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Audit.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Audit.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Availability`

<a id="class-netdotavailability"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### Availability.add_contact_as_notify_email

> ```add_contact_as_notify_email(self, data: netdot.dataclasses.users.Contact) -> netdot.dataclasses.users.Contact```

```
Add a Contact to this Availability.

Returns:
    netdot.Contact: The created Contact.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.add_contact_as_notify_pager

> ```add_contact_as_notify_pager(self, data: netdot.dataclasses.users.Contact) -> netdot.dataclasses.users.Contact```

```
Add a Contact to this Availability.

Returns:
    netdot.Contact: The created Contact.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.add_contact_as_notify_voice

> ```add_contact_as_notify_voice(self, data: netdot.dataclasses.users.Contact) -> netdot.dataclasses.users.Contact```

```
Add a Contact to this Availability.

Returns:
    netdot.Contact: The created Contact.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.add_entity

> ```add_entity(self, data: netdot.dataclasses.entity.Entity) -> netdot.dataclasses.entity.Entity```

```
Add a Entity to this Availability.

Returns:
    netdot.Entity: The created Entity.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.add_site

> ```add_site(self, data: netdot.dataclasses.site.Site) -> netdot.dataclasses.site.Site```

```
Add a Site to this Availability.

Returns:
    netdot.Site: The created Site.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Availability.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Availability.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Availability.load_entities

> ```load_entities(self, ignore_404=True) -> List[netdot.dataclasses.entity.Entity]```

```
Load the Entities associated to this Availability.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Entity]: All/Any Entities related to this Availability.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.load_notify_email_contacts

> ```load_notify_email_contacts(self, ignore_404=True) -> List[netdot.dataclasses.users.Contact]```

```
Load the Contacts associated to this Availability. (Via the `Contact.notify_email` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Contact]: All/Any Contacts related to this Availability.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.load_notify_pager_contacts

> ```load_notify_pager_contacts(self, ignore_404=True) -> List[netdot.dataclasses.users.Contact]```

```
Load the Contacts associated to this Availability. (Via the `Contact.notify_pager` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Contact]: All/Any Contacts related to this Availability.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.load_notify_voice_contacts

> ```load_notify_voice_contacts(self, ignore_404=True) -> List[netdot.dataclasses.users.Contact]```

```
Load the Contacts associated to this Availability. (Via the `Contact.notify_voice` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Contact]: All/Any Contacts related to this Availability.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.load_sites

> ```load_sites(self, ignore_404=True) -> List[netdot.dataclasses.site.Site]```

```
Load the Sites associated to this Availability.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Site]: All/Any Sites related to this Availability.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Availability.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Availability.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.BGPPeering`

<a id="class-netdotbgppeering"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| bgppeeraddr | str |  |
| bgppeerid | str |  |
| device_xlink | int |  |
| entity_xlink | int |  |
| monitored | bool | False |
| authkey | str |  |
| info | str |  |
| max_v4_prefixes | int |  |
| max_v6_prefixes | int |  |
| contactlist_xlink | int |  |
| last_changed | DateTime |  |
| peer_group | str |  |
| state | str |  |


### Methods

#### BGPPeering.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### BGPPeering.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### BGPPeering.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### BGPPeering.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this BGPPeering.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### BGPPeering.load_device

> ```load_device(self) -> netdot.dataclasses.device.Device```

```
Load the device (Device) associated to this BGPPeering.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### BGPPeering.load_entity

> ```load_entity(self) -> netdot.dataclasses.entity.Entity```

```
Load the entity (Entity) associated to this BGPPeering.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### BGPPeering.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### BGPPeering.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.BackboneCable`

<a id="class-netdotbackbonecable"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| end_closet_xlink | int |  |
| info | str |  |
| installdate | DateTime |  |
| length | str |  |
| name | str |  |
| owner_xlink | int |  |
| start_closet_xlink | int |  |
| type_xlink | int |  |


### Methods

#### BackboneCable.add_cablestrand

> ```add_cablestrand(self, data: netdot.dataclasses.cables.CableStrand) -> netdot.dataclasses.cables.CableStrand```

```
Add a CableStrand to this BackboneCable.

Returns:
    netdot.CableStrand: The created CableStrand.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### BackboneCable.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### BackboneCable.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### BackboneCable.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### BackboneCable.load_cablestrands

> ```load_cablestrands(self, ignore_404=True) -> List[netdot.dataclasses.cables.CableStrand]```

```
Load the CableStrands associated to this BackboneCable. (Via the `CableStrand.cable` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.CableStrand]: All/Any CableStrands related to this BackboneCable.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### BackboneCable.load_end_closet

> ```load_end_closet(self) -> netdot.dataclasses.site.Closet```

```
Load the end_closet (Closet) associated to this BackboneCable.

Returns:
    netdot.Closet: The full Closet object if available, else None.
```

#### BackboneCable.load_owner

> ```load_owner(self) -> netdot.dataclasses.entity.Entity```

```
Load the owner (Entity) associated to this BackboneCable.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### BackboneCable.load_start_closet

> ```load_start_closet(self) -> netdot.dataclasses.site.Closet```

```
Load the start_closet (Closet) associated to this BackboneCable.

Returns:
    netdot.Closet: The full Closet object if available, else None.
```

#### BackboneCable.load_type

> ```load_type(self) -> netdot.dataclasses.cables.CableType```

```
Load the type (CableType) associated to this BackboneCable.

Returns:
    netdot.CableType: The full CableType object if available, else None.
```

#### BackboneCable.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### BackboneCable.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.CableStrand`

<a id="class-netdotcablestrand"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| cable_xlink | int |  |
| circuit_id_xlink | int |  |
| description | str |  |
| fiber_type_xlink | int |  |
| info | str |  |
| name | str |  |
| number | int |  |
| status_xlink | int |  |


### Methods

#### CableStrand.add_splice_as_strand1

> ```add_splice_as_strand1(self, data: netdot.dataclasses.cables.Splice) -> netdot.dataclasses.cables.Splice```

```
Add a Splice to this CableStrand.

Returns:
    netdot.Splice: The created Splice.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CableStrand.add_splice_as_strand2

> ```add_splice_as_strand2(self, data: netdot.dataclasses.cables.Splice) -> netdot.dataclasses.cables.Splice```

```
Add a Splice to this CableStrand.

Returns:
    netdot.Splice: The created Splice.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CableStrand.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### CableStrand.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### CableStrand.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### CableStrand.load_cable

> ```load_cable(self) -> netdot.dataclasses.cables.BackboneCable```

```
Load the cable (BackboneCable) associated to this CableStrand.

Returns:
    netdot.BackboneCable: The full BackboneCable object if available, else None.
```

#### CableStrand.load_circuit

> ```load_circuit(self) -> netdot.dataclasses.cables.Circuit```

```
Load the circuit_id (Circuit) associated to this CableStrand.

Returns:
    netdot.Circuit: The full Circuit object if available, else None.
```

#### CableStrand.load_fiber_type

> ```load_fiber_type(self) -> netdot.dataclasses.cables.FiberType```

```
Load the fiber_type (FiberType) associated to this CableStrand.

Returns:
    netdot.FiberType: The full FiberType object if available, else None.
```

#### CableStrand.load_status

> ```load_status(self) -> netdot.dataclasses.cables.StrandStatus```

```
Load the status (StrandStatus) associated to this CableStrand.

Returns:
    netdot.StrandStatus: The full StrandStatus object if available, else None.
```

#### CableStrand.load_strand1_splices

> ```load_strand1_splices(self, ignore_404=True) -> List[netdot.dataclasses.cables.Splice]```

```
Load the Splices associated to this CableStrand. (Via the `Splice.strand1` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Splice]: All/Any Splices related to this CableStrand.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CableStrand.load_strand2_splices

> ```load_strand2_splices(self, ignore_404=True) -> List[netdot.dataclasses.cables.Splice]```

```
Load the Splices associated to this CableStrand. (Via the `Splice.strand2` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Splice]: All/Any Splices related to this CableStrand.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CableStrand.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### CableStrand.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.CableType`

<a id="class-netdotcabletype"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### CableType.add_backbonecable

> ```add_backbonecable(self, data: netdot.dataclasses.cables.BackboneCable) -> netdot.dataclasses.cables.BackboneCable```

```
Add a BackboneCable to this CableType.

Returns:
    netdot.BackboneCable: The created BackboneCable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CableType.add_horizontalcable

> ```add_horizontalcable(self, data: netdot.dataclasses.cables.HorizontalCable) -> netdot.dataclasses.cables.HorizontalCable```

```
Add a HorizontalCable to this CableType.

Returns:
    netdot.HorizontalCable: The created HorizontalCable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CableType.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### CableType.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### CableType.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### CableType.load_backbonecables

> ```load_backbonecables(self, ignore_404=True) -> List[netdot.dataclasses.cables.BackboneCable]```

```
Load the BackboneCables associated to this CableType. (Via the `BackboneCable.type` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.BackboneCable]: All/Any BackboneCables related to this CableType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CableType.load_horizontalcables

> ```load_horizontalcables(self, ignore_404=True) -> List[netdot.dataclasses.cables.HorizontalCable]```

```
Load the HorizontalCables associated to this CableType. (Via the `HorizontalCable.type` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.HorizontalCable]: All/Any HorizontalCables related to this CableType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CableType.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### CableType.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Circuit`

<a id="class-netdotcircuit"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| cid | str |  |
| info | str |  |
| installdate | DateTime |  |
| linkid_xlink | int |  |
| speed | str |  |
| status_xlink | int |  |
| type_xlink | int |  |
| vendor_xlink | int |  |
| datetested | DateTime |  |
| loss | str |  |


### Methods

#### Circuit.add_cablestrand

> ```add_cablestrand(self, data: netdot.dataclasses.cables.CableStrand) -> netdot.dataclasses.cables.CableStrand```

```
Add a CableStrand to this Circuit.

Returns:
    netdot.CableStrand: The created CableStrand.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Circuit.add_interface

> ```add_interface(self, data: netdot.dataclasses.interface.Interface) -> netdot.dataclasses.interface.Interface```

```
Add a Interface to this Circuit.

Returns:
    netdot.Interface: The created Interface.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Circuit.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Circuit.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Circuit.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Circuit.load_cablestrands

> ```load_cablestrands(self, ignore_404=True) -> List[netdot.dataclasses.cables.CableStrand]```

```
Load the CableStrands associated to this Circuit. (Via the `CableStrand.circuit_id` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.CableStrand]: All/Any CableStrands related to this Circuit.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Circuit.load_interfaces

> ```load_interfaces(self, ignore_404=True) -> List[netdot.dataclasses.interface.Interface]```

```
Load the Interfaces associated to this Circuit.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Interface]: All/Any Interfaces related to this Circuit.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Circuit.load_linkid

> ```load_linkid(self) -> netdot.dataclasses.site.SiteLink```

```
Load the linkid (SiteLink) associated to this Circuit.

Returns:
    netdot.SiteLink: The full SiteLink object if available, else None.
```

#### Circuit.load_status

> ```load_status(self) -> netdot.dataclasses.cables.CircuitStatus```

```
Load the status (CircuitStatus) associated to this Circuit.

Returns:
    netdot.CircuitStatus: The full CircuitStatus object if available, else None.
```

#### Circuit.load_type

> ```load_type(self) -> netdot.dataclasses.cables.CircuitType```

```
Load the type (CircuitType) associated to this Circuit.

Returns:
    netdot.CircuitType: The full CircuitType object if available, else None.
```

#### Circuit.load_vendor

> ```load_vendor(self) -> netdot.dataclasses.entity.Entity```

```
Load the vendor (Entity) associated to this Circuit.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### Circuit.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Circuit.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.CircuitStatus`

<a id="class-netdotcircuitstatus"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### CircuitStatus.add_circuit

> ```add_circuit(self, data: netdot.dataclasses.cables.Circuit) -> netdot.dataclasses.cables.Circuit```

```
Add a Circuit to this CircuitStatus.

Returns:
    netdot.Circuit: The created Circuit.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CircuitStatus.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### CircuitStatus.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### CircuitStatus.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### CircuitStatus.load_circuits

> ```load_circuits(self, ignore_404=True) -> List[netdot.dataclasses.cables.Circuit]```

```
Load the Circuits associated to this CircuitStatus. (Via the `Circuit.status` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Circuit]: All/Any Circuits related to this CircuitStatus.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CircuitStatus.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### CircuitStatus.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.CircuitType`

<a id="class-netdotcircuittype"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### CircuitType.add_circuit

> ```add_circuit(self, data: netdot.dataclasses.cables.Circuit) -> netdot.dataclasses.cables.Circuit```

```
Add a Circuit to this CircuitType.

Returns:
    netdot.Circuit: The created Circuit.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CircuitType.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### CircuitType.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### CircuitType.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### CircuitType.load_circuits

> ```load_circuits(self, ignore_404=True) -> List[netdot.dataclasses.cables.Circuit]```

```
Load the Circuits associated to this CircuitType. (Via the `Circuit.type` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Circuit]: All/Any Circuits related to this CircuitType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### CircuitType.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### CircuitType.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Closet`

<a id="class-netdotcloset"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| access_key_type | str |  |
| asbestos_tiles | bool | False |
| catv_taps | str |  |
| converted_patch_panels | bool | False |
| dimensions | str |  |
| ground_buss | bool | False |
| hvac_type | str |  |
| info | str |  |
| name | str |  |
| room_xlink | int |  |
| ot_blocks | str |  |
| outlets | str |  |
| pair_count | str |  |
| patch_panels | str |  |
| rack_type | str |  |
| racks | str |  |
| ru_avail | str |  |
| shared_with | str |  |
| ss_blocks | str |  |
| work_needed | str |  |


### Methods

#### Closet.add_backbonecable_as_end_closet

> ```add_backbonecable_as_end_closet(self, data: netdot.dataclasses.cables.BackboneCable) -> netdot.dataclasses.cables.BackboneCable```

```
Add a BackboneCable to this Closet.

Returns:
    netdot.BackboneCable: The created BackboneCable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Closet.add_backbonecable_as_start_closet

> ```add_backbonecable_as_start_closet(self, data: netdot.dataclasses.cables.BackboneCable) -> netdot.dataclasses.cables.BackboneCable```

```
Add a BackboneCable to this Closet.

Returns:
    netdot.BackboneCable: The created BackboneCable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Closet.add_horizontalcable

> ```add_horizontalcable(self, data: netdot.dataclasses.cables.HorizontalCable) -> netdot.dataclasses.cables.HorizontalCable```

```
Add a HorizontalCable to this Closet.

Returns:
    netdot.HorizontalCable: The created HorizontalCable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Closet.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Closet.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Closet.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Closet.load_end_closet_backbonecables

> ```load_end_closet_backbonecables(self, ignore_404=True) -> List[netdot.dataclasses.cables.BackboneCable]```

```
Load the BackboneCables associated to this Closet. (Via the `BackboneCable.end_closet` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.BackboneCable]: All/Any BackboneCables related to this Closet.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Closet.load_horizontalcables

> ```load_horizontalcables(self, ignore_404=True) -> List[netdot.dataclasses.cables.HorizontalCable]```

```
Load the HorizontalCables associated to this Closet.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.HorizontalCable]: All/Any HorizontalCables related to this Closet.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Closet.load_room

> ```load_room(self) -> netdot.dataclasses.site.Room```

```
Load the room (Room) associated to this Closet.

Returns:
    netdot.Room: The full Room object if available, else None.
```

#### Closet.load_start_closet_backbonecables

> ```load_start_closet_backbonecables(self, ignore_404=True) -> List[netdot.dataclasses.cables.BackboneCable]```

```
Load the BackboneCables associated to this Closet. (Via the `BackboneCable.start_closet` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.BackboneCable]: All/Any BackboneCables related to this Closet.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Closet.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Closet.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Contact`

<a id="class-netdotcontact"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| contactlist_xlink | int |  |
| contacttype_xlink | int |  |
| escalation_level | int |  |
| info | str |  |
| notify_email_xlink | int |  |
| notify_pager_xlink | int |  |
| notify_voice_xlink | int |  |
| person_xlink | int |  |


### Methods

#### Contact.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Contact.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Contact.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Contact.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this Contact.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### Contact.load_contacttype

> ```load_contacttype(self) -> netdot.dataclasses.users.ContactType```

```
Load the contacttype (ContactType) associated to this Contact.

Returns:
    netdot.ContactType: The full ContactType object if available, else None.
```

#### Contact.load_notify_email

> ```load_notify_email(self) -> netdot.dataclasses.misc.Availability```

```
Load the notify_email (Availability) associated to this Contact.

Returns:
    netdot.Availability: The full Availability object if available, else None.
```

#### Contact.load_notify_pager

> ```load_notify_pager(self) -> netdot.dataclasses.misc.Availability```

```
Load the notify_pager (Availability) associated to this Contact.

Returns:
    netdot.Availability: The full Availability object if available, else None.
```

#### Contact.load_notify_voice

> ```load_notify_voice(self) -> netdot.dataclasses.misc.Availability```

```
Load the notify_voice (Availability) associated to this Contact.

Returns:
    netdot.Availability: The full Availability object if available, else None.
```

#### Contact.load_person

> ```load_person(self) -> netdot.dataclasses.users.Person```

```
Load the person (Person) associated to this Contact.

Returns:
    netdot.Person: The full Person object if available, else None.
```

#### Contact.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Contact.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.ContactList`

<a id="class-netdotcontactlist"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### ContactList.add_accessright

> ```add_accessright(self, data: netdot.dataclasses.users.AccessRight) -> netdot.dataclasses.users.AccessRight```

```
Add a AccessRight to this ContactList (via GroupRight).

Args:
    data (netdot.AccessRight): The AccessRight to add to this ContactList.

Returns:
    netdot.GroupRight: The newly created GroupRight.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_bgppeering

> ```add_bgppeering(self, data: netdot.dataclasses.bgp.BGPPeering) -> netdot.dataclasses.bgp.BGPPeering```

```
Add a BGPPeering to this ContactList.

Returns:
    netdot.BGPPeering: The created BGPPeering.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_contact

> ```add_contact(self, data: netdot.dataclasses.users.Contact) -> netdot.dataclasses.users.Contact```

```
Add a Contact to this ContactList.

Returns:
    netdot.Contact: The created Contact.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this ContactList (via DeviceContacts).

Args:
    data (netdot.Device): The Device to add to this ContactList.

Returns:
    netdot.DeviceContacts: The newly created DeviceContacts.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_devicecontacts

> ```add_devicecontacts(self, data: netdot.dataclasses.device.DeviceContacts) -> netdot.dataclasses.device.DeviceContacts```

```
Add a DeviceContacts to this ContactList.

Returns:
    netdot.DeviceContacts: The created DeviceContacts.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_entity

> ```add_entity(self, data: netdot.dataclasses.entity.Entity) -> netdot.dataclasses.entity.Entity```

```
Add a Entity to this ContactList.

Returns:
    netdot.Entity: The created Entity.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_groupright

> ```add_groupright(self, data: netdot.dataclasses.users.GroupRight) -> netdot.dataclasses.users.GroupRight```

```
Add a GroupRight to this ContactList.

Returns:
    netdot.GroupRight: The created GroupRight.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_horizontalcable

> ```add_horizontalcable(self, data: netdot.dataclasses.cables.HorizontalCable) -> netdot.dataclasses.cables.HorizontalCable```

```
Add a HorizontalCable to this ContactList.

Returns:
    netdot.HorizontalCable: The created HorizontalCable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_interface

> ```add_interface(self, data: netdot.dataclasses.interface.Interface) -> netdot.dataclasses.interface.Interface```

```
Add a Interface to this ContactList.

Returns:
    netdot.Interface: The created Interface.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_ipservice

> ```add_ipservice(self, data: netdot.dataclasses.ipblock.IPService) -> netdot.dataclasses.ipblock.IPService```

```
Add a IPService to this ContactList.

Returns:
    netdot.IPService: The created IPService.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_site

> ```add_site(self, data: netdot.dataclasses.site.Site) -> netdot.dataclasses.site.Site```

```
Add a Site to this ContactList.

Returns:
    netdot.Site: The created Site.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.add_zone

> ```add_zone(self, data: netdot.dataclasses.dns.Zone) -> netdot.dataclasses.dns.Zone```

```
Add a Zone to this ContactList.

Returns:
    netdot.Zone: The created Zone.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### ContactList.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### ContactList.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### ContactList.load_accessrights

> ```load_accessrights(self, ignore_404=True) -> List[netdot.dataclasses.users.AccessRight]```

```
Load the accessrights (AccessRights) associated to this ContactList.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of AccessRights associated to this ContactList).

You might prefer :func:`load_grouprights` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.AccessRight]: Any/All AccessRights related to this ContactList, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_bgppeerings

> ```load_bgppeerings(self, ignore_404=True) -> List[netdot.dataclasses.bgp.BGPPeering]```

```
Load the BGPPeerings associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.BGPPeering]: All/Any BGPPeerings related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_contacts

> ```load_contacts(self, ignore_404=True) -> List[netdot.dataclasses.users.Contact]```

```
Load the Contacts associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Contact]: All/Any Contacts related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_devicecontacts

> ```load_devicecontacts(self, ignore_404=True) -> List[netdot.dataclasses.device.DeviceContacts]```

```
Load the DeviceContacts associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DeviceContacts]: All/Any DeviceContacts related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the devices (Devices) associated to this ContactList.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of Devices associated to this ContactList).

You might prefer :func:`load_devicecontacts` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: Any/All Devices related to this ContactList, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_entities

> ```load_entities(self, ignore_404=True) -> List[netdot.dataclasses.entity.Entity]```

```
Load the Entities associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Entity]: All/Any Entities related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_grouprights

> ```load_grouprights(self, ignore_404=True) -> List[netdot.dataclasses.users.GroupRight]```

```
Load the GroupRights associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.GroupRight]: All/Any GroupRights related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_horizontalcables

> ```load_horizontalcables(self, ignore_404=True) -> List[netdot.dataclasses.cables.HorizontalCable]```

```
Load the HorizontalCables associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.HorizontalCable]: All/Any HorizontalCables related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_interfaces

> ```load_interfaces(self, ignore_404=True) -> List[netdot.dataclasses.interface.Interface]```

```
Load the Interfaces associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Interface]: All/Any Interfaces related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_ipservices

> ```load_ipservices(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPService]```

```
Load the IPServices associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPService]: All/Any IPServices related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_sites

> ```load_sites(self, ignore_404=True) -> List[netdot.dataclasses.site.Site]```

```
Load the Sites associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Site]: All/Any Sites related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.load_zones

> ```load_zones(self, ignore_404=True) -> List[netdot.dataclasses.dns.Zone]```

```
Load the Zones associated to this ContactList.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Zone]: All/Any Zones related to this ContactList.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactList.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### ContactList.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.ContactType`

<a id="class-netdotcontacttype"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### ContactType.add_contact

> ```add_contact(self, data: netdot.dataclasses.users.Contact) -> netdot.dataclasses.users.Contact```

```
Add a Contact to this ContactType.

Returns:
    netdot.Contact: The created Contact.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactType.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### ContactType.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### ContactType.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### ContactType.load_contacts

> ```load_contacts(self, ignore_404=True) -> List[netdot.dataclasses.users.Contact]```

```
Load the Contacts associated to this ContactType.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Contact]: All/Any Contacts related to this ContactType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ContactType.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### ContactType.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DHCPAttr`

<a id="class-netdotdhcpattr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| name_xlink | int |  |
| scope_xlink | int |  |
| value | str |  |


### Methods

#### DHCPAttr.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DHCPAttr.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DHCPAttr.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DHCPAttr.load_name

> ```load_name(self) -> netdot.dataclasses.dhcp.DHCPAttrName```

```
Load the name (DHCPAttrName) associated to this DHCPAttr.

Returns:
    netdot.DHCPAttrName: The full DHCPAttrName object if available, else None.
```

#### DHCPAttr.load_scope

> ```load_scope(self) -> netdot.dataclasses.dhcp.DHCPScope```

```
Load the scope (DHCPScope) associated to this DHCPAttr.

Returns:
    netdot.DHCPScope: The full DHCPScope object if available, else None.
```

#### DHCPAttr.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DHCPAttr.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DHCPAttrName`

<a id="class-netdotdhcpattrname"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| code | int |  |
| format | str |  |
| info | str |  |
| name | str |  |


### Methods

#### DHCPAttrName.add_dhcpattr

> ```add_dhcpattr(self, data: netdot.dataclasses.dhcp.DHCPAttr) -> netdot.dataclasses.dhcp.DHCPAttr```

```
Add a DHCPAttr to this DHCPAttrName.

Returns:
    netdot.DHCPAttr: The created DHCPAttr.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPAttrName.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DHCPAttrName.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DHCPAttrName.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DHCPAttrName.load_dhcpattrs

> ```load_dhcpattrs(self, ignore_404=True) -> List[netdot.dataclasses.dhcp.DHCPAttr]```

```
Load the DHCPAttrs associated to this DHCPAttrName. (Via the `DHCPAttr.name` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DHCPAttr]: All/Any DHCPAttrs related to this DHCPAttrName.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPAttrName.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DHCPAttrName.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DHCPScope`

<a id="class-netdotdhcpscope"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| ipblock_xlink | int |  |
| text | str |  |
| name | str |  |
| container_xlink | int |  |
| physaddr_xlink | int |  |
| type_xlink | int |  |
| export_file | str |  |
| enable_failover | bool | False |
| failover_peer | str |  |
| active | bool | False |
| duid | str |  |
| version | int |  |


### Methods

#### DHCPScope.add_dhcpattr

> ```add_dhcpattr(self, data: netdot.dataclasses.dhcp.DHCPAttr) -> netdot.dataclasses.dhcp.DHCPAttr```

```
Add a DHCPAttr to this DHCPScope.

Returns:
    netdot.DHCPAttr: The created DHCPAttr.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScope.add_dhcpscope

> ```add_dhcpscope(self, data: netdot.dataclasses.dhcp.DHCPScope) -> netdot.dataclasses.dhcp.DHCPScope```

```
Add a DHCPScope to this DHCPScope.

Returns:
    netdot.DHCPScope: The created DHCPScope.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScope.add_dhcpscopeuse_as_scope

> ```add_dhcpscopeuse_as_scope(self, data: netdot.dataclasses.dhcp.DHCPScopeUse) -> netdot.dataclasses.dhcp.DHCPScopeUse```

```
Add a DHCPScopeUse to this DHCPScope.

Returns:
    netdot.DHCPScopeUse: The created DHCPScopeUse.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScope.add_dhcpscopeuse_as_template

> ```add_dhcpscopeuse_as_template(self, data: netdot.dataclasses.dhcp.DHCPScopeUse) -> netdot.dataclasses.dhcp.DHCPScopeUse```

```
Add a DHCPScopeUse to this DHCPScope.

Returns:
    netdot.DHCPScopeUse: The created DHCPScopeUse.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScope.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DHCPScope.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DHCPScope.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DHCPScope.load_container

> ```load_container(self) -> netdot.dataclasses.dhcp.DHCPScope```

```
Load the container (DHCPScope) associated to this DHCPScope.

Returns:
    netdot.DHCPScope: The full DHCPScope object if available, else None.
```

#### DHCPScope.load_dhcpattrs

> ```load_dhcpattrs(self, ignore_404=True) -> List[netdot.dataclasses.dhcp.DHCPAttr]```

```
Load the DHCPAttrs associated to this DHCPScope. (Via the `DHCPAttr.scope` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DHCPAttr]: All/Any DHCPAttrs related to this DHCPScope.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScope.load_dhcpscopes

> ```load_dhcpscopes(self, ignore_404=True) -> List[netdot.dataclasses.dhcp.DHCPScope]```

```
Load the DHCPScopes associated to this DHCPScope. (Via the `DHCPScope.container` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DHCPScope]: All/Any DHCPScopes related to this DHCPScope.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScope.load_ipblock

> ```load_ipblock(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the ipblock (IPBlock) associated to this DHCPScope.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### DHCPScope.load_physaddr

> ```load_physaddr(self) -> netdot.dataclasses.physaddr.PhysAddr```

```
Load the physaddr (PhysAddr) associated to this DHCPScope.

Returns:
    netdot.PhysAddr: The full PhysAddr object if available, else None.
```

#### DHCPScope.load_scope_dhcpscopeuses

> ```load_scope_dhcpscopeuses(self, ignore_404=True) -> List[netdot.dataclasses.dhcp.DHCPScopeUse]```

```
Load the DHCPScopeUses associated to this DHCPScope. (Via the `DHCPScopeUse.scope` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DHCPScopeUse]: All/Any DHCPScopeUses related to this DHCPScope.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScope.load_template_dhcpscopeuses

> ```load_template_dhcpscopeuses(self, ignore_404=True) -> List[netdot.dataclasses.dhcp.DHCPScopeUse]```

```
Load the DHCPScopeUses associated to this DHCPScope. (Via the `DHCPScopeUse.template` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DHCPScopeUse]: All/Any DHCPScopeUses related to this DHCPScope.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScope.load_type

> ```load_type(self) -> netdot.dataclasses.dhcp.DHCPScopeType```

```
Load the type (DHCPScopeType) associated to this DHCPScope.

Returns:
    netdot.DHCPScopeType: The full DHCPScopeType object if available, else None.
```

#### DHCPScope.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DHCPScope.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DHCPScopeType`

<a id="class-netdotdhcpscopetype"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### DHCPScopeType.add_dhcpscope

> ```add_dhcpscope(self, data: netdot.dataclasses.dhcp.DHCPScope) -> netdot.dataclasses.dhcp.DHCPScope```

```
Add a DHCPScope to this DHCPScopeType.

Returns:
    netdot.DHCPScope: The created DHCPScope.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScopeType.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DHCPScopeType.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DHCPScopeType.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DHCPScopeType.load_dhcpscopes

> ```load_dhcpscopes(self, ignore_404=True) -> List[netdot.dataclasses.dhcp.DHCPScope]```

```
Load the DHCPScopes associated to this DHCPScopeType. (Via the `DHCPScope.type` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DHCPScope]: All/Any DHCPScopes related to this DHCPScopeType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DHCPScopeType.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DHCPScopeType.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DHCPScopeUse`

<a id="class-netdotdhcpscopeuse"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| scope_xlink | int |  |
| template_xlink | int |  |


### Methods

#### DHCPScopeUse.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DHCPScopeUse.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DHCPScopeUse.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DHCPScopeUse.load_scope

> ```load_scope(self) -> netdot.dataclasses.dhcp.DHCPScope```

```
Load the scope (DHCPScope) associated to this DHCPScopeUse.

Returns:
    netdot.DHCPScope: The full DHCPScope object if available, else None.
```

#### DHCPScopeUse.load_template

> ```load_template(self) -> netdot.dataclasses.dhcp.DHCPScope```

```
Load the template (DHCPScope) associated to this DHCPScopeUse.

Returns:
    netdot.DHCPScope: The full DHCPScope object if available, else None.
```

#### DHCPScopeUse.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DHCPScopeUse.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Device`

<a id="class-netdotdevice"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| site_xlink | int |  |
| asset_id_xlink | int |  |
| monitorstatus_xlink | int |  |
| name_xlink | int |  |
| host_device_xlink | int |  |
| bgplocalas_xlink | int |  |
| snmp_target_xlink | int |  |
| room_xlink | int |  |
| owner_xlink | int |  |
| used_by_xlink | int |  |
| aliases | str |  |
| bgpid | str |  |
| canautoupdate | bool |  |
| collect_arp | bool |  |
| collect_fwt | bool |  |
| collect_stp | bool |  |
| community | str |  |
| customer_managed | bool |  |
| date_installed | DateTime |  |
| down_from | DateTime |  |
| down_until | DateTime |  |
| info | str |  |
| ipforwarding | bool |  |
| last_arp | DateTime |  |
| last_fwt | DateTime |  |
| last_updated | DateTime |  |
| layers | str |  |
| monitor_config | bool |  |
| monitor_config_group | str |  |
| monitored | bool |  |
| monitoring_path_cost | int |  |
| oobname | str |  |
| oobnumber | str |  |
| os | str |  |
| rack | str |  |
| snmp_authkey | str |  |
| snmp_authprotocol | str |  |
| snmp_bulk | bool |  |
| snmp_managed | bool |  |
| snmp_polling | bool |  |
| snmp_privkey | str |  |
| snmp_privprotocol | str |  |
| snmp_securitylevel | str |  |
| snmp_securityname | str |  |
| snmp_version | int |  |
| stp_enabled | bool |  |
| stp_mst_digest | str |  |
| stp_mst_region | str |  |
| stp_mst_rev | str |  |
| stp_type | str |  |
| sysdescription | str |  |
| syslocation | str |  |
| sysname | str |  |
| auto_dns | bool |  |
| extension | str |  |
| snmp_conn_attempts | int |  |
| snmp_down | bool |  |
| oobname_2 | str |  |
| oobnumber_2 | str |  |
| power_outlet | str |  |
| power_outlet_2 | str |  |
| monitoring_template | str |  |


### Methods

#### Device.add_arpcache

> ```add_arpcache(self, data: netdot.dataclasses.arpcache.ArpCache) -> netdot.dataclasses.arpcache.ArpCache```

```
Add a ArpCache to this Device.

Returns:
    netdot.ArpCache: The created ArpCache.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_bgppeering

> ```add_bgppeering(self, data: netdot.dataclasses.bgp.BGPPeering) -> netdot.dataclasses.bgp.BGPPeering```

```
Add a BGPPeering to this Device.

Returns:
    netdot.BGPPeering: The created BGPPeering.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_contactlist

> ```add_contactlist(self, data: netdot.dataclasses.users.ContactList) -> netdot.dataclasses.users.ContactList```

```
Add a ContactList to this Device (via DeviceContacts).

Args:
    data (netdot.ContactList): The ContactList to add to this Device.

Returns:
    netdot.DeviceContacts: The newly created DeviceContacts.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this Device.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_deviceattr

> ```add_deviceattr(self, data: netdot.dataclasses.device.DeviceAttr) -> netdot.dataclasses.device.DeviceAttr```

```
Add a DeviceAttr to this Device.

Returns:
    netdot.DeviceAttr: The created DeviceAttr.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_devicecontacts

> ```add_devicecontacts(self, data: netdot.dataclasses.device.DeviceContacts) -> netdot.dataclasses.device.DeviceContacts```

```
Add a DeviceContacts to this Device.

Returns:
    netdot.DeviceContacts: The created DeviceContacts.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_devicemodule

> ```add_devicemodule(self, data: netdot.dataclasses.device.DeviceModule) -> netdot.dataclasses.device.DeviceModule```

```
Add a DeviceModule to this Device.

Returns:
    netdot.DeviceModule: The created DeviceModule.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_fwtable

> ```add_fwtable(self, data: netdot.dataclasses.fwtable.FWTable) -> netdot.dataclasses.fwtable.FWTable```

```
Add a FWTable to this Device.

Returns:
    netdot.FWTable: The created FWTable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_interface

> ```add_interface(self, data: netdot.dataclasses.interface.Interface) -> netdot.dataclasses.interface.Interface```

```
Add a Interface to this Device.

Returns:
    netdot.Interface: The created Interface.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.add_stpinstance

> ```add_stpinstance(self, data: netdot.dataclasses.device.STPInstance) -> netdot.dataclasses.device.STPInstance```

```
Add a STPInstance to this Device.

Returns:
    netdot.STPInstance: The created STPInstance.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Device.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Device.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Device.infer_base_MAC

> ```infer_base_MAC(self) -> netdot.mac_address.MACAddress```

```
Infer the base_MAC address of this device from the asset_id (str, or Asset object).

Returns:
    MACAddress: The 'base MAC' of the device.

Raises:
    ValueError: If the asset_id does not contain a parsable MACAddress.
```

#### Device.infer_product

> ```infer_product(self) -> 'netdot.Product'```

```
Infer the Product of this device (based on its `asset_id` string returned from Netdot REST API).

> NOTE: One HTTP Request is made to retrieve all Products from Netdot.
> All subsequent calls to this method will use the cached results (see :func:`Repository.load_product_index`).

Returns:
    netdot.Product: The Product associated to this Device, or None (if there is no Product yet associated to this device).
```

#### Device.load_arpcaches

> ```load_arpcaches(self, ignore_404=True) -> List[netdot.dataclasses.arpcache.ArpCache]```

```
Load the ArpCaches associated to this Device.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.ArpCache]: All/Any ArpCaches related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_asset

> ```load_asset(self) -> netdot.dataclasses.asset.Asset```

```
Load the asset_id (Asset) associated to this Device.

Returns:
    netdot.Asset: The full Asset object if available, else None.
```

#### Device.load_bgplocalas

> ```load_bgplocalas(self) -> netdot.dataclasses.bgp.ASN```

```
Load the bgplocalas (ASN) associated to this Device.

Returns:
    netdot.ASN: The full ASN object if available, else None.
```

#### Device.load_bgppeerings

> ```load_bgppeerings(self, ignore_404=True) -> List[netdot.dataclasses.bgp.BGPPeering]```

```
Load the BGPPeerings associated to this Device.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.BGPPeering]: All/Any BGPPeerings related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_contactlists

> ```load_contactlists(self, ignore_404=True) -> List[netdot.dataclasses.users.ContactList]```

```
Load the contactlists (ContactLists) associated to this Device.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of ContactLists associated to this Device).

You might prefer :func:`load_devicecontacts` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.ContactList]: Any/All ContactLists related to this Device, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_deviceattrs

> ```load_deviceattrs(self, ignore_404=True) -> List[netdot.dataclasses.device.DeviceAttr]```

```
Load the DeviceAttrs associated to this Device.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DeviceAttr]: All/Any DeviceAttrs related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_devicecontacts

> ```load_devicecontacts(self, ignore_404=True) -> List[netdot.dataclasses.device.DeviceContacts]```

```
Load the DeviceContacts associated to this Device.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DeviceContacts]: All/Any DeviceContacts related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_devicemodules

> ```load_devicemodules(self, ignore_404=True) -> List[netdot.dataclasses.device.DeviceModule]```

```
Load the DeviceModules associated to this Device.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DeviceModule]: All/Any DeviceModules related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this Device. (Via the `Device.host_device` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_fwtables

> ```load_fwtables(self, ignore_404=True) -> List[netdot.dataclasses.fwtable.FWTable]```

```
Load the FWTables associated to this Device.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.FWTable]: All/Any FWTables related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_host_device

> ```load_host_device(self) -> netdot.dataclasses.device.Device```

```
Load the host_device (Device) associated to this Device.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### Device.load_interfaces

> ```load_interfaces(self, ignore_404=True) -> List[netdot.dataclasses.interface.Interface]```

```
Load the Interfaces associated to this Device.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Interface]: All/Any Interfaces related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_monitorstatus

> ```load_monitorstatus(self) -> netdot.dataclasses.misc.MonitorStatus```

```
Load the monitorstatus (MonitorStatus) associated to this Device.

Returns:
    netdot.MonitorStatus: The full MonitorStatus object if available, else None.
```

#### Device.load_name

> ```load_name(self) -> netdot.dataclasses.dns.RR```

```
Load the name (RR) associated to this Device.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### Device.load_owner

> ```load_owner(self) -> netdot.dataclasses.entity.Entity```

```
Load the owner (Entity) associated to this Device.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### Device.load_room

> ```load_room(self) -> netdot.dataclasses.site.Room```

```
Load the room (Room) associated to this Device.

Returns:
    netdot.Room: The full Room object if available, else None.
```

#### Device.load_site

> ```load_site(self) -> netdot.dataclasses.site.Site```

```
Load the site (Site) associated to this Device.

Returns:
    netdot.Site: The full Site object if available, else None.
```

#### Device.load_snmp_target

> ```load_snmp_target(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the snmp_target (IPBlock) associated to this Device.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### Device.load_stpinstances

> ```load_stpinstances(self, ignore_404=True) -> List[netdot.dataclasses.device.STPInstance]```

```
Load the STPInstances associated to this Device.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.STPInstance]: All/Any STPInstances related to this Device.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Device.load_used_by

> ```load_used_by(self) -> netdot.dataclasses.entity.Entity```

```
Load the used_by (Entity) associated to this Device.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### Device.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Device.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DeviceAttr`

<a id="class-netdotdeviceattr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| device_xlink | int |  |
| name_xlink | int |  |
| value | str |  |


### Methods

#### DeviceAttr.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DeviceAttr.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DeviceAttr.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DeviceAttr.load_device

> ```load_device(self) -> netdot.dataclasses.device.Device```

```
Load the device (Device) associated to this DeviceAttr.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### DeviceAttr.load_name

> ```load_name(self) -> netdot.dataclasses.device.DeviceAttrName```

```
Load the name (DeviceAttrName) associated to this DeviceAttr.

Returns:
    netdot.DeviceAttrName: The full DeviceAttrName object if available, else None.
```

#### DeviceAttr.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DeviceAttr.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DeviceAttrName`

<a id="class-netdotdeviceattrname"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### DeviceAttrName.add_deviceattr

> ```add_deviceattr(self, data: netdot.dataclasses.device.DeviceAttr) -> netdot.dataclasses.device.DeviceAttr```

```
Add a DeviceAttr to this DeviceAttrName.

Returns:
    netdot.DeviceAttr: The created DeviceAttr.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DeviceAttrName.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DeviceAttrName.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DeviceAttrName.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DeviceAttrName.load_deviceattrs

> ```load_deviceattrs(self, ignore_404=True) -> List[netdot.dataclasses.device.DeviceAttr]```

```
Load the DeviceAttrs associated to this DeviceAttrName. (Via the `DeviceAttr.name` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DeviceAttr]: All/Any DeviceAttrs related to this DeviceAttrName.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### DeviceAttrName.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DeviceAttrName.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DeviceContacts`

<a id="class-netdotdevicecontacts"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| contactlist_xlink | int |  |
| device_xlink | int |  |


### Methods

#### DeviceContacts.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DeviceContacts.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DeviceContacts.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DeviceContacts.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this DeviceContacts.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### DeviceContacts.load_device

> ```load_device(self) -> netdot.dataclasses.device.Device```

```
Load the device (Device) associated to this DeviceContacts.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### DeviceContacts.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DeviceContacts.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.DeviceModule`

<a id="class-netdotdevicemodule"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| class__KEYWORD_ESC | str |  |
| contained_in | int |  |
| date_installed | DateTime |  |
| description | str |  |
| device_xlink | int |  |
| fru | bool | False |
| fw_rev | str |  |
| hw_rev | str |  |
| last_updated | DateTime |  |
| model | str |  |
| name | str |  |
| number | int |  |
| pos | int |  |
| sw_rev | str |  |
| type | str |  |
| asset_id_xlink | int |  |


### Methods

#### DeviceModule.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### DeviceModule.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### DeviceModule.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### DeviceModule.load_asset

> ```load_asset(self) -> netdot.dataclasses.asset.Asset```

```
Load the asset_id (Asset) associated to this DeviceModule.

Returns:
    netdot.Asset: The full Asset object if available, else None.
```

#### DeviceModule.load_device

> ```load_device(self) -> netdot.dataclasses.device.Device```

```
Load the device (Device) associated to this DeviceModule.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### DeviceModule.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### DeviceModule.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Entity`

<a id="class-netdotentity"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| acctnumber | str |  |
| aliases | str |  |
| asname | str |  |
| asnumber | int |  |
| availability_xlink | int |  |
| contactlist_xlink | int |  |
| info | str |  |
| maint_contract | str |  |
| name | str |  |
| oid | str |  |
| short_name | str |  |
| config_type | str |  |


### Methods

#### Entity.add_backbonecable

> ```add_backbonecable(self, data: netdot.dataclasses.cables.BackboneCable) -> netdot.dataclasses.cables.BackboneCable```

```
Add a BackboneCable to this Entity.

Returns:
    netdot.BackboneCable: The created BackboneCable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_bgppeering

> ```add_bgppeering(self, data: netdot.dataclasses.bgp.BGPPeering) -> netdot.dataclasses.bgp.BGPPeering```

```
Add a BGPPeering to this Entity.

Returns:
    netdot.BGPPeering: The created BGPPeering.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_circuit

> ```add_circuit(self, data: netdot.dataclasses.cables.Circuit) -> netdot.dataclasses.cables.Circuit```

```
Add a Circuit to this Entity.

Returns:
    netdot.Circuit: The created Circuit.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_device_as_owner

> ```add_device_as_owner(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this Entity.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_device_as_used_by

> ```add_device_as_used_by(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this Entity.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_entityrole

> ```add_entityrole(self, data: netdot.dataclasses.entity.EntityRole) -> netdot.dataclasses.entity.EntityRole```

```
Add a EntityRole to this Entity.

Returns:
    netdot.EntityRole: The created EntityRole.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_entitysite

> ```add_entitysite(self, data: netdot.dataclasses.entity.EntitySite) -> netdot.dataclasses.entity.EntitySite```

```
Add a EntitySite to this Entity.

Returns:
    netdot.EntitySite: The created EntitySite.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_ipblock_as_owner

> ```add_ipblock_as_owner(self, data: netdot.dataclasses.ipblock.IPBlock) -> netdot.dataclasses.ipblock.IPBlock```

```
Add a IPBlock to this Entity.

Returns:
    netdot.IPBlock: The created IPBlock.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_ipblock_as_used_by

> ```add_ipblock_as_used_by(self, data: netdot.dataclasses.ipblock.IPBlock) -> netdot.dataclasses.ipblock.IPBlock```

```
Add a IPBlock to this Entity.

Returns:
    netdot.IPBlock: The created IPBlock.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_maintcontract

> ```add_maintcontract(self, data: netdot.dataclasses.misc.MaintContract) -> netdot.dataclasses.misc.MaintContract```

```
Add a MaintContract to this Entity.

Returns:
    netdot.MaintContract: The created MaintContract.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_person

> ```add_person(self, data: netdot.dataclasses.users.Person) -> netdot.dataclasses.users.Person```

```
Add a Person to this Entity.

Returns:
    netdot.Person: The created Person.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_product

> ```add_product(self, data: netdot.dataclasses.products.Product) -> netdot.dataclasses.products.Product```

```
Add a Product to this Entity.

Returns:
    netdot.Product: The created Product.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_site

> ```add_site(self, data: netdot.dataclasses.site.Site) -> netdot.dataclasses.site.Site```

```
Add a Site to this Entity (via EntitySite).

Args:
    data (netdot.Site): The Site to add to this Entity.

Returns:
    netdot.EntitySite: The newly created EntitySite.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_sitelink

> ```add_sitelink(self, data: netdot.dataclasses.site.SiteLink) -> netdot.dataclasses.site.SiteLink```

```
Add a SiteLink to this Entity.

Returns:
    netdot.SiteLink: The created SiteLink.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.add_type

> ```add_type(self, data: netdot.dataclasses.entity.EntityType) -> netdot.dataclasses.entity.EntityType```

```
Add a EntityType to this Entity (via EntityRole).

Args:
    data (netdot.EntityType): The EntityType to add to this Entity.

Returns:
    netdot.EntityRole: The newly created EntityRole.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Entity.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Entity.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Entity.load_availability

> ```load_availability(self) -> netdot.dataclasses.misc.Availability```

```
Load the availability (Availability) associated to this Entity.

Returns:
    netdot.Availability: The full Availability object if available, else None.
```

#### Entity.load_backbonecables

> ```load_backbonecables(self, ignore_404=True) -> List[netdot.dataclasses.cables.BackboneCable]```

```
Load the BackboneCables associated to this Entity. (Via the `BackboneCable.owner` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.BackboneCable]: All/Any BackboneCables related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_bgppeerings

> ```load_bgppeerings(self, ignore_404=True) -> List[netdot.dataclasses.bgp.BGPPeering]```

```
Load the BGPPeerings associated to this Entity.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.BGPPeering]: All/Any BGPPeerings related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_circuits

> ```load_circuits(self, ignore_404=True) -> List[netdot.dataclasses.cables.Circuit]```

```
Load the Circuits associated to this Entity. (Via the `Circuit.vendor` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Circuit]: All/Any Circuits related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this Entity.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### Entity.load_entityroles

> ```load_entityroles(self, ignore_404=True) -> List[netdot.dataclasses.entity.EntityRole]```

```
Load the EntityRoles associated to this Entity.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.EntityRole]: All/Any EntityRoles related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_entitysites

> ```load_entitysites(self, ignore_404=True) -> List[netdot.dataclasses.entity.EntitySite]```

```
Load the EntitySites associated to this Entity.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.EntitySite]: All/Any EntitySites related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_maintcontracts

> ```load_maintcontracts(self, ignore_404=True) -> List[netdot.dataclasses.misc.MaintContract]```

```
Load the MaintContracts associated to this Entity. (Via the `MaintContract.provider` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.MaintContract]: All/Any MaintContracts related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_owner_devices

> ```load_owner_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this Entity. (Via the `Device.owner` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_owner_ipblocks

> ```load_owner_ipblocks(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Load the IPBlocks associated to this Entity. (Via the `IPBlock.owner` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlock]: All/Any IPBlocks related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_persons

> ```load_persons(self, ignore_404=True) -> List[netdot.dataclasses.users.Person]```

```
Load the Persons associated to this Entity.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Person]: All/Any Persons related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_products

> ```load_products(self, ignore_404=True) -> List[netdot.dataclasses.products.Product]```

```
Load the Products associated to this Entity. (Via the `Product.manufacturer` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Product]: All/Any Products related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_sitelinks

> ```load_sitelinks(self, ignore_404=True) -> List[netdot.dataclasses.site.SiteLink]```

```
Load the SiteLinks associated to this Entity.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.SiteLink]: All/Any SiteLinks related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_sites

> ```load_sites(self, ignore_404=True) -> List[netdot.dataclasses.site.Site]```

```
Load the sites (Sites) associated to this Entity.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of Sites associated to this Entity).

You might prefer :func:`load_entitysites` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Site]: Any/All Sites related to this Entity, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_types

> ```load_types(self, ignore_404=True) -> List[netdot.dataclasses.entity.EntityType]```

```
Load the types (EntityTypes) associated to this Entity.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of EntityTypes associated to this Entity).

You might prefer :func:`load_entityroles` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.EntityType]: Any/All EntityTypes related to this Entity, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_used_by_devices

> ```load_used_by_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this Entity. (Via the `Device.used_by` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.load_used_by_ipblocks

> ```load_used_by_ipblocks(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Load the IPBlocks associated to this Entity. (Via the `IPBlock.used_by` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlock]: All/Any IPBlocks related to this Entity.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Entity.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Entity.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.EntityRole`

<a id="class-netdotentityrole"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| entity_xlink | int |  |
| type_xlink | int |  |


### Methods

#### EntityRole.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### EntityRole.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### EntityRole.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### EntityRole.load_entity

> ```load_entity(self) -> netdot.dataclasses.entity.Entity```

```
Load the entity (Entity) associated to this EntityRole.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### EntityRole.load_type

> ```load_type(self) -> netdot.dataclasses.entity.EntityType```

```
Load the type (EntityType) associated to this EntityRole.

Returns:
    netdot.EntityType: The full EntityType object if available, else None.
```

#### EntityRole.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### EntityRole.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.EntitySite`

<a id="class-netdotentitysite"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| entity_xlink | int |  |
| site_xlink | int |  |


### Methods

#### EntitySite.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### EntitySite.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### EntitySite.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### EntitySite.load_entity

> ```load_entity(self) -> netdot.dataclasses.entity.Entity```

```
Load the entity (Entity) associated to this EntitySite.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### EntitySite.load_site

> ```load_site(self) -> netdot.dataclasses.site.Site```

```
Load the site (Site) associated to this EntitySite.

Returns:
    netdot.Site: The full Site object if available, else None.
```

#### EntitySite.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### EntitySite.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.EntityType`

<a id="class-netdotentitytype"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### EntityType.add_entity

> ```add_entity(self, data: netdot.dataclasses.entity.Entity) -> netdot.dataclasses.entity.Entity```

```
Add a Entity to this EntityType (via EntityRole).

Args:
    data (netdot.Entity): The Entity to add to this EntityType.

Returns:
    netdot.EntityRole: The newly created EntityRole.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### EntityType.add_entityrole

> ```add_entityrole(self, data: netdot.dataclasses.entity.EntityRole) -> netdot.dataclasses.entity.EntityRole```

```
Add a EntityRole to this EntityType.

Returns:
    netdot.EntityRole: The created EntityRole.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### EntityType.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### EntityType.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### EntityType.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### EntityType.load_entities

> ```load_entities(self, ignore_404=True) -> List[netdot.dataclasses.entity.Entity]```

```
Load the entities (Entities) associated to this EntityType.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of Entities associated to this EntityType).

You might prefer :func:`load_entityroles` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Entity]: Any/All Entities related to this EntityType, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### EntityType.load_entityroles

> ```load_entityroles(self, ignore_404=True) -> List[netdot.dataclasses.entity.EntityRole]```

```
Load the EntityRoles associated to this EntityType. (Via the `EntityRole.type` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.EntityRole]: All/Any EntityRoles related to this EntityType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### EntityType.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### EntityType.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.FWTable`

<a id="class-netdotfwtable"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| device_xlink | int |  |
| tstamp | DateTime |  |


### Methods

#### FWTable.add_fwtableentry

> ```add_fwtableentry(self, data: netdot.dataclasses.fwtable.FWTableEntry) -> netdot.dataclasses.fwtable.FWTableEntry```

```
Add a FWTableEntry to this FWTable.

Returns:
    netdot.FWTableEntry: The created FWTableEntry.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### FWTable.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### FWTable.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### FWTable.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### FWTable.load_device

> ```load_device(self) -> netdot.dataclasses.device.Device```

```
Load the device (Device) associated to this FWTable.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### FWTable.load_fwtableentries

> ```load_fwtableentries(self, ignore_404=True) -> List[netdot.dataclasses.fwtable.FWTableEntry]```

```
Load the FWTableEntries associated to this FWTable.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.FWTableEntry]: All/Any FWTableEntries related to this FWTable.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### FWTable.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### FWTable.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.FWTableEntry`

<a id="class-netdotfwtableentry"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| fwtable_xlink | int |  |
| interface_xlink | int |  |
| physaddr_xlink | int |  |


### Methods

#### FWTableEntry.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### FWTableEntry.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### FWTableEntry.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### FWTableEntry.infer_timestamp

> ```infer_timestamp(self) -> datetime.datetime```

```
Infer the timestamp of this FWTableEntry (based on the 'fwtable' string returned to us by Netdot API).

Returns:
    datetime: The inferred timestamp.

Raises:
    ValueError: If unable to infer the timestamp.
```

#### FWTableEntry.load_fwtable

> ```load_fwtable(self) -> netdot.dataclasses.fwtable.FWTable```

```
Load the fwtable (FWTable) associated to this FWTableEntry.

Returns:
    netdot.FWTable: The full FWTable object if available, else None.
```

#### FWTableEntry.load_interface

> ```load_interface(self) -> netdot.dataclasses.interface.Interface```

```
Load the interface (Interface) associated to this FWTableEntry.

Returns:
    netdot.Interface: The full Interface object if available, else None.
```

#### FWTableEntry.load_physaddr

> ```load_physaddr(self) -> netdot.dataclasses.physaddr.PhysAddr```

```
Load the physaddr (PhysAddr) associated to this FWTableEntry.

Returns:
    netdot.PhysAddr: The full PhysAddr object if available, else None.
```

#### FWTableEntry.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### FWTableEntry.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.FiberType`

<a id="class-netdotfibertype"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### FiberType.add_cablestrand

> ```add_cablestrand(self, data: netdot.dataclasses.cables.CableStrand) -> netdot.dataclasses.cables.CableStrand```

```
Add a CableStrand to this FiberType.

Returns:
    netdot.CableStrand: The created CableStrand.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### FiberType.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### FiberType.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### FiberType.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### FiberType.load_cablestrands

> ```load_cablestrands(self, ignore_404=True) -> List[netdot.dataclasses.cables.CableStrand]```

```
Load the CableStrands associated to this FiberType. (Via the `CableStrand.fiber_type` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.CableStrand]: All/Any CableStrands related to this FiberType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### FiberType.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### FiberType.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Floor`

<a id="class-netdotfloor"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| level | str |  |
| site_xlink | int |  |


### Methods

#### Floor.add_room

> ```add_room(self, data: netdot.dataclasses.site.Room) -> netdot.dataclasses.site.Room```

```
Add a Room to this Floor.

Returns:
    netdot.Room: The created Room.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Floor.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Floor.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Floor.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Floor.load_rooms

> ```load_rooms(self, ignore_404=True) -> List[netdot.dataclasses.site.Room]```

```
Load the Rooms associated to this Floor.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Room]: All/Any Rooms related to this Floor.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Floor.load_site

> ```load_site(self) -> netdot.dataclasses.site.Site```

```
Load the site (Site) associated to this Floor.

Returns:
    netdot.Site: The full Site object if available, else None.
```

#### Floor.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Floor.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.GroupRight`

<a id="class-netdotgroupright"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| accessright_xlink | int |  |
| contactlist_xlink | int |  |


### Methods

#### GroupRight.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### GroupRight.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### GroupRight.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### GroupRight.load_accessright

> ```load_accessright(self) -> netdot.dataclasses.users.AccessRight```

```
Load the accessright (AccessRight) associated to this GroupRight.

Returns:
    netdot.AccessRight: The full AccessRight object if available, else None.
```

#### GroupRight.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this GroupRight.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### GroupRight.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### GroupRight.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.HorizontalCable`

<a id="class-netdothorizontalcable"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| account | str |  |
| closet_xlink | int |  |
| contactlist_xlink | int |  |
| datetested | DateTime |  |
| faceplateid | str |  |
| info | str |  |
| installdate | DateTime |  |
| jackid | str |  |
| length | str |  |
| room_xlink | int |  |
| testpassed | bool | False |
| type_xlink | int |  |


### Methods

#### HorizontalCable.add_interface

> ```add_interface(self, data: netdot.dataclasses.interface.Interface) -> netdot.dataclasses.interface.Interface```

```
Add a Interface to this HorizontalCable.

Returns:
    netdot.Interface: The created Interface.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### HorizontalCable.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### HorizontalCable.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### HorizontalCable.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### HorizontalCable.load_closet

> ```load_closet(self) -> netdot.dataclasses.site.Closet```

```
Load the closet (Closet) associated to this HorizontalCable.

Returns:
    netdot.Closet: The full Closet object if available, else None.
```

#### HorizontalCable.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this HorizontalCable.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### HorizontalCable.load_interfaces

> ```load_interfaces(self, ignore_404=True) -> List[netdot.dataclasses.interface.Interface]```

```
Load the Interfaces associated to this HorizontalCable. (Via the `Interface.jack` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Interface]: All/Any Interfaces related to this HorizontalCable.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### HorizontalCable.load_room

> ```load_room(self) -> netdot.dataclasses.site.Room```

```
Load the room (Room) associated to this HorizontalCable.

Returns:
    netdot.Room: The full Room object if available, else None.
```

#### HorizontalCable.load_type

> ```load_type(self) -> netdot.dataclasses.cables.CableType```

```
Load the type (CableType) associated to this HorizontalCable.

Returns:
    netdot.CableType: The full CableType object if available, else None.
```

#### HorizontalCable.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### HorizontalCable.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.HostAudit`

<a id="class-netdothostaudit"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| tstamp | DateTime |  |
| zone | str |  |
| scope | str |  |
| pending | bool | False |


### Methods

#### HostAudit.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### HostAudit.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### HostAudit.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### HostAudit.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### HostAudit.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.IPBlock`

<a id="class-netdotipblock"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| address | ip_address |  |
| description | str |  |
| first_seen | DateTime |  |
| info | str |  |
| interface_xlink | int |  |
| last_seen | DateTime |  |
| owner_xlink | int |  |
| parent_xlink | int |  |
| prefix | int |  |
| status_xlink | int |  |
| used_by_xlink | int |  |
| version | int |  |
| vlan_xlink | int |  |
| use_network_broadcast | bool | False |
| monitored | bool | False |
| rir | str |  |
| asn_xlink | int |  |


### Methods

#### IPBlock.add_arpcacheentry

> ```add_arpcacheentry(self, data: netdot.dataclasses.arpcache.ArpCacheEntry) -> netdot.dataclasses.arpcache.ArpCacheEntry```

```
Add a ArpCacheEntry to this IPBlock.

Returns:
    netdot.ArpCacheEntry: The created ArpCacheEntry.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this IPBlock.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_dhcpscope

> ```add_dhcpscope(self, data: netdot.dataclasses.dhcp.DHCPScope) -> netdot.dataclasses.dhcp.DHCPScope```

```
Add a DHCPScope to this IPBlock.

Returns:
    netdot.DHCPScope: The created DHCPScope.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_ipblock

> ```add_ipblock(self, data: netdot.dataclasses.ipblock.IPBlock) -> netdot.dataclasses.ipblock.IPBlock```

```
Add a IPBlock to this IPBlock.

Returns:
    netdot.IPBlock: The created IPBlock.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_ipblockattr

> ```add_ipblockattr(self, data: netdot.dataclasses.ipblock.IPBlockAttr) -> netdot.dataclasses.ipblock.IPBlockAttr```

```
Add a IPBlockAttr to this IPBlock.

Returns:
    netdot.IPBlockAttr: The created IPBlockAttr.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_ipservice

> ```add_ipservice(self, data: netdot.dataclasses.ipblock.IPService) -> netdot.dataclasses.ipblock.IPService```

```
Add a IPService to this IPBlock.

Returns:
    netdot.IPService: The created IPService.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_rraddr

> ```add_rraddr(self, data: netdot.dataclasses.dns.RRADDR) -> netdot.dataclasses.dns.RRADDR```

```
Add a RRADDR to this IPBlock.

Returns:
    netdot.RRADDR: The created RRADDR.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_rrptr

> ```add_rrptr(self, data: netdot.dataclasses.dns.RRPTR) -> netdot.dataclasses.dns.RRPTR```

```
Add a RRPTR to this IPBlock.

Returns:
    netdot.RRPTR: The created RRPTR.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_site

> ```add_site(self, data: netdot.dataclasses.site.Site) -> netdot.dataclasses.site.Site```

```
Add a Site to this IPBlock (via SiteSubnet).

Args:
    data (netdot.Site): The Site to add to this IPBlock.

Returns:
    netdot.SiteSubnet: The newly created SiteSubnet.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_sitesubnet

> ```add_sitesubnet(self, data: netdot.dataclasses.site.SiteSubnet) -> netdot.dataclasses.site.SiteSubnet```

```
Add a SiteSubnet to this IPBlock.

Returns:
    netdot.SiteSubnet: The created SiteSubnet.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.add_subnetzone

> ```add_subnetzone(self, data: netdot.dataclasses.ipblock.SubnetZone) -> netdot.dataclasses.ipblock.SubnetZone```

```
Add a SubnetZone to this IPBlock.

Returns:
    netdot.SubnetZone: The created SubnetZone.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### IPBlock.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### IPBlock.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### IPBlock.load_arpcacheentries

> ```load_arpcacheentries(self, ignore_404=True) -> List[netdot.dataclasses.arpcache.ArpCacheEntry]```

```
Load the ArpCacheEntries associated to this IPBlock. (Via the `ArpCacheEntry.ipaddr` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.ArpCacheEntry]: All/Any ArpCacheEntries related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_asn

> ```load_asn(self) -> netdot.dataclasses.bgp.ASN```

```
Load the asn (ASN) associated to this IPBlock.

Returns:
    netdot.ASN: The full ASN object if available, else None.
```

#### IPBlock.load_children

> ```load_children(self) -> List[ForwardRef('IPBlock')]```

```
Get the children of this IPBlock.

Returns:
    List[IPBlock]: The children of this IPBlock.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: If no results found. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this IPBlock. (Via the `Device.snmp_target` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_dhcpscopes

> ```load_dhcpscopes(self, ignore_404=True) -> List[netdot.dataclasses.dhcp.DHCPScope]```

```
Load the DHCPScopes associated to this IPBlock.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DHCPScope]: All/Any DHCPScopes related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_interface

> ```load_interface(self) -> netdot.dataclasses.interface.Interface```

```
Load the interface (Interface) associated to this IPBlock.

Returns:
    netdot.Interface: The full Interface object if available, else None.
```

#### IPBlock.load_ipblockattrs

> ```load_ipblockattrs(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlockAttr]```

```
Load the IPBlockAttrs associated to this IPBlock.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlockAttr]: All/Any IPBlockAttrs related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_ipblocks

> ```load_ipblocks(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Load the IPBlocks associated to this IPBlock. (Via the `IPBlock.parent` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlock]: All/Any IPBlocks related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_ipservices

> ```load_ipservices(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPService]```

```
Load the IPServices associated to this IPBlock. (Via the `IPService.ip` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPService]: All/Any IPServices related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_owner

> ```load_owner(self) -> netdot.dataclasses.entity.Entity```

```
Load the owner (Entity) associated to this IPBlock.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### IPBlock.load_parent

> ```load_parent(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the parent (IPBlock) associated to this IPBlock.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### IPBlock.load_rraddr

> ```load_rraddr(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRADDR]```

```
Load the RRADDR associated to this IPBlock.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRADDR]: All/Any RRADDR related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_rrptr

> ```load_rrptr(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRPTR]```

```
Load the RRPTR associated to this IPBlock.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRPTR]: All/Any RRPTR related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_sites

> ```load_sites(self, ignore_404=True) -> List[netdot.dataclasses.site.Site]```

```
Load the sites (Sites) associated to this IPBlock.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of Sites associated to this IPBlock).

You might prefer :func:`load_sitesubnets` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Site]: Any/All Sites related to this IPBlock, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_sitesubnets

> ```load_sitesubnets(self, ignore_404=True) -> List[netdot.dataclasses.site.SiteSubnet]```

```
Load the SiteSubnets associated to this IPBlock. (Via the `SiteSubnet.subnet` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.SiteSubnet]: All/Any SiteSubnets related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_status

> ```load_status(self) -> netdot.dataclasses.ipblock.IPBlockStatus```

```
Load the status (IPBlockStatus) associated to this IPBlock.

Returns:
    netdot.IPBlockStatus: The full IPBlockStatus object if available, else None.
```

#### IPBlock.load_subnetzones

> ```load_subnetzones(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.SubnetZone]```

```
Load the SubnetZones associated to this IPBlock. (Via the `SubnetZone.subnet` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.SubnetZone]: All/Any SubnetZones related to this IPBlock.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlock.load_used_by

> ```load_used_by(self) -> netdot.dataclasses.entity.Entity```

```
Load the used_by (Entity) associated to this IPBlock.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### IPBlock.load_vlan

> ```load_vlan(self) -> netdot.dataclasses.vlan.VLAN```

```
Load the vlan (VLAN) associated to this IPBlock.

Returns:
    netdot.VLAN: The full VLAN object if available, else None.
```

#### IPBlock.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### IPBlock.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.IPBlockAttr`

<a id="class-netdotipblockattr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| ipblock_xlink | int |  |
| name_xlink | int |  |
| value | str |  |


### Methods

#### IPBlockAttr.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### IPBlockAttr.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### IPBlockAttr.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### IPBlockAttr.load_ipblock

> ```load_ipblock(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the ipblock (IPBlock) associated to this IPBlockAttr.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### IPBlockAttr.load_name

> ```load_name(self) -> netdot.dataclasses.ipblock.IPBlockAttrName```

```
Load the name (IPBlockAttrName) associated to this IPBlockAttr.

Returns:
    netdot.IPBlockAttrName: The full IPBlockAttrName object if available, else None.
```

#### IPBlockAttr.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### IPBlockAttr.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.IPBlockAttrName`

<a id="class-netdotipblockattrname"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### IPBlockAttrName.add_ipblockattr

> ```add_ipblockattr(self, data: netdot.dataclasses.ipblock.IPBlockAttr) -> netdot.dataclasses.ipblock.IPBlockAttr```

```
Add a IPBlockAttr to this IPBlockAttrName.

Returns:
    netdot.IPBlockAttr: The created IPBlockAttr.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlockAttrName.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### IPBlockAttrName.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### IPBlockAttrName.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### IPBlockAttrName.load_ipblockattrs

> ```load_ipblockattrs(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlockAttr]```

```
Load the IPBlockAttrs associated to this IPBlockAttrName. (Via the `IPBlockAttr.name` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlockAttr]: All/Any IPBlockAttrs related to this IPBlockAttrName.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlockAttrName.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### IPBlockAttrName.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.IPBlockStatus`

<a id="class-netdotipblockstatus"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| name | str |  |


### Methods

#### IPBlockStatus.add_ipblock

> ```add_ipblock(self, data: netdot.dataclasses.ipblock.IPBlock) -> netdot.dataclasses.ipblock.IPBlock```

```
Add a IPBlock to this IPBlockStatus.

Returns:
    netdot.IPBlock: The created IPBlock.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlockStatus.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### IPBlockStatus.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### IPBlockStatus.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### IPBlockStatus.load_ipblocks

> ```load_ipblocks(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Load the IPBlocks associated to this IPBlockStatus. (Via the `IPBlock.status` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlock]: All/Any IPBlocks related to this IPBlockStatus.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### IPBlockStatus.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### IPBlockStatus.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.IPService`

<a id="class-netdotipservice"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| contactlist_xlink | int |  |
| ip_xlink | int |  |
| monitored | bool | False |
| monitorstatus_xlink | int |  |
| service_xlink | int |  |


### Methods

#### IPService.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### IPService.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### IPService.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### IPService.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this IPService.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### IPService.load_ip

> ```load_ip(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the ip (IPBlock) associated to this IPService.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### IPService.load_monitorstatus

> ```load_monitorstatus(self) -> netdot.dataclasses.misc.MonitorStatus```

```
Load the monitorstatus (MonitorStatus) associated to this IPService.

Returns:
    netdot.MonitorStatus: The full MonitorStatus object if available, else None.
```

#### IPService.load_service

> ```load_service(self) -> netdot.dataclasses.ipblock.Service```

```
Load the service (Service) associated to this IPService.

Returns:
    netdot.Service: The full Service object if available, else None.
```

#### IPService.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### IPService.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Interface`

<a id="class-netdotinterface"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| physaddr_xlink | int |  |
| oper_status | str |  |
| admin_status | str |  |
| admin_duplex | str |  |
| bpdu_filter_enabled | bool |  |
| bpdu_guard_enabled | bool |  |
| contactlist_xlink | int |  |
| description | str |  |
| device_xlink | str |  |
| doc_status | str |  |
| down_from | DateTime |  |
| down_until | DateTime |  |
| dp_remote_id | str |  |
| dp_remote_ip | str |  |
| dp_remote_port | str |  |
| dp_remote_type | str |  |
| info | str |  |
| jack_xlink | str |  |
| jack_char | str |  |
| loop_guard_enabled | bool |  |
| monitored | bool |  |
| monitorstatus_xlink | int |  |
| name | str |  |
| neighbor_xlink | int |  |
| neighbor_fixed | bool |  |
| neighbor_missed | int |  |
| number | str |  |
| oper_duplex | str |  |
| overwrite_descr | bool |  |
| room_char | str |  |
| root_guard_enabled | bool |  |
| snmp_managed | bool |  |
| speed | int |  |
| stp_id | str |  |
| type | str |  |
| ignore_ip | bool |  |
| auto_dns | bool |  |
| circuit_xlink | int |  |
| dlci | str |  |


### Methods

#### Interface.add_arpcacheentry

> ```add_arpcacheentry(self, data: netdot.dataclasses.arpcache.ArpCacheEntry) -> netdot.dataclasses.arpcache.ArpCacheEntry```

```
Add a ArpCacheEntry to this Interface.

Returns:
    netdot.ArpCacheEntry: The created ArpCacheEntry.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.add_fwtableentry

> ```add_fwtableentry(self, data: netdot.dataclasses.fwtable.FWTableEntry) -> netdot.dataclasses.fwtable.FWTableEntry```

```
Add a FWTableEntry to this Interface.

Returns:
    netdot.FWTableEntry: The created FWTableEntry.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.add_interface

> ```add_interface(self, data: netdot.dataclasses.interface.Interface) -> netdot.dataclasses.interface.Interface```

```
Add a Interface to this Interface.

Returns:
    netdot.Interface: The created Interface.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.add_interfacevlan

> ```add_interfacevlan(self, data: netdot.dataclasses.interface.InterfaceVLAN) -> netdot.dataclasses.interface.InterfaceVLAN```

```
Add a InterfaceVLAN to this Interface.

Returns:
    netdot.InterfaceVLAN: The created InterfaceVLAN.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.add_ipblock

> ```add_ipblock(self, data: netdot.dataclasses.ipblock.IPBlock) -> netdot.dataclasses.ipblock.IPBlock```

```
Add a IPBlock to this Interface.

Returns:
    netdot.IPBlock: The created IPBlock.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Interface.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Interface.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Interface.load_arpcacheentries

> ```load_arpcacheentries(self, ignore_404=True) -> List[netdot.dataclasses.arpcache.ArpCacheEntry]```

```
Load the ArpCacheEntries associated to this Interface.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.ArpCacheEntry]: All/Any ArpCacheEntries related to this Interface.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.load_circuit

> ```load_circuit(self) -> netdot.dataclasses.cables.Circuit```

```
Load the circuit (Circuit) associated to this Interface.

Returns:
    netdot.Circuit: The full Circuit object if available, else None.
```

#### Interface.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this Interface.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### Interface.load_device

> ```load_device(self) -> netdot.dataclasses.device.Device```

```
Load the device (Device) associated to this Interface.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### Interface.load_fwtableentries

> ```load_fwtableentries(self, ignore_404=True) -> List[netdot.dataclasses.fwtable.FWTableEntry]```

```
Load the FWTableEntries associated to this Interface.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.FWTableEntry]: All/Any FWTableEntries related to this Interface.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.load_interfaces

> ```load_interfaces(self, ignore_404=True) -> List[netdot.dataclasses.interface.Interface]```

```
Load the Interfaces associated to this Interface. (Via the `Interface.neighbor` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Interface]: All/Any Interfaces related to this Interface.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.load_interfacevlans

> ```load_interfacevlans(self, ignore_404=True) -> List[netdot.dataclasses.interface.InterfaceVLAN]```

```
Load the InterfaceVLANs associated to this Interface.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.InterfaceVLAN]: All/Any InterfaceVLANs related to this Interface.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.load_ipblocks

> ```load_ipblocks(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Load the IPBlocks associated to this Interface.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlock]: All/Any IPBlocks related to this Interface.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Interface.load_jack

> ```load_jack(self) -> netdot.dataclasses.cables.HorizontalCable```

```
Load the jack (HorizontalCable) associated to this Interface.

Returns:
    netdot.HorizontalCable: The full HorizontalCable object if available, else None.
```

#### Interface.load_monitorstatus

> ```load_monitorstatus(self) -> netdot.dataclasses.misc.MonitorStatus```

```
Load the monitorstatus (MonitorStatus) associated to this Interface.

Returns:
    netdot.MonitorStatus: The full MonitorStatus object if available, else None.
```

#### Interface.load_neighbor

> ```load_neighbor(self) -> netdot.dataclasses.interface.Interface```

```
Load the neighbor (Interface) associated to this Interface.

Returns:
    netdot.Interface: The full Interface object if available, else None.
```

#### Interface.load_physaddr

> ```load_physaddr(self) -> netdot.dataclasses.physaddr.PhysAddr```

```
Load the physaddr (PhysAddr) associated to this Interface.

Returns:
    netdot.PhysAddr: The full PhysAddr object if available, else None.
```

#### Interface.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Interface.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.InterfaceVLAN`

<a id="class-netdotinterfacevlan"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| interface_xlink | int |  |
| stp_des_bridge | str |  |
| stp_des_port | str |  |
| stp_instance_xlink | int |  |
| stp_state | str |  |
| vlan_xlink | int |  |


### Methods

#### InterfaceVLAN.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### InterfaceVLAN.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### InterfaceVLAN.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### InterfaceVLAN.load_interface

> ```load_interface(self) -> netdot.dataclasses.interface.Interface```

```
Load the interface (Interface) associated to this InterfaceVLAN.

Returns:
    netdot.Interface: The full Interface object if available, else None.
```

#### InterfaceVLAN.load_stp_instance

> ```load_stp_instance(self) -> netdot.dataclasses.device.STPInstance```

```
Load the stp_instance (STPInstance) associated to this InterfaceVLAN.

Returns:
    netdot.STPInstance: The full STPInstance object if available, else None.
```

#### InterfaceVLAN.load_vlan

> ```load_vlan(self) -> netdot.dataclasses.vlan.VLAN```

```
Load the vlan (VLAN) associated to this InterfaceVLAN.

Returns:
    netdot.VLAN: The full VLAN object if available, else None.
```

#### InterfaceVLAN.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### InterfaceVLAN.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.MaintContract`

<a id="class-netdotmaintcontract"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| number | str |  |
| provider_xlink | int |  |


### Methods

#### MaintContract.add_asset

> ```add_asset(self, data: netdot.dataclasses.asset.Asset) -> netdot.dataclasses.asset.Asset```

```
Add a Asset to this MaintContract.

Returns:
    netdot.Asset: The created Asset.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### MaintContract.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### MaintContract.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### MaintContract.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### MaintContract.load_assets

> ```load_assets(self, ignore_404=True) -> List[netdot.dataclasses.asset.Asset]```

```
Load the Assets associated to this MaintContract. (Via the `Asset.maint_contract` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Asset]: All/Any Assets related to this MaintContract.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### MaintContract.load_provider

> ```load_provider(self) -> netdot.dataclasses.entity.Entity```

```
Load the provider (Entity) associated to this MaintContract.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### MaintContract.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### MaintContract.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.MonitorStatus`

<a id="class-netdotmonitorstatus"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### MonitorStatus.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this MonitorStatus.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### MonitorStatus.add_interface

> ```add_interface(self, data: netdot.dataclasses.interface.Interface) -> netdot.dataclasses.interface.Interface```

```
Add a Interface to this MonitorStatus.

Returns:
    netdot.Interface: The created Interface.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### MonitorStatus.add_ipservice

> ```add_ipservice(self, data: netdot.dataclasses.ipblock.IPService) -> netdot.dataclasses.ipblock.IPService```

```
Add a IPService to this MonitorStatus.

Returns:
    netdot.IPService: The created IPService.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### MonitorStatus.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### MonitorStatus.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### MonitorStatus.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### MonitorStatus.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this MonitorStatus.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this MonitorStatus.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### MonitorStatus.load_interfaces

> ```load_interfaces(self, ignore_404=True) -> List[netdot.dataclasses.interface.Interface]```

```
Load the Interfaces associated to this MonitorStatus.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Interface]: All/Any Interfaces related to this MonitorStatus.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### MonitorStatus.load_ipservices

> ```load_ipservices(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPService]```

```
Load the IPServices associated to this MonitorStatus.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPService]: All/Any IPServices related to this MonitorStatus.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### MonitorStatus.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### MonitorStatus.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.OUI`

<a id="class-netdotoui"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| oui | str |  |
| vendor | str |  |


### Methods

#### OUI.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### OUI.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### OUI.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### OUI.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### OUI.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Person`

<a id="class-netdotperson"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| aliases | str |  |
| cell | str |  |
| email | str |  |
| emailpager | str |  |
| entity_xlink | int |  |
| extension | int |  |
| fax | str |  |
| firstname | str |  |
| home | str |  |
| info | str |  |
| lastname | str |  |
| location_xlink | int |  |
| office | str |  |
| pager | str |  |
| position | str |  |
| room_xlink | int |  |
| user_type_xlink | int |  |
| username | str |  |
| password | str |  |


### Methods

#### Person.add_accessright

> ```add_accessright(self, data: netdot.dataclasses.users.AccessRight) -> netdot.dataclasses.users.AccessRight```

```
Add a AccessRight to this Person (via UserRight).

Args:
    data (netdot.AccessRight): The AccessRight to add to this Person.

Returns:
    netdot.UserRight: The newly created UserRight.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Person.add_contact

> ```add_contact(self, data: netdot.dataclasses.users.Contact) -> netdot.dataclasses.users.Contact```

```
Add a Contact to this Person.

Returns:
    netdot.Contact: The created Contact.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Person.add_userright

> ```add_userright(self, data: netdot.dataclasses.users.UserRight) -> netdot.dataclasses.users.UserRight```

```
Add a UserRight to this Person.

Returns:
    netdot.UserRight: The created UserRight.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Person.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Person.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Person.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Person.load_accessrights

> ```load_accessrights(self, ignore_404=True) -> List[netdot.dataclasses.users.AccessRight]```

```
Load the accessrights (AccessRights) associated to this Person.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of AccessRights associated to this Person).

You might prefer :func:`load_userrights` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.AccessRight]: Any/All AccessRights related to this Person, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Person.load_contacts

> ```load_contacts(self, ignore_404=True) -> List[netdot.dataclasses.users.Contact]```

```
Load the Contacts associated to this Person.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Contact]: All/Any Contacts related to this Person.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Person.load_entity

> ```load_entity(self) -> netdot.dataclasses.entity.Entity```

```
Load the entity (Entity) associated to this Person.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### Person.load_location

> ```load_location(self) -> netdot.dataclasses.site.Site```

```
Load the location (Site) associated to this Person.

Returns:
    netdot.Site: The full Site object if available, else None.
```

#### Person.load_room

> ```load_room(self) -> netdot.dataclasses.site.Room```

```
Load the room (Room) associated to this Person.

Returns:
    netdot.Room: The full Room object if available, else None.
```

#### Person.load_user_type

> ```load_user_type(self) -> netdot.dataclasses.users.UserType```

```
Load the user_type (UserType) associated to this Person.

Returns:
    netdot.UserType: The full UserType object if available, else None.
```

#### Person.load_userrights

> ```load_userrights(self, ignore_404=True) -> List[netdot.dataclasses.users.UserRight]```

```
Load the UserRights associated to this Person.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.UserRight]: All/Any UserRights related to this Person.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Person.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Person.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.PhysAddr`

<a id="class-netdotphysaddr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| address | MACAddress |  |
| static | bool | False |
| first_seen | DateTime |  |
| last_seen | DateTime |  |


### Methods

#### PhysAddr.add_arpcacheentry

> ```add_arpcacheentry(self, data: netdot.dataclasses.arpcache.ArpCacheEntry) -> netdot.dataclasses.arpcache.ArpCacheEntry```

```
Add a ArpCacheEntry to this PhysAddr.

Returns:
    netdot.ArpCacheEntry: The created ArpCacheEntry.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.add_asset

> ```add_asset(self, data: netdot.dataclasses.asset.Asset) -> netdot.dataclasses.asset.Asset```

```
Add a Asset to this PhysAddr.

Returns:
    netdot.Asset: The created Asset.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.add_dhcpscope

> ```add_dhcpscope(self, data: netdot.dataclasses.dhcp.DHCPScope) -> netdot.dataclasses.dhcp.DHCPScope```

```
Add a DHCPScope to this PhysAddr.

Returns:
    netdot.DHCPScope: The created DHCPScope.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.add_fwtableentry

> ```add_fwtableentry(self, data: netdot.dataclasses.fwtable.FWTableEntry) -> netdot.dataclasses.fwtable.FWTableEntry```

```
Add a FWTableEntry to this PhysAddr.

Returns:
    netdot.FWTableEntry: The created FWTableEntry.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.add_interface

> ```add_interface(self, data: netdot.dataclasses.interface.Interface) -> netdot.dataclasses.interface.Interface```

```
Add a Interface to this PhysAddr.

Returns:
    netdot.Interface: The created Interface.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### PhysAddr.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### PhysAddr.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### PhysAddr.load_arpcacheentries

> ```load_arpcacheentries(self, ignore_404=True) -> List[netdot.dataclasses.arpcache.ArpCacheEntry]```

```
Load the ArpCacheEntries associated to this PhysAddr.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.ArpCacheEntry]: All/Any ArpCacheEntries related to this PhysAddr.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.load_assets

> ```load_assets(self, ignore_404=True) -> List[netdot.dataclasses.asset.Asset]```

```
Load the Assets associated to this PhysAddr.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Asset]: All/Any Assets related to this PhysAddr.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.load_dhcpscopes

> ```load_dhcpscopes(self, ignore_404=True) -> List[netdot.dataclasses.dhcp.DHCPScope]```

```
Load the DHCPScopes associated to this PhysAddr.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.DHCPScope]: All/Any DHCPScopes related to this PhysAddr.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.load_fwtableentries

> ```load_fwtableentries(self, ignore_404=True) -> List[netdot.dataclasses.fwtable.FWTableEntry]```

```
Load the FWTableEntries associated to this PhysAddr.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.FWTableEntry]: All/Any FWTableEntries related to this PhysAddr.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.load_interfaces

> ```load_interfaces(self, ignore_404=True) -> List[netdot.dataclasses.interface.Interface]```

```
Load the Interfaces associated to this PhysAddr.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Interface]: All/Any Interfaces related to this PhysAddr.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### PhysAddr.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### PhysAddr.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.PhysAddrAttr`

<a id="class-netdotphysaddrattr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| name | int |  |
| physaddr | int |  |
| value | str |  |


### Methods

#### PhysAddrAttr.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### PhysAddrAttr.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### PhysAddrAttr.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### PhysAddrAttr.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### PhysAddrAttr.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.PhysAddrAttrName`

<a id="class-netdotphysaddrattrname"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### PhysAddrAttrName.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### PhysAddrAttrName.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### PhysAddrAttrName.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### PhysAddrAttrName.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### PhysAddrAttrName.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Product`

<a id="class-netdotproduct"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| description | str |  |
| info | str |  |
| manufacturer_xlink | int |  |
| name | str |  |
| sysobjectid | str |  |
| type_xlink | int |  |
| latest_os | str |  |
| part_number | str |  |
| config_type | str |  |


### Methods

#### Product.add_asset

> ```add_asset(self, data: netdot.dataclasses.asset.Asset) -> netdot.dataclasses.asset.Asset```

```
Add a Asset to this Product.

Returns:
    netdot.Asset: The created Asset.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Product.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Product.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Product.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Product.load_assets

> ```load_assets(self, ignore_404=True) -> List[netdot.dataclasses.asset.Asset]```

```
Load the Assets associated to this Product. (Via the `Asset.product_id` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Asset]: All/Any Assets related to this Product.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Product.load_manufacturer

> ```load_manufacturer(self) -> netdot.dataclasses.entity.Entity```

```
Load the manufacturer (Entity) associated to this Product.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### Product.load_type

> ```load_type(self) -> netdot.dataclasses.products.ProductType```

```
Load the type (ProductType) associated to this Product.

Returns:
    netdot.ProductType: The full ProductType object if available, else None.
```

#### Product.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Product.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.ProductType`

<a id="class-netdotproducttype"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### ProductType.add_product

> ```add_product(self, data: netdot.dataclasses.products.Product) -> netdot.dataclasses.products.Product```

```
Add a Product to this ProductType.

Returns:
    netdot.Product: The created Product.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ProductType.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### ProductType.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### ProductType.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### ProductType.load_products

> ```load_products(self, ignore_404=True) -> List[netdot.dataclasses.products.Product]```

```
Load the Products associated to this ProductType. (Via the `Product.type` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Product]: All/Any Products related to this ProductType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### ProductType.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### ProductType.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RR`

<a id="class-netdotrr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| active | bool | False |
| auto_update | bool | False |
| expiration | DateTime |  |
| info | str |  |
| name | str |  |
| zone_xlink | int |  |
| created | DateTime |  |
| modified | DateTime |  |


### Methods

#### RR.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this RR.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rraddr

> ```add_rraddr(self, data: netdot.dataclasses.dns.RRADDR) -> netdot.dataclasses.dns.RRADDR```

```
Add a RRADDR to this RR.

Returns:
    netdot.RRADDR: The created RRADDR.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrcname

> ```add_rrcname(self, data: netdot.dataclasses.dns.RRCNAME) -> netdot.dataclasses.dns.RRCNAME```

```
Add a RRCNAME to this RR.

Returns:
    netdot.RRCNAME: The created RRCNAME.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrds

> ```add_rrds(self, data: netdot.dataclasses.dns.RRDS) -> netdot.dataclasses.dns.RRDS```

```
Add a RRDS to this RR.

Returns:
    netdot.RRDS: The created RRDS.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrhinfo

> ```add_rrhinfo(self, data: netdot.dataclasses.dns.RRHINFO) -> netdot.dataclasses.dns.RRHINFO```

```
Add a RRHINFO to this RR.

Returns:
    netdot.RRHINFO: The created RRHINFO.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrloc

> ```add_rrloc(self, data: netdot.dataclasses.dns.RRLOC) -> netdot.dataclasses.dns.RRLOC```

```
Add a RRLOC to this RR.

Returns:
    netdot.RRLOC: The created RRLOC.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrmx

> ```add_rrmx(self, data: netdot.dataclasses.dns.RRMX) -> netdot.dataclasses.dns.RRMX```

```
Add a RRMX to this RR.

Returns:
    netdot.RRMX: The created RRMX.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrnaptr

> ```add_rrnaptr(self, data: netdot.dataclasses.dns.RRNAPTR) -> netdot.dataclasses.dns.RRNAPTR```

```
Add a RRNAPTR to this RR.

Returns:
    netdot.RRNAPTR: The created RRNAPTR.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrns

> ```add_rrns(self, data: netdot.dataclasses.dns.RRNS) -> netdot.dataclasses.dns.RRNS```

```
Add a RRNS to this RR.

Returns:
    netdot.RRNS: The created RRNS.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrptr

> ```add_rrptr(self, data: netdot.dataclasses.dns.RRPTR) -> netdot.dataclasses.dns.RRPTR```

```
Add a RRPTR to this RR.

Returns:
    netdot.RRPTR: The created RRPTR.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrsrv

> ```add_rrsrv(self, data: netdot.dataclasses.dns.RRSRV) -> netdot.dataclasses.dns.RRSRV```

```
Add a RRSRV to this RR.

Returns:
    netdot.RRSRV: The created RRSRV.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.add_rrtxt

> ```add_rrtxt(self, data: netdot.dataclasses.dns.RRTXT) -> netdot.dataclasses.dns.RRTXT```

```
Add a RRTXT to this RR.

Returns:
    netdot.RRTXT: The created RRTXT.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RR.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RR.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RR.infer_FQDN

> ```infer_FQDN(self) -> str```

```
Infer the Fully Qualified Domain Name (FQDN) for this Resource Record (RR).

Raises:
    ValueError: If either `name` or `zone` are not set for this RR.

Returns:
    str: The FQDN for this RR.
```

#### RR.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this RR. (Via the `Device.name` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rraddr

> ```load_rraddr(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRADDR]```

```
Load the RRADDR associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRADDR]: All/Any RRADDR related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrcname

> ```load_rrcname(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRCNAME]```

```
Load the RRCNAME associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRCNAME]: All/Any RRCNAME related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrds

> ```load_rrds(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRDS]```

```
Load the RRDS associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRDS]: All/Any RRDS related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrhinfo

> ```load_rrhinfo(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRHINFO]```

```
Load the RRHINFO associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRHINFO]: All/Any RRHINFO related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrloc

> ```load_rrloc(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRLOC]```

```
Load the RRLOC associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRLOC]: All/Any RRLOC related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrmx

> ```load_rrmx(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRMX]```

```
Load the RRMX associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRMX]: All/Any RRMX related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrnaptr

> ```load_rrnaptr(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRNAPTR]```

```
Load the RRNAPTR associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRNAPTR]: All/Any RRNAPTR related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrns

> ```load_rrns(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRNS]```

```
Load the RRNS associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRNS]: All/Any RRNS related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrptr

> ```load_rrptr(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRPTR]```

```
Load the RRPTR associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRPTR]: All/Any RRPTR related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrsrv

> ```load_rrsrv(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRSRV]```

```
Load the RRSRV associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRSRV]: All/Any RRSRV related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_rrtxt

> ```load_rrtxt(self, ignore_404=True) -> List[netdot.dataclasses.dns.RRTXT]```

```
Load the RRTXT associated to this RR.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RRTXT]: All/Any RRTXT related to this RR.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### RR.load_zone

> ```load_zone(self) -> netdot.dataclasses.dns.Zone```

```
Load the zone (Zone) associated to this RR.

Returns:
    netdot.Zone: The full Zone object if available, else None.
```

#### RR.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RR.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRADDR`

<a id="class-netdotrraddr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| ipblock_xlink | int |  |
| rr_xlink | int |  |
| ttl | str |  |


### Methods

#### RRADDR.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRADDR.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRADDR.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRADDR.load_ipblock

> ```load_ipblock(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the ipblock (IPBlock) associated to this RRADDR.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### RRADDR.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRADDR.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRADDR.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRADDR.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRCNAME`

<a id="class-netdotrrcname"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| cname | str |  |
| rr_xlink | int |  |
| ttl | str |  |


### Methods

#### RRCNAME.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRCNAME.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRCNAME.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRCNAME.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRCNAME.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRCNAME.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRCNAME.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRDS`

<a id="class-netdotrrds"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| algorithm | int |  |
| digest | str |  |
| digest_type | int |  |
| key_tag | int |  |
| rr_xlink | int |  |
| ttl | str |  |


### Methods

#### RRDS.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRDS.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRDS.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRDS.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRDS.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRDS.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRDS.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRHINFO`

<a id="class-netdotrrhinfo"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| cpu | str |  |
| os | str |  |
| rr_xlink | int |  |
| ttl | str |  |


### Methods

#### RRHINFO.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRHINFO.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRHINFO.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRHINFO.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRHINFO.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRHINFO.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRHINFO.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRLOC`

<a id="class-netdotrrloc"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| altitude | int |  |
| horiz_pre | str |  |
| latitude | str |  |
| longitude | str |  |
| rr_xlink | int |  |
| size | str |  |
| ttl | str |  |
| vert_pre | str |  |


### Methods

#### RRLOC.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRLOC.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRLOC.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRLOC.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRLOC.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRLOC.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRLOC.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRMX`

<a id="class-netdotrrmx"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| exchange | str |  |
| preference | int |  |
| rr_xlink | int |  |
| ttl | str |  |


### Methods

#### RRMX.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRMX.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRMX.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRMX.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRMX.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRMX.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRMX.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRNAPTR`

<a id="class-netdotrrnaptr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| flags | str |  |
| order_field | int |  |
| preference | int |  |
| regexpr | str |  |
| replacement | str |  |
| rr_xlink | int |  |
| services | str |  |
| ttl | str |  |


### Methods

#### RRNAPTR.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRNAPTR.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRNAPTR.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRNAPTR.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRNAPTR.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRNAPTR.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRNAPTR.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRNS`

<a id="class-netdotrrns"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| nsdname | str |  |
| rr_xlink | int |  |
| ttl | str |  |


### Methods

#### RRNS.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRNS.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRNS.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRNS.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRNS.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRNS.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRNS.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRPTR`

<a id="class-netdotrrptr"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| ipblock_xlink | int |  |
| ptrdname | str |  |
| rr_xlink | int |  |
| ttl | str |  |


### Methods

#### RRPTR.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRPTR.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRPTR.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRPTR.load_ipblock

> ```load_ipblock(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the ipblock (IPBlock) associated to this RRPTR.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### RRPTR.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRPTR.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRPTR.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRPTR.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRSRV`

<a id="class-netdotrrsrv"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| port | int |  |
| priority | int |  |
| rr_xlink | int |  |
| target | str |  |
| ttl | str |  |
| weight | int |  |


### Methods

#### RRSRV.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRSRV.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRSRV.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRSRV.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRSRV.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRSRV.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRSRV.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.RRTXT`

<a id="class-netdotrrtxt"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| rr_xlink | int |  |
| ttl | str |  |
| txtdata | str |  |


### Methods

#### RRTXT.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### RRTXT.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### RRTXT.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### RRTXT.load_rr

> ```load_rr(self) -> netdot.dataclasses.dns.RR```

```
Load the rr (RR) associated to this RRTXT.

Returns:
    netdot.RR: The full RR object if available, else None.
```

#### RRTXT.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### RRTXT.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Room`

<a id="class-netdotroom"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| floor_xlink | int |  |
| name | str |  |


### Methods

#### Room.add_closet

> ```add_closet(self, data: netdot.dataclasses.site.Closet) -> netdot.dataclasses.site.Closet```

```
Add a Closet to this Room.

Returns:
    netdot.Closet: The created Closet.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Room.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this Room.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Room.add_horizontalcable

> ```add_horizontalcable(self, data: netdot.dataclasses.cables.HorizontalCable) -> netdot.dataclasses.cables.HorizontalCable```

```
Add a HorizontalCable to this Room.

Returns:
    netdot.HorizontalCable: The created HorizontalCable.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Room.add_person

> ```add_person(self, data: netdot.dataclasses.users.Person) -> netdot.dataclasses.users.Person```

```
Add a Person to this Room.

Returns:
    netdot.Person: The created Person.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Room.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Room.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Room.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Room.load_closets

> ```load_closets(self, ignore_404=True) -> List[netdot.dataclasses.site.Closet]```

```
Load the Closets associated to this Room.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Closet]: All/Any Closets related to this Room.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Room.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this Room.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this Room.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Room.load_floor

> ```load_floor(self) -> netdot.dataclasses.site.Floor```

```
Load the floor (Floor) associated to this Room.

Returns:
    netdot.Floor: The full Floor object if available, else None.
```

#### Room.load_horizontalcables

> ```load_horizontalcables(self, ignore_404=True) -> List[netdot.dataclasses.cables.HorizontalCable]```

```
Load the HorizontalCables associated to this Room.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.HorizontalCable]: All/Any HorizontalCables related to this Room.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Room.load_persons

> ```load_persons(self, ignore_404=True) -> List[netdot.dataclasses.users.Person]```

```
Load the Persons associated to this Room.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Person]: All/Any Persons related to this Room.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Room.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Room.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.STPInstance`

<a id="class-netdotstpinstance"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| bridge_priority | int |  |
| device_xlink | int |  |
| number | int |  |
| root_bridge | str |  |
| root_port | int |  |


### Methods

#### STPInstance.add_interfacevlan

> ```add_interfacevlan(self, data: netdot.dataclasses.interface.InterfaceVLAN) -> netdot.dataclasses.interface.InterfaceVLAN```

```
Add a InterfaceVLAN to this STPInstance.

Returns:
    netdot.InterfaceVLAN: The created InterfaceVLAN.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### STPInstance.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### STPInstance.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### STPInstance.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### STPInstance.load_device

> ```load_device(self) -> netdot.dataclasses.device.Device```

```
Load the device (Device) associated to this STPInstance.

Returns:
    netdot.Device: The full Device object if available, else None.
```

#### STPInstance.load_interfacevlans

> ```load_interfacevlans(self, ignore_404=True) -> List[netdot.dataclasses.interface.InterfaceVLAN]```

```
Load the InterfaceVLANs associated to this STPInstance. (Via the `InterfaceVLAN.stp_instance` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.InterfaceVLAN]: All/Any InterfaceVLANs related to this STPInstance.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### STPInstance.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### STPInstance.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.SavedQueries`

<a id="class-netdotsavedqueries"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| name | str |  |
| querytext | str |  |


### Methods

#### SavedQueries.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### SavedQueries.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### SavedQueries.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### SavedQueries.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### SavedQueries.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.SchemaInfo`

<a id="class-netdotschemainfo"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| version | str |  |


### Methods

#### SchemaInfo.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### SchemaInfo.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### SchemaInfo.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### SchemaInfo.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### SchemaInfo.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Service`

<a id="class-netdotservice"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### Service.add_ipservice

> ```add_ipservice(self, data: netdot.dataclasses.ipblock.IPService) -> netdot.dataclasses.ipblock.IPService```

```
Add a IPService to this Service.

Returns:
    netdot.IPService: The created IPService.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Service.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Service.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Service.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Service.load_ipservices

> ```load_ipservices(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPService]```

```
Load the IPServices associated to this Service.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPService]: All/Any IPServices related to this Service.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Service.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Service.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Site`

<a id="class-netdotsite"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| name | str |  |
| aliases | str |  |
| availability_xlink | int |  |
| contactlist_xlink | int |  |
| gsf | str |  |
| number | str |  |
| street1 | str |  |
| street2 | str |  |
| state | str |  |
| city | str |  |
| country | str |  |
| zip | str |  |
| pobox | str |  |
| info | str |  |


### Methods

#### Site.add_device

> ```add_device(self, data: netdot.dataclasses.device.Device) -> netdot.dataclasses.device.Device```

```
Add a Device to this Site.

Returns:
    netdot.Device: The created Device.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.add_entity

> ```add_entity(self, data: netdot.dataclasses.entity.Entity) -> netdot.dataclasses.entity.Entity```

```
Add a Entity to this Site (via EntitySite).

Args:
    data (netdot.Entity): The Entity to add to this Site.

Returns:
    netdot.EntitySite: The newly created EntitySite.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.add_entitysite

> ```add_entitysite(self, data: netdot.dataclasses.entity.EntitySite) -> netdot.dataclasses.entity.EntitySite```

```
Add a EntitySite to this Site.

Returns:
    netdot.EntitySite: The created EntitySite.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.add_floor

> ```add_floor(self, data: netdot.dataclasses.site.Floor) -> netdot.dataclasses.site.Floor```

```
Add a Floor to this Site.

Returns:
    netdot.Floor: The created Floor.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.add_person

> ```add_person(self, data: netdot.dataclasses.users.Person) -> netdot.dataclasses.users.Person```

```
Add a Person to this Site.

Returns:
    netdot.Person: The created Person.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.add_sitelink_as_farend

> ```add_sitelink_as_farend(self, data: netdot.dataclasses.site.SiteLink) -> netdot.dataclasses.site.SiteLink```

```
Add a SiteLink to this Site.

Returns:
    netdot.SiteLink: The created SiteLink.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.add_sitelink_as_nearend

> ```add_sitelink_as_nearend(self, data: netdot.dataclasses.site.SiteLink) -> netdot.dataclasses.site.SiteLink```

```
Add a SiteLink to this Site.

Returns:
    netdot.SiteLink: The created SiteLink.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.add_sitesubnet

> ```add_sitesubnet(self, data: netdot.dataclasses.site.SiteSubnet) -> netdot.dataclasses.site.SiteSubnet```

```
Add a SiteSubnet to this Site.

Returns:
    netdot.SiteSubnet: The created SiteSubnet.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.add_subnet

> ```add_subnet(self, data: netdot.dataclasses.ipblock.IPBlock) -> netdot.dataclasses.ipblock.IPBlock```

```
Add a IPBlock to this Site (via SiteSubnet).

Args:
    data (netdot.IPBlock): The IPBlock to add to this Site.

Returns:
    netdot.SiteSubnet: The newly created SiteSubnet.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Site.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Site.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Site.load_availability

> ```load_availability(self) -> netdot.dataclasses.misc.Availability```

```
Load the availability (Availability) associated to this Site.

Returns:
    netdot.Availability: The full Availability object if available, else None.
```

#### Site.load_closets

> ```load_closets(self) -> List[ForwardRef('Closet')]```

```
Load all closets for this site.

> NOTE: This will make approximately N+1 HTTP Requests (where N is the number of **rooms** in this site).
```

#### Site.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this Site.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### Site.load_devices

> ```load_devices(self, ignore_404=True) -> List[netdot.dataclasses.device.Device]```

```
Load the Devices associated to this Site.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Device]: All/Any Devices related to this Site.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.load_entities

> ```load_entities(self, ignore_404=True) -> List[netdot.dataclasses.entity.Entity]```

```
Load the entities (Entities) associated to this Site.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of Entities associated to this Site).

You might prefer :func:`load_entitysites` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Entity]: Any/All Entities related to this Site, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.load_entitysites

> ```load_entitysites(self, ignore_404=True) -> List[netdot.dataclasses.entity.EntitySite]```

```
Load the EntitySites associated to this Site.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.EntitySite]: All/Any EntitySites related to this Site.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.load_farend_sitelinks

> ```load_farend_sitelinks(self, ignore_404=True) -> List[netdot.dataclasses.site.SiteLink]```

```
Load the SiteLinks associated to this Site. (Via the `SiteLink.farend` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.SiteLink]: All/Any SiteLinks related to this Site.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.load_floors

> ```load_floors(self, ignore_404=True) -> List[netdot.dataclasses.site.Floor]```

```
Load the Floors associated to this Site.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Floor]: All/Any Floors related to this Site.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.load_nearend_sitelinks

> ```load_nearend_sitelinks(self, ignore_404=True) -> List[netdot.dataclasses.site.SiteLink]```

```
Load the SiteLinks associated to this Site. (Via the `SiteLink.nearend` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.SiteLink]: All/Any SiteLinks related to this Site.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.load_persons

> ```load_persons(self, ignore_404=True) -> List[netdot.dataclasses.users.Person]```

```
Load the Persons associated to this Site. (Via the `Person.location` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Person]: All/Any Persons related to this Site.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.load_rooms

> ```load_rooms(self) -> List[ForwardRef('Room')]```

```
Load all rooms for this site.

> NOTE: This will make N+1 HTTP Requests (where N is the number of **floors** in this site).
```

#### Site.load_sitesubnets

> ```load_sitesubnets(self, ignore_404=True) -> List[netdot.dataclasses.site.SiteSubnet]```

```
Load the SiteSubnets associated to this Site.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.SiteSubnet]: All/Any SiteSubnets related to this Site.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.load_subnets

> ```load_subnets(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Load the subnets (IPBlocks) associated to this Site.

> NOTE: This will make `N+1` HTTP Requests (where N is the number of IPBlocks associated to this Site).

You might prefer :func:`load_sitesubnets` over this method, if you want to load the many-to-many objects themselves. (and not make N+1 HTTP Requests)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlock]: Any/All IPBlocks related to this Site, or an empty list if none are found.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Site.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Site.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.SiteLink`

<a id="class-netdotsitelink"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| entity_xlink | int |  |
| farend_xlink | int |  |
| info | str |  |
| name | str |  |
| nearend_xlink | int |  |


### Methods

#### SiteLink.add_circuit

> ```add_circuit(self, data: netdot.dataclasses.cables.Circuit) -> netdot.dataclasses.cables.Circuit```

```
Add a Circuit to this SiteLink.

Returns:
    netdot.Circuit: The created Circuit.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### SiteLink.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### SiteLink.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### SiteLink.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### SiteLink.load_circuits

> ```load_circuits(self, ignore_404=True) -> List[netdot.dataclasses.cables.Circuit]```

```
Load the Circuits associated to this SiteLink. (Via the `Circuit.linkid` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Circuit]: All/Any Circuits related to this SiteLink.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### SiteLink.load_entity

> ```load_entity(self) -> netdot.dataclasses.entity.Entity```

```
Load the entity (Entity) associated to this SiteLink.

Returns:
    netdot.Entity: The full Entity object if available, else None.
```

#### SiteLink.load_farend

> ```load_farend(self) -> netdot.dataclasses.site.Site```

```
Load the farend (Site) associated to this SiteLink.

Returns:
    netdot.Site: The full Site object if available, else None.
```

#### SiteLink.load_nearend

> ```load_nearend(self) -> netdot.dataclasses.site.Site```

```
Load the nearend (Site) associated to this SiteLink.

Returns:
    netdot.Site: The full Site object if available, else None.
```

#### SiteLink.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### SiteLink.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.SiteSubnet`

<a id="class-netdotsitesubnet"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| site_xlink | int |  |
| subnet_xlink | int |  |


### Methods

#### SiteSubnet.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### SiteSubnet.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### SiteSubnet.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### SiteSubnet.load_site

> ```load_site(self) -> netdot.dataclasses.site.Site```

```
Load the site (Site) associated to this SiteSubnet.

Returns:
    netdot.Site: The full Site object if available, else None.
```

#### SiteSubnet.load_subnet

> ```load_subnet(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the subnet (IPBlock) associated to this SiteSubnet.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### SiteSubnet.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### SiteSubnet.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Splice`

<a id="class-netdotsplice"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| strand1_xlink | int |  |
| strand2_xlink | int |  |


### Methods

#### Splice.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Splice.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Splice.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Splice.load_strand1

> ```load_strand1(self) -> netdot.dataclasses.cables.CableStrand```

```
Load the strand1 (CableStrand) associated to this Splice.

Returns:
    netdot.CableStrand: The full CableStrand object if available, else None.
```

#### Splice.load_strand2

> ```load_strand2(self) -> netdot.dataclasses.cables.CableStrand```

```
Load the strand2 (CableStrand) associated to this Splice.

Returns:
    netdot.CableStrand: The full CableStrand object if available, else None.
```

#### Splice.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Splice.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.StrandStatus`

<a id="class-netdotstrandstatus"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### StrandStatus.add_cablestrand

> ```add_cablestrand(self, data: netdot.dataclasses.cables.CableStrand) -> netdot.dataclasses.cables.CableStrand```

```
Add a CableStrand to this StrandStatus.

Returns:
    netdot.CableStrand: The created CableStrand.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### StrandStatus.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### StrandStatus.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### StrandStatus.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### StrandStatus.load_cablestrands

> ```load_cablestrands(self, ignore_404=True) -> List[netdot.dataclasses.cables.CableStrand]```

```
Load the CableStrands associated to this StrandStatus. (Via the `CableStrand.status` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.CableStrand]: All/Any CableStrands related to this StrandStatus.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### StrandStatus.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### StrandStatus.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.SubnetZone`

<a id="class-netdotsubnetzone"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| subnet_xlink | int |  |
| zone_xlink | int |  |


### Methods

#### SubnetZone.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### SubnetZone.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### SubnetZone.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### SubnetZone.load_subnet

> ```load_subnet(self) -> netdot.dataclasses.ipblock.IPBlock```

```
Load the subnet (IPBlock) associated to this SubnetZone.

Returns:
    netdot.IPBlock: The full IPBlock object if available, else None.
```

#### SubnetZone.load_zone

> ```load_zone(self) -> netdot.dataclasses.dns.Zone```

```
Load the zone (Zone) associated to this SubnetZone.

Returns:
    netdot.Zone: The full Zone object if available, else None.
```

#### SubnetZone.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### SubnetZone.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.UserRight`

<a id="class-netdotuserright"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| accessright_xlink | int |  |
| person_xlink | int |  |


### Methods

#### UserRight.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### UserRight.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### UserRight.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### UserRight.load_accessright

> ```load_accessright(self) -> netdot.dataclasses.users.AccessRight```

```
Load the accessright (AccessRight) associated to this UserRight.

Returns:
    netdot.AccessRight: The full AccessRight object if available, else None.
```

#### UserRight.load_person

> ```load_person(self) -> netdot.dataclasses.users.Person```

```
Load the person (Person) associated to this UserRight.

Returns:
    netdot.Person: The full Person object if available, else None.
```

#### UserRight.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### UserRight.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.UserType`

<a id="class-netdotusertype"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |


### Methods

#### UserType.add_person

> ```add_person(self, data: netdot.dataclasses.users.Person) -> netdot.dataclasses.users.Person```

```
Add a Person to this UserType.

Returns:
    netdot.Person: The created Person.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### UserType.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### UserType.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### UserType.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### UserType.load_persons

> ```load_persons(self, ignore_404=True) -> List[netdot.dataclasses.users.Person]```

```
Load the Persons associated to this UserType. (Via the `Person.user_type` attribute)

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.Person]: All/Any Persons related to this UserType.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### UserType.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### UserType.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.VLAN`

<a id="class-netdotvlan"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| description | str |  |
| info | str |  |
| name | str |  |
| vid | int |  |
| vlangroup_xlink | int |  |


### Methods

#### VLAN.add_interfacevlan

> ```add_interfacevlan(self, data: netdot.dataclasses.interface.InterfaceVLAN) -> netdot.dataclasses.interface.InterfaceVLAN```

```
Add a InterfaceVLAN to this VLAN.

Returns:
    netdot.InterfaceVLAN: The created InterfaceVLAN.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### VLAN.add_ipblock

> ```add_ipblock(self, data: netdot.dataclasses.ipblock.IPBlock) -> netdot.dataclasses.ipblock.IPBlock```

```
Add a IPBlock to this VLAN.

Returns:
    netdot.IPBlock: The created IPBlock.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### VLAN.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### VLAN.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### VLAN.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### VLAN.load_interfacevlans

> ```load_interfacevlans(self, ignore_404=True) -> List[netdot.dataclasses.interface.InterfaceVLAN]```

```
Load the InterfaceVLANs associated to this VLAN.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.InterfaceVLAN]: All/Any InterfaceVLANs related to this VLAN.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### VLAN.load_ipblocks

> ```load_ipblocks(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.IPBlock]```

```
Load the IPBlocks associated to this VLAN.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.IPBlock]: All/Any IPBlocks related to this VLAN.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### VLAN.load_vlangroup

> ```load_vlangroup(self) -> netdot.dataclasses.vlan.VLANGroup```

```
Load the vlangroup (VLANGroup) associated to this VLAN.

Returns:
    netdot.VLANGroup: The full VLANGroup object if available, else None.
```

#### VLAN.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### VLAN.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.VLANGroup`

<a id="class-netdotvlangroup"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| description | str |  |
| end_vid | int |  |
| info | str |  |
| name | str |  |
| start_vid | int |  |


### Methods

#### VLANGroup.add_vlan

> ```add_vlan(self, data: netdot.dataclasses.vlan.VLAN) -> netdot.dataclasses.vlan.VLAN```

```
Add a VLAN to this VLANGroup.

Returns:
    netdot.VLAN: The created VLAN.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### VLANGroup.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### VLANGroup.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### VLANGroup.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### VLANGroup.load_vlans

> ```load_vlans(self, ignore_404=True) -> List[netdot.dataclasses.vlan.VLAN]```

```
Load the VLANs associated to this VLANGroup.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.VLAN]: All/Any VLANs related to this VLANGroup.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### VLANGroup.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### VLANGroup.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.Zone`

<a id="class-netdotzone"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| active | bool | False |
| contactlist_xlink | int |  |
| expire | int |  |
| info | str |  |
| minimum | int |  |
| name | str |  |
| refresh | int |  |
| retry | int |  |
| rname | str |  |
| serial | int |  |
| default_ttl | int |  |
| export_file | str |  |
| mname | str |  |
| include | str |  |


### Methods

#### Zone.add_rr

> ```add_rr(self, data: netdot.dataclasses.dns.RR) -> netdot.dataclasses.dns.RR```

```
Add a RR to this Zone.

Returns:
    netdot.RR: The created RR.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Zone.add_subnetzone

> ```add_subnetzone(self, data: netdot.dataclasses.ipblock.SubnetZone) -> netdot.dataclasses.ipblock.SubnetZone```

```
Add a SubnetZone to this Zone.

Returns:
    netdot.SubnetZone: The created SubnetZone.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Zone.add_zonealias

> ```add_zonealias(self, data: netdot.dataclasses.dns.ZoneAlias) -> netdot.dataclasses.dns.ZoneAlias```

```
Add a ZoneAlias to this Zone.

Returns:
    netdot.ZoneAlias: The created ZoneAlias.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Zone.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### Zone.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### Zone.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### Zone.load_contactlist

> ```load_contactlist(self) -> netdot.dataclasses.users.ContactList```

```
Load the contactlist (ContactList) associated to this Zone.

Returns:
    netdot.ContactList: The full ContactList object if available, else None.
```

#### Zone.load_rr

> ```load_rr(self, ignore_404=True) -> List[netdot.dataclasses.dns.RR]```

```
Load the RR associated to this Zone.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.RR]: All/Any RR related to this Zone.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Zone.load_subnetzones

> ```load_subnetzones(self, ignore_404=True) -> List[netdot.dataclasses.ipblock.SubnetZone]```

```
Load the SubnetZones associated to this Zone.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.SubnetZone]: All/Any SubnetZones related to this Zone.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Zone.load_zonealiases

> ```load_zonealiases(self, ignore_404=True) -> List[netdot.dataclasses.dns.ZoneAlias]```

```
Load the ZoneAliases associated to this Zone.

Args:
    ignore_404 (bool, optional): If True, will continue upon encountering HTTP 404 errors. Defaults to True.

Returns:
    List[netdot.ZoneAlias]: All/Any ZoneAliases related to this Zone.
    
Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: For any HTTP errors. (error details can be found in Netdot's apache server logs)
```

#### Zone.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### Zone.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```

## class `netdot.ZoneAlias`

<a id="class-netdotzonealias"></a>

### Attributes

| Attribute | Type | Default |
| --------- | ---- | ------- |
| info | str |  |
| name | str |  |
| zone_xlink | int |  |


### Methods

#### ZoneAlias.create

> ```create(self)```

```
Create this object in Netdot. (wrapper around :func:`create_or_update()`)

Raises:
    ValueError: If trying to create an object that already has an `id`.
```

#### ZoneAlias.create_or_update

> ```create_or_update(self: 'NetdotAPIDataclass')```

```
Create or update this object in Netdot.

> NOTE: Upon creation, this object's `id` will be populated.

Raises:
    ProtocolError: Can occur if your connection with Netdot has any issues.
    HTTPError: if the object cannot be created for any reason. (error details can be found in Netdot's apache server logs)
    As an example, expect a generic HTTP 400 when:
      * an object breaks 'uniqueness' rules (Ex. 2 Sites named "Test"),
      * an object is missing required fields (Ex. a Person without a `lastname`),
      * an object is improperly formatted (Ex. a Device with a `date_installed` that is not a datetime),
```

#### ZoneAlias.delete

> ```delete(self, **kwargs)```

```
Delete this object.

Args:
    See :func:`netdot.Repository.delete`

Requires:
    Must have repository attached. Use with_repository(...)
```

#### ZoneAlias.load_zone

> ```load_zone(self) -> netdot.dataclasses.dns.Zone```

```
Load the zone (Zone) associated to this ZoneAlias.

Returns:
    netdot.Zone: The full Zone object if available, else None.
```

#### ZoneAlias.replace

> ```replace(self, **kwargs)```

```
Return a new object replacing specified fields with new values.

Returns:
    NetdotAPIDataclass: A copy of this object with the specified fields replaced with new values.
```

#### ZoneAlias.update

> ```update(self)```

```
Update this object in Netdot.

Raises:
    ValueError: If trying to update an object that has no `id`.
```






# Netdot Python API Environment Variables
<a id="netdot-python-api-environment-variables"></a>

> Generated using `netdot.config.help()`
>
> Version 0.5.0 documentation generated on Nov 06, 2025 at 03:48PM 

```

  --terse TERSE         Print terse output (that generally tries to fit the
                        screen width). [env var: NETDOT_CLI_TERSE] (default:
                        False)
  --server-url SERVER_URL
                        The URL of the Netdot server. [env var:
                        NETDOT_CLI_SERVER_URL] (default:
                        https://nsdb.uoregon.edu)
  --truncate-min TRUNCATE_MIN_CHARS
                        The absolute minimum number of characters to print
                        before truncating. [env var:
                        NETDOT_CLI_TRUNCATE_MIN_CHARS] (default: 20)
  --terse-col-width TERSE_COL_WIDTH
                        The number of characters to use for each column when
                        printing terse output. [env var:
                        NETDOT_CLI_TERSE_COL_WIDTH] (default: 16)
  --terse-max-chars TERSE_MAX_CHARS
                        The maximum number of characters to print before
                        truncating. [env var: NETDOT_CLI_TERSE_MAX_CHARS]
                        (default: None)
  --display-full-objects DISPLAY_FULL_OBJECTS
                        Display the full objects instead of just the object
                        IDs. [env var: NETDOT_CLI_DISPLAY_FULL_OBJECTS]
                        (default: False)
  --skip-ssl SKIP_SSL   Skip SSL verification when making API requests.
                        **Never recommended in production.** Note: you must
                        reconnecting to Netdot for config to take effect.
                        (Used as a default arg for an __init__ method) [env
                        var: NETDOT_CLI_SKIP_SSL] (default: False)
  --connect-timeout CONNECT_TIMEOUT
                        The number of seconds to wait for connection to
                        establish with the Netdot server. Note: you must
                        reconnecting to Netdot for config to take effect.
                        (Used as a default arg for an __init__ method) [env
                        var: NETDOT_CLI_CONNECT_TIMEOUT] (default: 3)
  --timeout TIMEOUT     The number of seconds to wait for a response from the
                        API server. Note: you must reconnecting to Netdot for
                        config to take effect. (Used as a default arg for an
                        __init__ method) Note: "timeout is not a time limit on
                        the entire response download; rather, an exception is
                        raised if the server has not issued a response for
                        timeout seconds (more precisely, if no bytes have been
                        received on the underlying socket for timeout
                        seconds). If no timeout is specified explicitly,
                        requests do not time out." (from
                        requests.readthedocs.io) [env var: NETDOT_CLI_TIMEOUT]
                        (default: 20)
  --raise-parse-errors RAISE_FIELD_PARSE_ERRORS
                        Raise an exception if there is an error parsing any
                        server response (from Netdot). Else, log a warning and
                        continue, using the 'raw string' data. (These are
                        generally warnings that should be fixed by a netdot
                        python package developer) [env var:
                        NETDOT_CLI_RAISE_FIELD_PARSE_ERRORS] (default: False)
  --warn-missing-fields WARN_MISSING_FIELDS
                        Warn if a field is missing from the response from the
                        API server. [env var: NETDOT_CLI_WARN_MISSING_FIELDS]
                        (default: True)
  --threads THREADS     The number of threads to use when making
                        parallelizable API GET requests. Note: you must
                        reconnecting to Netdot for config to take effect.
                        (Used as a default arg for an __init__ method) [env
                        var: NETDOT_CLI_THREADS] (default: 1)
  --trace-downloads TRACE_DOWNLOADS
                        Intermittently log an "INFO" message saying how many
                        bytes have been downloaded from Netdot. Useful for
                        long-running download tasks. (Note: This *is* thread-
                        safe.) Note: you must reconnecting to Netdot for
                        config to take effect. (Used as a default arg for an
                        internal __init__ method) [env var:
                        NETDOT_CLI_TRACE_DOWNLOADS] (default: False)
  --trace-threshold TRACE_THRESHOLD
                        See TRACE_DOWNLOADS above. This threshold determines
                        how often messages should be logged -- the number of
                        bytes downloaded from Netdot before a log message is
                        printed. Note: you must reconnecting to Netdot for
                        config to take effect. (Used as a default arg for an
                        internal __init__ method) [env var:
                        NETDOT_CLI_TRACE_THRESHOLD] (default: 1000000)
  --save-as-file-on-error SAVE_AS_FILE_ON_ERROR
                        (Try to) Save the proposed changes to a file when an
                        error occurs. [env var:
                        NETDOT_CLI_SAVE_AS_FILE_ON_ERROR] (default: True)
  --error-pickle-filename ERROR_PICKLE_FILENAME
                        The filename to use when saving proposed changes to a
                        file when an error occurs. (timestamp based on when
                        the script is run) [env var:
                        NETDOT_CLI_ERROR_PICKLE_FILENAME] (default: netdot-
                        cli-0.5.0-proposed_changes-2025-11-06_15-48.pickle)

 In general, command-line values override environment variables which override
defaults.


⚠ NOTICE: These defaults are read from Environment Variables when 
`netdot.config` module is imported (via `netdot.config.parse_env_vars`). 
Look for "[env var: NETDOT_CLI_...]" above to discover the available 
Environment Variables.

Example: `export NETDOT_CLI_TERSE=True`

Import Env Vars anytime by calling: `netdot.config.parse_env_vars()`

Alternately, override these defaults by setting 
`netdot.config.<ENV_VAR_NAME>` directly in your Python code.

Example: `netdot.config.TERSE=True`

```


