# Copyright 2025 eelus1ve and the WaveTeam
#
# GitHub (author): https://github.com/eelus1ve
# GitHub (organization): https://github.com/WaveTeamDevs
# Repository: https://github.com/WaveTeamDevs/WaveSQL
# Website: https://waveteam.net
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import mysql
import colorama
import os
import traceback
import copy
import inspect
import shutil
import pathlib
import sys
import pprint
import yaml

from mysql.connector import connect
from mysql.connector.errorcode import ER_BAD_DB_ERROR, ER_DUP_ENTRY
from mysql.connector.connection import MySQLConnection
from mysql.connector.cursor import MySQLCursor
from typing import Literal, Any, Generator
from datetime import datetime
from colorama import Fore
from pathlib import Path


if __name__ == "__main__":
    from constants import PATH_DB_INIT_SCRIPTS, CONFIG_PATH, LOG_COLORS, USED_TABLES
    from sqlFileObject import SqlFileObject, SqlFileQueries
    from errors import DublicateKeyError
else:
    from .constants import PATH_DB_INIT_SCRIPTS, CONFIG_PATH, LOG_COLORS, USED_TABLES
    from .sqlFileObject import SqlFileObject, SqlFileQueries
    from .errors import DublicateKeyError

colorama.init(autoreset=True)


def read_files_to_vars(directory: str):
    dir_path = Path(directory)
    file_contents = {}
    
    if not dir_path.is_dir():
        raise NotADirectoryError(f"Folder {directory} not found")

    for file_path in dir_path.iterdir():
        if file_path.is_file():
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
                file_contents[file_path.name] = content

    return file_contents


class DBConnectionManager:
    def __init__(self, outer, edit_func, database: str | None, is_dictionary: bool | None):
        self.outer = outer
        self.database = database
        self.is_dictionary = is_dictionary if is_dictionary is not None else outer.is_dictionary
        self.connection = None
        self.cursor = None
        self.p_db = ""
        self.edit_func = edit_func

    def __enter__(self):
        if self.database is not None:
            if not isinstance(self.database, str):
                raise TypeError("Database name must be a string.")
            self.p_db = self.database
            mysql_config = copy.deepcopy(self.outer.config["mysql"])
            mysql_config["database"] = self.database
        else:
            mysql_config = self.outer.config["mysql"]
            self.p_db = mysql_config["database"]

        self.connection = connect(**mysql_config, use_unicode=True)
        self.cursor = self.connection.cursor(buffered=True, dictionary=self.is_dictionary)
        self.edit_func(True)
        return self.connection, self.cursor

    def __exit__(self, exc_type, exc_val, exc_tb):
        try:
            if exc_type is None:
                self.connection.commit()
            else:
                self.connection.rollback()
        except Exception:
            pass

        try:
            if self.cursor:
                self.cursor.close()
        except Exception:
            pass

        try:
            if self.connection:
                self.connection.close()
        except Exception:
            pass

        if exc_type:
            if self.edit_func():
                self.outer.log(
                    level=8,
                    module="DATABASE",
                    text=f"DB_CONNECT: {self.p_db}",
                    err=exc_val,
                    is_console_log=True
                )
            else:
                print(LOG_COLORS["LIGHTRED"] + f"[{datetime.now().strftime('%d-%m-%Y %H:%M:%S')}] [ERROR] [DATABASE] DB_CONNECT: {self.p_db}")

        return False  # ← обязательно, чтобы ошибка пробросилась


class WaveSQL:
    """Example:\n
        db = WaveDataBase(is_dictionary=True, is_console_log=True, is_log_backtrace=True, is_auto_start=True)\n
        db.log(level=3, text="All is good!")
        
    default_log_level : int, optional
        Logging level indicating the type of message. Default is 1 (INFO).

        Available levels:
        - 1: INFO — general information (color: CYAN)
        - 2: DEBUG — debug information (color: MAGENTA)
        - 3: OK — successful operation (color: GREEN)
        - 4: WARNING — warning that does not affect execution (color: YELLOW)
        - 5: FAILURE — logic or business error (color: RED)
        - 6: EXPECTED ERROR — an expected error (color: RED)
        - 7: UNEXPECTED ERROR — an unexpected error (color: RED)
        - 8: ERROR — general error (color: RED)
        - 9: FATAL ERROR — critical error requiring immediate attention (color: LIGHTRED)
    """
    def __init__(
        self, config: dict | str | None = None, path_to_sql: Path | str | None = None, is_dictionary: bool = False,
        executemany: bool = False, is_console_log: bool = False, is_log_backtrace: bool = False, raise_log_on_fail: bool = False,
        is_pprint: bool = False, is_protected: bool = True, is_auto_start: bool = False, is_try_update_db: bool = False,
        is_create_python_bridge: bool = False, is_try_update_python_bridge: bool = True, default_log_sep: str = " ",
        default_log_module: str = "DATABASE", default_log_level: int | Literal[1, 2, 3, 4, 5, 6, 7, 8, 9] = 1, debug_mode: bool = False
    ) -> None:
        self.config_save = config
        if isinstance(self.config_save, str):
            self.config_save = f'fr"{self.config_save}"'
        self.is_auto_start = is_auto_start
        if not isinstance(config, dict):
            if config is None:
                config = CONFIG_PATH
            with open(config, "r", encoding="utf-8") as f:
                self.config = yaml.safe_load(f)
        else:
            self.config = config
        if path_to_sql is None:
            self.path_to_sql = PATH_DB_INIT_SCRIPTS
        else:
            if not isinstance(path_to_sql, (Path, str)):
                raise TypeError(f"Expected 'path_to_sql' to be of type Path (Path or str), but got: {type(path_to_sql).__name__}")
            self.path_to_sql = path_to_sql if isinstance(path_to_sql, Path) else Path(path_to_sql)
        if not isinstance(is_dictionary, bool):
            raise TypeError(f"Expected 'is_dictionary' to be of type bool (True or False), but got: {type(is_dictionary).__name__}")
        self.is_dictionary = is_dictionary
        if not isinstance(executemany, bool):
            raise TypeError(f"Expected 'executemany' to be of type bool (True or False), but got: {type(executemany).__name__}")
        self.executemany = executemany
        if not isinstance(is_console_log, bool):
            raise TypeError(f"Expected 'is_console_log' to be of type bool (True or False), but got: {type(is_console_log).__name__}")
        self.is_console_log = is_console_log
        if not isinstance(is_log_backtrace, bool):
            raise TypeError(f"Expected 'is_log_backtrace' to be of type bool (True or False), but got: {type(is_log_backtrace).__name__}")
        self.is_log_backtrace = is_log_backtrace
        if not isinstance(raise_log_on_fail, bool):
            raise TypeError(f"Expected 'raise_log_on_fail' to be of type bool (True or False), but got: {type(raise_log_on_fail).__name__}")
        self.raise_log_on_fail = raise_log_on_fail
        if not isinstance(is_pprint, bool):
            raise TypeError(f"Expected 'is_pprint' to be of type bool (True or False), but got: {type(is_pprint).__name__}")
        self.is_pprint = is_pprint
        if not isinstance(is_protected, bool):
            raise TypeError(f"Expected 'is_protected' to be of type bool (True or False), but got: {type(is_protected).__name__}")
        self.is_protected = is_protected
        if not isinstance(is_try_update_db, bool):
            raise TypeError(f"Expected 'is_try_update_db' to be of type bool (True or False), but got: {type(is_try_update_db).__name__}")
        self.__is_try_update_db = is_try_update_db
        if not isinstance(is_create_python_bridge, bool):
            raise TypeError(f"Expected 'is_create_python_bridge' to be of type bool (True or False), but got: {type(is_create_python_bridge).__name__}")
        self.__is_create_python_bridge = is_create_python_bridge
        if not isinstance(is_try_update_python_bridge, bool):
            raise TypeError(f"Expected 'is_try_update_python_bridge' to be of type bool (True or False), but got: {type(is_try_update_python_bridge).__name__}")
        self.__is_try_update_python_bridge = is_try_update_python_bridge
        if not isinstance(default_log_sep, str):
            self.default_log_sep = str(default_log_sep)
        self.default_log_sep = default_log_sep
        if not isinstance(default_log_module, str):
            self.default_log_module = str(default_log_module)
        self.default_log_module = default_log_module
        if not isinstance(default_log_level, int):
            raise TypeError(f"Expected 'default_log_level' to be of type int (worked examples in (1, 2, 3, 4, 5, 6, 7, 8, 9)), but got: {type(default_log_level).__name__}")
        self.default_log_level = default_log_level
        if not isinstance(debug_mode, bool):
            raise TypeError(f"Expected 'debug_mode' to be of type bool (True or False), but got: {type(debug_mode).__name__}")
        self.debug_mode = debug_mode
        
        self.__db_init_succsess = False

        self.run_path: Path = pathlib.Path("/".join(str(sys.argv[0]).replace("\\", "/").split("/")[:-1])).resolve()
        self.local_dir: Path = Path(__file__).parent
        
        self.settings = {"dbname": self.config["mysql"]["database"]}

        # * autorun
        if is_auto_start:
            self.start()
    
    @staticmethod
    def __protected(method):
        def wrapper(self, *args, **kwargs):
            if self.is_protected:
                caller_frame = inspect.stack()[1].frame
                caller_self = caller_frame.f_locals.get('self')

                if caller_self is not None and isinstance(caller_self, self.__class__):
                    return method(self, *args, **kwargs)

                raise PermissionError(f"Async method '{method.__name__}' is protected and cannot be called from outside")
            
            return method(self, *args, **kwargs)
        
        return wrapper
    
    def start(self) -> None:
        all_sql_paths = sorted(self.path_to_sql.rglob("*_init_*.sql"), key=lambda x: int(Path(x).name.split("_")[0]))
        # required_files = {"0_init_db.sql", "1_init_logs.sql"}
        # print(f"local dir = {self.local_dir}")
        dir_path = self.local_dir / "sql"
        required_files = {f.name for f in dir_path.rglob('*') if f.is_file()}
        
        found_filenames = {p.name for p in all_sql_paths}
        missing_files = required_files - found_filenames
        sql_file_objects: list[SqlFileObject] = []
        if missing_files:
            local_sql_dir = self.local_dir / "sql"
            if not local_sql_dir.exists():
                raise FileNotFoundError(f"Local folder with default SQL files not found: {local_sql_dir}")
            for file_name in missing_files:
                src_file = local_sql_dir / file_name
                dest_file = self.path_to_sql / file_name
                if not src_file.exists():
                    raise FileNotFoundError(f"Local file missing: {src_file}")
                
                dest_file.parent.mkdir(parents=True, exist_ok=True)
                shutil.copy(src_file, dest_file)
                self.__print_log(backtrace=None, def_level="INFO", def_color="CYAN", def_module="DATABASE", def_msg=f"Copied file {file_name} from local folder to {dest_file}", is_raise_on_fail=False)
        else:
            self.__print_log(backtrace=None, def_level="INFO", def_color="CYAN", def_module="DATABASE", def_msg="All required SQL files found in the user path.", is_raise_on_fail=False)
            
        all_sql_paths = sorted(self.path_to_sql.rglob("*_init_*.sql"), key=lambda x: int(Path(x).name.split("_")[0]))
        sql_file_objects.extend([SqlFileObject(path=i, dict_of_values=self.settings, is_create_python=self.__is_create_python_bridge) for i in all_sql_paths])
        
        try:
            cnx = connect(**self.config["mysql"])
            if self.__is_try_update_db:
                # TODO write db update
                pass
            cnx.close()
            
            self.log(level=1, text="Initializing skipped!")
        except mysql.connector.Error as err:
            if err.errno == ER_BAD_DB_ERROR:
                try:
                    connection = connect(
                        host=self.config["mysql"]["host"],
                        user=self.config["mysql"]["user"],
                        password=self.config["mysql"]["password"]
                    )
                    cursor = connection.cursor()
                    for sql_file_object in sql_file_objects:
                        for sql_command in sql_file_object.sql_objects:
                            try:
                                cursor.execute(sql_command.code)
                            except Exception as ex:
                                error_text = str(ex)
                                msg = (
                                    f"[SQL INIT ERROR]\n"
                                    f"File: {sql_file_object.file_path}\n"
                                    f"Object: {sql_command.name}\n"
                                    f"SQL Code:\n"
                                    f"{sql_command.code.strip()}\n"
                                    f"{error_text}\n"
                                )
                                try:
                                    self.log(level=8, text=msg, err=ex, is_console_log=True, is_log_backtrace=False, is_raise_on_fail=True)
                                except Exception:
                                    self.__print_log(backtrace=ex, def_module="DATABASE", def_msg=msg, is_raise_on_fail=True)
                                quit(0)
                                
                    connection.commit()
                    cursor.close()
                    connection.close()
                except Exception as ex:
                    try:
                        self.log(level=8, text="Error initializing the database:", err=ex, is_console_log=True, is_log_backtrace=True, is_raise_on_fail=True)
                    except Exception:
                        self.__print_log(backtrace=ex, def_module="DATABASE", def_msg="Error initializing the database:", is_raise_on_fail=True)
                self.log(level=3, text="Database initialized!")
                
        if self.__is_create_python_bridge:
            queries_path = self.path_to_sql / "queries.sql"
            if os.path.exists(queries_path):
                files = read_files_to_vars(directory=self.local_dir / "python")
                
                query_file = SqlFileQueries(path=queries_path, create_python=self.__is_create_python_bridge, is_dictionary_default=self.is_dictionary, dict_of_values=self.settings)
                
                sql_queries_sync_lst = []
                sql_queries_async_lst = []
                
                for sql_file in sql_file_objects:
                    for sql_object in sql_file.sql_objects:
                        if sql_object.can_python:
                            sql_queries_sync_lst.append(sql_object.sync_python_code)
                            sql_queries_async_lst.append(sql_object.async_python_code)
                
                for sql_query in query_file.sql_queries:
                    if sql_query.can_python:
                        sql_queries_sync_lst.append(sql_query.sync_python_code)
                        sql_queries_async_lst.append(sql_query.async_python_code)
                        
                files["database.py"] += "\n" + "\n\n".join(sql_queries_sync_lst) + "\n\n\n" + f'db = DataBase(path_to_sql=fr"{self.path_to_sql}", config={self.config_save}, is_dictionary={self.is_dictionary}, is_console_log={self.is_console_log}, is_log_backtrace={self.is_log_backtrace}, is_auto_start={self.is_auto_start})\ndb.log(level=3, text="All is good!")\n'
                files["asyncdatabase.py"] += "\n" + "\n\n".join(sql_queries_async_lst) + "\n\n\n" + f'adb = AsyncDataBase(path_to_sql=fr"{self.path_to_sql}", config={self.config_save}, is_dictionary={self.is_dictionary}, is_console_log={self.is_console_log}, is_log_backtrace={self.is_log_backtrace}, is_auto_start={self.is_auto_start})\nadb.sync_log(level=3, text="All is good!")\n'
                
                for filename, content in files.items():
                    file_path = self.run_path / filename
                    with open(file_path, 'w', encoding='utf-8') as f:
                        f.write(content)

        self.log(level=3, text="All is good !")

    def __db_connect(
        self,
        database: str | None = None,
        is_dictionary: bool | None = None
    ) -> DBConnectionManager:
        """
        Establishes a connection to the MySQL database and returns a connection and cursor.

        Args:
            database (str, optional): Name of the target database. If None, the default from config is used.
            is_dictionary (bool, optional): If True, returns rows as dictionaries. If None, uses instance default.

        Returns:
            tuple[MySQLConnection, MySQLCursor]:
                A tuple containing the database connection and a cursor object.

        Raises:
            TypeError: If 'database' is provided but not a string.
            Exception: Any connection error is re-raised after logging.
        """
        return DBConnectionManager(self, self.__edit_func, database=database, is_dictionary=is_dictionary)

    @__protected
    def _db_query(
        self,
        query: str,
        inputs: tuple | Any = (),
        fetch: Literal[0, 1, 2] = 0,
        database: str | None = None,
        is_dictionary: bool | None = None,
        executemany: bool | None = None
    ) -> None | list[tuple] | list[dict] | tuple | dict:
        """
        Executes a SQL query with optional input parameters and fetch mode.

        Args:
            query (str): SQL query string to be executed.
            inputs (tuple or Any, optional): Parameters to pass with the query. Defaults to ().
            fetch (Literal[0, 1, 2], optional) Result retrieval mode
            - 0: Return None (no data expected).
            - 1: Return a single row.
            - 2: Return all rows.
            ---------------------
            database (str, optional): Target database name. Defaults to config value.
            is_dictionary (bool, optional): If True, results are returned as dicts. If False, as tuples. If None using default value `self.is_dictionary`.

        Returns
        ------------------
        None | tuple | dict | list[tuple] | list[dict]:
            - None: if fetch = 0,
            - tuple/dict: if fetch = 1,
            - list of tuple/dict: if fetch = 2

        Raises:
            Exception: Any error raised during SQL execution, re-raised after logging.
        """
        if executemany is None:
            executemany = self.executemany

        result = None
        ex = None

        if executemany != False and isinstance(inputs, (tuple, list)):
            if isinstance(inputs, tuple):
                inputs = list(inputs)
            if isinstance(inputs[0], list):
                inputs = tuple(inputs)
            if not isinstance(inputs[0], tuple):
                inputs = list(map(lambda x: (x, ), inputs))
            if executemany is None:
                executemany = True

        elif executemany is None:
            executemany = False
                
        if not isinstance(inputs, (tuple, list)):
            inputs = (inputs,)
        
        elif not (isinstance(inputs, tuple) or executemany):
            inputs = tuple(inputs)

        elif not isinstance(inputs, list) and executemany:
            inputs = list(inputs)


        with self.__db_connect(database, is_dictionary) as (connection, cursor):
            try:
                if executemany:
                    cursor.executemany(query, inputs)
                else:
                    cursor.execute(query, inputs)

                if fetch == 1:
                    result = cursor.fetchone()
                elif fetch == 2:
                    result = cursor.fetchall()

            except mysql.connector.Error as err:
                if err.errno == ER_DUP_ENTRY:
                    err = DublicateKeyError(err)

                ex = err

            except Exception as err:
                self.log(
                    level=8,
                    module="DATABASE",
                    text=f"DB_QUERY: {query}",
                    err=err,
                    is_console_log=True
                )

                ex = err

            finally:
                if ex:
                    raise ex
                
                return result

    @__protected
    def _db_call_procedure(
        self,
        procedure_name: str,
        inputs: tuple | Any = (),
        fetch: Literal[0, 1, 2] = 0,
        database: str | None = None,
        is_dictionary: bool | None = None
    ) -> None | list[tuple] | list[dict] | tuple | dict:
        """
        Calls a stored procedure in the MySQL database and optionally fetches results.

        Args:
            procedure_name (str): The name of the stored procedure to call.
            inputs (tuple or Any, optional): Parameters for the procedure. Defaults to ().
            fetch (Literal[0, 1, 2], optional): Result retrieval mode:
                - 0: Return None.
                - 1: Return a single row.
                - 2: Return all rows.
            database (str, optional): Target database name. Defaults to config value.
            is_dictionary (bool, optional): If True, results are returned as dicts. If False, as tuples. If None using default value `self.is_dictionary`.

        Returns:
            None | tuple | dict | list[tuple] | list[dict]:
                - None: if fetch = 0,
                - tuple/dict: if fetch = 1,
                - list of tuple/dict: if fetch = 2

        Raises:
            Exception: Any error during procedure execution is re-raised after logging.
        """
        
        result = None
        ex = None

        with self.__db_connect(database, is_dictionary) as (connection, cursor):
            try:
                if not isinstance(inputs, tuple):
                    inputs = (inputs,)

                cursor.callproc(procedure_name, inputs)

                for result_cursor in cursor.stored_results():
                    if fetch == 1:
                        result = result_cursor.fetchone()
                    elif fetch == 2:
                        result = result_cursor.fetchall()

            except mysql.connector.Error as err:
                if err.errno == ER_DUP_ENTRY:
                    err = DublicateKeyError(err)
                ex = err
            except Exception as err:
                self.log(
                    level=8,
                    module="DATABASE",
                    text=f"DB_PROCEDURE: {procedure_name}",
                    err=err,
                    is_console_log=True
                )
                ex = err
            finally:
                cursor.close()
                connection.close()
                if ex:
                    raise ex
                return result

    def __save_log_query(
        self, *, level: int, module: str,
        msg: str, backtrace: str
    ) -> dict:
        return self._db_call_procedure(
            "insert_log",
            (
                level,
                module,
                msg,
                backtrace
            ), fetch=1,
            is_dictionary=True
        )

    def log(
        self, text: str | Exception | Any = "", *args, level: int | None = None,
        sep: str | None = None, module: str = None, err: Exception | None = None,
        is_console_log: bool | None = None, is_log_backtrace: bool | None = None,
        is_raise_on_fail: bool | None = None, is_pprint: bool | None = None, **kwargs
    ) -> None | Exception:
        """
        Logs a message or exception to the database and, if necessary, to the console.

        Parameters
        ----------
        level : int, optional
            Logging level that determines the type of message. Default is 1 (INFO).
            
            Available levels:
            
            - 1: INFO — general information (color: CYAN)
            - 2: DEBUG — debugging information (color: MAGENTA)
            - 3: OK — successful operation completion (color: GREEN)
            - 4: WARNING — warning not affecting execution (color: YELLOW)
            - 5: FAILURE — logic or business error (color: RED)
            - 6: EXPECTED ERROR — expected error (color: RED)
            - 7: UNEXPECTED ERROR — unexpected error (color: RED)
            - 8: ERROR — standard error (color: RED)
            - 9: FATAL ERROR — critical error requiring immediate attention (color: LIGHTRED)

        text : Any, optional
            Message to log or exception object. Default is an empty string.

        module : str, optional
            Name of the module from which the log is sent. Default is "database".

        err : Exception, optional
            Exception to log (if passed separately from `text`). Default is None.

        is_console_log : bool, optional
            Explicitly indicates whether to output the log to the console. If None, uses the value of `self.is_console_log`.

        Behavior
        --------
        - Converts `text` to a string for logging.
        - If an exception is passed (`err` or `text` is an `Exception`), formats the stack trace.
        - If `is_console_log=True`, additionally outputs the log to the console via `self.__print_log(...)`.

        Notes
        -----
        The method uses `traceback.format_exception` to obtain the exception traceback string.
        """
        if is_console_log is None:
            is_console_log = self.is_console_log
        if is_log_backtrace is None:
            is_log_backtrace = self.is_log_backtrace
        if is_raise_on_fail is None:
            is_raise_on_fail = self.raise_log_on_fail
        if sep is None:
            sep = self.default_log_sep
        if level is None:
            level = self.default_log_level
        if module is None:
            module = self.default_log_module
        
        objects = {}
        save_text = ""
    
        if is_pprint:
            if isinstance(text, (dict, tuple, list)):
                objects["text"] = text
            parts = [str(text)] if text != "" else []
            save_text = str(text)
            if args:
                objects["args"] = args
                parts.extend(str(a) for a in args)
            if kwargs:
                objects["kwargs"] = kwargs
                kwargs_str = sep.join(f"{k}={v}" for k, v in kwargs.items())
                parts.append(kwargs_str)
            msg = sep.join(parts)
        else:
            parts = [str(text)] if text != "" else []
            parts.extend(str(a) for a in args)
            if kwargs:
                kwargs_str = sep.join(f"{k}={v}" for k, v in kwargs.items())
                parts.append(kwargs_str)
            msg = sep.join(parts)

        if isinstance(err, Exception):
            backtrace = "".join(traceback.format_exception(type(err), err, err.__traceback__))
        elif isinstance(text, Exception):
            backtrace = "".join(traceback.format_exception(type(text), text, text.__traceback__))
        else:
            backtrace = ""
        try:
            db_log: dict = self.__save_log_query(level=level, module=module, msg=msg, backtrace=backtrace)
            new_backtrace = None
        except Exception as ex:
            new_backtrace = ex
            db_log: dict = {}
            is_console_log = True
        if is_console_log:
            if db_log == {} or db_log is None:
                new_backtrace = (new_backtrace if new_backtrace else backtrace)
                return self.__print_log(log=db_log, backtrace=new_backtrace, def_msg=msg, is_raise_on_fail=is_raise_on_fail, is_pprint=is_pprint, objects=objects, save_text=save_text)
            else:
                return self.__print_log(log=db_log, backtrace=(backtrace if is_log_backtrace and backtrace else None), is_raise_on_fail=is_raise_on_fail, is_pprint=is_pprint, objects=objects, save_text=save_text)
        
    def __print_log(
        self, log: dict | None = None, backtrace: str | None = None, def_level: str | None = "ERROR",
        def_module: str | None = "logs", def_color: str | None = "LIGHTRED",
        def_time: datetime | None = datetime.now(), def_msg: str | None = "Unloggable error !!!",
        is_raise_on_fail: bool | None = None, is_pprint: bool | None = None, objects: dict = None, save_text: str | None = None
    ) -> None | Exception:
        if is_raise_on_fail is None:
            is_raise_on_fail = self.raise_log_on_fail
            
        if is_pprint is None:
            is_pprint = self.is_pprint
            
        if log is not None:
            color = LOG_COLORS.get(log.get("log_level_color_name", def_color), Fore.RED)
            log_time_str = log.get("log_date", def_time).strftime("%d-%m-%Y %H:%M:%S")
            log_level = log.get("log_level_name", def_level)
            module = log.get("log_module", def_module)
            msg = log.get("log_message", def_msg)
            # backtrace = log.get("", backtrace)
            if is_pprint and objects.get("text") is not None:
                print_mes = f"[{log_time_str}] [{log_level}] [{module}]\n"
            else:
                if save_text:
                    print_mes = f"[{log_time_str}] [{log_level}] [{module}] {save_text}"
                else:
                    print_mes = f"[{log_time_str}] [{log_level}] [{module}] {msg}"
        else:
            print_mes = f"[{datetime.now().strftime("%d-%m-%Y %H:%M:%S")}] [{def_level}] [{def_module}] {def_msg}"
            color = LOG_COLORS[f"{def_color}"]
            
        if backtrace is not None:
            print_mes = f"{print_mes}\n{backtrace}"
            
            # print(color)
            # print(print_mes)
            
        if is_pprint:
            print(color + print_mes)
            if len(list(objects.keys())) > 0:
                pprint.pprint(objects)
        else:
            print(color + print_mes)

    def __edit_func(self, _bool: bool | None = None) -> bool:
        if bool is None:
            return self.__db_init_succsess
        self.__db_init_succsess = _bool
        return self.__db_init_succsess

    def clear_tables(self, database: str | None = None, verbose: bool = True):
        """
        Clears all base tables in the given database by truncating them.

        Args:
            database (str): Target database name.
            verbose (bool): If True, logs cleared tables (only in debug_mode).
        """
        if database is None:
            database = self.settings.get("dbname")

        if not self.debug_mode:
            return
        
        if verbose:
            self.__print_log(def_module="DATABASE TESTS", def_msg="VERBOSE MODE ENABLED — SKIPPING TABLE CLEARING")
            return
        
        with self.__db_connect(database, False) as (connection, cursor):
            try:
                cursor.execute("SET FOREIGN_KEY_CHECKS = 0")
                cursor.execute("SELECT table_name FROM information_schema.tables WHERE table_schema = DATABASE() AND table_type='BASE TABLE'")
                tables = cursor.fetchall()
                for (table_name,) in tables:
                    if table_name in USED_TABLES:
                        continue
                    cursor.execute(f"TRUNCATE TABLE `{table_name}`")
                cursor.execute("SET FOREIGN_KEY_CHECKS = 1")
            except Exception as err:
                self.log(
                    level=8,
                    module="DATABASE",
                    text=f"Failed to clear tables in database '{database}'",
                    err=err,
                    is_console_log=True
                )
                raise err
            
    def drop_database(self, database: str | None = None, verbose: bool = True):
        """
        Drops the given database.

        Args:
            database (str | None): Name of the database to drop. If None, uses default from settings.
            verbose (bool): If True, logs the action but does not execute the drop.

        Raises:
            ValueError: If no database name is provided or available.
            Exception: If dropping fails.
        """
        if database is None:
            database = self.settings.get("dbname")

        if not self.debug_mode:
            return
        
        if verbose:
            self.__print_log(def_module="DATABASE TESTS", def_msg="VERBOSE MODE ENABLED — SKIPPING TABLE CLEARING")
            return
        
        with self.__db_connect(database, False) as (connection, cursor):
            try:
                cursor.execute(f"DROP DATABASE `{database}`")
            except Exception as err:
                self.log(
                    level=8,
                    module="DATABASE",
                    text=f"Failed to drop database '{database}'",
                    err=err,
                    is_console_log=True
                )
                raise err


if __name__ == "__main__":
    db = WaveSQL(is_dictionary=True, is_console_log=True, is_log_backtrace=True, is_auto_start=True)
    db.log(level=3, text="All is good!")


__all__ = ["WaveSQL"]
