"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2025

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This module is intended to distributed as part of automatically generated code by the
peakrdl-python tool. It provides a set of base classes used by the autogenerated code
"""


def swap_msb_lsb_ordering(width: int, value: int) -> int:
    """
    swaps the msb/lsb on a integer

    Returns:
        swapped value
    """
    value_to_return = 0
    for bit_positions in zip(range(0, width), range(width-1, -1, -1)):
        bit_value = (value >> bit_positions[0]) & 0x1
        value_to_return |= bit_value << bit_positions[1]

    return value_to_return


class UnsupportedWidthError(Exception):
    """
    Exception for width that can not be supported in the legacy array based API
    """


def get_array_typecode(width: int) -> str:
    """
        python array typecode

        Args:
            width: in bits

        Returns:
            string to pass into the array generator

        """
    if width == 32:
        return 'L'

    if width == 64:
        return 'Q'

    if width == 16:
        return 'I'

    if width == 8:
        return 'B'

    raise UnsupportedWidthError(f'unhandled width {width:d}, consider using the new new list '
                                f'based API and callbacks rather than the legacy Array versions')


def is_power_two(value: int) -> bool:
    """
    efficient algorithm for checking if something is a power of two
    """
    return (value != 0) and ((value & (value - 1)) == 0)


def legal_register_width(width_in_bits: int) -> bool:
    """
    check if a register width is legal, see the system RDL specification 10.1 e) and f)
    must be a power of 2 and greater than 8
    """
    return (width_in_bits >= 8) and is_power_two(width_in_bits)


def calculate_bitmask(high:int, low:int) -> int:
    """
    Calculate an integer bitmask based on the high and low bit positions
    """
    if not isinstance(high, int):
        raise TypeError(f'high must be an int, got {type(high)}')
    if high < 0:
        raise ValueError('high must be greater than 0')
    if not isinstance(low, int):
        raise TypeError(f'low must be an int, got {type(low)}')
    if low < 0:
        raise ValueError('low must be greater than 0')
    if high < low:
        raise ValueError(f'low must be great than or equal to high, got: {low=}, {high=}')

    return ((1 << (high - low + 1)) - 1) << low
