from typing import List, Dict, Literal, TypedDict, Union
import numpy as np
from numpy.typing import NDArray

from fc_value import decode, encode


FCElementTypeLiteral = Literal[
    'NONE',
    'LUMPMASS3D',
    'LUMPMASS6D',
    'LUMPMASS2D',
    'POINT3D',
    'POINT2D',
    'POINT6D',
    'LUMPMASS2DR',
    'BEAM26',
    'BEAM36',
    'SPRING3D',
    'SPRING6D',
    'BEAM27',
    'BEAM37',
    'BAR2',
    'BAR3',
    'CABLE2',
    'CABLE3',
    'TRI3',
    'TRI6',
    'QUAD4',
    'QUAD8',
    'MITC3',
    'MITC6',
    'MITC4',
    'MITC8',
    'TETRA4',
    'TETRA10',
    'HEX8',
    'HEX20',
    'TETRA4S',
    'TETRA10S',
    'HEX8S',
    'HEX20S',
    'WEDGE6',
    'WEDGE15',
    'WEDGE6S',
    'WEDGE15S',
    'PYR5',
    'PYR13',
    'PYR5S',
    'PYR13S',
    'TRI3S',
    'TRI6S',
    'QUAD4S',
    'QUAD8S',
    'SPRING2D',
    'SHELL3S',
    'SHELL4S',
    'SHELL6S',
    'SHELL8S',
    'BEAM26S',
    'BEAM36S',
    'BEAM27S',
    'BEAM37S'
]


class FCElementType(TypedDict):
    name: FCElementTypeLiteral 
    fc_id: int 
    dim: int 
    order: int 
    nodes: int 
    edges: List[List[int]]
    facets: List[List[int]]
    tetras: List[List[int]]


FC_ELEMENT_TYPES: List[FCElementType] = [
    {
        'name': 'NONE',
        'fc_id': 0,
        'dim': 0,
        'order': 0,
        'nodes': 0,
        'edges': [],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'LUMPMASS3D',
        'fc_id': 38,
        'dim': 0,
        'order': 1,
        'nodes': 1,
        'edges': [],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'LUMPMASS6D',
        'fc_id': 40,
        'dim': 0,
        'order': 1,
        'nodes': 1,
        'edges': [],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'LUMPMASS2D',
        'fc_id': 82,
        'dim': 0,
        'order': 1,
        'nodes': 1,
        'edges': [],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'POINT3D',
        'fc_id': 99,
        'dim': 0,
        'order': 1,
        'nodes': 1,
        'edges': [],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'POINT2D',
        'fc_id': 100,
        'dim': 0,
        'order': 1,
        'nodes': 1,
        'edges': [],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'POINT6D',
        'fc_id': 101,
        'dim': 0,
        'order': 1,
        'nodes': 1,
        'edges': [],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'LUMPMASS2DR',
        'fc_id': 105,
        'dim': 0,
        'order': 1,
        'nodes': 1,
        'edges': [],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BEAM26',
        'fc_id': 36,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BEAM36',
        'fc_id': 37,
        'dim': 1,
        'order': 2,
        'nodes': 3,
        'edges': [[0, 2, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'SPRING3D',
        'fc_id': 39,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'SPRING6D',
        'fc_id': 41,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BEAM27',
        'fc_id': 89,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BEAM37',
        'fc_id': 90,
        'dim': 1,
        'order': 2,
        'nodes': 3,
        'edges': [[0, 2, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BAR2',
        'fc_id': 107,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BAR3',
        'fc_id': 108,
        'dim': 1,
        'order': 2,
        'nodes': 3,
        'edges': [[0, 2, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'CABLE2',
        'fc_id': 109,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'CABLE3',
        'fc_id': 110,
        'dim': 1,
        'order': 2,
        'nodes': 3,
        'edges': [[0, 2, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'TRI3',
        'fc_id': 10,
        'dim': 2,
        'order': 1,
        'nodes': 3,
        'edges': [[0, 1, 2, 0]],
        'facets': [[0, 1, 2]],
        'tetras': [],
    },
    {
        'name': 'TRI6',
        'fc_id': 11,
        'dim': 2,
        'order': 2,
        'nodes': 6,
        'edges': [[0, 3, 1, 4, 2, 5, 0]],
        'facets': [[0, 3, 1, 4, 2, 5]],
        'tetras': [],
    },
    {
        'name': 'QUAD4',
        'fc_id': 12,
        'dim': 2,
        'order': 1,
        'nodes': 4,
        'edges': [[0, 1, 2, 3, 0]],
        'facets': [[0, 1, 2, 3]],
        'tetras': [],
    },
    {
        'name': 'QUAD8',
        'fc_id': 13,
        'dim': 2,
        'order': 2,
        'nodes': 8,
        'edges': [[0, 4, 1, 5, 2, 6, 3, 7, 0]],
        'facets': [[0, 4, 1, 5, 2, 6, 3, 7]],
        'tetras': [],
    },
    {
        'name': 'MITC3',
        'fc_id': 29,
        'dim': 2,
        'order': 1,
        'nodes': 3,
        'edges': [[0, 1, 2, 0]],
        'facets': [[0, 1, 2]],
        'tetras': [],
    },
    {
        'name': 'MITC6',
        'fc_id': 30,
        'dim': 2,
        'order': 2,
        'nodes': 6,
        'edges': [[0, 3, 1, 4, 2, 5, 0]],
        'facets': [[0, 3, 1, 4, 2, 5]],
        'tetras': [],
    },
    {
        'name': 'MITC4',
        'fc_id': 31,
        'dim': 2,
        'order': 1,
        'nodes': 4,
        'edges': [[0, 1, 2, 3, 0]],
        'facets': [[0, 1, 2, 3]],
        'tetras': [],
    },
    {
        'name': 'MITC8',
        'fc_id': 32,
        'dim': 2,
        'order': 2,
        'nodes': 8,
        'edges': [[0, 4, 1, 5, 2, 6, 3, 7, 0]],
        'facets': [[0, 4, 1, 5, 2, 6, 3, 7]],
        'tetras': [],
    },
    {
        'name': 'TETRA4',
        'fc_id': 1,
        'dim': 3,
        'order': 1,
        'nodes': 4,
        'edges': [[0, 1, 2, 0], [0, 3], [1, 3], [2, 3]],
        'facets': [[0, 2, 1], [0, 1, 3], [1, 2, 3], [2, 0, 3]],
        'tetras': [[0, 1, 2, 3]],
    },
    {
        'name': 'TETRA10',
        'fc_id': 2,
        'dim': 3,
        'order': 2,
        'nodes': 10,
        'edges': [[0, 4, 1, 5, 2, 6, 0], [0, 7, 3], [1, 8, 3], [2, 9, 3]],
        'facets': [[0, 6, 2, 5, 1, 4], [0, 4, 1, 8, 3, 5], [1, 5, 2, 9, 3, 8], [2, 6, 0, 5, 3, 9]],
        'tetras': [],
    },
    {
        'name': 'HEX8',
        'fc_id': 3,
        'dim': 3,
        'order': 1,
        'nodes': 8,
        'edges': [[0, 1, 2, 3, 0], [4, 5, 6, 7, 4], [0, 4], [1, 5], [2, 6], [3, 7]],
        'facets': [[3, 2, 1, 0], [4, 5, 6, 7], [1, 2, 6, 5], [0, 1, 5, 4], [0, 4, 7, 3], [2, 3, 7, 6]],
        'tetras': [[1, 3, 4, 6], [3, 1, 4, 0], [1, 3, 6, 2], [4, 1, 6, 5], [3, 4, 6, 7]],
    },
    {
        'name': 'HEX20',
        'fc_id': 4,
        'dim': 3,
        'order': 2,
        'nodes': 20,
        'edges': [[0, 8, 1, 9, 2, 10, 3, 11, 0], [4, 12, 5, 13, 6, 14, 7, 15, 4],
                  [0, 0, 4], [1, 0, 5], [2, 0, 6], [3, 0, 7]],
        'facets': [[3, 10, 2, 9, 1, 8, 0, 11], [4, 12, 5, 13, 6, 14, 7, 15], [1, 9, 2, 18, 6, 13, 5, 17],
                   [0, 8, 1, 17, 5, 12, 4, 16], [0, 16, 4, 15, 7, 19, 3, 11], [2, 10, 3, 19, 7, 14, 6, 18]],
        'tetras': [],
    },
    {
        'name': 'TETRA4S',
        'fc_id': 15,
        'dim': 3,
        'order': 1,
        'nodes': 4,
        'edges': [[0, 1, 2, 0], [0, 3], [1, 3], [2, 3]],
        'facets': [[0, 2, 1], [0, 1, 3], [1, 2, 3], [2, 0, 3]],
        'tetras': [[0, 1, 2, 3]],
    },
    {
        'name': 'TETRA10S',
        'fc_id': 16,
        'dim': 3,
        'order': 2,
        'nodes': 10,
        'edges': [[0, 4, 1, 5, 2, 6, 0], [0, 7, 3], [1, 8, 3], [2, 9, 3]],
        'facets': [[0, 6, 2, 5, 1, 4], [0, 4, 1, 8, 3, 5], [1, 5, 2, 9, 3, 8], [2, 6, 0, 5, 3, 9]],
        'tetras': [],
    },
    {
        'name': 'HEX8S',
        'fc_id': 17,
        'dim': 3,
        'order': 1,
        'nodes': 8,
        'edges': [[0, 1, 2, 3, 0], [4, 5, 6, 7, 4], [0, 4], [1, 5], [2, 6], [3, 7]],
        'facets': [[3, 2, 1, 0], [4, 5, 6, 7], [1, 2, 6, 5], [0, 1, 5, 4], [0, 4, 7, 3], [2, 3, 7, 6]],
        'tetras': [[1, 3, 4, 6], [3, 1, 4, 0], [1, 3, 6, 2], [4, 1, 6, 5], [3, 4, 6, 7]],
    },
    {
        'name': 'HEX20S',
        'fc_id': 18,
        'dim': 3,
        'order': 2,
        'nodes': 20,
        'edges': [[0, 8, 1, 9, 2, 10, 3, 11, 0], [4, 12, 5, 13, 6, 14, 7, 15, 4],
                  [0, 0, 4], [1, 0, 5], [2, 0, 6], [3, 0, 7]],
        'facets': [[3, 10, 2, 9, 1, 8, 0, 11], [4, 12, 5, 13, 6, 14, 7, 15], [1, 9, 2, 18, 6, 13, 5, 17],
                   [0, 8, 1, 17, 5, 12, 4, 16], [0, 16, 4, 15, 7, 19, 3, 11], [2, 10, 3, 19, 7, 14, 6, 18]],
        'tetras': [],
    },
    {
        'name': 'WEDGE6',
        'fc_id': 6,
        'dim': 3,
        'order': 1,
        'nodes': 6,
        'edges': [[0, 1, 2, 0], [3, 4, 5, 3], [0, 3], [1, 4], [2, 5]],
        'facets': [[0, 1, 2], [5, 4, 3], [0, 2, 5, 3], [0, 3, 4, 1], [1, 4, 5, 2]],
        'tetras': [[0, 5, 4, 3], [0, 4, 2, 1], [0, 2, 4, 5]],
    },
    {
        'name': 'WEDGE15',
        'fc_id': 7,
        'dim': 3,
        'order': 2,
        'nodes': 15,
        'edges': [[0, 5, 1, 6, 2, 7, 3, 8, 0], [0, 9, 4], [1, 10, 4], [2, 11, 4], [3, 12, 4]],
        'facets': [[3, 7, 2, 6, 1, 5, 0, 8],
                   [0, 5, 1, 10, 4, 9], [1, 6, 2, 11, 4, 10], [2, 7, 3, 12, 4, 11], [3, 8, 0, 9, 4, 12]],
        'tetras': [],
    },
    {
        'name': 'WEDGE6S',
        'fc_id': 20,
        'dim': 3,
        'order': 1,
        'nodes': 6,
        'edges': [[0, 1, 2, 0], [3, 4, 5, 3], [0, 3], [1, 4], [2, 5]],
        'facets': [[0, 1, 2], [5, 4, 3], [0, 2, 5, 3], [0, 3, 4, 1], [1, 4, 5, 2]],
        'tetras': [[0, 5, 4, 3], [0, 4, 2, 1], [0, 2, 4, 5]],
    },
    {
        'name': 'WEDGE15S',
        'fc_id': 21,
        'dim': 3,
        'order': 2,
        'nodes': 15,
        'edges': [[0, 5, 1, 6, 2, 7, 3, 8, 0], [0, 9, 4], [1, 10, 4], [2, 11, 4], [3, 12, 4]],
        'facets': [[3, 7, 2, 6, 1, 5, 0, 8],
                   [0, 5, 1, 10, 4, 9], [1, 6, 2, 11, 4, 10], [2, 7, 3, 12, 4, 11], [3, 8, 0, 9, 4, 12]],
        'tetras': [],
    },
    {
        'name': 'PYR5',
        'fc_id': 8,
        'dim': 3,
        'order': 1,
        'nodes': 5,
        'edges': [[0, 1, 2, 3, 0], [0, 4], [1, 4], [2, 4], [3, 4]],
        'facets': [[3, 2, 1, 0], [0, 1, 4], [1, 2, 4], [2, 3, 4], [3, 0, 4]],
        'tetras': [[1, 3, 4, 0], [3, 4, 1, 2]],
    },
    {
        'name': 'PYR13',
        'fc_id': 9,
        'dim': 3,
        'order': 2,
        'nodes': 13,
        'edges': [[0, 5, 1, 6, 2, 7, 3, 8, 0], [0, 9, 4], [1, 10, 4], [2, 11, 4], [3, 12, 4]],
        'facets': [[3, 7, 2, 6, 1, 5, 0, 8],
                   [0, 5, 1, 10, 4, 9], [1, 6, 2, 11, 4, 10], [2, 7, 3, 12, 4, 11], [3, 8, 0, 9, 4, 12]],
        'tetras': [],
    },
    {
        'name': 'PYR5S',
        'fc_id': 22,
        'dim': 3,
        'order': 1,
        'nodes': 5,
        'edges': [[0, 1, 2, 3, 0], [0, 4], [1, 4], [2, 4], [3, 4]],
        'facets': [[3, 2, 1, 0], [0, 1, 4], [1, 2, 4], [2, 3, 4], [3, 0, 4]],
        'tetras': [[1, 3, 4, 0], [3, 4, 1, 2]],
    },
    {
        'name': 'PYR13S',
        'fc_id': 23,
        'dim': 3,
        'order': 2,
        'nodes': 13,
        'edges': [[0, 5, 1, 6, 2, 7, 3, 8, 0], [0, 9, 4], [1, 10, 4], [2, 11, 4], [3, 12, 4]],
        'facets': [[3, 7, 2, 6, 1, 5, 0, 8],
                   [0, 5, 1, 10, 4, 9], [1, 6, 2, 11, 4, 10], [2, 7, 3, 12, 4, 11], [3, 8, 0, 9, 4, 12]],
        'tetras': [],
    },
    {
        'name': 'TRI3S',
        'fc_id': 24,
        'dim': 2,
        'order': 1,
        'nodes': 3,
        'edges': [[0, 1, 2, 0]],
        'facets': [[0, 1, 2]],
        'tetras': [],
    },
    {
        'name': 'TRI6S',
        'fc_id': 25,
        'dim': 2,
        'order': 2,
        'nodes': 6,
        'edges': [[0, 3, 1, 4, 2, 5, 0]],
        'facets': [[0, 3, 1, 4, 2, 5]],
        'tetras': [],
    },
    {
        'name': 'QUAD4S',
        'fc_id': 26,
        'dim': 2,
        'order': 1,
        'nodes': 4,
        'edges': [[0, 1, 2, 3, 0]],
        'facets': [[0, 1, 2, 3]],
        'tetras': [],
    },
    {
        'name': 'QUAD8S',
        'fc_id': 27,
        'dim': 2,
        'order': 2,
        'nodes': 8,
        'edges': [[0, 4, 1, 5, 2, 6, 3, 7, 0]],
        'facets': [[0, 4, 1, 5, 2, 6, 3, 7]],
        'tetras': [],
    },
    {
        'name': 'SPRING2D',
        'fc_id': 83,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'SHELL3S',
        'fc_id': 84,
        'dim': 2,
        'order': 1,
        'nodes': 3,
        'edges': [[0, 1, 2, 0]],
        'facets': [[0, 1, 2]],
        'tetras': [],
    },
    {
        'name': 'SHELL4S',
        'fc_id': 85,
        'dim': 2,
        'order': 1,
        'nodes': 4,
        'edges': [[0, 1, 2, 3, 0]],
        'facets': [[0, 1, 2, 3]],
        'tetras': [],
    },
    {
        'name': 'SHELL6S',
        'fc_id': 86,
        'dim': 2,
        'order': 2,
        'nodes': 6,
        'edges': [[0, 3, 1, 4, 2, 5, 0]],
        'facets': [[0, 3, 1, 4, 2, 5]],
        'tetras': [],
    },
    {
        'name': 'SHELL8S',
        'fc_id': 87,
        'dim': 2,
        'order': 2,
        'nodes': 8,
        'edges': [[0, 4, 1, 5, 2, 6, 3, 7, 0]],
        'facets': [[0, 4, 1, 5, 2, 6, 3, 7]],
        'tetras': [],
    },
    {
        'name': 'BEAM26S',
        'fc_id': 95,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BEAM36S',
        'fc_id': 96,
        'dim': 1,
        'order': 2,
        'nodes': 3,
        'edges': [[0, 2, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BEAM27S',
        'fc_id': 97,
        'dim': 1,
        'order': 1,
        'nodes': 2,
        'edges': [[0, 1]],
        'facets': [],
        'tetras': [],
    },
    {
        'name': 'BEAM37S',
        'fc_id': 98,
        'dim': 1,
        'order': 2,
        'nodes': 3,
        'edges': [[0, 2, 1]],
        'facets': [],
        'tetras': [],
    }
]

FC_ELEMENT_TYPES_KEYID: Dict[int, FCElementType] = {
    element_type['fc_id']:element_type for element_type in FC_ELEMENT_TYPES
}

FC_ELEMENT_TYPES_KEYNAME: Dict[FCElementTypeLiteral, FCElementType] = {
    element_type['name']:element_type for element_type in FC_ELEMENT_TYPES
}


class FCSrcElement(TypedDict):
    id: int
    block: int
    parent_id: int
    type: FCElementTypeLiteral
    nodes: List[int]
    order: int


class FCElement:
    """
    Определяет один конечный элемент в сетке.
    """
    id: int
    block: int
    parent_id: int
    type: FCElementTypeLiteral
    nodes: List[int]
    order: int

    def __init__(self, src_element: FCSrcElement):
        """
        Инициализатор для FCElem.
        """

        self.id = src_element['id']
        self.block = src_element['block']
        self.parent_id = src_element['parent_id']
        self.type = src_element['type']
        self.nodes = src_element['nodes']
        self.order = src_element['order']
       

    def dump(self):
        return {
            'id': self.id,
            'block': self.block,
            'parent_id': self.parent_id,
            'type': self.type,
            'nodes': self.nodes,
            'order': self.order,
        }


class FCSrcMesh(TypedDict):
    elem_blocks: str
    elem_orders: str
    elem_parent_ids: str
    elem_types: str
    elemids: str
    elems: str
    elems_count: int
    nids: str
    nodes: str
    nodes_count: int


class FCMesh:
    """
    Контейнер для хранения всех элементов модели, сгруппированных по типам.

    Внутри коллекции элементы хранятся в словаре `elements`,
    где ключами являются строковые имена типов элементов (e.g., 'HEX8', 'TETRA4'),
    а значениями — словари `Dict[int, FCElement]` соответствующего типа.

    Этот класс также управляет общей кодировкой и декодировкой всего набора
    элементов в/из формата .fc.
    """

    nodes_ids: NDArray[np.int32]
    nodes_xyz: NDArray[np.float64]

    elements: Dict[FCElementTypeLiteral, Dict[int, FCElement]]


    def __init__(self):

        # INSERT_YOUR_CODE
        self.nodes_ids = np.array([], dtype=np.int32)
        self.nodes_xyz = np.array([], dtype=np.float64)

        self.elements = {}


    def decode(self, src_mesh: FCSrcMesh):

        self.nodes_ids = decode(src_mesh['nids'], np.dtype('int32'))
        nodes_raw = decode(src_mesh['nodes'], np.dtype('float64'))
        if nodes_raw.size % 3 != 0:
            raise ValueError(f"mesh.nodes length must be divisible by 3, got {nodes_raw.size}")
        self.nodes_xyz = nodes_raw.reshape(-1, 3)

        # basic consistency: nodes_count must match ids/xyz lengths
        if src_mesh['nodes_count'] != len(self.nodes_ids):
            raise ValueError(
                f"nodes_count mismatch: header {src_mesh['nodes_count']} vs decoded {len(self.nodes_ids)}"
            )
        if self.nodes_xyz.shape[0] != len(self.nodes_ids):
            raise ValueError(
                f"nodes xyz count mismatch: {self.nodes_xyz.shape[0]} rows vs {len(self.nodes_ids)} ids"
            )

        elem_blocks = decode(src_mesh['elem_blocks'])
        elem_orders = decode(src_mesh['elem_orders'])
        elem_parent_ids = decode(src_mesh['elem_parent_ids'])
        elem_types = decode(src_mesh['elem_types'], np.dtype('uint8'))
        elem_ids = decode(src_mesh['elemids'])
        elem_nodes = decode(src_mesh['elems'])

        # basic consistency: elems_count must match arrays' lengths
        elems_count = src_mesh['elems_count']
        for name, arr in (
            ("elemids", elem_ids),
            ("elem_blocks", elem_blocks),
            ("elem_orders", elem_orders),
            ("elem_parent_ids", elem_parent_ids),
            ("elem_types", elem_types),
        ):
            if len(arr) != elems_count:
                raise ValueError(f"{name} length {len(arr)} != elems_count {elems_count}")

        elem_sizes = np.vectorize(lambda t: FC_ELEMENT_TYPES_KEYID[t]['nodes'])(elem_types)
        total_nodes = int(np.sum(elem_sizes))
        if len(elem_nodes) != total_nodes:
            raise ValueError(
                f"elems (flattened nodes) length {len(elem_nodes)} != expected {total_nodes} from elem_types"
            )
        elem_offsets = [0, *np.cumsum(elem_sizes)]

        for i, eid in enumerate(elem_ids):
            fc_type_name = FC_ELEMENT_TYPES_KEYID[elem_types[i]]['name']

            if fc_type_name not in self.elements:
                self.elements[fc_type_name] = {}

            element = FCElement({
                'id': eid,
                'type': fc_type_name,
                'nodes': elem_nodes[elem_offsets[i]:elem_offsets[i+1]].tolist(),
                'parent_id': elem_parent_ids[i],
                'block': elem_blocks[i],
                'order': elem_orders[i],
            })

            self.elements[fc_type_name][eid] = element


    def encode(self) -> FCSrcMesh:

        elems_count = len(self)

        elem_ids: NDArray = np.zeros(elems_count, np.int32)
        elem_blocks: NDArray = np.zeros(elems_count, np.int32)
        elem_orders: NDArray = np.zeros(elems_count, np.int32)
        elem_parent_ids: NDArray = np.zeros(elems_count, np.int32)
        elem_types: NDArray = np.zeros(elems_count, np.int8)

        # basic consistency: nodes arrays
        if self.nodes_xyz.ndim != 2 or self.nodes_xyz.shape[1] != 3:
            raise ValueError("nodes must be a 2D array of shape (N,3)")
        if len(self.nodes_ids) != self.nodes_xyz.shape[0]:
            raise ValueError(
                f"nodes ids length {len(self.nodes_ids)} != nodes xyz rows {self.nodes_xyz.shape[0]}"
            )

        for i, elem in enumerate(self):
            elem_ids[i] = elem.id
            elem_blocks[i] = elem.block
            elem_parent_ids[i] = elem.parent_id
            elem_orders[i] = elem.order
            elem_types[i] = FC_ELEMENT_TYPES_KEYNAME[elem.type]['fc_id']

        # basic consistency: each element nodes count must match its type definition
        expected_nodes_total = 0
        for elem in self:
            expected_nodes_total += FC_ELEMENT_TYPES_KEYNAME[elem.type]['nodes']
        elem_nodes: NDArray = np.array(self.nodes_list, np.int32)
        if len(elem_nodes) != expected_nodes_total:
            raise ValueError(
                f"flattened nodes length {len(elem_nodes)} != expected {expected_nodes_total} from element types"
            )

        src_mesh: FCSrcMesh = {
            "elem_blocks": encode(elem_blocks),
            "elem_orders": encode(elem_orders),
            "elem_parent_ids": encode(elem_parent_ids),
            "elem_types": encode(elem_types),
            "elemids": encode(elem_ids),
            "elems": encode(elem_nodes),
            "elems_count": elems_count,
            "nids": encode(self.nodes_ids), 
            "nodes": encode(self.nodes_xyz),
            "nodes_count": len(self.nodes_ids)           
        }

        return src_mesh


    def __len__(self):
        return sum([len(self.elements[typename]) for typename in self.elements])


    def __bool__(self):
        return len(self) > 0


    def __iter__(self):
        for typename in self.elements:
            for elem in self.elements[typename].values():
                yield elem


    def __contains__(self, key):
        for tp in self.elements:
            if key in self.elements[tp]:
                return True
        return False


    def __getitem__(self, key:Union[int, FCElementTypeLiteral]):
        if isinstance(key, str):
            return self.elements[key]
        elif isinstance(key, int):
            for typename in self.elements:
                if key in self.elements[typename]:
                    return self.elements[typename][key]
        raise KeyError(f'{key}')


    def __setitem__(self, key:int, item: FCElement):

        if item.type not in self.elements:
            self.elements[item.type] = {}
        self.elements[item.type][item.id] = item


    @property
    def nodes_list(self):
        return [node for elem in self for node in elem.nodes]


    def compress(self):
        index_map = {elem.id: i + 1 for i, elem in enumerate(self)}
        self.reindex(index_map)
        return index_map


    def reindex(self, index_map):
        for typename in list(self.elements.keys()):
            new_bucket: Dict[int, FCElement] = {}
            for elem in self.elements[typename].values():
                if elem.id in index_map:
                    elem.id = index_map[elem.id]
                new_bucket[elem.id] = elem
            self.elements[typename] = new_bucket


    @property
    def max_id(self):
        max_id = 0
        for tp in self.elements:
            if self.elements[tp]:
                local_max = max(self.elements[tp].keys())
                if max_id < local_max:
                    max_id = local_max
        return max_id


    def add(self, item: FCElement):
        if item.type not in self.elements:
            self.elements[item.type] = {}
        if item.id in self or item.id < 1:
            item.id = self.max_id + 1
        self.elements[item.type][item.id] = item
        return item.id

