"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExistingZoneDomain = exports.DelegationDomain = exports.CrossAccountDelegationDomain = exports.BaseDomain = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_certificatemanager_1 = require("aws-cdk-lib/aws-certificatemanager");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const constructs_1 = require("constructs");
/**
 * A DNS domain and its wildcard X.509 certificate.
 */
class BaseDomain extends constructs_1.Construct {
}
exports.BaseDomain = BaseDomain;
_a = JSII_RTTI_SYMBOL_1;
BaseDomain[_a] = { fqn: "shady-island.networking.BaseDomain", version: "0.1.62" };
/**
 * Provides a domain using delegation from a parent zone in another account.
 *
 * This construct creates a new Route 53 hosted zone for the subdomain, a zone
 * delegation record, and a new wildcard ACM certificate for the subdomain.
 */
class CrossAccountDelegationDomain extends BaseDomain {
    constructor(scope, id, props) {
        super(scope, id);
        const { delegationRole, removalPolicy, parentHostedZoneName } = props;
        const zoneName = `${props.subdomain}.${parentHostedZoneName}`;
        this.name = zoneName;
        this.hostedZone = new aws_route53_1.PublicHostedZone(this, "HostedZone", { zoneName });
        this.delegationRecord = new aws_route53_1.CrossAccountZoneDelegationRecord(this, "Delegation", {
            delegatedZone: this.hostedZone,
            delegationRole,
            parentHostedZoneId: props.parentHostedZoneId,
            parentHostedZoneName,
            removalPolicy,
            ttl: props.ttl,
        });
        this.certificate = new aws_certificatemanager_1.Certificate(this, "Certificate", {
            domainName: `*.${this.hostedZone.zoneName}`,
            validation: aws_certificatemanager_1.CertificateValidation.fromDns(this.hostedZone),
        });
        // Make sure we don't try to create the certificate before the delegation.
        this.certificate.node.addDependency(this.delegationRecord);
        if (removalPolicy) {
            this.hostedZone.applyRemovalPolicy(removalPolicy);
            this.certificate.applyRemovalPolicy(removalPolicy);
        }
    }
}
exports.CrossAccountDelegationDomain = CrossAccountDelegationDomain;
_b = JSII_RTTI_SYMBOL_1;
CrossAccountDelegationDomain[_b] = { fqn: "shady-island.networking.CrossAccountDelegationDomain", version: "0.1.62" };
/**
 * Provides a domain using delegation from a parent zone in the same account.
 *
 * This construct creates a new Route 53 hosted zone for the subdomain, a zone
 * delegation record, and a new wildcard ACM certificate for the subdomain.
 */
class DelegationDomain extends BaseDomain {
    constructor(scope, id, props) {
        super(scope, id);
        const { parentHostedZone, subdomain, removalPolicy } = props;
        const zoneName = `${subdomain}.${parentHostedZone.zoneName}`;
        this.name = zoneName;
        this.hostedZone = new aws_route53_1.PublicHostedZone(this, "HostedZone", { zoneName });
        this.delegationRecord = new aws_route53_1.ZoneDelegationRecord(this, "Record", {
            recordName: subdomain,
            nameServers: this.hostedZone.hostedZoneNameServers,
            zone: parentHostedZone,
        });
        this.certificate = new aws_certificatemanager_1.Certificate(this, "Certificate", {
            domainName: `*.${this.hostedZone.zoneName}`,
            validation: aws_certificatemanager_1.CertificateValidation.fromDns(this.hostedZone),
        });
        // Make sure we don't try to create the certificate before the delegation.
        this.certificate.node.addDependency(this.delegationRecord);
        if (removalPolicy) {
            this.applyRemovalPolicy(removalPolicy);
        }
    }
    applyRemovalPolicy(policy) {
        this.hostedZone.applyRemovalPolicy(policy);
        this.certificate.applyRemovalPolicy(policy);
        this.delegationRecord.applyRemovalPolicy(policy);
    }
}
exports.DelegationDomain = DelegationDomain;
_c = JSII_RTTI_SYMBOL_1;
DelegationDomain[_c] = { fqn: "shady-island.networking.DelegationDomain", version: "0.1.62" };
/**
 * Provides a domain using an existing hosted zone.
 *
 * This construct will create a new wildcard ACM certificate using the existing
 * hosted zone name.
 */
class ExistingZoneDomain extends BaseDomain {
    /**
     * Returns an ExistingZoneDomain using the provided attributes.
     *
     * @param scope - The scope in which to define this construct.
     * @param id - The scoped construct ID.
     * @param attrs - The provided attributes
     */
    static fromDomainAttributes(scope, id, attrs) {
        class Import extends constructs_1.Construct {
            constructor() {
                super(...arguments);
                this.hostedZone = attrs.hostedZone;
                this.certificate = attrs.certificate;
            }
            get name() {
                return this.hostedZone.zoneName;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        const { hostedZone } = props;
        this.hostedZone = hostedZone;
        this.certificate = new aws_certificatemanager_1.Certificate(this, "Certificate", {
            domainName: `*.${this.hostedZone.zoneName}`,
            validation: aws_certificatemanager_1.CertificateValidation.fromDns(this.hostedZone),
        });
    }
    get name() {
        return this.hostedZone.zoneName;
    }
}
exports.ExistingZoneDomain = ExistingZoneDomain;
_d = JSII_RTTI_SYMBOL_1;
ExistingZoneDomain[_d] = { fqn: "shady-island.networking.ExistingZoneDomain", version: "0.1.62" };
//# sourceMappingURL=data:application/json;base64,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