"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3 = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * (experimental) Saves the received message to an Amazon S3 bucket and, optionally, publishes a notification to Amazon SNS.
 *
 * @experimental
 */
class S3 {
    /**
     * @experimental
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * (experimental) (experimental) Returns the receipt rule action specification.
     *
     * @experimental
     */
    bind(rule) {
        // Allow SES to write to S3 bucket
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-s3
        const keyPattern = this.props.objectKeyPrefix || '';
        const s3Statement = new iam.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
            resources: [this.props.bucket.arnForObjects(`${keyPattern}*`)],
            conditions: {
                StringEquals: {
                    'aws:Referer': cdk.Aws.ACCOUNT_ID,
                },
            },
        });
        this.props.bucket.addToResourcePolicy(s3Statement);
        const policy = this.props.bucket.node.tryFindChild('Policy');
        if (policy) { // The bucket could be imported
            rule.node.addDependency(policy);
        }
        else {
            cdk.Annotations.of(rule).addWarning('This rule is using a S3 action with an imported bucket. Ensure permission is given to SES to write to that bucket.');
        }
        // Allow SES to use KMS master key
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-kms
        if (this.props.kmsKey && !/alias\/aws\/ses$/.test(this.props.kmsKey.keyArn)) {
            const kmsStatement = new iam.PolicyStatement({
                actions: ['kms:Encrypt', 'kms:GenerateDataKey'],
                principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
                resources: ['*'],
                conditions: {
                    Null: {
                        'kms:EncryptionContext:aws:ses:rule-name': 'false',
                        'kms:EncryptionContext:aws:ses:message-id': 'false',
                    },
                    StringEquals: {
                        'kms:EncryptionContext:aws:ses:source-account': cdk.Aws.ACCOUNT_ID,
                    },
                },
            });
            this.props.kmsKey.addToResourcePolicy(kmsStatement);
        }
        return {
            s3Action: {
                bucketName: this.props.bucket.bucketName,
                kmsKeyArn: this.props.kmsKey ? this.props.kmsKey.keyArn : undefined,
                objectKeyPrefix: this.props.objectKeyPrefix,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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