"""Implementation of a locking ServerProxy for XML-RPC communication."""

from __future__ import annotations

from collections.abc import Mapping
from concurrent.futures import ThreadPoolExecutor
from enum import Enum, IntEnum, StrEnum
import errno
import logging
from ssl import SSLError
from typing import Any, Final
import xmlrpc.client

from hahomematic import central as hmcu
from hahomematic.async_support import Looper
from hahomematic.const import ISO_8859_1
from hahomematic.exceptions import (
    AuthFailure,
    BaseHomematicException,
    ClientException,
    NoConnectionException,
    UnsupportedException,
)
from hahomematic.support import extract_exc_args, get_tls_context

_LOGGER: Final = logging.getLogger(__name__)

_CONTEXT: Final = "context"
_TLS: Final = "tls"
_VERIFY_TLS: Final = "verify_tls"


class _XmlRpcMethod(StrEnum):
    """Enum for homematic json rpc methods types."""

    GET_VERSION = "getVersion"
    HOMEGEAR_INIT = "clientServerInitialized"
    INIT = "init"
    PING = "ping"
    SYSTEM_LIST_METHODS = "system.listMethods"


_VALID_XMLRPC_COMMANDS_ON_NO_CONNECTION: Final[tuple[str, ...]] = (
    _XmlRpcMethod.GET_VERSION,
    _XmlRpcMethod.HOMEGEAR_INIT,
    _XmlRpcMethod.INIT,
    _XmlRpcMethod.PING,
    _XmlRpcMethod.SYSTEM_LIST_METHODS,
)

_SSL_ERROR_CODES: Final[dict[int, str]] = {
    errno.ENOEXEC: "EOF occurred in violation of protocol",
}

_OS_ERROR_CODES: Final[dict[int, str]] = {
    errno.ECONNREFUSED: "Connection refused",
    errno.EHOSTUNREACH: "No route to host",
    errno.ENETUNREACH: "Network is unreachable",
    errno.ENOEXEC: "Exec",
    errno.ETIMEDOUT: "Operation timed out",
}


# noinspection PyProtectedMember,PyUnresolvedReferences
class XmlRpcProxy(xmlrpc.client.ServerProxy):
    """ServerProxy implementation with ThreadPoolExecutor when request is executing."""

    def __init__(
        self,
        max_workers: int,
        interface_id: str,
        connection_state: hmcu.CentralConnectionState,
        *args: Any,
        **kwargs: Any,
    ) -> None:
        """Initialize new proxy for server and get local ip."""
        self.interface_id: Final = interface_id
        self._connection_state: Final = connection_state
        self._looper: Final = Looper()
        self._proxy_executor: Final = (
            ThreadPoolExecutor(max_workers=max_workers, thread_name_prefix=interface_id) if max_workers > 0 else None
        )
        self._tls: Final[bool] = kwargs.pop(_TLS, False)
        self._verify_tls: Final[bool] = kwargs.pop(_VERIFY_TLS, True)
        self._supported_methods: tuple[str, ...] = ()
        if self._tls:
            kwargs[_CONTEXT] = get_tls_context(self._verify_tls)
        xmlrpc.client.ServerProxy.__init__(  # type: ignore[misc]
            self,
            encoding=ISO_8859_1,
            *args,  # noqa: B026
            **kwargs,
        )

    async def do_init(self) -> None:
        """Init the xml rpc proxy."""
        if supported_methods := await self.system.listMethods():
            # ping is missing in VirtualDevices interface but can be used.
            supported_methods.append(_XmlRpcMethod.PING)
            self._supported_methods = tuple(supported_methods)

    @property
    def supported_methods(self) -> tuple[str, ...]:
        """Return the supported methods."""
        return self._supported_methods

    async def __async_request(self, *args, **kwargs):  # type: ignore[no-untyped-def]
        """Call method on server side."""
        parent = xmlrpc.client.ServerProxy
        try:
            method = args[0]
            if self._supported_methods and method not in self._supported_methods:
                raise UnsupportedException(f"__ASYNC_REQUEST: method '{method} not supported by backend.")

            if method in _VALID_XMLRPC_COMMANDS_ON_NO_CONNECTION or not self._connection_state.has_issue(
                issuer=self, iid=self.interface_id
            ):
                args = _cleanup_args(*args)
                _LOGGER.debug("__ASYNC_REQUEST: %s", args)
                result = await self._looper.async_add_executor_job(
                    # pylint: disable=protected-access
                    parent._ServerProxy__request,  # type: ignore[attr-defined]
                    self,
                    *args,
                    name="xmp_rpc_proxy",
                    executor=self._proxy_executor,
                )
                self._connection_state.remove_issue(issuer=self, iid=self.interface_id)
                return result
            raise NoConnectionException(f"No connection to {self.interface_id}")
        except BaseHomematicException:
            raise
        except SSLError as sslerr:
            message = f"SSLError on {self.interface_id}: {extract_exc_args(exc=sslerr)}"
            if sslerr.args[0] in _SSL_ERROR_CODES:
                _LOGGER.debug(message)
            else:
                _LOGGER.error(message)
            raise NoConnectionException(message) from sslerr
        except OSError as oserr:
            message = f"OSError on {self.interface_id}: {extract_exc_args(exc=oserr)}"
            if oserr.args[0] in _OS_ERROR_CODES:
                if self._connection_state.add_issue(issuer=self, iid=self.interface_id):
                    _LOGGER.error(message)
                else:
                    _LOGGER.debug(message)
            else:
                _LOGGER.error(message)
            raise NoConnectionException(message) from oserr
        except xmlrpc.client.Fault as flt:
            raise ClientException(f"XMLRPC Fault from backend: {flt.faultCode} {flt.faultString}") from flt
        except TypeError as terr:
            raise ClientException(terr) from terr
        except xmlrpc.client.ProtocolError as perr:
            if not self._connection_state.has_issue(issuer=self, iid=self.interface_id):
                if perr.errmsg == "Unauthorized":
                    raise AuthFailure(perr) from perr
                raise NoConnectionException(perr.errmsg) from perr
        except Exception as exc:
            raise ClientException(exc) from exc

    def __getattr__(self, *args, **kwargs):  # type: ignore[no-untyped-def]
        """Magic method dispatcher."""
        return xmlrpc.client._Method(self.__async_request, *args, **kwargs)

    async def stop(self) -> None:
        """Stop depending services."""
        await self._looper.block_till_done()
        if self._proxy_executor:
            self._proxy_executor.shutdown()


def _cleanup_args(*args: Any) -> Any:
    """Cleanup the type of args."""
    if len(args[1]) == 0:
        return args
    if len(args) == 2:
        new_args: list[Any] = []
        for data in args[1]:
            if isinstance(data, dict):
                new_args.append(_cleanup_paramset(paramset=data))
            else:
                new_args.append(_cleanup_item(item=data))
        return (args[0], tuple(new_args))
    _LOGGER.error("XmlRpcProxy command: Too many arguments")
    return args


def _cleanup_item(item: Any) -> Any:
    """Cleanup a single item."""
    if isinstance(item, StrEnum):
        return str(item)
    if isinstance(item, IntEnum):
        return int(item)
    if isinstance(item, Enum):
        _LOGGER.error("XmlRpcProxy command: Enum is not supported as parameter value")
    return item


def _cleanup_paramset(paramset: Mapping[str, Any]) -> Mapping[str, Any]:
    """Cleanup a paramset."""
    new_paramset: dict[str, Any] = {}
    for name, value in paramset.items():
        new_paramset[_cleanup_item(item=name)] = _cleanup_item(item=value)
    return new_paramset
