"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cw_actions = require("@aws-cdk/aws-cloudwatch-actions");
const ec2 = require("@aws-cdk/aws-ec2");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
class IntegTesting {
    constructor(props = {}) {
        const app = new cdk.App();
        const env = {
            account: process.env.CDK_DEFAULT_ACCOUNT,
            region: process.env.CDK_DEFAULT_REGION,
        };
        const stack = new cdk.Stack(app, 'testing-stack', { env });
        const vpc = props.vpc ?? new ec2.Vpc(stack, 'Vpc', { natGateways: 1 });
        const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        securityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22));
        const instance = new ec2.Instance(stack, 'Instance', {
            vpc,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.T3A, ec2.InstanceSize.NANO),
            machineImage: ec2.MachineImage.latestAmazonLinux({
                generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            securityGroup,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
        });
        instance.userData.addCommands('amazon-linux-extras install epel -y', 'yum install stress-ng tmux htop -y');
        new cdk.CfnOutput(stack, 'instance', {
            value: instance.instancePublicDnsName,
        });
        const topic = new sns.Topic(stack, 'Topic');
        const metric = new cloudwatch.Metric({
            namespace: 'AWS/EC2',
            metricName: 'CPUUtilization',
            dimensions: {
                InstanceId: instance.instanceId,
            },
            period: cdk.Duration.minutes(1),
        });
        const alarm = new cloudwatch.Alarm(stack, 'Alarm', {
            metric,
            threshold: 5,
            evaluationPeriods: 1,
        });
        alarm.addAlarmAction(new cw_actions.SnsAction(topic));
        const snsLineNotify = new index_1.SnsNotify(stack, 'sns-line-notify', {
            lineNotifyToken: 'lineNotifyToken',
        });
        topic.addSubscription(snsLineNotify.lambdaSubscription);
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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