"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Validators = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
class Validators {
    /**
     * Validates whether a supplied value conforms to the allowedPattern, granted neither is an unresolved token.
     *
     * @param value          the value to be validated.
     * @param allowedPattern the regular expression to use for validation.
     *
     * @throws if the ``value`` does not conform to the ``allowedPattern`` and neither is an unresolved token (per
     *         ``cdk.unresolved``).
     */
    static parameterValue(value, allowedPattern) {
        if (aws_cdk_lib_1.Token.isUnresolved(value) || aws_cdk_lib_1.Token.isUnresolved(allowedPattern)) {
            // Unable to perform validations against unresolved tokens
            return;
        }
        if (!new RegExp(allowedPattern).test(value)) {
            throw new Error(`The supplied value (${value}) does not match the specified allowedPattern (${allowedPattern})`);
        }
    }
    static parameterName(parameterName) {
        if (aws_cdk_lib_1.Token.isUnresolved(parameterName)) {
            throw new Error("Unable to determine parameter name for SSM parameter since the parameter name is an unresolved token.");
        }
        if (parameterName.length > 2048) {
            throw new Error("Name cannot be longer than 2048 characters.");
        }
        if (!parameterName.match(/^\/[\/\w.-]+$/)) {
            throw new Error(`Name must begin with / and must only contain letters, numbers, and the following 4 symbols .-_/; got ${parameterName}`);
        }
    }
    static description(description) {
        if (description && description?.length > 1024) {
            throw new Error("Description cannot be longer than 1024 characters.");
        }
    }
}
exports.Validators = Validators;
//# sourceMappingURL=data:application/json;base64,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