#!/usr/bin/python
# -*- coding: utf-8 -*-
"""ORM for index server."""
import logging
from time import sleep
import peewee
from playhouse.db_url import connect
from pacifica.uniqueid.config import get_config

DATABASE_CONNECT_ATTEMPTS = 20
DATABASE_WAIT = 5
DB = connect(get_config().get('database', 'peewee_url'))


class UniqueIndex(peewee.Model):
    """Auto-generated by pwiz maps a python record to a mysql table."""

    idid = peewee.CharField(primary_key=True, db_column='id')
    index = peewee.BigIntegerField(db_column='index')

    # pylint: disable=too-few-public-methods
    class Meta(object):
        """Map to the database connected above."""

        database = DB
        only_save_dirty = True
    # pylint: enable=too-few-public-methods

    @classmethod
    def atomic(cls):
        """Get the atomic context or decorator."""
        # pylint: disable=no-member
        return cls._meta.database.atomic()
        # pylint: enable=no-member

    @classmethod
    def database_connect(cls):
        """
        Make sure database is connected.

        Trying to connect a second time doesnt cause any problems.
        """
        peewee_logger = logging.getLogger('peewee')
        peewee_logger.debug('Connecting to database.')
        # pylint: disable=no-member
        if not cls._meta.database.is_closed():  # pragma no cover
            cls._meta.database.close()
        cls._meta.database.connect()
        # pylint: enable=no-member

    @classmethod
    def database_close(cls):
        """
        Close the database connection.

        Closing already closed database throws an error so catch it and continue on.
        """
        peewee_logger = logging.getLogger('peewee')
        peewee_logger.debug('Closing database connection.')
        # pylint: disable=no-member
        if not cls._meta.database.is_closed():  # pragma no cover
            cls._meta.database.close()
        # pylint: enable=no-member


def update_index(id_range, id_mode):
    """Update the index for a mode and returns a unique start and stop index."""
    index = -1
    id_range = id_range
    with UniqueIndex.atomic():
        if id_range and id_mode and id_range > 0:
            record = UniqueIndex.get_or_create(
                idid=id_mode, defaults={'index': 1})[0]
            index = int(record.index)
            record.index = index + id_range
            record.save()
        else:
            index = -1
            id_range = int(-1)
    return (index, id_range)


def try_db_connect(attempts=0):
    """Try connecting to the db."""
    try:
        UniqueIndex.database_connect()
    except peewee.OperationalError as ex:
        if attempts < DATABASE_CONNECT_ATTEMPTS:
            sleep(DATABASE_WAIT)
            attempts += 1
            try_db_connect(attempts)
        else:
            raise ex


def database_setup():
    """Create the tables nessisary."""
    try_db_connect()
    if not UniqueIndex.table_exists():
        UniqueIndex.create_table()
