"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const kinesis = require("@aws-cdk/aws-kinesis");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const dests = require("../lib");
test('stream can be subscription destination', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const stream = new kinesis.Stream(stack, 'MyStream');
    const logGroup = new logs.LogGroup(stack, 'LogGroup');
    // WHEN
    new logs.SubscriptionFilter(stack, 'Subscription', {
        logGroup,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents()
    });
    // THEN: subscription target is Stream
    expect(stack).toHaveResource('AWS::Logs::SubscriptionFilter', {
        DestinationArn: { "Fn::GetAtt": ["MyStream5C050E93", "Arn"] },
        RoleArn: { "Fn::GetAtt": ["SubscriptionCloudWatchLogsCanPutRecords9C1223EC", "Arn"] },
    });
    // THEN: we have a role to write to the Stream
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [{
                    Action: "sts:AssumeRole",
                    Effect: 'Allow',
                    Principal: {
                        Service: {
                            "Fn::Join": ["", [
                                    "logs.",
                                    { Ref: "AWS::Region" },
                                    ".",
                                    { Ref: "AWS::URLSuffix" }
                                ]
                            ]
                        }
                    }
                }],
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Action: [
                        "kinesis:DescribeStream",
                        "kinesis:PutRecord",
                        "kinesis:PutRecords",
                    ],
                    Effect: "Allow",
                    Resource: { "Fn::GetAtt": ["MyStream5C050E93", "Arn"] }
                },
                {
                    Action: "iam:PassRole",
                    Effect: "Allow",
                    Resource: { "Fn::GetAtt": ["SubscriptionCloudWatchLogsCanPutRecords9C1223EC", "Arn"] }
                }
            ],
        }
    });
});
test('stream can be subscription destination twice, without duplicating permissions', () => {
    // GIVEN
    const stack = new cdk.Stack();
    const stream = new kinesis.Stream(stack, 'MyStream');
    const logGroup1 = new logs.LogGroup(stack, 'LogGroup');
    const logGroup2 = new logs.LogGroup(stack, 'LogGroup2');
    // WHEN
    new logs.SubscriptionFilter(stack, 'Subscription', {
        logGroup: logGroup1,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents()
    });
    new logs.SubscriptionFilter(stack, 'Subscription2', {
        logGroup: logGroup2,
        destination: new dests.KinesisDestination(stream),
        filterPattern: logs.FilterPattern.allEvents()
    });
    // THEN: subscription target is Stream
    expect(stack).toHaveResource('AWS::Logs::SubscriptionFilter', {
        DestinationArn: { "Fn::GetAtt": ["MyStream5C050E93", "Arn"] },
        RoleArn: { "Fn::GetAtt": ["SubscriptionCloudWatchLogsCanPutRecords9C1223EC", "Arn"] },
    });
    // THEN: we have a role to write to the Stream
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Version: '2012-10-17',
            Statement: [{
                    Action: "sts:AssumeRole",
                    Effect: 'Allow',
                    Principal: {
                        Service: {
                            "Fn::Join": ["", [
                                    "logs.",
                                    { Ref: "AWS::Region" },
                                    ".",
                                    { Ref: "AWS::URLSuffix" }
                                ]
                            ]
                        }
                    }
                }],
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Version: '2012-10-17',
            Statement: [
                {
                    Action: [
                        "kinesis:DescribeStream",
                        "kinesis:PutRecord",
                        "kinesis:PutRecords",
                    ],
                    Effect: "Allow",
                    Resource: { "Fn::GetAtt": ["MyStream5C050E93", "Arn"] }
                },
                {
                    Action: "iam:PassRole",
                    Effect: "Allow",
                    Resource: { "Fn::GetAtt": ["SubscriptionCloudWatchLogsCanPutRecords9C1223EC", "Arn"] }
                }
            ],
        }
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2luZXNpcy50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsia2luZXNpcy50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsZ0NBQThCO0FBQzlCLGdEQUFnRDtBQUNoRCwwQ0FBMEM7QUFDMUMscUNBQXFDO0FBQ3JDLGdDQUFnQztBQUVoQyxJQUFJLENBQUMsd0NBQXdDLEVBQUUsR0FBRyxFQUFFO0lBQ2xELFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sUUFBUSxHQUFHLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFFdEQsT0FBTztJQUNQLElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7UUFDakQsUUFBUTtRQUNSLFdBQVcsRUFBRSxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLENBQUM7UUFDakQsYUFBYSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxFQUFFO0tBQzlDLENBQUMsQ0FBQztJQUVILHNDQUFzQztJQUN0QyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLCtCQUErQixFQUFFO1FBQzVELGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGtCQUFrQixFQUFFLEtBQUssQ0FBRSxFQUFFO1FBQy9ELE9BQU8sRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGlEQUFpRCxFQUFFLEtBQUssQ0FBRSxFQUFFO0tBQ3hGLENBQUMsQ0FBQztJQUVILDhDQUE4QztJQUM5QyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1FBQzdDLHdCQUF3QixFQUFFO1lBQ3hCLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFNBQVMsRUFBRSxDQUFDO29CQUNWLE1BQU0sRUFBRSxnQkFBZ0I7b0JBQ3hCLE1BQU0sRUFBRSxPQUFPO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxPQUFPLEVBQUU7NEJBQ1AsVUFBVSxFQUFFLENBQUUsRUFBRSxFQUFFO29DQUNkLE9BQU87b0NBQ1AsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29DQUN0QixHQUFHO29DQUNILEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO2lDQUMxQjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRixDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtCQUFrQixFQUFFO1FBQy9DLGNBQWMsRUFBRTtZQUNkLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04sd0JBQXdCO3dCQUN4QixtQkFBbUI7d0JBQ25CLG9CQUFvQjtxQkFDckI7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsa0JBQWtCLEVBQUUsS0FBSyxDQUFFLEVBQUU7aUJBQzFEO2dCQUNEO29CQUNFLE1BQU0sRUFBRSxjQUFjO29CQUN0QixNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxpREFBaUQsRUFBRSxLQUFLLENBQUUsRUFBRTtpQkFDekY7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsK0VBQStFLEVBQUUsR0FBRyxFQUFFO0lBQ3pGLFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sU0FBUyxHQUFHLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFDdkQsTUFBTSxTQUFTLEdBQUcsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztJQUV4RCxPQUFPO0lBQ1AsSUFBSSxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtRQUNqRCxRQUFRLEVBQUUsU0FBUztRQUNuQixXQUFXLEVBQUUsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsTUFBTSxDQUFDO1FBQ2pELGFBQWEsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsRUFBRTtLQUM5QyxDQUFDLENBQUM7SUFFSCxJQUFJLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1FBQ2xELFFBQVEsRUFBRSxTQUFTO1FBQ25CLFdBQVcsRUFBRSxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLENBQUM7UUFDakQsYUFBYSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxFQUFFO0tBQzlDLENBQUMsQ0FBQztJQUVILHNDQUFzQztJQUN0QyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLCtCQUErQixFQUFFO1FBQzVELGNBQWMsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGtCQUFrQixFQUFFLEtBQUssQ0FBRSxFQUFFO1FBQy9ELE9BQU8sRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGlEQUFpRCxFQUFFLEtBQUssQ0FBRSxFQUFFO0tBQ3hGLENBQUMsQ0FBQztJQUVILDhDQUE4QztJQUM5QyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1FBQzdDLHdCQUF3QixFQUFFO1lBQ3hCLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFNBQVMsRUFBRSxDQUFDO29CQUNWLE1BQU0sRUFBRSxnQkFBZ0I7b0JBQ3hCLE1BQU0sRUFBRSxPQUFPO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxPQUFPLEVBQUU7NEJBQ1AsVUFBVSxFQUFFLENBQUUsRUFBRSxFQUFFO29DQUNkLE9BQU87b0NBQ1AsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29DQUN0QixHQUFHO29DQUNILEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO2lDQUMxQjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRixDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtCQUFrQixFQUFFO1FBQy9DLGNBQWMsRUFBRTtZQUNkLE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFNBQVMsRUFBRTtnQkFDVDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04sd0JBQXdCO3dCQUN4QixtQkFBbUI7d0JBQ25CLG9CQUFvQjtxQkFDckI7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsa0JBQWtCLEVBQUUsS0FBSyxDQUFFLEVBQUU7aUJBQzFEO2dCQUNEO29CQUNFLE1BQU0sRUFBRSxjQUFjO29CQUN0QixNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSxpREFBaUQsRUFBRSxLQUFLLENBQUUsRUFBRTtpQkFDekY7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGtpbmVzaXMgZnJvbSAnQGF3cy1jZGsvYXdzLWtpbmVzaXMnO1xuaW1wb3J0ICogYXMgbG9ncyBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBkZXN0cyBmcm9tICcuLi9saWInO1xuXG50ZXN0KCdzdHJlYW0gY2FuIGJlIHN1YnNjcmlwdGlvbiBkZXN0aW5hdGlvbicsICgpID0+IHtcbiAgLy8gR0lWRU5cbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHN0cmVhbSA9IG5ldyBraW5lc2lzLlN0cmVhbShzdGFjaywgJ015U3RyZWFtJyk7XG4gIGNvbnN0IGxvZ0dyb3VwID0gbmV3IGxvZ3MuTG9nR3JvdXAoc3RhY2ssICdMb2dHcm91cCcpO1xuXG4gIC8vIFdIRU5cbiAgbmV3IGxvZ3MuU3Vic2NyaXB0aW9uRmlsdGVyKHN0YWNrLCAnU3Vic2NyaXB0aW9uJywge1xuICAgIGxvZ0dyb3VwLFxuICAgIGRlc3RpbmF0aW9uOiBuZXcgZGVzdHMuS2luZXNpc0Rlc3RpbmF0aW9uKHN0cmVhbSksXG4gICAgZmlsdGVyUGF0dGVybjogbG9ncy5GaWx0ZXJQYXR0ZXJuLmFsbEV2ZW50cygpXG4gIH0pO1xuXG4gIC8vIFRIRU46IHN1YnNjcmlwdGlvbiB0YXJnZXQgaXMgU3RyZWFtXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6TG9nczo6U3Vic2NyaXB0aW9uRmlsdGVyJywge1xuICAgIERlc3RpbmF0aW9uQXJuOiB7IFwiRm46OkdldEF0dFwiOiBbIFwiTXlTdHJlYW01QzA1MEU5M1wiLCBcIkFyblwiIF0gfSxcbiAgICBSb2xlQXJuOiB7IFwiRm46OkdldEF0dFwiOiBbIFwiU3Vic2NyaXB0aW9uQ2xvdWRXYXRjaExvZ3NDYW5QdXRSZWNvcmRzOUMxMjIzRUNcIiwgXCJBcm5cIiBdIH0sXG4gIH0pO1xuXG4gIC8vIFRIRU46IHdlIGhhdmUgYSByb2xlIHRvIHdyaXRlIHRvIHRoZSBTdHJlYW1cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OiB7XG4gICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICBTdGF0ZW1lbnQ6IFt7XG4gICAgICAgIEFjdGlvbjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgIFNlcnZpY2U6IHtcbiAgICAgICAgICAgIFwiRm46OkpvaW5cIjogWyBcIlwiLCBbXG4gICAgICAgICAgICAgICAgXCJsb2dzLlwiLFxuICAgICAgICAgICAgICAgIHsgUmVmOiBcIkFXUzo6UmVnaW9uXCIgfSxcbiAgICAgICAgICAgICAgICBcIi5cIixcbiAgICAgICAgICAgICAgICB7IFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiIH1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfV0sXG4gICAgfVxuICB9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgXCJraW5lc2lzOkRlc2NyaWJlU3RyZWFtXCIsXG4gICAgICAgICAgICBcImtpbmVzaXM6UHV0UmVjb3JkXCIsXG4gICAgICAgICAgICBcImtpbmVzaXM6UHV0UmVjb3Jkc1wiLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJNeVN0cmVhbTVDMDUwRTkzXCIsIFwiQXJuXCIgXSB9XG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBBY3Rpb246IFwiaWFtOlBhc3NSb2xlXCIsXG4gICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgUmVzb3VyY2U6IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJTdWJzY3JpcHRpb25DbG91ZFdhdGNoTG9nc0NhblB1dFJlY29yZHM5QzEyMjNFQ1wiLCBcIkFyblwiIF0gfVxuICAgICAgICB9XG4gICAgICBdLFxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgnc3RyZWFtIGNhbiBiZSBzdWJzY3JpcHRpb24gZGVzdGluYXRpb24gdHdpY2UsIHdpdGhvdXQgZHVwbGljYXRpbmcgcGVybWlzc2lvbnMnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBzdHJlYW0gPSBuZXcga2luZXNpcy5TdHJlYW0oc3RhY2ssICdNeVN0cmVhbScpO1xuICBjb25zdCBsb2dHcm91cDEgPSBuZXcgbG9ncy5Mb2dHcm91cChzdGFjaywgJ0xvZ0dyb3VwJyk7XG4gIGNvbnN0IGxvZ0dyb3VwMiA9IG5ldyBsb2dzLkxvZ0dyb3VwKHN0YWNrLCAnTG9nR3JvdXAyJyk7XG5cbiAgLy8gV0hFTlxuICBuZXcgbG9ncy5TdWJzY3JpcHRpb25GaWx0ZXIoc3RhY2ssICdTdWJzY3JpcHRpb24nLCB7XG4gICAgbG9nR3JvdXA6IGxvZ0dyb3VwMSxcbiAgICBkZXN0aW5hdGlvbjogbmV3IGRlc3RzLktpbmVzaXNEZXN0aW5hdGlvbihzdHJlYW0pLFxuICAgIGZpbHRlclBhdHRlcm46IGxvZ3MuRmlsdGVyUGF0dGVybi5hbGxFdmVudHMoKVxuICB9KTtcblxuICBuZXcgbG9ncy5TdWJzY3JpcHRpb25GaWx0ZXIoc3RhY2ssICdTdWJzY3JpcHRpb24yJywge1xuICAgIGxvZ0dyb3VwOiBsb2dHcm91cDIsXG4gICAgZGVzdGluYXRpb246IG5ldyBkZXN0cy5LaW5lc2lzRGVzdGluYXRpb24oc3RyZWFtKSxcbiAgICBmaWx0ZXJQYXR0ZXJuOiBsb2dzLkZpbHRlclBhdHRlcm4uYWxsRXZlbnRzKClcbiAgfSk7XG5cbiAgLy8gVEhFTjogc3Vic2NyaXB0aW9uIHRhcmdldCBpcyBTdHJlYW1cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpMb2dzOjpTdWJzY3JpcHRpb25GaWx0ZXInLCB7XG4gICAgRGVzdGluYXRpb25Bcm46IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJNeVN0cmVhbTVDMDUwRTkzXCIsIFwiQXJuXCIgXSB9LFxuICAgIFJvbGVBcm46IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJTdWJzY3JpcHRpb25DbG91ZFdhdGNoTG9nc0NhblB1dFJlY29yZHM5QzEyMjNFQ1wiLCBcIkFyblwiIF0gfSxcbiAgfSk7XG5cbiAgLy8gVEhFTjogd2UgaGF2ZSBhIHJvbGUgdG8gd3JpdGUgdG8gdGhlIFN0cmVhbVxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgIFN0YXRlbWVudDogW3tcbiAgICAgICAgQWN0aW9uOiBcInN0czpBc3N1bWVSb2xlXCIsXG4gICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgUHJpbmNpcGFsOiB7XG4gICAgICAgICAgU2VydmljZToge1xuICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbIFwiXCIsIFtcbiAgICAgICAgICAgICAgICBcImxvZ3MuXCIsXG4gICAgICAgICAgICAgICAgeyBSZWY6IFwiQVdTOjpSZWdpb25cIiB9LFxuICAgICAgICAgICAgICAgIFwiLlwiLFxuICAgICAgICAgICAgICAgIHsgUmVmOiBcIkFXUzo6VVJMU3VmZml4XCIgfVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XSxcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICB7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICBcImtpbmVzaXM6RGVzY3JpYmVTdHJlYW1cIixcbiAgICAgICAgICAgIFwia2luZXNpczpQdXRSZWNvcmRcIixcbiAgICAgICAgICAgIFwia2luZXNpczpQdXRSZWNvcmRzXCIsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZTogeyBcIkZuOjpHZXRBdHRcIjogWyBcIk15U3RyZWFtNUMwNTBFOTNcIiwgXCJBcm5cIiBdIH1cbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogXCJpYW06UGFzc1JvbGVcIixcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZTogeyBcIkZuOjpHZXRBdHRcIjogWyBcIlN1YnNjcmlwdGlvbkNsb3VkV2F0Y2hMb2dzQ2FuUHV0UmVjb3JkczlDMTIyM0VDXCIsIFwiQXJuXCIgXSB9XG4gICAgICAgIH1cbiAgICAgIF0sXG4gICAgfVxuICB9KTtcbn0pO1xuIl19