"""
API call detection - find API calls in frontend/client code
"""
import re
from pathlib import Path
from typing import List, Dict


# Common API call patterns
API_CALL_PATTERNS = [
    # fetch()
    (r'fetch\(["\']([^"\']+)["\']', 'fetch'),
    
    # axios
    (r'axios\.(get|post|put|delete|patch)\(["\']([^"\']+)["\']', 'axios'),
    
    # requests (Python)
    (r'requests\.(get|post|put|delete|patch)\(["\']([^"\']+)["\']', 'requests'),
    
    # http.get/post (Go)
    (r'http\.(Get|Post|Put|Delete)\(["\']([^"\']+)["\']', 'http'),
]


def find_api_calls(file_path: Path, content: str) -> List[Dict]:
    """
    Find API calls in file content
    
    Args:
        file_path: Path to file
        content: File content
        
    Returns:
        List of API call dictionaries
    """
    api_calls = []
    
    for pattern, library in API_CALL_PATTERNS:
        matches = re.finditer(pattern, content, re.IGNORECASE)
        
        for match in matches:
            if library == 'fetch':
                method = 'GET'  # Default for fetch
                url = match.group(1)
            else:
                method = match.group(1).upper()
                url = match.group(2)
            
            # Get line number
            line_num = content[:match.start()].count('\n') + 1
            
            api_calls.append({
                "method": method,
                "url": url,
                "file": str(file_path),
                "line": line_num,
                "library": library,
                "type": "api_call"
            })
    
    return api_calls
