"""
Connection matcher - match API calls to endpoints
"""
from typing import List, Dict
import re


def normalize_path(path: str) -> str:
    """Normalize API path for matching"""
    # Remove query parameters
    path = path.split('?')[0]
    
    # Remove trailing slash
    path = path.rstrip('/')
    
    # Remove leading slash if present
    if path.startswith('/'):
        path = path[1:]
    
    return path


def paths_match(endpoint_path: str, call_url: str) -> bool:
    """
    Check if endpoint path matches API call URL
    
    Handles:
    - Exact matches
    - Path parameters (:id, {id}, <id>)
    - Partial URL matches
    """
    # Normalize both paths
    endpoint = normalize_path(endpoint_path)
    call = normalize_path(call_url)
    
    # Extract path from full URL if present
    if '://' in call:
        # Extract path from URL
        parts = call.split('://', 1)[1].split('/', 1)
        if len(parts) > 1:
            call = parts[1]
        else:
            return False
    
    # Exact match
    if endpoint == call:
        return True
    
    # Convert path parameters to regex
    # :id, {id}, <id> -> [^/]+
    endpoint_pattern = re.sub(r'[:{<][^>}:]+[>}:]?', r'[^/]+', endpoint)
    
    # Try regex match
    if re.fullmatch(endpoint_pattern, call):
        return True
    
    return False


def match_connections(endpoints: List[Dict], api_calls: List[Dict]) -> List[Dict]:
    """
    Match API calls to endpoints
    
    Args:
        endpoints: List of endpoint dicts
        api_calls: List of API call dicts
        
    Returns:
        List of connection dicts
    """
    connections = []
    
    for call in api_calls:
        for endpoint in endpoints:
            # Check if methods match
            if call["method"] != endpoint["method"]:
                continue
            
            # Check if paths match
            if paths_match(endpoint["path"], call["url"]):
                connections.append({
                    "endpoint": endpoint,
                    "api_call": call,
                    "confidence": "high",
                    "match_type": "path_and_method"
                })
    
    return connections
