"""
Report Comparison - Compare reports over time to track changes
"""
import json
from pathlib import Path
from typing import Dict, List, Optional
from datetime import datetime


def load_report(report_path: Path) -> Optional[Dict]:
    """Load a report from file"""
    try:
        with open(report_path, 'r') as f:
            return json.load(f)
    except:
        return None


def find_previous_reports(reports_dir: Path, current_report: str) -> List[Path]:
    """Find all previous reports, sorted by date"""
    if not reports_dir.exists():
        return []
    
    reports = []
    for report_file in reports_dir.glob("report_*.json"):
        if report_file.name != current_report:
            reports.append(report_file)
    
    # Sort by modification time (newest first)
    reports.sort(key=lambda x: x.stat().st_mtime, reverse=True)
    return reports


def compare_reports(current: Dict, previous: Dict) -> Dict:
    """
    Compare two reports and return differences
    
    Returns dict with:
    - added_endpoints: New endpoints
    - removed_endpoints: Deleted endpoints
    - added_api_calls: New API calls
    - removed_api_calls: Deleted API calls
    - added_connections: New connections
    - removed_connections: Broken connections
    - impact_changes: Changes in blast radius
    - confidence_changes: Changes in confidence scores
    """
    # Compare endpoints
    current_endpoints = {f"{e['method']}:{e['path']}" for e in current.get('endpoints', [])}
    previous_endpoints = {f"{e['method']}:{e['path']}" for e in previous.get('endpoints', [])}
    
    added_endpoints = current_endpoints - previous_endpoints
    removed_endpoints = previous_endpoints - current_endpoints
    
    # Compare API calls
    current_calls = {f"{c['method']}:{c['url']}:{c['file']}:{c['line']}" 
                     for c in current.get('api_calls', [])}
    previous_calls = {f"{c['method']}:{c['url']}:{c['file']}:{c['line']}" 
                      for c in previous.get('api_calls', [])}
    
    added_calls = current_calls - previous_calls
    removed_calls = previous_calls - current_calls
    
    # Compare connections
    current_connections = {
        f"{c['endpoint']['method']}:{c['endpoint']['path']}->{c['api_call']['url']}"
        for c in current.get('connections', [])
    }
    previous_connections = {
        f"{c['endpoint']['method']}:{c['endpoint']['path']}->{c['api_call']['url']}"
        for c in previous.get('connections', [])
    }
    
    added_connections = current_connections - previous_connections
    removed_connections = previous_connections - current_connections
    
    # Compare summary metrics
    current_summary = current.get('summary', {})
    previous_summary = previous.get('summary', {})
    
    metric_changes = {
        'endpoints': current_summary.get('total_endpoints', 0) - previous_summary.get('total_endpoints', 0),
        'api_calls': current_summary.get('total_api_calls', 0) - previous_summary.get('total_api_calls', 0),
        'connections': current_summary.get('total_connections', 0) - previous_summary.get('total_connections', 0),
        'high_impact_nodes': current_summary.get('high_impact_nodes', 0) - previous_summary.get('high_impact_nodes', 0)
    }
    
    # Compare accuracy
    current_accuracy = current.get('accuracy', {})
    previous_accuracy = previous.get('accuracy', {})
    
    accuracy_change = {
        'current': current_accuracy.get('estimated_accuracy', 'N/A'),
        'previous': previous_accuracy.get('estimated_accuracy', 'N/A'),
        'percentage_change': current_accuracy.get('accuracy_percentage', 0) - previous_accuracy.get('accuracy_percentage', 0)
    }
    
    # Generate change summary
    change_summary = generate_change_summary(
        added_endpoints, removed_endpoints,
        added_calls, removed_calls,
        added_connections, removed_connections,
        metric_changes
    )
    
    return {
        'comparison_date': datetime.utcnow().isoformat() + 'Z',
        'current_report_date': current.get('generated_at', 'unknown'),
        'previous_report_date': previous.get('generated_at', 'unknown'),
        'added_endpoints': list(added_endpoints),
        'removed_endpoints': list(removed_endpoints),
        'added_api_calls': list(added_calls),
        'removed_api_calls': list(removed_calls),
        'added_connections': list(added_connections),
        'removed_connections': list(removed_connections),
        'metric_changes': metric_changes,
        'accuracy_change': accuracy_change,
        'change_summary': change_summary,
        'has_changes': any([
            added_endpoints, removed_endpoints,
            added_calls, removed_calls,
            added_connections, removed_connections
        ])
    }


def generate_change_summary(added_endpoints, removed_endpoints,
                           added_calls, removed_calls,
                           added_connections, removed_connections,
                           metric_changes) -> str:
    """Generate human-readable change summary"""
    parts = []
    
    if added_endpoints:
        parts.append(f"+{len(added_endpoints)} endpoint{'s' if len(added_endpoints) != 1 else ''}")
    if removed_endpoints:
        parts.append(f"-{len(removed_endpoints)} endpoint{'s' if len(removed_endpoints) != 1 else ''}")
    
    if added_calls:
        parts.append(f"+{len(added_calls)} API call{'s' if len(added_calls) != 1 else ''}")
    if removed_calls:
        parts.append(f"-{len(removed_calls)} API call{'s' if len(removed_calls) != 1 else ''}")
    
    if added_connections:
        parts.append(f"+{len(added_connections)} connection{'s' if len(added_connections) != 1 else ''}")
    if removed_connections:
        parts.append(f"-{len(removed_connections)} connection{'s' if len(removed_connections) != 1 else ''}")
    
    if not parts:
        return "No changes detected"
    
    return ", ".join(parts)


def print_comparison(comparison: Dict):
    """Print comparison results in a readable format"""
    print("\n" + "="*60)
    print("REPORT COMPARISON")
    print("="*60)
    
    print(f"\nCurrent:  {comparison['current_report_date']}")
    print(f"Previous: {comparison['previous_report_date']}")
    
    if not comparison['has_changes']:
        print("\n✓ No changes detected")
        return
    
    print(f"\nSummary: {comparison['change_summary']}")
    
    # Metric changes
    print("\n" + "-"*60)
    print("METRIC CHANGES")
    print("-"*60)
    
    for metric, change in comparison['metric_changes'].items():
        if change != 0:
            sign = "+" if change > 0 else ""
            print(f"  {metric.replace('_', ' ').title()}: {sign}{change}")
    
    # Accuracy change
    acc_change = comparison['accuracy_change']
    if acc_change['percentage_change'] != 0:
        print(f"\n  Accuracy: {acc_change['previous']} → {acc_change['current']}")
        print(f"  Change: {acc_change['percentage_change']:+.1f}%")
    
    # Added endpoints
    if comparison['added_endpoints']:
        print("\n" + "-"*60)
        print(f"ADDED ENDPOINTS ({len(comparison['added_endpoints'])})")
        print("-"*60)
        for endpoint in comparison['added_endpoints'][:10]:
            print(f"  + {endpoint}")
        if len(comparison['added_endpoints']) > 10:
            print(f"  ... and {len(comparison['added_endpoints']) - 10} more")
    
    # Removed endpoints
    if comparison['removed_endpoints']:
        print("\n" + "-"*60)
        print(f"REMOVED ENDPOINTS ({len(comparison['removed_endpoints'])})")
        print("-"*60)
        for endpoint in comparison['removed_endpoints'][:10]:
            print(f"  - {endpoint}")
        if len(comparison['removed_endpoints']) > 10:
            print(f"  ... and {len(comparison['removed_endpoints']) - 10} more")
    
    # Added connections
    if comparison['added_connections']:
        print("\n" + "-"*60)
        print(f"NEW CONNECTIONS ({len(comparison['added_connections'])})")
        print("-"*60)
        for conn in comparison['added_connections'][:5]:
            print(f"  + {conn}")
        if len(comparison['added_connections']) > 5:
            print(f"  ... and {len(comparison['added_connections']) - 5} more")
    
    # Broken connections
    if comparison['removed_connections']:
        print("\n" + "-"*60)
        print(f"BROKEN CONNECTIONS ({len(comparison['removed_connections'])})")
        print("-"*60)
        for conn in comparison['removed_connections'][:5]:
            print(f"  - {conn}")
        if len(comparison['removed_connections']) > 5:
            print(f"  ... and {len(comparison['removed_connections']) - 5} more")
    
    print("\n" + "="*60 + "\n")
