import math
import json
import random
from datetime import datetime, date, time, timedelta
import lstpressure
GLOBAL_VARS_FILE = "tmp/global_vars.json"

SECONDS_PER_DAY: int = 86400
SIDEREAL_DAY_SECONDS: int = 86164.0905
J2000: datetime = datetime(2000, 1, 1, 12)  # Reference epoch for JD

# Constants for SKA site
SKA_LATITUDE_STR: str = "-30:42:39.8"
"""Latitude of the SKA site as a string. This is the same value used in OPT.
"""
SKA_LONGITUDE_STR: str = "21:26:38.0"
"""Longitude of the SKA site as a string. This is the same value used in OPT.
"""

MINUTES_IN_SIDEREAL_DAY = (
    23 * 60 + 56
)  # 23 hours and 56 minutes (and 4 seconds technically)
"""Number of minutes in a sidereal day. Used for time calculations."""

MINUTES_IN_SOLAR_DAY = 24 * 60  # 24 hours
"""Number of minutes in a solar day. Used for time calculations."""

ZENITH = 90 + 50 / 60  # Official zenith for sunrise/sunset in degrees

# MATH CONSTANT
TO_RAD = math.pi/180.0

# PLANE TIME CONSTANTS
PLANE_WEEK_DAY: int = 2  # Wednesday = 2 according to date.weekday()
PLANE_ARRIVAL_TIME_BLOCK: tuple[time, time] = (time(6, 0, 0), time(7, 0, 0)) # These values are in UTC
PLANE_DEPARTURE_TIME_BLOCK: tuple[time, time] = (time(13, 0, 0), time(15, 0, 0)) # These values are in UTC

def degrees_string_to_float(degrees: str) -> float:
    """
    Convert a string in the format "hh:mm:ss.s" to a float in degrees.

    Args:
        degrees (str): String in the format "hh:mm:ss.s".

    Returns:
        float: Float representation in degrees.
    """
    sign = -1 if degrees.startswith("-") else 1
    if sign == -1:
        degrees = degrees[1:]
    h, m, s = map(float, degrees.split(":"))
    return sign * (h + m / 60 + s / 3600)

SKA_LATITUDE: float = degrees_string_to_float(SKA_LATITUDE_STR)
SKA_LONGITUDE: float = degrees_string_to_float(SKA_LONGITUDE_STR)

def get_global_vars() -> tuple[date, date, list[dict]]:
    """
    Get the current values of the global variables from the JSON file.

    Args:
        None

    Returns:
        tuple[date, date, list[dict]]: The current values of START_DATE, END_DATE, and PROPOSALS.
    """
    try:
        with open(GLOBAL_VARS_FILE, "r") as file:
            data = json.load(file)
            start_date = date.fromisoformat(data["START_DATE"])
            end_date = date.fromisoformat(data["END_DATE"])
            proposals = data["PROPOSALS"]
            return start_date, end_date, proposals
    except FileNotFoundError:
        return date(2024, 1, 1), date(2024, 1, 22), []

def update_global_vars(start_date: date = date.today(), end_date: date = date.today(),proposals: list[dict] = []) -> None:
    """
    Update the global variables and write them to the JSON file.

    Args:
        start_date (date): The new start date.
        end_date (date): The new end date.
        proposals (list[dict]): The new list of proposal data.

    Returns:
        None
    """
    data = {
        "START_DATE": start_date.isoformat(),
        "END_DATE": end_date.isoformat(),
        "PROPOSALS": proposals
    }
    with open(GLOBAL_VARS_FILE, "w") as file:
        json.dump(data, file)

def parse_time(time_str: str) -> time:
    """
    Parses a time string in the format "HH:MM" and returns a datetime.time object.

    Args:
        time_str (str): A time string in the format "HH:MM".

    Returns:
        time: A datetime.time object representing the parsed time.
    """
    hour, minute = map(int, time_str.split(":"))
    return time(hour, minute)

def julian_date(date_obj: datetime) -> float:
    """
    Convert a datetime to Julian Date.
    """
    a = (14 - date_obj.month) // 12
    y = date_obj.year + 4800 - a
    m = date_obj.month + 12 * a - 3

    jdn = date_obj.day + ((153 * m + 2) // 5) + 365 * y + y // 4 - y // 100 + y // 400 - 32045
    day_frac = (date_obj.hour - 12) / 24 + date_obj.minute / 1440 + date_obj.second / 86400
    return jdn + day_frac

def gmst_at_0h_utc(jd: float) -> float:
    """
    Compute GMST at 0h UTC in decimal hours.
    """
    d = jd - 2451545.0  # Days since J2000
    gmst = 6.697374558 + 0.06570982441908 * d + 1.00273790935 * 0
    return gmst % 24

def lst_to_utc(date_obj: date, lst_time: time, longitude: float = SKA_LONGITUDE) -> datetime:
    """
    Convert Local Sidereal Time (LST) to UTC datetime (approximate method).

    Args:
        date_obj (date): UTC calendar date.
        lst_time (time): LST as time object.
        longitude (float): Observer longitude in degrees East (default is SKA site).

    Returns:
        datetime: Approximate UTC datetime corresponding to the given LST.
    """
    # 1. Convert LST to decimal hours
    lst_hours: float = lst_time.hour + lst_time.minute / 60 + lst_time.second / 3600

    # 2. Convert observer longitude from degrees to hours
    longitude_hours: float = longitude / 15.0

    # 3. Compute Julian Date at 0h UTC for the given date
    jd_0h: float = julian_date(datetime.combine(date_obj, time(0, 0, 0)))

    # 4. Compute GMST at 0h UTC
    gmst0: float = gmst_at_0h_utc(jd_0h)

    # 5. Calculate Greenwich Sidereal Time (GST) from Local Sidereal Time (LST)
    gst: float = (lst_hours - longitude_hours) % 24

    # 6. Difference between GST and GMST at 0h UTC (in sidereal hours)
    delta_sidereal_hours: float = (gst - gmst0) % 24

    # 7. Convert sidereal time to solar (UTC) time
    SIDEREAL_TO_SOLAR: float = 0.9972695663  # conversion factor
    delta_utc_hours: float = delta_sidereal_hours * SIDEREAL_TO_SOLAR

    # 8. Compute the UTC datetime
    utc_datetime: datetime = datetime.combine(date_obj, time(0, 0, 0)) + timedelta(hours=delta_utc_hours)

    return utc_datetime.replace(microsecond=0)

def LST_to_UTC_time(lst_hours: float, date: datetime, longitude: float = SKA_LONGITUDE) -> int:
    """Convert Local Sidereal Time (LST) to UTC.

    Args:
        lst_hours (float): Time in LST, in sidereal hours since 12am LST.
        date (datetime): Date on which to calculate the UTC time.
        longitude (float): Longitude used to calculate UTC. Must be given as a float in degrees. For example, -30.0 for 30 degrees west. Defaults to the longitude of the SKA site as used in OPT (21.44389).

    Returns:
        UTC time (int) : A UTC timestamp that corresponds to the given LST.
    """

    # print(f"Converting LST {lst_hours} hours to UTC on date {date} with longitude {longitude}")

    return lstpressure.sun.location_providers.meerkat_provider.DateTimeUtils.DateTimeUtil.lst2ut(
        lst_hours, longitude, date
    )
def get_sunrise_sunset(date: date, latitude: float = SKA_LATITUDE, longitude: float = SKA_LONGITUDE) -> tuple[datetime, datetime]:
    """
    Calculate sunrise and sunset times for a given date, latitude, and longitude.

    Args:
        date (date): The date for which to calculate the sunrise and sunset.
        latitude (float): The latitude of the location.
        longitude (float): The longitude of the location.

    Returns:
        tuple[datetime, datetime]: The sunrise and sunset times as naive datetime objects.
    """

    # 1. Get the day of the year
    N = date.timetuple().tm_yday

    # 2. Convert the longitude to hour value and calculate an approximate time
    lng_hour = longitude / 15.0
    t_rise = N + ((6 - lng_hour) / 24)  # For sunrise
    t_set = N + ((18 - lng_hour) / 24)  # For sunset

    # 3a. Calculate the Sun's mean anomaly
    M_rise = (0.9856 * t_rise) - 3.289
    M_set = (0.9856 * t_set) - 3.289

    # 3b. Calculate the Sun's true longitude
    L_rise = M_rise + (1.916 * math.sin(TO_RAD * M_rise)) + (0.020 * math.sin(TO_RAD * 2 * M_rise)) + 282.634
    L_set = M_set + (1.916 * math.sin(TO_RAD * M_set)) + (0.020 * math.sin(TO_RAD * 2 * M_set)) + 282.634

    # Adjust L into the range [0, 360)
    L_rise = force_range(L_rise, 360)
    L_set = force_range(L_set, 360)

    # 4a. Calculate the Sun's declination
    sinDec_rise = 0.39782 * math.sin(TO_RAD * L_rise)
    cosDec_rise = math.cos(math.asin(sinDec_rise))

    sinDec_set = 0.39782 * math.sin(TO_RAD * L_set)
    cosDec_set = math.cos(math.asin(sinDec_set))

    # 4b. Calculate the Sun's local hour angle
    cosH_rise = (math.cos(TO_RAD * ZENITH) - (sinDec_rise * math.sin(TO_RAD * latitude))) / (cosDec_rise * math.cos(TO_RAD * latitude))
    cosH_set = (math.cos(TO_RAD * ZENITH) - (sinDec_set * math.sin(TO_RAD * latitude))) / (cosDec_set * math.cos(TO_RAD * latitude))

    # Check if the sun never rises or sets
    if cosH_rise > 1:
        return None, None  # The sun never rises
    if cosH_set < -1:
        return None, None  # The sun never sets

    # 4c. Finish calculating H and convert into hours
    H_rise = 360 - (1 / TO_RAD) * math.acos(cosH_rise)
    H_set = (1 / TO_RAD) * math.acos(cosH_set)

    H_rise /= 15
    H_set /= 15

    # 5a. Calculate the Sun's right ascension
    RA_rise = (1 / TO_RAD) * math.atan(0.91764 * math.tan(TO_RAD * L_rise))
    RA_set = (1 / TO_RAD) * math.atan(0.91764 * math.tan(TO_RAD * L_set))

    # Adjust RA into the range [0, 360)
    RA_rise = force_range(RA_rise, 360)
    RA_set = force_range(RA_set, 360)

    # 5b. Right ascension value needs to be in the same quadrant as L
    L_quadrant_rise = (math.floor(L_rise / 90)) * 90
    RA_quadrant_rise = (math.floor(RA_rise / 90)) * 90
    RA_rise += (L_quadrant_rise - RA_quadrant_rise)

    L_quadrant_set = (math.floor(L_set / 90)) * 90
    RA_quadrant_set = (math.floor(RA_set / 90)) * 90
    RA_set += (L_quadrant_set - RA_quadrant_set)

    # 5c. Right ascension value needs to be converted into hours
    RA_rise /= 15
    RA_set /= 15

    # 6. Calculate local mean time of rising/setting
    T_rise = H_rise + RA_rise - (0.06571 * t_rise) - 6.622
    T_set = H_set + RA_set - (0.06571 * t_set) - 6.622

    # 7. Adjust back to UTC
    UT_rise = T_rise - lng_hour
    UT_set = T_set - lng_hour

    # 7c. rounding and impose range bounds
    UT_rise = round(UT_rise, 2)
    UT_set = round(UT_set, 2)

    UT_rise = force_range(UT_rise, 24.0)
    UT_set = force_range(UT_set, 24.0)

    # Convert UT to naive datetime objects
    sunrise = datetime.combine(date, time(0, 0, 0)) + timedelta(hours=UT_rise)
    sunset = datetime.combine(date, time(0, 0, 0)) + timedelta(hours=UT_set)

    return sunrise.replace(second=0, microsecond=0), sunset.replace(second=0, microsecond=0)

def force_range(value: float, max_value: float):
    """
    Adjusts the value to wrap around within the range [0, max).

    Args:
        value (float): The value to adjust.
        max_value (float): The exclusive upper bound of the range.

    Returns:
        float: The adjusted value within the range [0, max).
    """
    if value < 0:
        return value + max_value
    elif value >= max_value:
        return value - max_value
    return value

def get_night_window(date: date, latitude: float = SKA_LATITUDE, longitude: float = SKA_LONGITUDE) -> tuple[datetime, datetime]:
    """
    Calculate the night window for a given date and geographic location. The night window is defined as starting 5 minutes after sunset on the specified date and ending 5 minutes before sunrise on the following day.

    Args:
        date (date): The date for which to calculate the night window. 
        latitude (float): The latitude of the location for which the night window is calculated. Default is set to SKA_LATITUDE.
        longitude (float): The longitude of the location for which the night window is calculated. Default is set to SKA_LONGITUDE.

    Returns:
        tuple[datetime, datetime]: A tuple containing night start and end datetime.
    """
    # Getting sunset and sunrise of the current and next day
    _, current_day_sunset = get_sunrise_sunset(date, latitude, longitude)
    next_day_sunsrise, _ = get_sunrise_sunset(date + timedelta(days=1), latitude, longitude)
    
    # Assuming night starts 5 min after sunset of the current day and ends 5 minutes before sunrise the next day
    return (current_day_sunset + timedelta(minutes=5), next_day_sunsrise - timedelta(minutes=5))

def get_plane_arrival_and_departure_blocks(
    date: date,
    plane_weekday: int = PLANE_WEEK_DAY,
    plane_arrival_time_block: tuple[time, time] = PLANE_ARRIVAL_TIME_BLOCK,
    plane_departure_time_block: tuple[time, time] = PLANE_DEPARTURE_TIME_BLOCK
) -> list[tuple[datetime, datetime]]:
    """
    Retrieve the plane arrival and departure time blocks for a specified date.

    Args:
        date (date): The date for which to retrieve the plane arrival and departure blocks.
        plane_weekday (int): The weekday (0=Monday, 6=Sunday) when the plane operates. 
                             Defaults to PLANE_WEEK_DAY.
        plane_arrival_time_block (tuple[time, time]): A tuple representing the start and end time 
                                                      for plane arrivals. Defaults to PLANE_ARRIVAL_TIME_BLOCK.
        plane_departure_time_block (tuple[time, time]): A tuple representing the start and end time 
                                                       for plane departures. Defaults to PLANE_DEPARTURE_TIME_BLOCK.

    Returns:
        list[tuple[datetime, datetime]]: A list of tuples, each containing:
            - Arrival or departure start time as a datetime object.
            - Arrival or departure end time as a datetime object.
    """
    # Initialize an empty list to hold the arrival and departure blocks
    plane_arrival_and_departure_blocks: list[tuple[datetime, datetime]] = []

    # Check if the specified date is the correct weekday for operations
    if date.weekday() == plane_weekday:
        # Define the time blocks for arrival and departure
        time_blocks = [plane_arrival_time_block, plane_departure_time_block]

        # Combine the date with each time block and append to the list
        for (start_time, end_time) in time_blocks:
            start_datetime = datetime.combine(date, start_time)
            end_datetime = datetime.combine(date, end_time)
            plane_arrival_and_departure_blocks.append((start_datetime, end_datetime))

    return plane_arrival_and_departure_blocks

def _degrees_string_to_float(degrees: str) -> float:
    """Convert a string in the format "hh:mm:ss.s" to a float in degrees.

    Args:
        degrees (str): String in the format "hh:mm:ss.s".

    Returns:
        float: Float in degrees.
    """
    sign = 1 if degrees[0] != "-" else -1
    if sign == -1:
        degrees = degrees[1:]
    h, m, s = map(float, degrees.split(":"))
    return sign * (h + m / 60 + s / 3600)

SKA_LATITUDE = _degrees_string_to_float(SKA_LATITUDE_STR)
"""Latitude of the SKA site in degrees. This is the same value used in OPT.
"""

SKA_LONGITUDE = _degrees_string_to_float(SKA_LONGITUDE_STR)
"""Longitude of the SKA site in degrees. This is the same value used in OPT.
"""

def get_UTC_sunrise_sunset_times(
    date: datetime,
    num_days: int = 1,
    latitude: float = SKA_LATITUDE,
    longitude: float = SKA_LONGITUDE,
):
    """Get the sunrise and sunset times at the SKA site in UTC on a specific day.

    Args:
        date (datetime.datetime): The date to query for
        num_days (int, optional): How many days to fetch sunrise/sunset times for. Defaults to 1.
        latitude (float, optional): Latitude to use when getting sunrise/sunset times. Defaults to the latitude of the SKA site as used in OPT (-30.71105).
        longitude (float, optional): Longitude to use when getting sunrise/sunset times. Defaults to the longitude of the SKA site as used in OPT (21.44389).

    Returns:
        list: List of dicts which each represent a date, with `date`, `sunrise` and `sunset` properties. All times are in UTC.
    """

    # If needs be, we can get sunrise/sunset times for the whole week
    date_start = date
    date_end = date + timedelta(days=num_days - 1)

    calendar = [
        d.dt.strftime("%Y-%m-%d")
        for d in lstpressure.LSTCalendar(date_start, date_end).dates
    ]

    times = []

    for date in calendar:

        sun_info = lstpressure.Sun(
            yyyymmdd=date.replace(
                "-", ""
            ),  # change format from YYYY-MM-DD to YYYYMMDD which is what lstpressure expects
            latitude=latitude,
            longitude=longitude,
        )
        times.append(
            {
                "date": date,  # returned date is still in YYYY-MM-DD format though
                "sunrise": sun_info.sunrise,
                "sunset": sun_info.sunset,
            }
        )

    times = sorted(times, key=lambda x: x["date"])

    return times


if __name__ == "__main__":
    # Example Usage of lst_to_utc function 
    date = date.today()
    lst_time = time(12, 30, 45)
    utc_time = lst_to_utc(date, lst_time, SKA_LONGITUDE)
    print(f"UTC Time: {utc_time}")

    # Example Usage of LST_TO_UTC_hours function 
    lst_time = 1 + 59 / 60 + 0 / 3600
    date = datetime.now()
    utc_time = LST_to_UTC_time(lst_time, date, SKA_LONGITUDE)
    print(f"UTC Time: {int(utc_time)}:{int((utc_time % 1) * 60):02}")

    