from aleph_lang import KetInt, KetBool, sample, prepare

def assert_sample(kets, valid):
    u = sample(kets)
    assert u in valid

def assert_universe(kets, rows):
    u = prepare(kets)
    assert u['state']['rows'] == rows

def test_literal_bool():
    k1 = KetBool()
    assert_universe([k1], [[0],[1]])

def test_literal_int():
    k1 = KetInt()
    assert 3 == k1.width
    assert_universe([k1], [[0],[1],[2],[3],[4],[5],[6],[7]])

    k2 = KetInt(width=2)
    assert 2 == k2.width
    assert_universe([k2], [[0],[1],[2],[3]])

def test_where():
    k1 = KetInt()
    assert_universe([k1.where_equals(5)], [[5,5,1]])
    assert_universe([k1.where_greater_than(5)], [[6,5,1], [7,5,1]])
    assert_universe([k1.where_less_than_equals(5)], [[0,5,1],[1,5,1],[2,5,1],[3,5,1],[4,5,1],[5,5,1]])
    assert_universe([k1, k1.greater_than(5).where_true()], [[6,5,1], [7,5,1]])
    assert_universe([k1.where_in([1,2,5])], [[1,1], [2,1], [5,1]])
    assert_universe([k1.where_in(range(1,7))], [[1,1], [2,1], [3,1], [4,1], [5,1], [6,1]])
    
    assert_sample([k1.where_equals(5)], [[5]])
    assert_sample([k1.where_greater_than(5)], [[6], [7]])


def test_add():
    k1 = KetInt(width=2)
    k2 = KetInt(width=2)
    k3 = k1 + k2
    assert 3 == k3.width

    assert_universe([k1 + 2, 2 + k1, k1 + k2, k2 + k1, k1.add(k2, width=2)], [
        [0,2,2,2,2,0,0,0,0],
        [0,2,2,2,2,1,1,1,1],
        [0,2,2,2,2,2,2,2,2],
        [0,2,2,2,2,3,3,3,3],
        
        [1,2,3,2,3,0,1,1,1],
        [1,2,3,2,3,1,2,2,2],
        [1,2,3,2,3,2,3,3,3],
        [1,2,3,2,3,3,4,4,0],
        
        [2,2,4,2,4,0,2,2,2],
        [2,2,4,2,4,1,3,3,3],
        [2,2,4,2,4,2,4,4,0],
        [2,2,4,2,4,3,5,5,1],
        
        [3,2,5,2,5,0,3,3,3],
        [3,2,5,2,5,1,4,4,0],
        [3,2,5,2,5,2,5,5,1],
        [3,2,5,2,5,3,6,6,2],
    ])

    assert_sample([k1, k3], [
        [0,0],
        [0,1],
        [0,2],
        [0,3],
        [1,1],
        [1,2],
        [1,3],
        [1,4],
        [2,2],
        [2,3],
        [2,4],
        [2,5],
        [3,3],
        [3,4],
        [3,5],
        [3,6],
    ])

def test_multiply():
    k1 = KetInt(width=2)
    k2 = KetInt(width=2)
    k3 = k1 * k2
    assert 4 == k3.width

    assert_universe([k1 * 2, 2 * k1, k1 * k2, k2 * k1, k1.multiply(k2, width=3)], [
        [0,2,0,2,0,0,0,0,0],
        [0,2,0,2,0,1,0,0,0],
        [0,2,0,2,0,2,0,0,0],
        [0,2,0,2,0,3,0,0,0],
        
        [1,2,2,2,2,0,0,0,0],
        [1,2,2,2,2,1,1,1,1],
        [1,2,2,2,2,2,2,2,2],
        [1,2,2,2,2,3,3,3,3],
        
        [2,2,4,2,4,0,0,0,0],
        [2,2,4,2,4,1,2,2,2],
        [2,2,4,2,4,2,4,4,4],
        [2,2,4,2,4,3,6,6,6],
        
        [3,2,6,2,6,0,0,0,0],
        [3,2,6,2,6,1,3,3,3],
        [3,2,6,2,6,2,6,6,6],
        [3,2,6,2,6,3,9,9,1],
    ])
    
def test_equals():
    k1 = KetInt(width=2)
    k2 = KetInt(width=2)
    k3 = k1 == k2
    assert 1 == k3.width

    assert_universe([k1 == 2, 2 == k1, k1 == k2, k2 == k1, k1.equals(k2)], [
        [0,2,0,2,0,0,1,1,1],
        [0,2,0,2,0,1,0,0,0],
        [0,2,0,2,0,2,0,0,0],
        [0,2,0,2,0,3,0,0,0],
        
        [1,2,0,2,0,0,0,0,0],
        [1,2,0,2,0,1,1,1,1],
        [1,2,0,2,0,2,0,0,0],
        [1,2,0,2,0,3,0,0,0],
        
        [2,2,1,2,1,0,0,0,0],
        [2,2,1,2,1,1,0,0,0],
        [2,2,1,2,1,2,1,1,1],
        [2,2,1,2,1,3,0,0,0],
        
        [3,2,0,2,0,0,0,0,0],
        [3,2,0,2,0,1,0,0,0],
        [3,2,0,2,0,2,0,0,0],
        [3,2,0,2,0,3,1,1,1],
    ])
    
def test_not_equals():
    k1 = KetInt(width=2)
    k2 = KetInt(width=2)
    k3 = k1 != k2
    assert 1 == k3.width

    assert_universe([k1 != 2, 2 != k1, k1 != k2, k2 != k1, k1.not_equals(k2)], [
        [0,2,0,1,2,0,1,0,1,0,1,0,1,0],
        [0,2,0,1,2,0,1,1,0,1,0,1,0,1],
        [0,2,0,1,2,0,1,2,0,1,0,1,0,1],
        [0,2,0,1,2,0,1,3,0,1,0,1,0,1],
        
        [1,2,0,1,2,0,1,0,0,1,0,1,0,1],
        [1,2,0,1,2,0,1,1,1,0,1,0,1,0],
        [1,2,0,1,2,0,1,2,0,1,0,1,0,1],
        [1,2,0,1,2,0,1,3,0,1,0,1,0,1],
        
        [2,2,1,0,2,1,0,0,0,1,0,1,0,1],
        [2,2,1,0,2,1,0,1,0,1,0,1,0,1],
        [2,2,1,0,2,1,0,2,1,0,1,0,1,0],
        [2,2,1,0,2,1,0,3,0,1,0,1,0,1],
        
        [3,2,0,1,2,0,1,0,0,1,0,1,0,1],
        [3,2,0,1,2,0,1,1,0,1,0,1,0,1],
        [3,2,0,1,2,0,1,2,0,1,0,1,0,1],
        [3,2,0,1,2,0,1,3,1,0,1,0,1,0],
    ])

def test_less_than_equals():
    k1 = KetInt(width=2)
    k2 = KetInt(width=2)
    k3 = k1 <= k2
    assert 1 == k3.width

    assert_universe([k1 <= 2, 2 <= k1, k1 <= k2, k2 <= k1, k1.less_than_equals(k2)], [
        [0,2,1,2,0,0,1,1,1],
        [0,2,1,2,0,1,1,0,1],
        [0,2,1,2,0,2,1,0,1],
        [0,2,1,2,0,3,1,0,1],
        
        [1,2,1,2,0,0,0,1,0],
        [1,2,1,2,0,1,1,1,1],
        [1,2,1,2,0,2,1,0,1],
        [1,2,1,2,0,3,1,0,1],
        
        [2,2,1,2,1,0,0,1,0],
        [2,2,1,2,1,1,0,1,0],
        [2,2,1,2,1,2,1,1,1],
        [2,2,1,2,1,3,1,0,1],
        
        [3,2,0,2,1,0,0,1,0],
        [3,2,0,2,1,1,0,1,0],
        [3,2,0,2,1,2,0,1,0],
        [3,2,0,2,1,3,1,1,1],
    ])

def test_greater_than():
    k1 = KetInt(width=2)
    k2 = KetInt(width=2)
    k3 = k1 > k2
    assert 1 == k3.width

    assert_universe([k1 > 2, 2 > k1, k1 > k2, k2 > k1, k1.greater_than(k2)], [
        [0,2,0,2,1,0,0,0,0],
        [0,2,0,2,1,1,0,1,0],
        [0,2,0,2,1,2,0,1,0],
        [0,2,0,2,1,3,0,1,0],
        
        [1,2,0,2,1,0,1,0,1],
        [1,2,0,2,1,1,0,0,0],
        [1,2,0,2,1,2,0,1,0],
        [1,2,0,2,1,3,0,1,0],
        
        [2,2,0,2,0,0,1,0,1],
        [2,2,0,2,0,1,1,0,1],
        [2,2,0,2,0,2,0,0,0],
        [2,2,0,2,0,3,0,1,0],
        
        [3,2,1,2,0,0,1,0,1],
        [3,2,1,2,0,1,1,0,1],
        [3,2,1,2,0,2,1,0,1],
        [3,2,1,2,0,3,0,0,0],
    ])

def test_and():
    k1 = KetBool()
    k2 = KetBool()
    k3 = k1 & k2
    assert 1 == k3.width

    assert_universe([k1 & True, True & k1, k1 & k2, k2 & k1, k1.And(k2)], [
        [0,1,0,1,0,0,0,0,0],
        [0,1,0,1,0,1,0,0,0],
        [1,1,1,1,1,0,0,0,0],
        [1,1,1,1,1,1,1,1,1],
    ])
    
def test_or():
    k1 = KetBool()
    k2 = KetBool()
    k3 = k1 | k2
    assert 1 == k3.width

    assert_universe([k1 | True, True | k1, k1 | k2, k2 | k1, k1.Or(k2)], [
        [0,1,1,1,1,0,0,0,0],
        [0,1,1,1,1,1,1,1,1],
        [1,1,1,1,1,0,1,1,1],
        [1,1,1,1,1,1,1,1,1],
    ])


def test_not():
    k1 = KetBool()
    k2 = KetBool()
    k3 = k1.Not()
    assert 1 == k3.width

    assert_universe([k1.Not(), (k1 | k2).Not()], [
        [0,1,0,0,1],
        [0,1,1,1,0],
        [1,0,0,1,0],
        [1,0,1,1,0],
    ])