"""Contains the csVuln dataclass"""

from dataclasses import dataclass
from datetime import datetime
from typing import Union

from .software import csSoftware
from ...base.base_class import BaseClass
from ...base.base_list import BaseList
from ...base import DONT_EXPAND


@dataclass
class csVuln(BaseClass):
    """
    Represents a vulnerability in Qualys Container Security.
    """

    # NOTE: there are more fields from other API calls that have not been added yet.
    # They will be added at a later date.

    qid: int = None
    title: str = None
    result: str = None
    severity: int = None
    customerSeverity: int = None
    qdsScore: int = None
    cvssInfo: dict = None
    patchAvailable: bool = None
    published: Union[str, datetime] = None
    # cvssInfo is parsed into below fields:
    cvssInfo_baseScore: float = None
    cvssInfo_temporalScore: float = None
    cvssInfo_accessVector: str = None
    # End cvssInfo fields
    cvss3Info: dict = None
    # cvss3Info is parsed into below fields:
    cvss3Info_baseScore: float = None
    cvss3Info_temporalScore: float = None
    # End cvss3Info fields
    port: int = None
    status: str = None
    risk: int = None
    category: str = None
    discoveryType: BaseList[str] = None
    authType: BaseList[str] = None
    supportedBy: BaseList[str] = None
    product: BaseList[str] = None
    vendor: BaseList[str] = None
    cveids: BaseList[str] = None
    threatIntel: Union[dict[str, bool], BaseList[str]] = None
    software: BaseList[csSoftware] = None
    lastFound: Union[str, datetime] = None
    firstFound: Union[str, datetime] = None
    typeDetected: str = None
    scanType: Union[list[str], BaseList[str]] = None
    source: str = None
    reason: str = None
    imageResult: str = None
    containerResult: str = None
    containerSha: str = None
    vulnerability: str = None
    isExempted: bool = None
    vendorData: dict = None  # unknown

    def __post_init__(self):
        # Parse the software field into its components:
        if not DONT_EXPAND.flag:
            if self.software:
                bl = BaseList()
                for software in self.software:
                    bl.append(csSoftware.from_dict(software))
                setattr(self, "software", bl)

            for info_field, keys in {
                "cvssInfo": [
                    "cvssInfo_baseScore",
                    "cvssInfo_temporalScore",
                    "cvssInfo_accessVector",
                ],
                "cvss3Info": [
                    "cvss3Info_baseScore",
                    "cvss3Info_temporalScore",
                ],
            }.items():
                info_data = getattr(self, info_field, None)
                if info_data:
                    for key in keys:
                        if key in info_data:
                            setattr(self, key, info_data[key])
                delattr(self, info_field)

            for field in [
                "discoveryType",
                "authType",
                "supportedBy",
                "product",
                "vendor",
                "cveids",
                "scanType",
            ]:
                if getattr(self, field):
                    setattr(self, field, BaseList(getattr(self, field)))

            if self.threatIntel:
                # threatIntel is a dict with either true or null values
                intelBl = BaseList()
                for k, v in self.threatIntel.items():
                    if v:
                        intelBl.append(k)
                setattr(self, "threatIntel", intelBl)

        for dt_field in ["lastFound", "firstFound", "published"]:
            if getattr(self, dt_field):
                # published is already an int
                if isinstance(getattr(self, dt_field), str) or dt_field == "published":
                    setattr(
                        self,
                        dt_field,
                        datetime.fromtimestamp(int(getattr(self, dt_field)) / 1000),
                    )

    def __str__(self):
        return str(self.qid)

    def __int__(self):
        return self.qid

    def __float__(self):
        return float(self.qid)

    def __gt__(self, other):
        try:
            return self.qid > int(other.qid)
        except:
            return self.qid > int(other)

    def __ge__(self, other):
        try:
            return self.qid >= int(other.qid)
        except:
            return self.qid >= int(other)

    def __le__(self, other):
        try:
            return self.qid <= int(other.qid)
        except:
            return self.qid <= int(other)

    def __lt__(self, other):
        try:
            return self.qid < int(other.qid)
        except:
            return self.qid < int(other)
