"""
Simplified visualization for Runnable pipeline execution.

This module provides lightweight, reusable components that understand
the composite pipeline structure documented in the run logs.
"""

import json
from dataclasses import dataclass
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple, Union


@dataclass
class ParameterInfo:
    """Enhanced parameter representation with categorization."""

    key_inputs: List[str]  # Most important inputs for this step
    step_outputs: List[str]  # Parameters generated by this step
    context_params: List[str]  # Pipeline/inherited parameters
    iteration_vars: List[str]  # Map/loop variables (chunk, etc.)
    all_inputs: List[str]  # Complete input list
    all_outputs: List[str]  # Complete output list


@dataclass
class StepInfo:
    """Clean representation of a pipeline step."""

    name: str
    internal_name: str
    status: str
    step_type: str
    start_time: Optional[datetime]
    end_time: Optional[datetime]
    duration_ms: float
    level: int  # 0=top-level, 1=branch, 2=nested
    parent: Optional[str]
    branch: Optional[str]
    command: str
    command_type: str
    input_params: List[str]
    output_params: List[str]
    catalog_ops: Dict[str, List[Dict[str, str]]]
    # Enhanced parameter information
    parameters: Optional["ParameterInfo"] = None
    # Error and attempt information
    error_message: Optional[str] = None
    attempt_count: int = 0
    max_attempts: int = 1


class StepHierarchyParser:
    """Parse internal names to understand pipeline hierarchy."""

    @staticmethod
    def parse_internal_name(internal_name: str) -> Dict[str, str]:
        """
        Parse internal name into components.

        Examples:
        - "hello" -> {"step": "hello"}
        - "parallel_step.branch1.hello_stub" -> {
            "composite": "parallel_step",
            "branch": "branch1",
            "step": "hello_stub"
          }
        """
        parts = internal_name.split(".")

        if len(parts) == 1:
            return {"step": parts[0]}
        elif len(parts) == 2:
            return {"composite": parts[0], "branch": parts[1]}
        elif len(parts) == 3:
            return {"composite": parts[0], "branch": parts[1], "step": parts[2]}
        else:
            # Handle deeper nesting if needed
            return {
                "composite": parts[0],
                "branch": ".".join(parts[1:-1]),
                "step": parts[-1],
            }

    @staticmethod
    def get_step_level(internal_name: str) -> int:
        """Determine hierarchy level from internal name."""
        parts = internal_name.split(".")
        if len(parts) == 1:
            return 0  # Top-level step
        elif len(parts) == 2:
            return 1  # Branch level (for composite step parent)
        else:
            return 2  # Branch step


class TimelineExtractor:
    """Extract chronological timeline from run log."""

    def __init__(self, run_log_data: Dict[str, Any]):
        self.run_log_data = run_log_data
        self.dag_nodes = (
            run_log_data.get("run_config", {}).get("dag", {}).get("nodes", {})
        )

    def parse_time(self, time_str: str) -> Optional[datetime]:
        """Parse ISO timestamp string."""
        try:
            return datetime.fromisoformat(time_str) if time_str else None
        except (ValueError, TypeError):
            return None

    def get_step_timing(
        self, step_data: Dict[str, Any]
    ) -> Tuple[Optional[datetime], Optional[datetime], float]:
        """Extract timing from step attempts."""
        attempts = step_data.get("attempts", [])
        if not attempts:
            return None, None, 0

        attempt = attempts[0]
        start = self.parse_time(attempt.get("start_time"))
        end = self.parse_time(attempt.get("end_time"))

        if start and end:
            duration_ms = (end - start).total_seconds() * 1000
            return start, end, duration_ms

        return None, None, 0

    def find_dag_node(self, internal_name: str, clean_name: str) -> Dict[str, Any]:
        """Find DAG node info for command details."""
        # Try direct lookup first
        if clean_name in self.dag_nodes:
            return self.dag_nodes[clean_name]

        # For composite steps, look in branch structures
        hierarchy = StepHierarchyParser.parse_internal_name(internal_name)
        if "composite" in hierarchy:
            composite_node = self.dag_nodes.get(hierarchy["composite"], {})
            if composite_node.get("is_composite"):
                branches = composite_node.get("branches", {})
                branch_key = hierarchy.get("branch", "")

                if branch_key in branches:
                    branch_nodes = branches[branch_key].get("nodes", {})
                    if clean_name in branch_nodes:
                        return branch_nodes[clean_name]
                # For map nodes, the branch structure might be different
                elif "branch" in composite_node:  # Map node structure
                    branch_nodes = composite_node["branch"].get("nodes", {})
                    if clean_name in branch_nodes:
                        return branch_nodes[clean_name]

        return {}

    def _format_parameter_value(self, value: Any, kind: str) -> str:
        """Format parameter value for display."""
        if kind == "metric":
            if isinstance(value, (int, float)):
                return f"{value:.3g}"
            return str(value)

        if isinstance(value, str):
            # Truncate long strings
            if len(value) > 50:
                return f'"{value[:47]}..."'
            return f'"{value}"'
        elif isinstance(value, (list, tuple)):
            if len(value) > 3:
                preview = ", ".join(str(v) for v in value[:3])
                return f"[{preview}, ...+{len(value)-3}]"
            return str(value)
        elif isinstance(value, dict):
            if len(value) > 2:
                keys = list(value.keys())[:2]
                preview = ", ".join(f'"{k}": {value[k]}' for k in keys)
                return f"{{{preview}, ...+{len(value)-2}}}"
            return str(value)
        else:
            return str(value)

    def _categorize_parameters(
        self,
        input_params: List[str],
        output_params: List[str],
        step_name: str,
        internal_name: str,
    ) -> ParameterInfo:
        """Categorize parameters for smarter display."""
        # Common pipeline parameter patterns
        pipeline_params = {"integer", "floater", "stringer", "pydantic_param", "chunks"}

        # Categorize inputs
        key_inputs = []
        context_params = []
        iteration_vars = []

        for param in input_params:
            param_name = param.split("=")[0]

            # Check for iteration variables (chunk, index, etc.)
            if param_name in {"chunk", "index", "iteration", "item"}:
                iteration_vars.append(param)
            # Check for step-specific parameters (not empty and not pipeline defaults)
            elif (
                param_name not in pipeline_params
                and "=" in param
                and param.split("=")[1] not in ['""', "''", "[]", "{}"]
            ):
                key_inputs.append(param)
            # Pipeline/context parameters
            elif param_name in pipeline_params:
                context_params.append(param)
            else:
                context_params.append(param)

        # Limit key inputs to most relevant ones
        if len(key_inputs) > 3:
            key_inputs = key_inputs[:3]

        # Add iteration vars to key inputs if present
        key_inputs.extend(iteration_vars)

        # Identify step outputs (parameters generated by this step)
        step_outputs = []
        for param in output_params:
            param_name = param.split("=")[0]
            # Step outputs often have prefixes like "1_processed_python"
            if param_name.startswith(
                f"{step_name.split('.')[-1]}_"
            ) or param_name.startswith(f"{internal_name.split('.')[-1]}_"):
                step_outputs.append(param)
            else:
                step_outputs.append(param)

        return ParameterInfo(
            key_inputs=key_inputs,
            step_outputs=step_outputs,
            context_params=context_params,
            iteration_vars=iteration_vars,
            all_inputs=input_params,
            all_outputs=output_params,
        )

    def extract_timeline(self) -> List[StepInfo]:
        """Extract all steps in chronological order."""
        steps = []

        # Process top-level steps
        for step_name, step_data in self.run_log_data.get("steps", {}).items():
            step_info = self._create_step_info(step_name, step_data)
            steps.append(step_info)

            # Process branches if they exist
            branches = step_data.get("branches", {})
            for branch_name, branch_data in branches.items():
                # Add branch steps
                for sub_step_name, sub_step_data in branch_data.get(
                    "steps", {}
                ).items():
                    sub_step_info = self._create_step_info(
                        sub_step_name,
                        sub_step_data,
                        parent=step_name,
                        branch=branch_name,
                    )
                    steps.append(sub_step_info)

        # Sort by start time for chronological order
        return sorted(steps, key=lambda x: x.start_time or datetime.min)

    def _create_step_info(
        self,
        step_name: str,
        step_data: Dict[str, Any],
        parent: Optional[str] = None,
        branch: Optional[str] = None,
    ) -> StepInfo:
        """Create StepInfo from raw step data."""
        internal_name = step_data.get("internal_name", step_name)
        clean_name = step_data.get("name", step_name)

        # Get timing
        start, end, duration = self.get_step_timing(step_data)

        # Get command info from DAG
        dag_node = self.find_dag_node(internal_name, clean_name)
        command = dag_node.get("command", "")
        command_type = dag_node.get("command_type", "")

        # Extract parameters with detailed metadata (exclude pickled/object types)
        input_params = []
        output_params = []
        catalog_ops: Dict[str, List[Dict[str, str]]] = {"put": [], "get": []}

        attempts = step_data.get("attempts", [])
        if attempts:
            attempt = attempts[0]
            input_param_data = attempt.get("input_parameters", {})
            output_param_data = attempt.get("output_parameters", {})

            # Process input parameters (exclude object/pickled types)
            for name, param in input_param_data.items():
                if isinstance(param, dict):
                    kind = param.get("kind", "")
                    if kind in ("json", "metric"):
                        value = param.get("value", "")
                        # Format value for display
                        formatted_value = self._format_parameter_value(value, kind)
                        input_params.append(f"{name}={formatted_value}")
                    # Skip object/pickled parameters entirely

            # Process output parameters (exclude object/pickled types)
            for name, param in output_param_data.items():
                if isinstance(param, dict):
                    kind = param.get("kind", "")
                    if kind in ("json", "metric"):
                        value = param.get("value", "")
                        # Format value for display
                        formatted_value = self._format_parameter_value(value, kind)
                        output_params.append(f"{name}={formatted_value}")
                    # Skip object/pickled parameters entirely

        # Extract error information and attempt details
        error_message = None
        attempt_count = len(attempts)
        max_attempts = 1  # Default, will be updated from DAG if available

        # Get max_attempts from DAG node
        max_attempts = dag_node.get("max_attempts", 1)

        # Extract error message from failed attempts
        if attempts and step_data.get("status") == "FAIL":
            # Find the failed attempt (usually the last one)
            failed_attempt = None
            for attempt in reversed(attempts):
                if attempt.get("status") == "FAIL":
                    failed_attempt = attempt
                    break

            if failed_attempt and failed_attempt.get("message"):
                error_message = failed_attempt["message"].strip()

        # Extract catalog operations with detailed information
        catalog_data = step_data.get("data_catalog", [])
        for item in catalog_data:
            stage = item.get("stage", "")
            if stage in ("put", "get"):
                catalog_info = {
                    "name": item.get("name", ""),
                    "data_hash": item.get("data_hash", "")[:8] + "..."
                    if item.get("data_hash")
                    else "",  # Show first 8 chars
                    "catalog_path": item.get("catalog_relative_path", ""),
                    "stage": stage,
                }
                catalog_ops[stage].append(catalog_info)

        # Create parameter categorization for better display
        parameters = self._categorize_parameters(
            input_params, output_params, clean_name, internal_name
        )

        return StepInfo(
            name=clean_name,
            internal_name=internal_name,
            status=step_data.get("status", "UNKNOWN"),
            step_type=step_data.get("step_type", "task"),
            start_time=start,
            end_time=end,
            duration_ms=duration,
            level=StepHierarchyParser.get_step_level(internal_name),
            parent=parent,
            branch=branch,
            command=command,
            command_type=command_type,
            input_params=input_params,
            output_params=output_params,
            catalog_ops=catalog_ops,
            parameters=parameters,
            error_message=error_message,
            attempt_count=attempt_count,
            max_attempts=max_attempts,
        )


class SimpleVisualizer:
    """Simple, lightweight pipeline visualizer."""

    def __init__(self, run_log_path: Union[str, Path]):
        self.run_log_path = Path(run_log_path)
        self.run_log_data = self._load_run_log()
        self.extractor = TimelineExtractor(self.run_log_data)
        self.timeline = self.extractor.extract_timeline()

    def _load_run_log(self) -> Dict[str, Any]:
        """Load run log JSON."""
        if not self.run_log_path.exists():
            raise FileNotFoundError(f"Run log not found: {self.run_log_path}")

        with open(self.run_log_path, "r") as f:
            return json.load(f)

    def print_simple_timeline(self) -> None:
        """Print a clean console timeline."""
        run_id = self.run_log_data.get("run_id", "unknown")
        status = self.run_log_data.get("status", "UNKNOWN")

        print(f"\n🔄 Pipeline Timeline - {run_id}")
        print(f"Status: {status}")
        print("=" * 80)

        # Group by composite steps for better display
        current_composite = None
        current_branch = None

        for step in self.timeline:
            # Skip composite steps themselves (they have no timing)
            if (
                step.step_type in ["parallel", "map", "conditional"]
                and not step.start_time
            ):
                continue

            # Detect composite/branch changes
            hierarchy = StepHierarchyParser.parse_internal_name(step.internal_name)
            composite = hierarchy.get("composite")
            branch = hierarchy.get("branch")

            # Show composite header
            if composite and composite != current_composite:
                print(f"\n🔀 {composite} ({self._get_composite_type(composite)})")
                current_composite = composite
                current_branch = None

            # Show branch header
            if branch and branch != current_branch:
                branch_display = self._format_branch_name(composite or "", branch)
                print(f"  ├─ Branch: {branch_display}")
                current_branch = branch

            # Show step
            indent = (
                "  " if step.level == 0 else "    " if step.level == 1 else "      "
            )
            status_emoji = (
                "✅"
                if step.status == "SUCCESS"
                else "❌"
                if step.status == "FAIL"
                else "⏸️"
            )

            # Type icon
            type_icons = {
                "task": "⚙️",
                "stub": "📝",
                "success": "✅",
                "fail": "❌",
                "parallel": "🔀",
                "map": "🔁",
                "conditional": "🔀",
            }
            type_icon = type_icons.get(step.step_type, "⚙️")

            timing = f"({step.duration_ms:.1f}ms)" if step.duration_ms > 0 else ""

            print(f"{indent}{type_icon} {status_emoji} {step.name} {timing}")

            # Show error information for failed steps
            if step.status == "FAIL" and step.error_message:
                error_lines = step.error_message.split("\n")
                # Show first line of error, truncated if too long
                error_preview = (
                    error_lines[0][:100] + "..."
                    if len(error_lines[0]) > 100
                    else error_lines[0]
                )
                print(f"{indent}   💥 Error: {error_preview}")
                if step.attempt_count > 1:
                    print(
                        f"{indent}   🔄 Failed after {step.attempt_count}/{step.max_attempts} attempts"
                    )

            # Show metadata for tasks
            if step.step_type == "task" and (
                step.command
                or step.input_params
                or step.output_params
                or step.catalog_ops["put"]
                or step.catalog_ops["get"]
            ):
                if step.command:
                    cmd_short = (
                        step.command[:50] + "..."
                        if len(step.command) > 50
                        else step.command
                    )
                    print(f"{indent}   📝 {step.command_type.upper()}: {cmd_short}")

                # Show input parameters - compact horizontal display
                if step.input_params:
                    params_display = " • ".join(step.input_params)
                    print(f"{indent}   📥 {params_display}")

                # Show output parameters - compact horizontal display
                if step.output_params:
                    params_display = " • ".join(step.output_params)
                    print(f"{indent}   📤 {params_display}")

                # Show catalog operations - compact horizontal display
                if step.catalog_ops.get("put") or step.catalog_ops.get("get"):
                    catalog_items = []
                    if step.catalog_ops.get("put"):
                        catalog_items.extend(
                            [f"PUT:{item['name']}" for item in step.catalog_ops["put"]]
                        )
                    if step.catalog_ops.get("get"):
                        catalog_items.extend(
                            [f"GET:{item['name']}" for item in step.catalog_ops["get"]]
                        )
                    if catalog_items:
                        catalog_display = " • ".join(catalog_items)
                        print(f"{indent}   💾 {catalog_display}")

        print("=" * 80)

    def _get_composite_type(self, composite_name: str) -> str:
        """Get composite node type from DAG."""
        dag_nodes = (
            self.run_log_data.get("run_config", {}).get("dag", {}).get("nodes", {})
        )
        node = dag_nodes.get(composite_name, {})
        return node.get("node_type", "composite")

    def _format_branch_name(self, composite: str, branch: str) -> str:
        """Format branch name based on composite type."""
        # Remove composite prefix if present
        if branch.startswith(f"{composite}."):
            branch_clean = branch[len(f"{composite}.") :]
        else:
            branch_clean = branch

        # Check if it's a map iteration (numeric)
        if branch_clean.isdigit():
            return f"Iteration {branch_clean}"

        return branch_clean

    def print_execution_summary(self) -> None:
        """Print execution summary table."""
        run_id = self.run_log_data.get("run_id", "unknown")

        print(f"\n📊 Execution Summary - {run_id}")
        print("=" * 80)

        # Filter to actual executed steps (with timing)
        executed_steps = [step for step in self.timeline if step.start_time]

        if not executed_steps:
            print("No executed steps found")
            return

        # Table header
        print(f"{'Step':<30} {'Status':<10} {'Duration':<12} {'Type':<10}")
        print("-" * 80)

        total_duration = 0
        success_count = 0

        for step in executed_steps:
            status_emoji = (
                "✅"
                if step.status == "SUCCESS"
                else "❌"
                if step.status == "FAIL"
                else "⏸️"
            )
            duration_text = (
                f"{step.duration_ms:.1f}ms" if step.duration_ms > 0 else "0.0ms"
            )

            # Truncate long names
            display_name = step.name[:28] + ".." if len(step.name) > 30 else step.name

            print(
                f"{display_name:<30} {status_emoji}{step.status:<9} {duration_text:<12} {step.step_type:<10}"
            )

            total_duration += int(step.duration_ms)
            if step.status == "SUCCESS":
                success_count += 1

        print("-" * 80)
        success_rate = (
            (success_count / len(executed_steps)) * 100 if executed_steps else 0
        )
        overall_status = self.run_log_data.get("status", "UNKNOWN")
        overall_emoji = "✅" if overall_status == "SUCCESS" else "❌"

        print(
            f"Total Duration: {total_duration:.1f}ms | Success Rate: {success_rate:.1f}% | Status: {overall_emoji} {overall_status}"
        )

    def generate_html_timeline(
        self, output_path: Optional[Union[str, Path]] = None
    ) -> str:
        """
        Generate an interactive HTML timeline visualization.

        This creates a lightweight HTML version with:
        - Clean timeline layout
        - Hover tooltips with metadata
        - Expandable composite sections
        - Timing bars proportional to execution duration
        """
        run_id = self.run_log_data.get("run_id", "unknown")
        status = self.run_log_data.get("status", "UNKNOWN")

        # Calculate total timeline for proportional bars
        executed_steps = [step for step in self.timeline if step.start_time]
        if executed_steps:
            earliest = min(
                step.start_time for step in executed_steps if step.start_time
            )
            latest = max(step.end_time for step in executed_steps if step.end_time)
            total_duration_ms = (
                (latest - earliest).total_seconds() * 1000 if latest and earliest else 1
            )
        else:
            total_duration_ms = 1

        html_content = f"""<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pipeline Timeline - {run_id}</title>
    <style>
        * {{ margin: 0; padding: 0; box-sizing: border-box; }}

        body {{
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', system-ui, sans-serif;
            background: #f8fafc;
            color: #1e293b;
            line-height: 1.6;
        }}

        .header {{
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem 0;
            text-align: center;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }}

        .container {{
            max-width: 1600px;
            margin: 2rem auto;
            padding: 0 1rem;
            display: flex;
            gap: 2rem;
        }}

        .main-content {{
            flex: 1;
            min-width: 0;
        }}

        .sidebar {{
            width: 400px;
            position: sticky;
            top: 2rem;
            height: fit-content;
            max-height: calc(100vh - 4rem);
            overflow-y: auto;
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            transition: transform 0.3s ease;
        }}

        .sidebar.hidden {{
            transform: translateX(100%);
        }}

        .sidebar-header {{
            background: #f8fafc;
            padding: 1.5rem;
            border-bottom: 1px solid #e2e8f0;
            border-radius: 12px 12px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }}

        .sidebar-content {{
            padding: 1.5rem;
            max-height: 60vh;
            overflow-y: auto;
        }}

        .sidebar-section {{
            margin-bottom: 1.5rem;
        }}

        .sidebar-section h4 {{
            color: #374151;
            font-size: 0.875rem;
            font-weight: 600;
            margin-bottom: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }}

        .param-grid {{
            display: grid;
            gap: 0.5rem;
        }}

        .param-item {{
            padding: 0.5rem 0.75rem;
            border-radius: 6px;
            font-family: monospace;
            font-size: 0.8rem;
            border-left: 3px solid;
        }}

        .param-item.input {{
            background: #f0fdf4;
            border-left-color: #22c55e;
            color: #166534;
        }}

        .param-item.output {{
            background: #fef2f2;
            border-left-color: #ef4444;
            color: #991b1b;
        }}

        .param-item.context {{
            background: #f8fafc;
            border-left-color: #64748b;
            color: #475569;
        }}

        .param-item.iteration {{
            background: #fefce8;
            border-left-color: #eab308;
            color: #854d0e;
        }}

        .step-clickable {{
            cursor: pointer;
            transition: background-color 0.2s ease;
        }}

        .step-clickable:hover {{
            background-color: #f1f5f9 !important;
        }}

        .step-clickable.selected {{
            background-color: #dbeafe !important;
            border-left: 3px solid #3b82f6;
        }}

        .close-sidebar {{
            background: none;
            border: none;
            color: #64748b;
            cursor: pointer;
            font-size: 1.2rem;
            padding: 0.25rem;
        }}

        .close-sidebar:hover {{
            color: #374151;
        }}

        .timeline-card {{
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            overflow: hidden;
            margin-bottom: 2rem;
        }}

        .timeline-header {{
            background: #f8fafc;
            padding: 1.5rem;
            border-bottom: 1px solid #e2e8f0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }}

        .timeline-content {{
            padding: 1rem;
        }}

        .step-row {{
            display: grid;
            grid-template-columns: 300px 1fr 80px;
            align-items: center;
            padding: 0.5rem 0;
            border-bottom: 1px solid #f1f5f9;
            transition: background 0.2s ease;
            gap: 1rem;
            min-height: 40px;
            overflow: visible;
        }}

        .step-row:hover {{
            background: #f8fafc;
        }}

        .step-info {{
            display: flex;
            flex-direction: column;
            gap: 0.25rem;
            font-weight: 500;
            min-height: 24px;
            justify-content: flex-start;
            overflow: visible;
            word-wrap: break-word;
            overflow-wrap: break-word;
        }}

        .step-level-0 {{ padding-left: 0; }}
        .step-level-1 {{ padding-left: 1rem; }}
        .step-level-2 {{ padding-left: 2rem; }}

        .composite-header {{
            background: #e0f2fe !important;
            border-left: 4px solid #0277bd;
            font-weight: 600;
            color: #01579b;
        }}

        .branch-header {{
            background: #f3e5f5 !important;
            border-left: 4px solid #7b1fa2;
            font-weight: 600;
            color: #4a148c;
        }}

        .gantt-container {{
            position: relative;
            height: 30px;
            background: #f8fafc;
            border: 1px solid #e2e8f0;
            border-radius: 4px;
            min-width: 100%;
            overflow: hidden;
        }}

        .gantt-bar {{
            position: absolute;
            top: 3px;
            height: 24px;
            border-radius: 3px;
            transition: all 0.2s ease;
            cursor: pointer;
            border: 1px solid rgba(255,255,255,0.3);
        }}

        .gantt-bar:hover {{
            transform: scaleY(1.1);
            z-index: 10;
            box-shadow: 0 2px 8px rgba(0,0,0,0.2);
        }}

        .time-grid {{
            position: absolute;
            top: 0;
            bottom: 0;
            border-left: 1px solid #e2e8f0;
            opacity: 0.3;
        }}

        .time-scale {{
            position: relative;
            height: 20px;
            background: #f1f5f9;
            border-bottom: 1px solid #d1d5db;
            font-size: 0.75rem;
            color: #6b7280;
        }}

        .time-marker {{
            position: absolute;
            top: 0;
            height: 100%;
            display: flex;
            align-items: center;
            padding-left: 4px;
            font-weight: 500;
        }}

        .timeline-bar:hover {{
            transform: scaleY(1.1);
            z-index: 10;
        }}

        .bar-success {{ background: linear-gradient(90deg, #22c55e, #16a34a); }}
        .bar-fail {{ background: linear-gradient(90deg, #ef4444, #dc2626); }}
        .bar-unknown {{ background: linear-gradient(90deg, #f59e0b, #d97706); }}

        .duration-text {{
            font-family: monospace;
            font-size: 0.875rem;
            font-weight: 600;
        }}

        .duration-fast {{ color: #16a34a; }}
        .duration-medium {{ color: #f59e0b; }}
        .duration-slow {{ color: #dc2626; }}

        .status-success {{ color: #16a34a; }}
        .status-fail {{ color: #dc2626; }}
        .status-unknown {{ color: #f59e0b; }}

        .expandable {{
            cursor: pointer;
            user-select: none;
        }}

        .expandable:hover {{
            background: #e2e8f0 !important;
        }}

        .step-header.expandable {{
            padding: 0.25rem;
            border-radius: 4px;
            margin: -0.25rem;
        }}

        .step-header.expandable:hover {{
            background: #f1f5f9 !important;
        }}

        .collapsible-content {{
            max-height: none;
            overflow: visible;
            transition: max-height 0.3s ease;
        }}

        .collapsible-content.collapsed {{
            max-height: 0;
            overflow: hidden;
        }}

        .summary-grid {{
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-top: 2rem;
        }}

        .summary-card {{
            background: white;
            padding: 1.5rem;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            text-align: center;
        }}

        .summary-number {{
            font-size: 2rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
        }}

        .summary-label {{
            color: #64748b;
            font-size: 0.875rem;
        }}
    </style>
</head>
<body>
    <div class="header">
        <h1>🔄 Pipeline Timeline Visualization</h1>
        <p>Interactive execution analysis for {run_id}</p>
    </div>

    <div class="container">
        <div class="main-content">
            <div class="timeline-card">
                <div class="timeline-header">
                    <div>
                        <h2>Run ID: {run_id}</h2>
                        <p>Status: <span class="status-{status.lower()}">{status}</span></p>
                    </div>
                    <div>
                        <span class="duration-text">Total: {total_duration_ms:.1f}ms</span>
                    </div>
                </div>

                <div class="timeline-content">
                    {self._generate_html_timeline_rows(total_duration_ms)}
                </div>
            </div>

            {self._generate_html_summary()}
        </div>

        <div class="sidebar hidden" id="parameterSidebar">
            <div class="sidebar-header">
                <div>
                    <h3>📊 Step Details</h3>
                    <p id="sidebarStepName" style="color: #64748b; font-size: 0.875rem; margin: 0;">Select a step to view details</p>
                </div>
                <button class="close-sidebar" onclick="closeSidebar()">×</button>
            </div>
            <div class="sidebar-content" id="sidebarContent">
                <div class="sidebar-section">
                    <p style="color: #64748b; text-align: center; padding: 2rem;">Click on any step in the timeline to view its detailed parameters and metadata.</p>
                </div>
            </div>
        </div>
    </div>

    <script>
        let currentSelectedStep = null;

        // Collapsible sections
        document.querySelectorAll('.expandable').forEach(element => {{
            element.addEventListener('click', (e) => {{
                e.stopPropagation(); // Prevent triggering step selection
                let content;

                // Handle step metadata (using data-target)
                if (element.dataset.target) {{
                    content = document.getElementById(element.dataset.target);
                }} else {{
                    // Handle composite sections (using nextElementSibling)
                    content = element.nextElementSibling;
                }}

                if (content && content.classList.contains('collapsible-content')) {{
                    content.classList.toggle('collapsed');

                    // Update expand/collapse indicator
                    const indicator = element.querySelector('.expand-indicator');
                    if (indicator) {{
                        indicator.textContent = content.classList.contains('collapsed') ? '▶' : '▼';
                    }}
                }}
            }});
        }});

        function showStepDetails(internalName, startTime) {{
            const sidebar = document.getElementById('parameterSidebar');
            const stepNameElement = document.getElementById('sidebarStepName');
            const contentElement = document.getElementById('sidebarContent');

            // Get step data from embedded JSON
            const dataId = `step-data-${{internalName.replace(/\\./g, '-')}}-${{startTime}}`;
            const dataElement = document.getElementById(dataId);

            if (!dataElement) {{
                console.error('Step data not found for:', dataId);
                return;
            }}

            try {{
                const stepData = JSON.parse(dataElement.textContent);

                // Update step selection visual feedback
                if (currentSelectedStep) {{
                    currentSelectedStep.classList.remove('selected');
                }}

                const stepRow = event.currentTarget;
                stepRow.classList.add('selected');
                currentSelectedStep = stepRow;

                // Update sidebar content
                stepNameElement.textContent = `${{stepData.name}} (${{stepData.duration_ms.toFixed(1)}}ms)`;

                const perfEmoji = stepData.performance_class === 'fast' ? '🟢' :
                                stepData.performance_class === 'medium' ? '🟡' : '🔴';

                const statusEmoji = stepData.status === 'SUCCESS' ? '✅' :
                                  stepData.status === 'FAIL' ? '❌' : '⏸️';

                contentElement.innerHTML = `
                    <div class="sidebar-section">
                        <h4>📋 Step Information</h4>
                        <div style="display: grid; gap: 0.5rem; font-size: 0.875rem;">
                            <div><strong>Status:</strong> ${{statusEmoji}} ${{stepData.status}}</div>
                            <div><strong>Type:</strong> ${{stepData.step_type}}</div>
                            <div><strong>Duration:</strong> ${{perfEmoji}} ${{stepData.duration_ms.toFixed(1)}}ms</div>
                            <div><strong>Time:</strong> ${{stepData.start_time}} → ${{stepData.end_time}}</div>
                            ${{stepData.attempt_count > 1 ? `<div><strong>Attempts:</strong> ${{stepData.attempt_count}}/${{stepData.max_attempts}}</div>` : ''}}
                        </div>
                    </div>

                    ${{stepData.error_message ? `
                    <div class="sidebar-section">
                        <h4 style="color: #dc2626;">💥 Error Details</h4>
                        <div class="param-item" style="background: #fef2f2; border-left-color: #dc2626; color: #991b1b; white-space: pre-wrap; font-family: monospace; font-size: 0.75rem; line-height: 1.4;">
                            ${{stepData.error_message}}
                        </div>
                        ${{stepData.attempt_count > 1 ? `
                        <div style="margin-top: 0.5rem; font-size: 0.8rem; color: #6b7280;">
                            ⚠️ Step failed after ${{stepData.attempt_count}} attempt(s)
                        </div>
                        ` : ''}}
                    </div>
                    ` : ''}}


                    ${{stepData.command ? `
                    <div class="sidebar-section">
                        <h4>🔧 Command</h4>
                        <div class="param-item context">
                            <strong>${{stepData.command_type.toUpperCase()}}:</strong><br>
                            ${{stepData.command}}
                        </div>
                    </div>
                    ` : ''}}

                    ${{stepData.parameters.key_inputs.length > 0 ? `
                    <div class="sidebar-section">
                        <h4>🎯 Key Inputs</h4>
                        <div class="param-grid">
                            ${{stepData.parameters.key_inputs.map(param =>
                                `<div class="param-item input">${{param}}</div>`
                            ).join('')}}
                        </div>
                    </div>
                    ` : ''}}

                    ${{stepData.parameters.step_outputs.length > 0 ? `
                    <div class="sidebar-section">
                        <h4>📤 Step Outputs</h4>
                        <div class="param-grid">
                            ${{stepData.parameters.step_outputs.map(param =>
                                `<div class="param-item output">${{param}}</div>`
                            ).join('')}}
                        </div>
                    </div>
                    ` : ''}}

                    ${{stepData.parameters.iteration_vars.length > 0 ? `
                    <div class="sidebar-section">
                        <h4>🔁 Iteration Variables</h4>
                        <div class="param-grid">
                            ${{stepData.parameters.iteration_vars.map(param =>
                                `<div class="param-item iteration">${{param}}</div>`
                            ).join('')}}
                        </div>
                    </div>
                    ` : ''}}

                    ${{(stepData.catalog_ops.put.length > 0 || stepData.catalog_ops.get.length > 0) ? `
                    <div class="sidebar-section">
                        <h4>💾 Data Operations</h4>
                        ${{stepData.catalog_ops.get.length > 0 ? `
                        <div style="margin-bottom: 1rem;">
                            <h5 style="color: #059669; font-size: 0.8rem; font-weight: 600; margin-bottom: 0.5rem;">📥 CATALOG INPUTS</h5>
                            <div class="param-grid">
                                ${{stepData.catalog_ops.get.map(item =>
                                    `<div class="param-item input" style="font-family: monospace;">
                                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${{item.name}}</div>
                                        <div style="font-size: 0.7rem; opacity: 0.8;">Hash: ${{item.data_hash}}</div>
                                        <div style="font-size: 0.7rem; opacity: 0.8;">Path: ${{item.catalog_path}}</div>
                                    </div>`
                                ).join('')}}
                            </div>
                        </div>
                        ` : ''}}
                        ${{stepData.catalog_ops.put.length > 0 ? `
                        <div>
                            <h5 style="color: #dc2626; font-size: 0.8rem; font-weight: 600; margin-bottom: 0.5rem;">📤 CATALOG OUTPUTS</h5>
                            <div class="param-grid">
                                ${{stepData.catalog_ops.put.map(item =>
                                    `<div class="param-item output" style="font-family: monospace;">
                                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${{item.name}}</div>
                                        <div style="font-size: 0.7rem; opacity: 0.8;">Hash: ${{item.data_hash}}</div>
                                        <div style="font-size: 0.7rem; opacity: 0.8;">Path: ${{item.catalog_path}}</div>
                                    </div>`
                                ).join('')}}
                            </div>
                        </div>
                        ` : ''}}
                    </div>
                    ` : ''}}

                    ${{stepData.parameters.context_params.length > 0 ? `
                    <div class="sidebar-section">
                        <details>
                            <summary style="cursor: pointer; padding: 0.5rem 0; font-weight: 600; color: #374151;">
                                📊 Pipeline Context (${{stepData.parameters.context_params.length}} params)
                            </summary>
                            <div class="param-grid" style="margin-top: 0.5rem;">
                                ${{stepData.parameters.context_params.map(param =>
                                    `<div class="param-item context">${{param}}</div>`
                                ).join('')}}
                            </div>
                        </details>
                    </div>
                    ` : ''}}
                `;

                // Show sidebar
                sidebar.classList.remove('hidden');

            }} catch (error) {{
                console.error('Error parsing step data:', error);
                contentElement.innerHTML = `<div class="sidebar-section"><p style="color: #ef4444;">Error loading step details</p></div>`;
                sidebar.classList.remove('hidden');
            }}
        }}

        function closeSidebar() {{
            const sidebar = document.getElementById('parameterSidebar');
            sidebar.classList.add('hidden');

            // Clear step selection
            if (currentSelectedStep) {{
                currentSelectedStep.classList.remove('selected');
                currentSelectedStep = null;
            }}
        }}

        // Close sidebar when clicking outside
        document.addEventListener('click', (e) => {{
            const sidebar = document.getElementById('parameterSidebar');
            const isClickInsideSidebar = sidebar.contains(e.target);
            const isClickOnStep = e.target.closest('.step-clickable');

            if (!isClickInsideSidebar && !isClickOnStep && !sidebar.classList.contains('hidden')) {{
                closeSidebar();
            }}
        }});

        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => {{
            if (e.key === 'Escape') {{
                closeSidebar();
            }}
        }});
    </script>
</body>
</html>"""

        # Save to file if path provided
        if output_path:
            Path(output_path).write_text(html_content)
            print(f"HTML timeline saved to: {output_path}")

        return html_content

    def _generate_html_timeline_rows(self, total_duration_ms: float) -> str:
        """Generate HTML rows for the Gantt chart timeline display."""
        executed_steps = [
            step for step in self.timeline if step.start_time and step.end_time
        ]

        if not executed_steps:
            return "<div>No executed steps found</div>"

        # Calculate the absolute timeline
        earliest_start = min(
            step.start_time for step in executed_steps if step.start_time
        )
        latest_end = max(step.end_time for step in executed_steps if step.end_time)
        total_timeline_ms = (
            (latest_end - earliest_start).total_seconds() * 1000
            if latest_end and earliest_start
            else 1
        )

        # Generate time scale and Gantt rows
        time_scale_html = self._generate_time_scale(total_timeline_ms)
        gantt_rows_html = self._generate_gantt_rows(
            executed_steps, earliest_start, total_timeline_ms
        )

        return time_scale_html + "\n" + gantt_rows_html

    def _generate_time_scale(self, total_timeline_ms: float) -> str:
        """Generate the time scale header for the Gantt chart."""
        # Create time markers at regular intervals
        num_markers = 10
        interval_ms = total_timeline_ms / num_markers

        markers_html = []
        for i in range(num_markers + 1):
            time_ms = i * interval_ms
            position_percent = (time_ms / total_timeline_ms) * 100
            time_display = (
                f"{time_ms:.0f}ms" if time_ms < 1000 else f"{time_ms/1000:.1f}s"
            )

            markers_html.append(f"""
                <div class="time-marker" style="left: {position_percent:.1f}%;">
                    {time_display}
                </div>
            """)

            # Add grid line (except for the first one)
            if i > 0:
                markers_html.append(
                    f'<div class="time-grid" style="left: {position_percent:.1f}%;"></div>'
                )

        return f"""
            <div class="step-row" style="border-bottom: 2px solid #d1d5db;">
                <div class="step-info">
                    <strong>Timeline</strong>
                </div>
                <div class="time-scale">
                    {"".join(markers_html)}
                </div>
                <div></div>
            </div>
        """

    def _generate_gantt_rows(
        self, executed_steps: List, earliest_start, total_timeline_ms: float
    ) -> str:
        """Generate HTML rows for the Gantt chart display."""
        html_parts = []

        # Group by composite steps for better display
        current_composite = None
        current_branch = None

        for step in executed_steps:
            # Calculate timing positions for Gantt chart
            start_offset_ms = (step.start_time - earliest_start).total_seconds() * 1000
            start_percent = (start_offset_ms / total_timeline_ms) * 100
            width_percent = (step.duration_ms / total_timeline_ms) * 100

            # Detect composite/branch changes
            hierarchy = StepHierarchyParser.parse_internal_name(step.internal_name)
            composite = hierarchy.get("composite")
            branch = hierarchy.get("branch")

            # Show composite header
            if composite and composite != current_composite:
                composite_type = self._get_composite_type(composite)
                composite_id = f"composite-{composite.replace(' ', '-')}"

                html_parts.append(f"""
                    <div class="step-row composite-header expandable" data-composite="{composite}">
                        <div class="step-info step-level-0">
                            <div style="display: flex; align-items: center; gap: 0.5rem;">
                                <span class="expand-indicator">▼</span>
                                🔀 <strong>{composite}</strong> ({composite_type})
                            </div>
                        </div>
                        <div class="gantt-container"></div>
                        <div></div>
                    </div>
                """)

                # Start collapsible content
                html_parts.append(
                    f'<div class="collapsible-content" id="{composite_id}">'
                )
                current_composite = composite
                current_branch = None

            # Show branch header for parallel/map steps
            if branch and branch != current_branch:
                branch_display = self._format_branch_name(composite or "", branch)

                html_parts.append(f"""
                    <div class="step-row branch-header">
                        <div class="step-info step-level-1">
                            <div style="display: flex; align-items: center; gap: 0.5rem;">
                                🌿 <strong>Branch: {branch_display}</strong>
                            </div>
                        </div>
                        <div class="gantt-container"></div>
                        <div></div>
                    </div>
                """)
                current_branch = branch

            # Status styling
            status_class = step.status.lower()
            bar_class = (
                f"bar-{status_class}"
                if status_class in ["success", "fail"]
                else "bar-unknown"
            )

            # Type icon and status emoji
            type_icons = {
                "task": "⚙️",
                "stub": "📝",
                "success": "✅",
                "fail": "❌",
                "parallel": "🔀",
                "map": "🔁",
                "conditional": "🔀",
            }
            type_icon = type_icons.get(step.step_type, "⚙️")
            status_emoji = (
                "✅"
                if step.status == "SUCCESS"
                else "❌"
                if step.status == "FAIL"
                else "⏸️"
            )

            # Build parameter display - compact horizontal format
            param_info = []

            if step.input_params:
                params_text = " • ".join(step.input_params)
                param_info.append(
                    f'<div style="color: #059669; font-size: 0.7rem; margin-top: 0.2rem; font-family: monospace; word-break: break-all; line-height: 1.3;">📥 {params_text}</div>'
                )

            if step.output_params:
                params_text = " • ".join(step.output_params)
                param_info.append(
                    f'<div style="color: #dc2626; font-size: 0.7rem; margin-top: 0.2rem; font-family: monospace; word-break: break-all; line-height: 1.3;">📤 {params_text}</div>'
                )

            if step.catalog_ops.get("put") or step.catalog_ops.get("get"):
                catalog_items = []
                if step.catalog_ops.get("put"):
                    catalog_items.extend(
                        [f"PUT:{item['name']}" for item in step.catalog_ops["put"]]
                    )
                if step.catalog_ops.get("get"):
                    catalog_items.extend(
                        [f"GET:{item['name']}" for item in step.catalog_ops["get"]]
                    )
                if catalog_items:
                    catalog_text = " • ".join(catalog_items)
                    param_info.append(
                        f'<div style="color: #7c3aed; font-size: 0.7rem; margin-top: 0.2rem; font-family: monospace; word-break: break-all; line-height: 1.3;">💾 {catalog_text}</div>'
                    )

            # Generate sidebar data for this step
            sidebar_data = self._generate_step_sidebar_data(step)

            html_parts.append(f"""
                <div class="step-row step-clickable" onclick="showStepDetails('{step.internal_name}', '{step.start_time.isoformat()}')">
                    <div class="step-info step-level-{step.level}">
                        <div style="display: flex; align-items: center; gap: 0.5rem;">
                            {type_icon} {status_emoji} <strong>{step.name}</strong>
                            <span style="font-size: 0.7rem; color: #64748b;">📊 details</span>
                        </div>
                        {f'<div style="color: #64748b; font-size: 0.75rem; margin-top: 0.25rem;"><strong>{step.command_type.upper()}:</strong> {step.command[:40]}{"..." if len(step.command) > 40 else ""}</div>' if step.command else ''}
                        {'<div style="color: #059669; font-size: 0.65rem; margin-top: 0.25rem;">🎯 ' + (', '.join(step.parameters.key_inputs[:2]) if step.parameters and step.parameters.key_inputs else 'No key inputs') + ('...' if step.parameters and len(step.parameters.key_inputs) > 2 else '') + '</div>' if step.parameters else ''}
                    </div>
                    <div class="gantt-container">
                        <div class="gantt-bar {bar_class}"
                             style="left: {start_percent:.2f}%; width: {max(width_percent, 0.5):.2f}%;"
                             title="{step.name}: {step.duration_ms:.1f}ms">
                        </div>
                    </div>
                    <div style="font-family: monospace; font-size: 0.75rem; color: #6b7280;">
                        {step.duration_ms:.1f}ms
                    </div>
                </div>

                <!-- Hidden step data for sidebar -->
                <script type="application/json" id="step-data-{step.internal_name.replace('.', '-')}-{step.start_time.isoformat()}">{sidebar_data}</script>
            """)

        # Close any open composite sections
        if current_composite:
            html_parts.append("</div>")  # Close collapsible-content

        return "\n".join(html_parts)

    def _generate_step_sidebar_data(self, step: StepInfo) -> str:
        """Generate JSON data for step sidebar display."""
        import json

        if not step.parameters:
            return json.dumps({"error": "No parameter data available"})

        # Performance classification
        perf_class = "fast"
        if step.duration_ms > 1000:
            perf_class = "slow"
        elif step.duration_ms > 100:
            perf_class = "medium"

        data = {
            "name": step.name,
            "internal_name": step.internal_name,
            "status": step.status,
            "step_type": step.step_type,
            "command": step.command,
            "command_type": step.command_type,
            "duration_ms": step.duration_ms,
            "performance_class": perf_class,
            "start_time": step.start_time.strftime("%H:%M:%S.%f")[:-3]
            if step.start_time
            else None,
            "end_time": step.end_time.strftime("%H:%M:%S.%f")[:-3]
            if step.end_time
            else None,
            "parameters": {
                "key_inputs": step.parameters.key_inputs,
                "step_outputs": step.parameters.step_outputs,
                "context_params": step.parameters.context_params,
                "iteration_vars": step.parameters.iteration_vars,
            },
            "catalog_ops": step.catalog_ops,
            "error_message": step.error_message,
            "attempt_count": step.attempt_count,
            "max_attempts": step.max_attempts,
        }

        return json.dumps(data)

    def _generate_html_summary(self) -> str:
        """Generate HTML summary cards."""
        executed_steps = [step for step in self.timeline if step.start_time]
        total_duration = sum(step.duration_ms for step in executed_steps)
        success_count = sum(1 for step in executed_steps if step.status == "SUCCESS")
        success_rate = (
            (success_count / len(executed_steps)) * 100 if executed_steps else 0
        )

        # Find slowest step
        slowest_step = (
            max(executed_steps, key=lambda x: x.duration_ms) if executed_steps else None
        )

        return f"""
        <div class="summary-grid">
            <div class="summary-card">
                <div class="summary-number status-success">{len(executed_steps)}</div>
                <div class="summary-label">Total Steps</div>
            </div>
            <div class="summary-card">
                <div class="summary-number duration-medium">{total_duration:.1f}ms</div>
                <div class="summary-label">Total Duration</div>
            </div>
            <div class="summary-card">
                <div class="summary-number {'status-success' if success_rate == 100 else 'status-fail'}">{success_rate:.1f}%</div>
                <div class="summary-label">Success Rate</div>
            </div>
            <div class="summary-card">
                <div class="summary-number duration-slow">{'%.1fms' % slowest_step.duration_ms if slowest_step else 'N/A'}</div>
                <div class="summary-label">Slowest Step<br><small>{slowest_step.name if slowest_step else 'N/A'}</small></div>
            </div>
        </div>
        """


def visualize_simple(
    run_id: str, show_summary: bool = False, output_html: Optional[str] = None
) -> None:
    """
    Simple visualization of a pipeline run.

    Args:
        run_id: Run ID to visualize
        show_summary: Whether to show execution summary (deprecated, timeline has enough info)
        output_html: Optional path to save HTML timeline
    """
    # Find run log file
    run_log_dir = Path(".run_log_store")
    log_file = run_log_dir / f"{run_id}.json"

    if not log_file.exists():
        # Try partial match
        matching_files = [f for f in run_log_dir.glob("*.json") if run_id in f.stem]
        if matching_files:
            log_file = matching_files[0]
        else:
            print(f"❌ Run log not found for: {run_id}")
            return

    print(f"📊 Visualizing: {log_file.stem}")

    viz = SimpleVisualizer(log_file)
    viz.print_simple_timeline()

    if show_summary:
        viz.print_execution_summary()

    # Generate HTML if requested
    if output_html:
        viz.generate_html_timeline(output_html)


def generate_html_timeline(
    run_id: str, output_file: str, open_browser: bool = True
) -> None:
    """
    Generate HTML timeline for a specific run ID.

    Args:
        run_id: The run ID to visualize
        output_file: Output HTML file path
        open_browser: Whether to open the result in browser
    """
    from pathlib import Path

    # Find run log file
    run_log_dir = Path(".run_log_store")
    log_file = run_log_dir / f"{run_id}.json"

    if not log_file.exists():
        # Try partial match
        matching_files = [f for f in run_log_dir.glob("*.json") if run_id in f.stem]
        if matching_files:
            log_file = matching_files[0]
        else:
            print(f"❌ Run log not found for: {run_id}")
            return

    print(f"🌐 Generating HTML timeline for: {log_file.stem}")

    # Create visualizer and generate HTML
    viz = SimpleVisualizer(log_file)
    viz.generate_html_timeline(output_file)

    if open_browser:
        import webbrowser

        file_path = Path(output_file).absolute()
        print(f"🌐 Opening timeline in browser: {file_path.name}")
        webbrowser.open(file_path.as_uri())


if __name__ == "__main__":
    import sys

    if len(sys.argv) > 1:
        if len(sys.argv) > 2 and sys.argv[2].endswith(".html"):
            # Generate HTML: python viz_simple.py <run_id> <output.html>
            generate_html_timeline(sys.argv[1], sys.argv[2])
        else:
            # Console visualization: python viz_simple.py <run_id>
            visualize_simple(sys.argv[1])
    else:
        print("Usage: python viz_simple.py <run_id> [output.html]")
