import numpy as np
import pytest

from fiberpy.mechanics import A2Eij, FiberComposite

# RVE data for Moldflow A218V50
rve_data = {
    "rho0": 1.14e-9,
    "E0": 631.66,
    "nu0": 0.42925,
    "alpha0": 5.86e-5,
    "rho1": 2.55e-9,
    "E1": 72000,
    "nu1": 0.22,
    "alpha1": 5e-6,
    "mf": 0.5,
    "aspect_ratio": 17.983,
}
fiber = FiberComposite(rve_data)

# Reference data generated by Digimat
Eij_Digimat_2d = np.array(
    [
        [
            3.92735e03,
            3.92735e03,
            1.88993e03,
            1.45310e03,
            4.05540e02,
            4.05540e02,
            3.51416e-01,
            4.52022e-01,
            2.17523e-01,
        ],
        [
            4.30739e03,
            3.57553e03,
            1.88402e03,
            1.44280e03,
            4.04380e02,
            4.06710e02,
            3.83972e-01,
            4.76465e-01,
            1.87011e-01,
        ],
        [
            4.71828e03,
            3.24903e03,
            1.86616e03,
            1.41190e03,
            4.03220e02,
            4.07870e02,
            4.15790e-01,
            5.00628e-01,
            1.59619e-01,
        ],
        [
            5.16350e03,
            2.94569e03,
            1.83650e03,
            1.36030e03,
            4.02050e02,
            4.09030e02,
            4.45986e-01,
            5.24343e-01,
            1.35381e-01,
        ],
        [
            5.64650e03,
            2.66348e03,
            1.79517e03,
            1.28790e03,
            4.00890e02,
            4.10200e02,
            4.73286e-01,
            5.47486e-01,
            1.14346e-01,
        ],
        [
            6.17084e03,
            2.40032e03,
            1.74230e03,
            1.19470e03,
            3.99730e02,
            4.11360e02,
            4.95920e-01,
            5.70016e-01,
            9.65277e-02,
        ],
        [
            6.73974e03,
            2.15467e03,
            1.67830e03,
            1.08070e03,
            3.98570e02,
            4.12520e02,
            5.11390e-01,
            5.91794e-01,
            8.19434e-02,
        ],
        [
            7.35542e03,
            1.92498e03,
            1.60340e03,
            9.45940e02,
            3.97400e02,
            4.13680e02,
            5.15989e-01,
            6.12829e-01,
            7.06132e-02,
        ],
        [
            8.01847e03,
            1.70961e03,
            1.51795e03,
            7.90370e02,
            3.96240e02,
            4.14850e02,
            5.04619e-01,
            6.33106e-01,
            6.24842e-02,
        ],
        [
            8.72648e03,
            1.50733e03,
            1.42233e03,
            6.14080e02,
            3.95080e02,
            4.16010e02,
            4.69675e-01,
            6.52636e-01,
            5.75241e-02,
        ],
        [
            9.47002e03,
            1.31677e03,
            1.31677e03,
            4.17170e02,
            3.93910e02,
            4.17170e02,
            4.00298e-01,
            6.71456e-01,
            5.56600e-02,
        ],
    ]
)
Eij_Digimat_3d = np.array(
    [
        [
            2.61296e03,
            2.61315e03,
            2.61296e03,
            9.61870e02,
            9.61870e02,
            9.61940e02,
            3.58230e-01,
            3.58255e-01,
            3.58301e-01,
        ],
        [
            2.98581e03,
            2.45197e03,
            2.45197e03,
            9.91210e02,
            8.84030e02,
            9.91210e02,
            3.78803e-01,
            3.86391e-01,
            3.11077e-01,
        ],
        [
            3.42892e03,
            2.30287e03,
            2.30295e03,
            9.99500e02,
            8.11290e02,
            9.99440e02,
            3.96589e-01,
            4.17909e-01,
            2.66343e-01,
        ],
        [
            3.94302e03,
            2.16221e03,
            2.16207e03,
            9.87760e02,
            7.43410e02,
            9.87820e02,
            4.11234e-01,
            4.51260e-01,
            2.25542e-01,
        ],
        [
            4.52855e03,
            2.02706e03,
            2.02706e03,
            9.57170e02,
            6.80460e02,
            9.57170e02,
            4.22578e-01,
            4.85352e-01,
            1.89153e-01,
        ],
        [
            5.18506e03,
            1.89661e03,
            1.89673e03,
            9.08520e02,
            6.22170e02,
            9.08470e02,
            4.30197e-01,
            5.19321e-01,
            1.57327e-01,
        ],
        [
            5.91133e03,
            1.77060e03,
            1.77056e03,
            8.42520e02,
            5.68240e02,
            8.42570e02,
            4.33604e-01,
            5.52602e-01,
            1.29899e-01,
        ],
        [
            6.70593e03,
            1.64880e03,
            1.64880e03,
            7.59950e02,
            5.18740e02,
            7.59950e02,
            4.32738e-01,
            5.84774e-01,
            1.06398e-01,
        ],
        [
            7.56615e03,
            1.53196e03,
            1.53196e03,
            6.61270e02,
            4.73310e02,
            6.61250e02,
            4.27111e-01,
            6.15479e-01,
            8.64795e-02,
        ],
        [
            8.48867e03,
            1.42098e03,
            1.42098e03,
            5.46900e02,
            4.31750e02,
            5.46920e02,
            4.16393e-01,
            6.44459e-01,
            6.97028e-02,
        ],
        [
            9.47002e03,
            1.31677e03,
            1.31677e03,
            4.17170e02,
            3.93910e02,
            4.17170e02,
            4.00298e-01,
            6.71456e-01,
            5.56600e-02,
        ],
    ]
)


@pytest.mark.parametrize("fiber_orientation_type", ["2d", "3d"])
def test_elastic_properties(fiber_orientation_type):
    if fiber_orientation_type == "2d":
        ref = Eij_Digimat_2d
        a11 = np.linspace(0.5, 1, len(ref))

        def a_from_a11(a11):
            return np.array([a11, 1 - a11, 0])

    else:
        ref = Eij_Digimat_3d
        a11 = np.linspace(1 / 3, 1, len(ref))

        def a_from_a11(a11):
            return np.array([a11, (1 - a11) / 2, (1 - a11) / 2])

    Eij = np.zeros((len(a11), 9))
    for i in range(len(a11)):
        a = a_from_a11(a11[i])
        Eij[i, :] = A2Eij(fiber.ABar(a))
        assert np.allclose(Eij[i], ref[i], rtol=1e-3)


def test_thermal_properties():
    alpha = fiber.alphaBar(fiber.TandonWeng())
    ref = (8.29181165e-06, 5.44419385e-05, 5.44419385e-05)
    assert np.allclose(alpha, ref)
