from zoneinfo import ZoneInfo

import pytest
from datetime import time, date, datetime
from temporals.pydatetime.periods import TimePeriod, DatePeriod, AbsolutePeriod, WallClockPeriod
from temporals.exceptions import TimeAmbiguityError


class TestAbsolutePeriod:

    def test_constructor_valid(self):
        # Valid objects
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)
        assert isinstance(self.period, AbsolutePeriod)

    def test_constructor_invalid(self):
        # Time
        self.start = time(13, 1, 1)
        self.end = time(14, 1, 2)
        with pytest.raises(ValueError):
            AbsolutePeriod(start=self.start, end=self.end)

        # Date
        self.start = date(2024, 1, 1)
        self.end = date(2024, 1, 1)
        with pytest.raises(ValueError):
            AbsolutePeriod(start=self.start, end=self.end)

    def test_nonrepeating_time(self):
        """ This tests the internal _time_repeats method """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.time = time(7, 0)
        assert self.period._time_repeats(self.time) is False
        self.time = time(13, 0)
        assert self.period._time_repeats(self.time) is False

        self.period1 = TimePeriod(start=time(7, 0), end=time(10, 0))
        assert self.period._time_repeats(self.period1) is False
        self.period1 = TimePeriod(start=time(9, 0), end=time(13, 0))
        assert self.period._time_repeats(self.period1) is False

        # 48h period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 3, 11, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # TimePeriod that starts before it and ends after it
        self.period1 = TimePeriod(start=time(7, 0), end=time(12, 0))
        assert self.period._time_repeats(self.period1) is False

    def test_repeating_time(self):
        """ This tests the internal _time_repeat method """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.time = time(10, 0)
        assert self.period._time_repeats(self.time) is True
        self.period1 = TimePeriod(start=time(9, 0), end=time(11, 0))
        assert self.period._time_repeats(self.period1) is True

        # More than 24h period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 3, 11, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.time = time(7, 0)
        assert self.period._time_repeats(self.time) is True
        # TimePeriod that starts before it but also ends before it
        self.period1 = TimePeriod(start=time(7, 0), end=time(10, 0))
        assert self.period._time_repeats(self.period1) is True
        # TimePeriod that starts after it but also ends after it
        self.period1 = TimePeriod(start=time(9, 0), end=time(12, 0))
        assert self.period._time_repeats(self.period1) is True
        # TimePeriod that starts and ends within it
        self.period1 = TimePeriod(start=time(9, 0), end=time(10, 0))
        assert self.period._time_repeats(self.period1) is True
        # TimePeriod that has start time and end time after this period - not repeating
        self.period1 = TimePeriod(start=time(7, 0), end=time(12, 0))
        assert self.period._time_repeats(self.period1) is False

    def test_timeperiod_eq(self):
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period1 = TimePeriod(start=self.start.time(), end=self.end.time())
        self.period2 = AbsolutePeriod(start=self.start, end=self.end)
        assert self.period1 == self.period2

    def test_dateperiod_eq(self):
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period1 = DatePeriod(start=self.start.date(), end=self.end.date())
        self.period2 = AbsolutePeriod(start=self.start, end=self.end)
        assert self.period1 == self.period2

        self.different_end = date(2024, 1, 10)
        self.period2 = DatePeriod(start=self.start.date(), end=self.different_end)
        assert self.period1 != self.period2

    def test_datetimeperiod_eq(self):
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)
        self.start_dt = datetime(2024, 1, 1, 8, 0)
        self.end_dt = datetime(2024, 1, 1, 12, 0)
        self.period_dt = AbsolutePeriod(start=self.start_dt, end=self.end_dt)
        assert self.period == self.period_dt

    def test_invalid_eq(self):
        self.random_dt = datetime(2024, 1, 1, 8, 0)
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)
        assert self.random_dt != self.period

    def test_timeshift_eq(self):
        # shift forward by 1 hour
        period = AbsolutePeriod(start=datetime(2025, 3, 30, 1, 0, tzinfo=ZoneInfo(key='Europe/Paris')),
                                end=datetime(2025, 3, 31, 2, 0, tzinfo=ZoneInfo(key='Europe/Paris')))
        other_period = WallClockPeriod(start=datetime(2025, 3, 30, 1, 0, tzinfo=ZoneInfo(key='Europe/Paris')),
                                       end=datetime(2025, 3, 31, 2, 0, tzinfo=ZoneInfo(key='Europe/Paris')))
        assert not period == other_period

    def test_valid_membership_time(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time test
        self.time = time(10, 0)
        assert self.time in self.period

    def test_valid_membership_date(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Date test
        self.date = date(2024, 1, 1)
        assert self.date in self.period

    def test_valid_membership_datetime(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Datetime test
        self.datetime = datetime(2024, 1, 1, 12, 0)
        assert self.datetime in self.period

    def test_valid_membership_timeperiod(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time period test
        self.time_period = TimePeriod(
            start=time(9, 0),
            end=time(11, 0)
        )
        assert self.time_period in self.period

    def test_valid_membership_absolute(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # AbsolutePeriod test
        self.dt_period = AbsolutePeriod(
            start=datetime(2024, 1, 1, 9, 0),
            end=datetime(2024, 1, 1, 11, 0)
        )
        assert self.dt_period in self.period

    def test_valid_membership_24h_time(self):
        # 24h period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time test
        self.time = time(7, 0)
        assert self.time in self.period
        self.time = time(13, 0)
        assert self.time in self.period

    def test_valid_membership_24h_date(self):
        # 24h period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Date test
        self.date = date(2024, 1, 2)
        assert self.date in self.period

    def test_valid_membership_24h_timeperiod(self):
        # 24h period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time period test
        self.time_period = TimePeriod(
            start=time(7, 0),
            end=time(11, 0)
        )
        assert self.time_period in self.period

        self.time_period = TimePeriod(
            start=time(9, 0),
            end=time(13, 0)
        )
        assert self.time_period in self.period

    def test_invalid_membership_time(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time test
        self.time = time(17, 0)
        assert self.time not in self.period

    def test_invalid_membership_date(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Date test
        self.date = date(2024, 1, 2)
        assert self.date not in self.period

    def test_invalid_membership_datetime(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Datetime test
        self.datetime = datetime(2024, 1, 1, 13, 0)
        assert self.datetime not in self.period

    def test_invalid_membership_overlap(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time period test - overlapping
        self.time_period = TimePeriod(
            start=time(9, 0),
            end=time(14, 0)
        )
        assert self.time_period not in self.period

    def test_invalid_membership_equal(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)
        self.eq_period_wc = WallClockPeriod(start=self.start, end=self.end)
        self.eq_period_abs = AbsolutePeriod(start=self.start, end=self.end)

        assert self.eq_period_wc not in self.period
        assert self.eq_period_abs not in self.period

    def test_is_before(self):
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)
        assert self.period.is_before(date(2024, 1, 2)) is True
        assert self.period.is_before(date(2024, 1, 1)) is False
        assert self.period.is_before(datetime(2024, 1, 1, 12, 0)) is True
        assert self.period.is_before(datetime(2024, 1, 1, 11, 59)) is False

        self.other_start = date(2024, 1, 2)
        self.other_end = date(2024, 1, 3)
        self.other_period = DatePeriod(start=self.other_start, end=self.other_end)
        assert self.period.is_before(self.other_period) is True

        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 5, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)
        self.other_start = date(2024, 1, 5)
        self.other_end = date(2024, 1, 10)
        self.other_period = DatePeriod(start=self.other_start, end=self.other_end)
        assert self.period.is_before(self.other_period) is False

        self.other_start = datetime(2024, 1, 5, 12, 0)
        self.other_end = datetime(2024, 3, 1, 12, 0)
        self.other_period = AbsolutePeriod(start=self.other_start, end=self.other_end)
        assert self.period.is_before(self.other_period) is True

    def test_is_after(self):
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)
        assert self.period.is_after(date(2023, 12, 31)) is True
        assert self.period.is_after(date(2024, 1, 1)) is False
        assert self.period.is_after(datetime(2024, 1, 1, 8, 0)) is True
        assert self.period.is_after(datetime(2024, 1, 1, 8, 1)) is False

        self.other_start = date(2023, 12, 1)
        self.other_end = date(2023, 12, 31)
        self.other_period = DatePeriod(start=self.other_start, end=self.other_end)
        assert self.period.is_after(self.other_period) is True

        self.other_start = date(2023, 12, 1)
        self.other_end = date(2024, 1, 1)
        self.other_period = DatePeriod(start=self.other_start, end=self.other_end)
        assert self.period.is_after(self.other_period) is False

        self.other_start = datetime(2024, 1, 1, 7, 0)
        self.other_end = datetime(2024, 1, 1, 8, 0)
        self.other_period = AbsolutePeriod(start=self.other_start, end=self.other_end)
        assert self.period.is_after(self.other_period) is True

    def test_overlaps_time(self):
        """
            Same day:

           0800 Period 1         1200
            |=====================|
                                |===================|
                              1100 Period 2        1700

            Period 1 is overlapped by Period 2
            Period 2 overlaps Period 1
            Period 1 does not overlap Period 2
            Period 2 is not overlapped by Period 1
        """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = time(10, 0)
        self.other_end = time(14, 0)
        self.other_period = TimePeriod(start=self.other_start, end=self.other_end)

        assert self.period.overlapped_by(self.other_period) is True
        assert self.other_period.overlaps_with(self.period) is True
        assert self.period.overlaps_with(self.other_period) is False
        assert self.other_period.overlapped_by(self.period) is False

    def test_overlaps_date(self):
        """
        2024-01-01 0800        Period 1         2024-01-03 1200
            |=========================================|
                                |===================================================|
                            2024-01-02                   Period 2              2024-01-04

            Period 1 is overlapped by Period 2
            Period 2 overlaps Period 1
            Period 1 does not overlap Period 2
            Period 2 is not overlapped by Period 1
        """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 3, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = date(2024, 1, 2)
        self.other_end = date(2024, 1, 4)
        self.other_period = DatePeriod(start=self.other_start, end=self.other_end)

        assert self.period.overlapped_by(self.other_period) is True
        assert self.other_period.overlaps_with(self.period) is True
        assert self.period.overlaps_with(self.other_period) is False
        assert self.other_period.overlapped_by(self.period) is False

    def test_overlaps_datetime(self):
        """
            Same day:

           0800 Period 1         1200
            |=====================|
                                |===================|
                              1100 Period 2        1700

            Period 1 is overlapped by Period 2
            Period 2 overlaps Period 1
            Period 1 does not overlap Period 2
            Period 2 is not overlapped by Period 1
        """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = datetime(2024, 1, 1, 11, 0)
        self.other_end = datetime(2024, 1, 1, 17, 0)
        self.other_period = AbsolutePeriod(start=self.other_start, end=self.other_end)
        assert self.period.overlapped_by(self.other_period) is True
        assert self.other_period.overlaps_with(self.period) is True
        assert self.period.overlaps_with(self.other_period) is False
        assert self.other_period.overlapped_by(self.period) is False

        """
            Different day:

            2024-01-01 0800 Period 1        2024-02-01 1200
                    |==============================|
                                        |===============================|
                                    2024-01-15 0800 Period 2  2024-03-01 1200

            Period 1 is overlapped by Period 2
            Period 2 overlaps Period 1
            Period 1 does not overlap Period 2
            Period 2 is not overlapped by Period 1
        """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 2, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = datetime(2024, 1, 15, 11, 0)
        self.other_end = datetime(2024, 3, 1, 12, 0)
        self.other_period = AbsolutePeriod(start=self.other_start, end=self.other_end)
        assert self.period.overlapped_by(self.other_period) is True
        assert self.other_period.overlaps_with(self.period) is True
        assert self.period.overlaps_with(self.other_period) is False
        assert self.other_period.overlapped_by(self.period) is False

    def test_ambiguity_time(self):
        """ All tests below must raise a TimeAmbiguityError """

        # Membership ambiguity
        """ 2 day long period
                2024-01-01 0800                                         2024-01-02 1200
                    /==========================================================/
                       /=======/                                       /======/
                    1000     1200                                   1000     1200
        """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.time = time(9, 0)
        with pytest.raises(TimeAmbiguityError):
            self.time in self.period

        """ 1 day long period 
                2024-01-01 0800                                         2024-01-02 1200
                    /==========================================================/
                       /=======/                                       /======/
                    1000     1200                                   1000     1200
        """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = time(10, 0)
        self.other_end = time(12, 0)
        self.other_period = TimePeriod(start=self.other_start, end=self.other_end)

        with pytest.raises(TimeAmbiguityError):
            self.other_period in self.period

        # Overlaps with ambiguity
        """ 2 days long period 
                2024-01-01 0800                                         2024-01-02 1200
                    /==========================================================/
                        /=========/                                        /=========/
                    1000       1400                                      1000      1400
        """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 3, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = time(10, 0)
        self.other_end = time(14, 0)
        self.other_period = TimePeriod(start=self.other_start, end=self.other_end)

        with pytest.raises(TimeAmbiguityError):
            self.period.overlaps_with(self.other_period)

        self.other_start = time(6, 0)
        self.other_end = time(14, 0)
        self.other_period = TimePeriod(start=self.other_start, end=self.other_end)

        with pytest.raises(TimeAmbiguityError):
            self.period.overlaps_with(self.other_period)

        """ Overnight period 
                2024-01-01 1000              Midnight                   2024-01-02 0800
                    /==========================================================/
                /=======/                                                   /======/
              0700    1200                                                0700    1200
        """
        self.start = datetime(2024, 1, 1, 10, 0)
        self.end = datetime(2024, 1, 2, 8, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = time(7, 0)
        self.other_end = time(12, 0)
        self.other_period = TimePeriod(start=self.other_start, end=self.other_end)

        with pytest.raises(TimeAmbiguityError):
            self.period.overlaps_with(self.other_period)

        # Overlapped by ambiguity
        """ 2 days long period 
                2024-01-01 0800                                         2024-01-02 1200
                    /==========================================================/
                        /=========/                                        /=========/
                      1000       1400                                    1000      1400
        """
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 2, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = time(10, 0)
        self.other_end = time(14, 0)
        self.other_period = TimePeriod(start=self.other_start, end=self.other_end)

        with pytest.raises(TimeAmbiguityError):
            self.period.overlapped_by(self.other_period)

        """ Overnight period 
                2024-01-01 0600                Midnight               2024-01-02 1000
                    /==========================================================/
                        /=========/                                        /=========/
                      1000       1400                                    1000      1400
        """
        self.start = datetime(2024, 1, 1, 6, 0)
        self.end = datetime(2024, 1, 2, 10, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.other_start = time(10, 0)
        self.other_end = time(14, 0)
        self.other_period = TimePeriod(start=self.other_start, end=self.other_end)

        with pytest.raises(TimeAmbiguityError):
            self.period.overlapped_by(self.other_period)

    def test_get_interim(self):
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)
        assert self.period.get_interim(datetime(2023, 12, 15, 13, 0)) == AbsolutePeriod(
            datetime(2023, 12, 15, 13, 0), datetime(2024, 1, 1, 8, 0)
        )
        assert self.period.get_interim(datetime(2024, 2, 20, 8, 0)) == AbsolutePeriod(
            datetime(2024, 1, 1, 12, 0), datetime(2024, 2, 20, 8, 0)
        )

        self.other_period = AbsolutePeriod(
            start=datetime(2023, 12, 15, 10, 0),
            end=datetime(2023, 12, 30, 8, 0)
        )
        assert self.period.get_interim(self.other_period) == AbsolutePeriod(datetime(2023, 12, 30, 8, 0),
                                                                            datetime(2024, 1, 1, 8, 0)
                                                                            )

        self.other_period = AbsolutePeriod(
            start=datetime(2024, 1, 1, 12, 0, 5),
            end=datetime(2024, 1, 1, 16, 0)
        )
        assert self.period.get_interim(self.other_period) == AbsolutePeriod(datetime(2024, 1, 1, 12, 0, 0),
                                                                            datetime(2024, 1, 1, 12, 0, 5)
                                                                            )

    def test_get_overlap(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time period test
        self.time_period = TimePeriod(
            start=time(9, 0),
            end=time(13, 0)
        )
        self.overlap = self.period.get_overlap(self.time_period)
        assert self.overlap == TimePeriod(start=time(9, 0), end=time(12, 0))
        # TODO: re-do this once the get_overlap method for TimePeriod has been refactored
        self.other_overlap = self.time_period.get_overlap(self.period)
        assert self.other_overlap == TimePeriod(start=time(9, 0), end=time(12, 0))

        # 2 day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 3, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Date period test
        self.date_period = DatePeriod(
            start=date(2024, 1, 2),
            end=date(2024, 1, 4)
        )
        self.overlap = self.period.get_overlap(self.date_period)
        assert self.overlap == DatePeriod(start=date(2024, 1, 2),
                                          end=date(2024, 1, 3))

        # Datetime period test
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 3, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.dt_period = AbsolutePeriod(
            start=datetime(2024, 1, 2, 10, 0),
            end=datetime(2024, 1, 4, 8, 0)
        )
        self.overlap = self.period.get_overlap(self.date_period)
        assert self.overlap == AbsolutePeriod(
            start=datetime(2024, 1, 2, 10, 0),
            end=datetime(2024, 1, 3, 12, 0)
        )

    def test_get_disconnect(self):
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time period test
        self.time_period = TimePeriod(
            start=time(9, 0),
            end=time(13, 0)
        )
        self.dc = self.period.get_disconnect(self.time_period)
        assert self.dc == TimePeriod(start=time(8, 0), end=time(9, 0))
        self.other_dc = self.time_period.get_disconnect(self.period)
        assert self.other_dc == TimePeriod(start=time(12, 0), end=time(13, 0))

        # 2 day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 3, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Date period test
        self.date_period = DatePeriod(
            start=date(2024, 1, 2),
            end=date(2024, 1, 4)
        )
        self.dc = self.period.get_disconnect(self.date_period)
        assert self.dc == DatePeriod(start=date(2024, 1, 1),
                                     end=date(2024, 1, 2))
        self.other_dc = self.date_period.get_disconnect(self.period)
        assert self.other_dc == DatePeriod(start=date(2024, 1, 3),
                                           end=date(2024, 1, 4))

        # Datetime period test
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 3, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.dt_period = AbsolutePeriod(
            start=datetime(2024, 1, 2, 10, 0),
            end=datetime(2024, 1, 4, 8, 0)
        )
        self.dc = self.period.get_disconnect(self.date_period)
        assert self.dc == AbsolutePeriod(
            start=datetime(2024, 1, 1, 8, 0),
            end=datetime(2024, 1, 2, 10, 0)
        )
        self.other_dc = self.dt_period.get_disconnect(self.period)
        assert self.other_dc == AbsolutePeriod(
            start=datetime(2024, 1, 3, 12, 0),
            end=datetime(2024, 1, 4, 8, 0)
        )

        # Non overlapping periods
        # Same day period
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 1, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        # Time period test
        self.time_period = TimePeriod(
            start=time(13, 0),
            end=time(14, 0)
        )
        self.dc = self.period.get_disconnect(self.time_period)
        assert self.dc is None
        self.other_dc = self.time_period.get_disconnect(self.period)
        assert self.other_dc is None

        # Date period test
        self.start = datetime(2024, 1, 1, 8, 0)
        self.end = datetime(2024, 1, 5, 12, 0)
        self.period = AbsolutePeriod(start=self.start, end=self.end)

        self.date_period = DatePeriod(
            start=date(2024, 1, 10),
            end=date(2024, 1, 15)
        )
        self.dc = self.period.get_disconnect(self.date_period)
        assert self.dc is None
        self.other_dc = self.date_period.get_disconnect(self.period)
        assert self.other_dc is None

        # Datetime period test
        self.dt_period = AbsolutePeriod(
            start=datetime(2023, 12, 10, 10, 0),
            end=datetime(2023, 12, 15, 8, 0)
        )
        self.dc = self.period.get_disconnect(self.dt_period)
        assert self.dc is None
        self.other_dc = self.dt_period.get_disconnect(self.period)
        assert self.other_dc is None
