"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmplifyAssetDeploymentHandler = void 0;
const client_s3_1 = require("@aws-sdk/client-s3");
const s3_request_presigner_1 = require("@aws-sdk/s3-request-presigner");
const common_1 = require("./common");
class AmplifyAssetDeploymentHandler extends common_1.ResourceHandler {
    constructor(amplify, s3, event) {
        super(event);
        this.props = parseProps(this.event.ResourceProperties);
        this.amplify = amplify;
        this.s3 = s3;
    }
    // ------
    // CREATE
    // ------
    async onCreate() {
        // eslint-disable-next-line no-console
        console.log('deploying to Amplify with options:', JSON.stringify(this.props, undefined, 2));
        // Verify no jobs are currently running.
        const jobs = await this.amplify
            .listJobs({
            appId: this.props.AppId,
            branchName: this.props.BranchName,
            maxResults: 1,
        });
        if (jobs.jobSummaries &&
            jobs.jobSummaries.find(summary => summary.status === 'PENDING')) {
            return Promise.reject('Amplify job already running. Aborting deployment.');
        }
        // Create a pre-signed get URL of the asset so Amplify can retrieve it.
        const command = new client_s3_1.GetObjectCommand({
            Bucket: this.props.S3BucketName,
            Key: this.props.S3ObjectKey,
        });
        const assetUrl = await (0, s3_request_presigner_1.getSignedUrl)(this.s3, command);
        // Deploy the asset to Amplify.
        const deployment = await this.amplify
            .startDeployment({
            appId: this.props.AppId,
            branchName: this.props.BranchName,
            sourceUrl: assetUrl,
        });
        return {
            AmplifyJobId: deployment.jobSummary?.jobId,
        };
    }
    async isCreateComplete() {
        return this.isActive(this.event.AmplifyJobId);
    }
    // ------
    // DELETE
    // ------
    async onDelete() {
        // We can't delete this resource as it's a deployment.
        return;
    }
    async isDeleteComplete() {
        // We can't delete this resource as it's a deployment.
        return {
            IsComplete: true,
        };
    }
    // ------
    // UPDATE
    // ------
    async onUpdate() {
        return this.onCreate();
    }
    async isUpdateComplete() {
        return this.isActive(this.event.AmplifyJobId);
    }
    async isActive(jobId) {
        if (!jobId) {
            throw new Error('Unable to determine Amplify job status without job id');
        }
        const job = await this.amplify
            .getJob({
            appId: this.props.AppId,
            branchName: this.props.BranchName,
            jobId: jobId,
        });
        if (job.job?.summary?.status === 'SUCCEED') {
            return {
                IsComplete: true,
                Data: {
                    JobId: jobId,
                    Status: job.job.summary.status,
                },
            };
        }
        if (job.job?.summary?.status === 'FAILED' || job.job?.summary?.status === 'CANCELLED') {
            throw new Error(`Amplify job failed with status: ${job.job?.summary?.status}`);
        }
        else {
            return {
                IsComplete: false,
            };
        }
    }
}
exports.AmplifyAssetDeploymentHandler = AmplifyAssetDeploymentHandler;
function parseProps(props) {
    return props;
}
//# sourceMappingURL=data:application/json;base64,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