from http import HTTPStatus
from typing import Any, Optional, Union, cast

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.wrapped_doc import WrappedDoc
from ...models.wrapped_doc_update import WrappedDocUpdate
from ...types import Response


def _get_kwargs(
    id: str,
    *,
    body: WrappedDocUpdate,
) -> dict[str, Any]:
    headers: dict[str, Any] = {}

    _kwargs: dict[str, Any] = {
        "method": "put",
        "url": "/docs/{id}".format(
            id=id,
        ),
    }

    _body = body.to_dict()

    _kwargs["json"] = _body
    headers["Content-Type"] = "application/json"

    _kwargs["headers"] = headers
    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, WrappedDoc]]:
    if response.status_code == 200:
        response_200 = WrappedDoc.from_dict(response.json())

        return response_200
    if response.status_code == 400:
        response_400 = cast(Any, None)
        return response_400
    if response.status_code == 404:
        response_404 = cast(Any, None)
        return response_404
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, WrappedDoc]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: WrappedDocUpdate,
) -> Response[Union[Any, WrappedDoc]]:
    """Update an existing doc

     Update certain properties of an existing doc. This will save the doc in Dart for later access,
    search, etc. Any properties that are not specified will not be changed.

    Args:
        id (str):
        body (WrappedDocUpdate):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, WrappedDoc]]
    """

    kwargs = _get_kwargs(
        id=id,
        body=body,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: WrappedDocUpdate,
) -> Optional[Union[Any, WrappedDoc]]:
    """Update an existing doc

     Update certain properties of an existing doc. This will save the doc in Dart for later access,
    search, etc. Any properties that are not specified will not be changed.

    Args:
        id (str):
        body (WrappedDocUpdate):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, WrappedDoc]
    """

    return sync_detailed(
        id=id,
        client=client,
        body=body,
    ).parsed


async def asyncio_detailed(
    id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: WrappedDocUpdate,
) -> Response[Union[Any, WrappedDoc]]:
    """Update an existing doc

     Update certain properties of an existing doc. This will save the doc in Dart for later access,
    search, etc. Any properties that are not specified will not be changed.

    Args:
        id (str):
        body (WrappedDocUpdate):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, WrappedDoc]]
    """

    kwargs = _get_kwargs(
        id=id,
        body=body,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    id: str,
    *,
    client: Union[AuthenticatedClient, Client],
    body: WrappedDocUpdate,
) -> Optional[Union[Any, WrappedDoc]]:
    """Update an existing doc

     Update certain properties of an existing doc. This will save the doc in Dart for later access,
    search, etc. Any properties that are not specified will not be changed.

    Args:
        id (str):
        body (WrappedDocUpdate):

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, WrappedDoc]
    """

    return (
        await asyncio_detailed(
            id=id,
            client=client,
            body=body,
        )
    ).parsed
