import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as iam from '@aws-cdk/aws-iam';
import * as logs from '@aws-cdk/aws-logs';
import * as sns from '@aws-cdk/aws-sns';
import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * Properties for a new Slack channel configuration.
 *
 * @stability stable
 */
export interface SlackChannelConfigurationProps {
    /**
     * The name of Slack channel configuration.
     *
     * @stability stable
     */
    readonly slackChannelConfigurationName: string;
    /**
     * The permission role of Slack channel configuration.
     *
     * @default - A role will be created.
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The ID of the Slack workspace authorized with AWS Chatbot.
     *
     * To get the workspace ID, you must perform the initial authorization flow with Slack in the AWS Chatbot console.
     * Then you can copy and paste the workspace ID from the console.
     * For more details, see steps 1-4 in Setting Up AWS Chatbot with Slack in the AWS Chatbot User Guide.
     *
     * @see https://docs.aws.amazon.com/chatbot/latest/adminguide/setting-up.html#Setup_intro
     * @stability stable
     */
    readonly slackWorkspaceId: string;
    /**
     * The ID of the Slack channel.
     *
     * To get the ID, open Slack, right click on the channel name in the left pane, then choose Copy Link.
     * The channel ID is the 9-character string at the end of the URL. For example, ABCBBLZZZ.
     *
     * @stability stable
     */
    readonly slackChannelId: string;
    /**
     * The SNS topics that deliver notifications to AWS Chatbot.
     *
     * @default None
     * @stability stable
     */
    readonly notificationTopics?: sns.ITopic[];
    /**
     * Specifies the logging level for this configuration.
     *
     * This property affects the log entries pushed to Amazon CloudWatch Logs.
     *
     * @default LoggingLevel.NONE
     * @stability stable
     */
    readonly loggingLevel?: LoggingLevel;
    /**
     * The number of days log events are kept in CloudWatch Logs.
     *
     * When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `INFINITE`.
     *
     * @default logs.RetentionDays.INFINITE
     * @stability stable
     */
    readonly logRetention?: logs.RetentionDays;
    /**
     * The IAM role for the Lambda function associated with the custom resource that sets the retention policy.
     *
     * @default - A new role is created.
     * @stability stable
     */
    readonly logRetentionRole?: iam.IRole;
    /**
     * When log retention is specified, a custom resource attempts to create the CloudWatch log group.
     *
     * These options control the retry policy when interacting with CloudWatch APIs.
     *
     * @default - Default AWS SDK retry options.
     * @stability stable
     */
    readonly logRetentionRetryOptions?: logs.LogRetentionRetryOptions;
}
/**
 * Logging levels include ERROR, INFO, or NONE.
 *
 * @stability stable
 */
export declare enum LoggingLevel {
    /**
     * ERROR.
     *
     * @stability stable
     */
    ERROR = "ERROR",
    /**
     * INFO.
     *
     * @stability stable
     */
    INFO = "INFO",
    /**
     * NONE.
     *
     * @stability stable
     */
    NONE = "NONE"
}
/**
 * Represents a Slack channel configuration.
 *
 * @stability stable
 */
export interface ISlackChannelConfiguration extends cdk.IResource, iam.IGrantable {
    /**
     * The ARN of the Slack channel configuration In the form of arn:aws:chatbot:{region}:{account}:chat-configuration/slack-channel/{slackChannelName}.
     *
     * @stability stable
     * @attribute true
     */
    readonly slackChannelConfigurationArn: string;
    /**
     * The name of Slack channel configuration.
     *
     * @stability stable
     * @attribute true
     */
    readonly slackChannelConfigurationName: string;
    /**
     * The permission role of Slack channel configuration.
     *
     * @default - A role will be created.
     * @stability stable
     * @attribute true
     */
    readonly role?: iam.IRole;
    /**
     * Adds a statement to the IAM role.
     *
     * @stability stable
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Return the given named metric for this SlackChannelConfiguration.
     *
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * Either a new or imported Slack channel configuration
 */
declare abstract class SlackChannelConfigurationBase extends cdk.Resource implements ISlackChannelConfiguration {
    abstract readonly slackChannelConfigurationArn: string;
    abstract readonly slackChannelConfigurationName: string;
    abstract readonly grantPrincipal: iam.IPrincipal;
    abstract readonly role?: iam.IRole;
    /**
     * Adds extra permission to iam-role of Slack channel configuration.
     *
     * @stability stable
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Return the given named metric for this SlackChannelConfiguration.
     *
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * A new Slack channel configuration.
 *
 * @stability stable
 */
export declare class SlackChannelConfiguration extends SlackChannelConfigurationBase {
    /**
     * Import an existing Slack channel configuration provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param slackChannelConfigurationArn configuration ARN (i.e. arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack).
     * @returns a reference to the existing Slack channel configuration
     * @stability stable
     */
    static fromSlackChannelConfigurationArn(scope: Construct, id: string, slackChannelConfigurationArn: string): ISlackChannelConfiguration;
    /**
     * Return the given named metric for All SlackChannelConfigurations.
     *
     * @stability stable
     */
    static metricAll(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The ARN of the Slack channel configuration In the form of arn:aws:chatbot:{region}:{account}:chat-configuration/slack-channel/{slackChannelName}.
     *
     * @stability stable
     */
    readonly slackChannelConfigurationArn: string;
    /**
     * The name of Slack channel configuration.
     *
     * @stability stable
     */
    readonly slackChannelConfigurationName: string;
    /**
     * The permission role of Slack channel configuration.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SlackChannelConfigurationProps);
}
export {};
