"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackChannelConfiguration = exports.LoggingLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const chatbot_generated_1 = require("./chatbot.generated");
/**
 * Logging levels include ERROR, INFO, or NONE.
 *
 * @stability stable
 */
var LoggingLevel;
(function (LoggingLevel) {
    LoggingLevel["ERROR"] = "ERROR";
    LoggingLevel["INFO"] = "INFO";
    LoggingLevel["NONE"] = "NONE";
})(LoggingLevel = exports.LoggingLevel || (exports.LoggingLevel = {}));
/**
 * Either a new or imported Slack channel configuration
 */
class SlackChannelConfigurationBase extends cdk.Resource {
    /**
     * Adds extra permission to iam-role of Slack channel configuration.
     *
     * @stability stable
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Return the given named metric for this SlackChannelConfiguration.
     *
     * @stability stable
     */
    metric(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            dimensions: {
                ConfigurationName: this.slackChannelConfigurationName,
            },
            metricName,
            ...props,
        });
    }
}
/**
 * A new Slack channel configuration.
 *
 * @stability stable
 */
class SlackChannelConfiguration extends SlackChannelConfigurationBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.slackChannelConfigurationName,
        });
        this.role = props.role || new iam.Role(this, 'ConfigurationRole', {
            assumedBy: new iam.ServicePrincipal('chatbot.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const configuration = new chatbot_generated_1.CfnSlackChannelConfiguration(this, 'Resource', {
            configurationName: props.slackChannelConfigurationName,
            iamRoleArn: this.role.roleArn,
            slackWorkspaceId: props.slackWorkspaceId,
            slackChannelId: props.slackChannelId,
            snsTopicArns: (_b = props.notificationTopics) === null || _b === void 0 ? void 0 : _b.map(topic => topic.topicArn),
            loggingLevel: (_c = props.loggingLevel) === null || _c === void 0 ? void 0 : _c.toString(),
        });
        // Log retention
        // AWS Chatbot publishes logs to us-east-1 regardless of stack region https://docs.aws.amazon.com/chatbot/latest/adminguide/cloudwatch-logs.html
        if (props.logRetention) {
            new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/chatbot/${props.slackChannelConfigurationName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logGroupRegion: 'us-east-1',
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
        }
        this.slackChannelConfigurationArn = configuration.ref;
        this.slackChannelConfigurationName = props.slackChannelConfigurationName;
    }
    /**
     * Import an existing Slack channel configuration provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param slackChannelConfigurationArn configuration ARN (i.e. arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack).
     * @returns a reference to the existing Slack channel configuration
     * @stability stable
     */
    static fromSlackChannelConfigurationArn(scope, id, slackChannelConfigurationArn) {
        const re = /^slack-channel\//;
        const resourceName = cdk.Stack.of(scope).parseArn(slackChannelConfigurationArn).resourceName;
        if (!re.test(resourceName)) {
            throw new Error('The ARN of a Slack integration must be in the form: arn:aws:chatbot:{region}:{account}:chat-configuration/slack-channel/{slackChannelName}');
        }
        class Import extends SlackChannelConfigurationBase {
            constructor(s, i) {
                super(s, i);
                /**
                 * @attribute
                 */
                this.slackChannelConfigurationArn = slackChannelConfigurationArn;
                this.role = undefined;
                /**
                 * Returns a name of Slack channel configuration
                 *
                 * NOTE:
                 * For example: arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack
                 * The ArnComponents API will return `slack-channel/my-slack`
                 * It need to handle that to gets a correct name.`my-slack`
                 */
                this.slackChannelConfigurationName = resourceName.substring('slack-channel/'.length);
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for All SlackChannelConfigurations.
     *
     * @stability stable
     */
    static metricAll(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            metricName,
            ...props,
        });
    }
}
exports.SlackChannelConfiguration = SlackChannelConfiguration;
_a = JSII_RTTI_SYMBOL_1;
SlackChannelConfiguration[_a] = { fqn: "@aws-cdk/aws-chatbot.SlackChannelConfiguration", version: "1.98.0" };
//# sourceMappingURL=data:application/json;base64,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