from __future__ import annotations

import logging
import types

from typing import Any

from mknodes import mknav
from mknodes.basenodes import mknode
from mknodes.pages import mkpage
from mknodes.utils import linkprovider, reprhelpers


logger = logging.getLogger(__name__)


class MkLink(mknode.MkNode):
    """A simple Link."""

    ICON = "octicons/link-24"
    REQUIRED_EXTENSIONS = ["attr_list"]  # for buttons

    def __init__(
        self,
        target: str | mkpage.MkPage | mknav.MkNav | type | types.ModuleType,
        title: str | None = None,
        icon: str | None = None,
        as_button: bool = False,
        primary_color: bool = False,
        **kwargs: Any,
    ):
        """Constructor.

        Arguments:
            target: Link target
            title: Title used for link
            icon: Optional icon to be displayed in front of title
            as_button: Whether link should be rendered as button
            primary_color: If rendered as button, use primary color as background.
            kwargs: keyword arguments passed to parent
        """
        super().__init__(**kwargs)
        self.target = target
        self.title = title
        self.as_button = as_button
        self.primary_color = primary_color
        icon = icon or ""
        self.icon = icon
        if as_button:
            self.add_css_class("md-button")
        if primary_color:
            self.add_css_class("md-button--primary")

    def __repr__(self):
        return reprhelpers.get_repr(
            self,
            target=self.target,
            title=self.title,
            icon=self.icon,
            as_button=self.as_button,
            primary_color=self.primary_color,
            _filter_empty=True,
            _filter_false=True,
        )

    @property
    def url(self) -> str:  # type: ignore[return]
        if self.associated_project:
            return self.associated_project.linkprovider.get_url(self.target)
        return linkprovider.LinkProvider().get_url(self.target)

    def _to_markdown(self) -> str:
        title = self.target if self.title is None else self.title
        icon = (
            self.icon
            if not self.icon or self.icon.startswith(":")
            else f':{self.icon.replace("/", "-")}:'
        )
        prefix = f"{icon} " if self.icon else ""
        return f"[{prefix}{title}]({self.url})"

    @staticmethod
    def create_example_page(page):
        import mknodes

        url = "http://www.google.de"
        node = mknodes.MkLink(url, "This is a link.")
        page += mknodes.MkReprRawRendered(node, header="### Regular")
        node = mknodes.MkLink(url, "Disguised as button.", as_button=True)
        page += mknodes.MkReprRawRendered(node, header="### Button")
        node = mknodes.MkLink(url, "Colored.", as_button=True, primary_color=True)
        page += mknodes.MkReprRawRendered(node, header="### Colored")
        node = mknodes.MkLink(url, "With icon.", icon="octicons/link-24")
        page += mknodes.MkReprRawRendered(node, header="### With icon")
        node = mknodes.MkLink(page.parent.index_page, "To page.")
        page += mknodes.MkReprRawRendered(node, header="###To page")


if __name__ == "__main__":
    link = MkLink("www.test.de")
