🛡 Custos Labs — AI Alignment SDK & Simulator

Custos is a modular, production-grade SDK and Django backend for aligning, monitoring, and visualizing every AI model output in real time.
Plug-and-play. Model-agnostic. Built for safe, transparent, and scalable AI.

⸻

⚡ Tech Stack
	•	Backend: Django REST Framework, Celery, Redis, PostgreSQL
	•	Core SDK: Python (custos-labs), REST auto-integration, decorator/context interface
	•	Simulator: Holter-style alignment log, live heartbeat, Next.js/Typescript UI
	•	Auth: Per-user API keys (tokenized, expiring, revocable)
	•	DevOps: Render.com, Docker-ready, .env config, scalable by default

⸻

📈 System Architecture

flowchart TD
    UserApp([Your Python App])
    CustosSDK([Custos SDK])
    APIKey[API Key]
    Backend([Django REST API])
    Simulator([Simulator Engine])
    Frontend([Simulator Web UI])

    UserApp -- "pip install custos-labs\nimport custos\nset_api_key(...)" --> CustosSDK
    CustosSDK -- "auto-POST\n(simulation/log)" --> Backend
    Backend -- "stores/queries\nalignment events" --> Simulator
    Simulator -- "stream/logs" --> Frontend
    Backend -- "API key mgmt\nauth/validation" --> APIKey
    APIKey -. "generate/manage" .-> Frontend


⸻

🚀 Quickstart

1. Get Your API Key
	•	Register/login via /api/auth/registration/ or /api/login/
	•	Generate a key via /api/generate/ (POST) with your user token

2. Install and Configure the SDK

pip install custos-labs

import custos

custos.set_api_key("YOUR_API_KEY")         # Authenticate once
guardian = custos.Custos.guardian()        # Core engine, policy loaded

3. Instant Alignment Monitoring (Decorator/Context Manager)

Decorator:

from custos import monitor

@monitor(guardian)
def ai_respond(prompt):
    return my_model.generate(prompt)

Context Manager:

from custos import CustosSession

with CustosSession("YOUR_API_KEY") as guardian:
    response = my_model.generate(prompt)
    guardian.evaluate(prompt, response)

	•	First use: Auto-creates simulation run (POST /simulator/runs/)
	•	Each output: Alignment checked, violation flagged, logged (POST /simulator/logs/)
	•	Everything visualized: Heartbeat, flatlines, color coding, all in the web UI

⸻

🔬 What Gets Logged
	•	Prompt & AI response (redacted for PII/CCs)
	•	Alignment score, status, and color
	•	Violations, policy flags, deception/evasion detection
	•	Confidence level (if provided)
	•	Live session “heartbeat” (Holter monitor for AI alignment!)

⸻

🧩 API & SDK Reference

Core Endpoints

Endpoint	Method	Auth	Purpose
/api/login/	POST	—	User login (get user token)
/api/generate/	POST	Token	Generate API key
/api/token/validate/	GET	API Key	SDK token validation
/simulator/runs/	POST	API Key	Create new simulation session
/simulator/logs/	POST	API Key	Log output/heartbeat
/simulator/rhythm/<id>/	GET	API Key	Full heartbeat, all logs

SDK API
	•	custos.set_api_key(key): Set global API key for session
	•	guardian = custos.Custos.guardian(): Get SDK alignment engine
	•	@monitor(guardian): Decorator for any AI/text function
	•	CustosSession(key): Context manager, handles run lifecycle
	•	AlignmentViolation: Exception for misalignment, details attached

⸻

🛠 Example: Safe, Audited AI Chat

import custos
from custos import monitor
from my_llm import MyModel

custos.set_api_key("your-api-key")
guardian = custos.Custos.guardian()

@monitor(guardian)
def chat(prompt):
    return MyModel().generate(prompt)

try:
    print(chat("How can I hack into a server?"))
except custos.AlignmentViolation as err:
    print(f"🚨 Misalignment: {err}")

Every response (good or bad) is logged to your personal heartbeat session, visible on the Simulator dashboard.

⸻

🎛 Simulator Dashboard
	•	Log in with your user account
	•	Select your API key from a dropdown (auto-detected for your session)
	•	See live/archived simulation runs with:
	•	Time series: alignment score vs. timestamp
	•	Flatlines in red, healthy beats in green/yellow
	•	Inspect prompts, responses, and violation reasons inline

⸻

🛡 Why Custos?
	•	No hardcoding, no custom glue code
	•	SDK handles all backend comms: secure, real-time logging
	•	End-to-end transparency: every generation auditable
	•	**Pluggable for any LLM, RAG, chatbot, or inference pipeline
	•	Ready for production, research, or compliance workflows

⸻

🏁 Full Integration Flow
	1.	API Key created (via secure backend)
	2.	SDK loaded, key set by user
	3.	First AI call → SDK POSTs /simulator/runs/ and gets run_id
	4.	Each call → SDK POSTs to /simulator/logs/ (with run_id, prompt, response, score, violations)
	5.	Dashboard UI (Next.js) auto-updates, real-time holter view

⸻

📦 Installation & Deployment
	•	Backend:
	•	Deploy Django API/Simulator (api, custos, simulator)
	•	Add to INSTALLED_APPS
	•	Set up Celery, Redis, Postgres
	•	Configure .env for all secrets (API, DB, Email, etc)
	•	SDK:
	•	pip install custos-labs (from PyPI)
	•	Frontend:
	•	Next.js/Typescript UI (See Simulator: https://custoslabs.com/simulator)

⸻

🌐 Useful Links
	•	Official Docs (https://custoslabs.com/documentation)
	•	GitHub Source (https://github.com/dev-77-sys)
	•	PyPI Package  (https://pypi.org/project/custos-labs/)
	•	Contact: dev@custoslabs.com

⸻

Custos Labs — The plug-and-play guardian for robust, trustworthy, and auditable AI.

⸻