import pyperclip

themes = {
    "Алгоритмы умножения матриц": {
        "Наивное умножение матрицы на вектор и умножение матриц": {
            "code": "## numpy\n\nimport numpy as np\n\n\ndef matmul(a, b):\n    n = a.shape[0]\n    k = a.shape[1]\n    m = b.shape[1]\n    res = np.zeros((n, m))\n\n    for i in range(n):\n        for j in range(m):\n            for s in range(k):\n                res[i, j] += a[i, s] * b[s, j]\n\n    return res\n\n# python\n\ndef matmul(A, B):\n    n = len(A)\n    k_A = len(A[0])\n    k_B = len(B)\n    m = len(B[0])\n\n    assert k_A == k_B, f\"Can't multiply {n}x{k_A} on {k_B}x{m}\"\n\n    result = [[0 for j in range(m)] for i in range(n)]\n\n    for i in range(n):\n        for j in range(m):\n            for k in range(k_A):\n                result[i][j] += A[i][k] * B[k][j]\n\n    return result\n\n# пример\n\nA = [[1, 2], [3, 4]]\n\nB = [[5, 6], [7, 8]]\n\nC = [[5, 6]]\n\n# for numpy\nA = np.array(A)\nB = np.array(B)\nC = np.array(C)\n\n\n# vector by matrix\nprint(matmul(C, B))\n\n# matrix by matrix\nprint(matmul(A, B))"
        },
        "Алгоритм Штрассена": {
            "markdown": "Алгоритм Штрассена, предложенный Фолькером Штрассеном в 1969 году, предназначен для быстрого умножения матриц. \nВ отличие от традиционного алгоритма с временной сложностью O(n³), алгоритм Штрассена выполняет умножение за время O(n^2.81), \nчто обеспечивает выигрыш при работе с большими плотными матрицами.\n\nf1 = (a11 + a22) * (b11 + b22)\n\nf2 = (a21 + a22) * b11\n\nf3 = a11 * (b12 - b22)\n\nf4 = a22 * (b21 - b11)\n\nf5 = (a11 + a12) * b22\n\nf6 = (a21 - a11) * (b11 + b12)\n\nf7 = (a12 - a22) * (b21 + b22)\n\n\nс11 = f1 + f4 - f5 + f7\n\nс12 = f3 + f5\n\nc21 = f2 + f4\n\nc22 = f1 - f2 + f3 + f6\n\nСхема алгоритма:\n\n1. Разбиваем матрицы A и B размера n x n на 4 блока n/2 x n/2\n2. Вычисляем произведения по приведённым выше формулам рекурсивно",
            "code": "import numpy as np\n\n\ndef strassen(A, B):\n    n = len(A)\n\n    if n <= 2:\n        return np.dot(A, B)\n\n    mid = n // 2\n\n    A11 = A[:mid, :mid]\n    A12 = A[:mid, mid:]\n    A21 = A[mid:, :mid]\n    A22 = A[mid:, mid:]\n    B11 = B[:mid, :mid]\n    B12 = B[:mid, mid:]\n    B21 = B[mid:, :mid]\n    B22 = B[mid:, mid:]\n\n    f1 = strassen(A11 + A22, B11 + B22)\n    f2 = strassen(A21 + A22, B11)\n    f3 = strassen(A11, B12 - B22)\n    f4 = strassen(A22, B21 - B11)\n    f5 = strassen(A11 + A12, B22)\n    f6 = strassen(A21 - A11, B11 + B12)\n    f7 = strassen(A12 - A22, B21 + B22)\n\n    C11 = f1 + f4 - f5 + f7\n    C12 = f3 + f5\n    C21 = f2 + f4\n    C22 = f1 - f2 + f3 + f6\n\n    C = np.vstack((np.hstack((C11, C12)), np.hstack((C21, C22))))\n\n    return C\n\n# пример\n\nn = 4\na = np.random.randn(n, n)\nb = np.random.randn(n, n)\n\nstrassen(a, b)\n\n## python only\n\n\ndef add_matrix(X, Y):\n    return [[X[i][j] + Y[i][j] for j in range(len(X[0]))] for i in range(len(X))]\n\n\ndef subtract_matrix(X, Y):\n    return [[X[i][j] - Y[i][j] for j in range(len(X[0]))] for i in range(len(X))]\n\n\ndef merge_matrices(C11, C12, C21, C22):\n    top = [C11[i] + C12[i] for i in range(len(C11))]\n    bottom = [C21[i] + C22[i] for i in range(len(C21))]\n    return top + bottom\n\n\ndef strassen(A, B):\n    n = len(A)\n\n    if n == 1:\n        return [[A[0][0] * B[0][0]]]\n\n    mid = n // 2\n    A11 = [row[:mid] for row in A[:mid]]\n    A12 = [row[mid:] for row in A[:mid]]\n    A21 = [row[:mid] for row in A[mid:]]\n    A22 = [row[mid:] for row in A[mid:]]\n\n    B11 = [row[:mid] for row in B[:mid]]\n    B12 = [row[mid:] for row in B[:mid]]\n    B21 = [row[:mid] for row in B[mid:]]\n    B22 = [row[mid:] for row in B[mid:]]\n\n    f1 = strassen(add_matrix(A11, A22), add_matrix(B11, B22))\n    f2 = strassen(add_matrix(A21, A22), B11)\n    f3 = strassen(A11, subtract_matrix(B12, B22))\n    f4 = strassen(A22, subtract_matrix(B21, B11))\n    f5 = strassen(add_matrix(A11, A12), B22)\n    f6 = strassen(subtract_matrix(A21, A11), add_matrix(B11, B12))\n    f7 = strassen(subtract_matrix(A12, A22), add_matrix(B21, B22))\n\n    C11 = add_matrix(subtract_matrix(add_matrix(f1, f4), f5), f7)\n    C12 = add_matrix(f3, f5)\n    C21 = add_matrix(f2, f4)\n    C22 = add_matrix(subtract_matrix(add_matrix(f1, f3), f2), f6)\n\n    return merge_matrices(C11, C12, C21, C22)"
        }
    },
    "Собственные значения и вектора": {
        "Собственные векторы собственные значения важность Google PageRank": {
            "markdown": "### Собственные векторы и собственные значения\n\n**Собственные значения и собственные векторы** — это ключевые понятия линейной алгебры, которые помогают понять свойства матриц и операций над ними.  \n\n- **Собственное значение ($ \\lambda $)** показывает, насколько масштабируется собственный вектор при умножении на матрицу.  \n- **Собственный вектор ($ v $)** — это ненулевой вектор, направление которого остаётся неизменным при умножении на матрицу.\n\nИх связь выражается уравнением:\n$ A \\cdot v = \\lambda \\cdot v $,\nгде $ A $ — квадратная матрица, $ v $ — собственный вектор, $ \\lambda $ — собственное значение.\n\nСобственные значения и векторы используются для:\n1. **Анализа систем**: В инженерии и науке для анализа устойчивости систем (например, механические вибрации, электрические цепи).\n2. **Сжатия данных**: В машинном обучении (например, метод главных компонент — PCA).\n3. **Графов и сетей**: Для анализа графов, например, в алгоритме PageRank.\n4. **Диагонализации матриц**: Упрощение вычислений, особенно для степенных матриц или экспоненциальных функций матриц.\n\n### Пример нахождения собственных значений и векторов\n\nРассмотрим матрицу:\n$\nA = \n\\begin{pmatrix}\n4 & 2 \\\\\n1 & 3\n\\end{pmatrix}\n$\n\n1. **Составим характеристическое уравнение:**\n   - Для нахождения собственных значений решается уравнение:\n     $ \\det(A - \\lambda I) = 0 $,\n     где $ I $ — единичная матрица.\n\n   - Подставляем $ A $ и $ I $:\n     $\n     \\det \\begin{pmatrix}\n     4 - \\lambda & 2 \\\\\n     1 & 3 - \\lambda\n     \\end{pmatrix} = 0\n     $\n\n   - Находим определитель:\n     $\n     (4 - \\lambda)(3 - \\lambda) - 2 \\cdot 1 = 0\n     $\n\n   - Упрощаем:\n     $\n     \\lambda^2 - 7\\lambda + 10 = 0\n     $\n\n   - Решаем квадратное уравнение:\n     $\n     \\lambda = 5, \\lambda = 2\n     $\n\n2. **Найдём собственные векторы:**\n   - Для каждого значения $ \\lambda $, решаем уравнение $ (A - \\lambda I) \\cdot v = 0 $ для $ v $.\n\n   - Например, для $ \\lambda = 5 $:\n     $\n     \\begin{pmatrix}\n     -1 & 2 \\\\\n     1 & -2\n     \\end{pmatrix}\n     \\cdot\n     \\begin{pmatrix}\n     x \\\\\n     y\n     \\end{pmatrix}\n     = 0\n     $\n\n     Решая, получаем $ v = (2, 1)^T $.\n\n\nВ реальной практике собственные значения и векторы для больших матриц редко ищутся аналитически, так как это сложно для высоких размерностей. Вместо этого используются численные методы (например, QR-разложение, степенной метод), которые реализованы в популярных библиотеках, таких как NumPy или MATLAB.\n\n## Алгоритм PageRank\n\nАлгоритм PageRank вычисляет глобальный вектор важности страниц для всего веб-графа, учитывая только структуру ссылок между страницами и не завися от пользовательских запросов. <br>\nЭтот вектор, который обновляется периодически, используется как базовый показатель важности страниц. <br>\nКогда пользователь вводит запрос, Google применяет дополнительные алгоритмы для отбора релевантных страниц, основываясь на ключевых словах, семантике и других факторах. Среди этих релевантных страниц PageRank выступает как один из факторов, определяющих их приоритет в результатах поиска.\n\nАлгоритм PageRank находит собственный вектор для собственного значения 1 для матрицы переходов, построенной по ссылкам из графа webgraph",
            "code": "import numpy as np\n\n\ndef page_rank(A, alpha=0.85, tolerance=1e-6, max_iterations=1000):\n    # нормализованная по столбцам матрица ссылок (смежности) web страниц из графа webgraph\n    A = A / A.sum(axis=0)\n    A[np.isnan(A)] = 0\n\n    # добавление демпфирования (случайный переход между страницами), коэффициент демпфирования = 0.85\n    N = A.shape[0]\n    M = alpha * A + (1 - alpha) * 1 / N * np.ones((N, N))\n\n    # начальное равномерное распределение\n    R = np.full((N, 1), 1 / N)\n    iteration = 0\n\n    # итеративный процесс нахождения собственного вектора\n    while iteration < max_iterations:\n        new_R = np.dot(M, R)\n\n        if np.linalg.norm(new_R - R, 1) < tolerance:\n            break\n\n        R = new_R\n        iteration += 1\n\n    R_normalized = R / np.sum(R)\n\n    return R_normalized, iteration\n\n\n# Пример вызова\n\nA = np.array([[0, 0, 1], [1, 0, 0], [1, 1, 0]])\n\nR_normalized, iteration_count = page_rank(A)\n\nR_normalized, iteration_count\n\n## нахождение собственного вектора, соответствующего собственному значению 1 через numpy\n\nA_ = A / A.sum(axis=0)\nA_[np.isnan(A)] = 0\n\nalpha = 0.85\n\nN = A_.shape[0]\nM = alpha * A_ + (1 - alpha) * 1 / N * np.ones((N, N))\n\n\neigenvalues, eigenvectors = np.linalg.eig(M)\n\nindex = np.argmax(np.isclose(eigenvalues, 1))\n\nprincipal_eigenvector = eigenvectors[:, index]\n\n\nprincipal_eigenvector_normalized = principal_eigenvector / np.sum(\n    principal_eigenvector\n\n)  # сумму элементов равна 1\n\n\nR_normalized = principal_eigenvector_normalized.real\n\nR_normalized"
        },
        "Разложение Шура и QR-алгоритм": {
            "markdown": "### Разложение Шура\n\n**Разложение Шура** — это представление квадратной матрицы $A$ в виде произведения унитарной матрицы $U$ ($U^* U = I$) и верхней треугольной матрицы $T$:\n\n$ A = U T U^* $\n\nДиагональные элементы матрицы $T$ являются собственными значениями матрицы $A$.\nРазложение Шура можно получить из QR-алгоритма, верхняя треугольная матрица T - это результат QR-алгоритма, унитарная матрица U составляется из накопленных матриц Q_k за все итерации.\n\n### QR-алгоритм\n\n**QR-алгоритм** — это итеративный численный метод для нахождения всех собственных чисел и собственных векторов квадратной матрицы. Он основан на последовательном применении QR-разложения и обновлении матрицы.\n\nОпишем и реализуем QR-алгоритм:\n\n1. Пусть $A$ - данная матрица, для которой надо найти собственные значения и векторы\n\n2. Для $k$-го шага применить QR-разложение для $A_k$:\n    - Размер $Q$ - $n \\times m$ - матрица с ортонормированными столбцами, размер $R$ - $m \\times m$ - верхнетреугольная матрица\n    - Выполнить процесс Грама-Шмидта:\n        - Положить оператор проекции как: \n            - $proj_b a = \\frac{\\langle a, b \\rangle}{\\langle b, b \\rangle}b$\n        - Для каждого столбца $b_i$ изначальной матрицы $A$:\n            - $b_i = a_i - \\sum_{j=1}^{i} (proj_{b_{i-1}} a_i)$ \n        - Получить нормированный вектор $e_i$:\n            - $e_i = \\frac{b_i}{||b_i||}$\n        - Полученный вектор $e_i$ есть $i$-ый столбец матрицы $Q$\n    - Получить матрицу $R = Q^T A$\n\n3. Посчитать $A_{k+1} = R_k Q_k$\n\n4. Повторять 2 и 3 шаги до тех пор, пока $A_k$ не станет \"достаточно треугольной\" - например, норма нижней треугольной матрицы не будет достаточно низка\n",
            "code": "import copy\nimport math\n\n\ndef transpose(A: list[list[float]]) -> list[list[float]]:\n    \"\"\"Транспонирует матрицу\"\"\"\n    return [[A[j][i] for j in range(len(A))] for i in range(len(A[0]))]\n\n\ndef matmul(A: list[list[float]], B: list[list[float]]) -> list[list[float]]:\n    \"\"\"Функция матричного умножения\"\"\"\n    n = len(A)\n    k_A = len(A[0])\n    k_B = len(B)\n    m = len(B[0])\n\n    assert k_A == k_B, f\"Can't multiply {n}x{k_A} on {k_B}x{m}\"\n\n    result = [[0 for j in range(m)] for i in range(n)]\n\n    for i in range(n):\n        for j in range(m):\n            for k in range(k_A):\n                result[i][j] += A[i][k] * B[k][j]\n\n    return result\n\n\ndef get_diag(A: list[list[float]]) -> list[float]:\n    \"\"\"Возвращает диагональные элементы\"\"\"\n    return [A[i][i] for i in range(len(A))]\n\n\ndef norm_vector(vector: list[float]) -> float:\n    \"\"\"Находит норму вектора\"\"\"\n    return math.sqrt(sum(el**2 for el in vector))\n\n\ndef get_proj(vector1: list[float], vector2: list[float]):\n    \"\"\"Находит проекцию одного вектора на другой\"\"\"\n    return sum([vector1[i] * vector2[i] for i in range(len(vector1))])\n\n\ndef lower_norm(A: list[list[float]]) -> float:\n    \"\"\"Подсчитывает норму нижней области под главной диагональю\"\"\"\n    summ = 0\n\n    for i in range(1, len(A)):\n        for j in range(i):\n            summ += A[i][j] ** 2\n\n    return math.sqrt(summ)\n\ndef QR_decomposition(\n    A: list[list[float]],\n) -> tuple[list[list[float]], list[list[float]]]:\n    \"\"\"Метод QR-разложение с помощью процесса Грама-Шмидта\"\"\"\n    n = len(A)\n    m = len(A[0])\n\n    Q = [[0] * m for _ in range(n)]\n\n    # процесс Грама-Шмидта\n    for j in range(m):\n        b = [A[i][j] for i in range(n)]\n\n        for i in range(j):\n            a = [Q[k][i] for k in range(n)]\n            proj = get_proj(a, b)\n            b = [b[k] - proj * a[k] for k in range(n)]\n\n        normed = norm_vector(b)\n\n        e = [b[k] / normed for k in range(n)]\n\n        for i in range(n):\n            Q[i][j] = e[i]\n\n    R = matmul(transpose(Q), A)\n\n    return Q, R\n\ndef QR_algorithm(A: list[list[float]], eps: float = 1e-5):\n    A_ = copy.deepcopy(A)\n\n    k = 0\n\n    Qs = []\n\n    while lower_norm(A_) > eps:\n        Q, R = QR_decomposition(A_)\n        Qs.append(Q)\n        A_ = matmul(R, Q)\n        k += 1\n\n    eingvals = get_diag(A_)\n\n    eingvectors = Qs[0]\n    for Q in Qs[1:]:\n        eingvectors = matmul(eingvectors, Q)\n\n    return eingvals, eingvectors, k\n\n# пример\n\nA = [\n    [-3, 4, 6],\n    [4, 2, 1],\n    [6, 1, 5],\n]\n\neingvals, eingvectors, k = QR_algorithm(A)"
        },
        "Степенной метод": {
            "markdown": "Степенной метод, или метод степенных итераций, — это итерационный алгоритм, предназначенный для нахождения собственного значения матрицы с наибольшей абсолютной величиной и соответствующего ему собственного вектора.\n\nБерём некоторый случайный вектор $b$ и начинаем действовать на него оператором $A$ (умножая его на эту матрицу), при этом нормируя:\n\n$b_{i+1} = \\frac{A \\cdot b_i}{||A||}$\n\nПовторяем до тех пор, пока изменение вектора $b$ не будет меньше заданного значения $\\varepsilon$. При достижении этого условия, считаем, что мы нашли собственный вектор соответствующей наибольшему собственному значению. ",
            "code": "import numpy as np\n\n\ndef power_method(A, x0, tol=1e-6, max_iter=100):\n    x = x0 / np.linalg.norm(x0)\n    lam_prev = 0\n\n    for i in range(max_iter):\n        x = A @ x / np.linalg.norm(A @ x)\n        lam = (x.T @ A @ x) / (x.T @ x)\n\n        if np.abs(lam - lam_prev) < tol:\n            return lam, x, i + 1\n\n        lam_prev = lam\n\n    return lam, x, max_iter\n\n\nA = np.array([[2, 1], [1, 2]])  # квадратная матрица\n\nx0 = np.array([[1, 2]]).T  # приближённый собственный вектор\n\nlam, x, iterations = power_method(A, x0)\n\nprint(\"Собственное значение:\", lam)\nprint(\"Собственный вектор:\\n\", x)\nprint(\"Количество итераций:\", iterations)"
        },
        "Круги Гершгорина": {
            "markdown": "Круги Гершгорина — это геометрический способ локализации собственных значений квадратной матрицы. Для заданной матрицы $ A = [a_{ij}] $ радиусы и центры кругов определяются следующим образом:\n\n- **Центр диска** для $ i $-й строки:\n  $\n  c_i = a_{i}\n  $\n  где $ a_{i} $ — диагональный элемент матрицы.\n\n- **Радиус диска**:\n  $\n  r_i = \\sum_{j \\neq i} |a_{ij}|\n  $\n  где сумма берётся по всем элементам строки, кроме диагонального.\n\n#### Теорема Гершгорина:\nВсе собственные значения матрицы $ A $ лежат внутри объединения всех кругов Гершгорина:\n$\n\\bigcup_{i=1}^n \\{ z \\in \\mathbb{C} : |z - c_i| \\leq r_i \\}\n$\n\nКруги Гершгорина дают приближённую область, где находятся собственные значения матрицы. Они используется для проверки точности спектральных методов и оценки матриц.",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\n\n\ndef plot_gershgorin_circles(a, randomize=True, scale=2):\n    n = a.shape[0]\n    fig, ax = plt.subplots(1, 1)\n\n    if randomize:\n        a = a + scale * np.random.randn(n, n)\n\n    xg = np.diag(a).real\n    yg = np.diag(a).imag\n    rg = np.zeros(n)\n    ev = np.linalg.eigvals(a)\n\n    for i in range(n):\n        rg[i] = np.sum(np.abs(a[i, :])) - np.abs(a[i, i])\n        crc = plt.Circle((xg[i], yg[i]), radius=rg[i], fill=False)\n        ax.add_patch(crc)\n\n    plt.scatter(ev.real, ev.imag, color=\"r\", label=\"Eigenvalues\")\n    plt.legend()\n    plt.axis(\"equal\")\n    plt.show()\n\n\na = np.array([[5, 1, 1], [1, 0, 0.5], [2, 0, 10]])\nplot_gershgorin_circles(a)"
        },
        "Разложение Шура теорема Шура": {
            "markdown": "### Разложение Шура и теорема Шура\n\n#### Разложение Шура\nРазложение Шура — это представление любой квадратной матрицы $ A $ в виде произведения унитарной матрицы $ U $ и верхней треугольной матрицы $ T $:\n\n$\nA = U T U^*\n$\n\nгде:\n- $ U $ — унитарная матрица ($ U^* U = I $),\n- $ T $ — верхняя треугольная матрица,\n- $ U^* $ — эрмитово-сопряжённая матрица (транспонированная и комплексно сопряжённая).\n\nДиагональные элементы матрицы $ T $ являются собственными значениями матрицы $ A $.\n\n#### Теорема Шура\nТеорема Шура утверждает, что для любой квадратной матрицы $ A $ существует унитарная матрица $ U $, такая что $ A = U T U^* $, где $ T $ — верхняя треугольная матрица. \n\n##### Особенности:\n- Если $ A $ является нормальной матрицей ($ A^* A = A A^* $), то $ T $ становится диагональной, и разложение Шура совпадает с разложением в собственные значения.\n\n#### Связь с QR-алгоритмом:\nQR-алгоритм приводит матрицу $ A $ к верхней треугольной форме $ T $, при этом унитарная матрица $ U $ является накопленным произведением матриц $ Q_k $ на каждой итерации QR-разложения.\n",
            "code": "## numpy\n\nimport numpy as np\nfrom scipy.linalg import schur\n\nA = [[4, 1, 1], [1, 3, 1], [1, 1, 2]]\nA = np.array(A)\n\nT, U = schur(A, output=\"real\")\n\nprint(\"Матрица T (верхняя треугольная):\")\nprint(T)\n\nprint(\"\\nМатрица U (унитарная):\")\nprint(U)\n\n# Проверка разложения\nA_reconstructed = U @ T @ np.conjugate(U).T\nprint(\"\\nВосстановленная матрица A:\")\nprint(A_reconstructed)\n\n## python\n\nimport copy\nimport math\n\n\ndef transpose(A: list[list[float]]) -> list[list[float]]:\n    \"\"\"Транспонирует матрицу\"\"\"\n    return [[A[j][i] for j in range(len(A))] for i in range(len(A[0]))]\n\n\ndef matmul(A: list[list[float]], B: list[list[float]]) -> list[list[float]]:\n    \"\"\"Функция матричного умножения\"\"\"\n    n = len(A)\n    k_A = len(A[0])\n    k_B = len(B)\n    m = len(B[0])\n\n    assert k_A == k_B, f\"Can't multiply {n}x{k_A} on {k_B}x{m}\"\n\n    result = [[0 for j in range(m)] for i in range(n)]\n\n    for i in range(n):\n        for j in range(m):\n            for k in range(k_A):\n                result[i][j] += A[i][k] * B[k][j]\n\n    return result\n\n\ndef get_diag(A: list[list[float]]) -> list[float]:\n    \"\"\"Возвращает диагональные элементы\"\"\"\n    return [A[i][i] for i in range(len(A))]\n\n\ndef norm_vector(vector: list[float]) -> float:\n    \"\"\"Находит норму вектора\"\"\"\n    return math.sqrt(sum(el**2 for el in vector))\n\n\ndef get_proj(vector1: list[float], vector2: list[float]):\n    \"\"\"Находит проекцию одного вектора на другой\"\"\"\n    return sum([vector1[i] * vector2[i] for i in range(len(vector1))])\n\n\ndef lower_norm(A: list[list[float]]) -> float:\n    \"\"\"Подсчитывает норму нижней области под главной диагональю\"\"\"\n    summ = 0\n\n    for i in range(1, len(A)):\n        for j in range(i):\n            summ += A[i][j] ** 2\n\n    return math.sqrt(summ)\n\n\ndef QR_decomposition(\n    A: list[list[float]],\n) -> tuple[list[list[float]], list[list[float]]]:\n    \"\"\"Метод QR-разложение с помощью процесса Грама-Шмидта\"\"\"\n    n = len(A)\n    m = len(A[0])\n\n    Q = [[0] * m for _ in range(n)]\n\n    # процесс Грама-Шмидта\n    for j in range(m):\n        b = [A[i][j] for i in range(n)]\n\n        for i in range(j):\n            a = [Q[k][i] for k in range(n)]\n            proj = get_proj(a, b)\n            b = [b[k] - proj * a[k] for k in range(n)]\n\n        normed = norm_vector(b)\n\n        e = [b[k] / normed for k in range(n)]\n\n        for i in range(n):\n            Q[i][j] = e[i]\n\n    R = matmul(transpose(Q), A)\n\n    return Q, R\n\n\ndef schur_decomposition(A: list[list[float]], eps: float = 1e-5):\n    A_ = copy.deepcopy(A)\n    Qs = []\n\n    while lower_norm(A_) > eps:\n        Q, R = QR_decomposition(A_)\n        Qs.append(Q)\n        A_ = matmul(R, Q)\n\n    U = Qs[0]\n    for Q in Qs[1:]:\n        U = matmul(U, Q)\n\n    return U, A_\n\n# пример использования\nA = [[4, 1, 1], [1, 3, 1], [1, 1, 2]]\n\nU, T = schur_decomposition(A)\nU, T\n\n# реконструкция разложения для проверки\n\nA_reconstructed = matmul(\n    matmul(U, T), [[U[j][i] for j in range(len(U))] for i in range(len(U[0]))]\n)\nA_reconstructed"
        },
        "Спектр и псевдоспектр": {
            "markdown": "Спектром матрицы $ A $ называется множество всех её собственных значений. Собственные значения определяются как корни характеристического уравнения:\n\n$\n\\det(A - \\lambda I) = 0,\n$\n\nгде $ \\lambda $ — собственное значение, $ I $ — единичная матрица, а $ \\det $ обозначает детерминант. Спектр матрицы предоставляет информацию о её фундаментальных свойствах, таких как устойчивость и динамика соответствующих систем.\n\n**Псевдоспектр матрицы**\n\nПсевдоспектр матрицы расширяет понятие спектра, учитывая чувствительность собственных значений к малым возмущениям. Для заданного положительного числа $ \\varepsilon $ $ (\\varepsilon > 0) $ - псевдоспектр матрицы $ A $ определяется как множество комплексных чисел $ z $, для которых обратная матрица $ (A - zI)^{-1} $ существует и её норма превышает $ \\varepsilon^{-1} $:\n\n\n$\n\\Lambda_\\varepsilon(A) = \\{ z \\in \\mathbb{C} \\mid \\| (A - zI)^{-1} \\| > \\varepsilon^{-1} \\}.\n$\n\nИными словами, псевдоспектр матрицы $ A $ включает точки $ \\lambda $ в комплексной плоскости, где матрица $ (A - \\lambda I) $ близка к необратимой, что указывает на потенциальную чувствительность системы к малым возмущениям.",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\nfrom scipy.linalg import svdvals\n\nA = np.array([[1, 2], [3, 4]])\n\neigenvalues = np.linalg.eigvals(A)\n\nprint(\"Cобственные значения (спектр) матрицы A:\", eigenvalues)\n\n\nx = np.linspace(-1, 3, 100)\ny = np.linspace(-1, 3, 100)\nX, Y = np.meshgrid(x, y)\nZ = X + 1j * Y  # массив для задания сетки\n\n# вычисляем минимальное сингулярное число (квадратный корень из минимального собственного значения матрицы A* @ A) для каждой точки\nsigma_min = np.zeros_like(Z, dtype=float)\nfor i in range(Z.shape[0]):\n    for j in range(Z.shape[1]):\n        sigma_min[i, j] = min(svdvals(A - Z[i, j] * np.eye(A.shape[0])))\n\n# Строим контурный график псевдоспектра\nplt.contour(X, Y, sigma_min, levels=[0.1, 0.5, 1, 2, 5], colors=\"blue\")\nplt.xlabel(\"Re(λ)\")\nplt.ylabel(\"Im(λ)\")\nplt.title(\"Псевдоспектр матрицы A\")\nplt.colorbar(label=\"Минимальное сингулярное число\")\nplt.grid()\nplt.show()"
        },
        "Неявный QR алгоритм со сдвигами": {
            "markdown": "**Неявный QR-алгоритм со сдвигами** — это численный метод для нахождения собственных значений матрицы, который сочетает в себе QR-разложение и технику сдвигов, обеспечивая быструю и надёжную сходимость. Позволяет находить как собственные значения, так и собственные векторы матрицы.\n- **QR-разложение:** Представление матрицы $ A $ в виде произведения ортогональной матрицы $ Q $ и верхнетреугольной матрицы $ R $:\n  $\n  A = Q \\cdot R\n  $\n\n- **Сдвиги:** Использование сдвига $ \\mu $, близкого к собственному значению, для ускорения сходимости:\n  $\n  A - \\mu I = Q \\cdot R\n  $\n\nОпишем и реализуем QR-алгоритм:\n\n1. Пусть $A$ - данная матрица, для которой надо найти собственные значения и векторы\n\n2. Для $k$-го шага применить QR-разложение для $A_k$:\n    - Размер $Q$ - $n \\times m$ - матрица с ортонормированными столбцами, размер $R$ - $m \\times m$ - верхнетреугольная матрица\n    - Выполнить процесс Грама-Шмидта:\n        - Положить оператор проекции как: \n            - $proj_b a = \\frac{\\langle a, b \\rangle}{\\langle b, b \\rangle}b$\n        - Для каждого столбца $b_i$ изначальной матрицы $A$:\n            - $b_i = a_i - \\sum_{j=1}^{i} (proj_{b_{i-1}} a_i)$ \n        - Получить нормированный вектор $e_i$:\n            - $e_i = \\frac{b_i}{||b_i||}$\n        - Полученный вектор $e_i$ есть $i$-ый столбец матрицы $Q$\n    - Получить матрицу $R = Q^T A$\n\n3. Посчитать $A_{k+1} = R_k Q_k$\n\n4. Повторять 2 и 3 шаги до тех пор, пока $A_k$ не станет \"достаточно треугольной\" - например, норма нижней треугольной матрицы не будет достаточно низка\n\n\nОпишем процесс QR-алгоритма со сдвигами:\n\n1. Пусть $A$ - данная матрица, для которой надо найти собственные значения и векторы\n\n2. Для $k$-го шага применить QR-разложение для $A_k - v_k E$:\n    - $v_k$ - сдвиг, который можно выбрать, например, последним элементов диагонали матрицы $A_k$\n    - Осуществить разложение для матрицы $A_k - v_k E$\n\n3. Посчитать $A_{k+1} = R_k Q_k + v_k E$\n\n4. Повторять 2 и 3 шаги до тех пор, пока $A_k$ не станет \"достаточно треугольной\" - например, норма нижней треугольной матрицы не будет достаточно низка",
            "code": "import copy\nimport math\n\n\ndef transpose(A: list[list[float]]) -> list[list[float]]:\n    \"\"\"Транспонирует матрицу\"\"\"\n    return [[A[j][i] for j in range(len(A))] for i in range(len(A[0]))]\n\n\ndef matmul(A: list[list[float]], B: list[list[float]]) -> list[list[float]]:\n    \"\"\"Функция матричного умножения\"\"\"\n    n = len(A)\n    k_A = len(A[0])\n    k_B = len(B)\n    m = len(B[0])\n\n    assert k_A == k_B, f\"Can't multiply {n}x{k_A} on {k_B}x{m}\"\n\n    result = [[0 for j in range(m)] for i in range(n)]\n\n    for i in range(n):\n        for j in range(m):\n            for k in range(k_A):\n                result[i][j] += A[i][k] * B[k][j]\n\n    return result\n\n\ndef get_diag(A: list[list[float]]) -> list[float]:\n    \"\"\"Возвращает диагональные элементы\"\"\"\n    return [A[i][i] for i in range(len(A))]\n\n\ndef norm_vector(vector: list[float]) -> float:\n    \"\"\"Находит норму вектора\"\"\"\n    return math.sqrt(sum(el**2 for el in vector))\n\n\ndef get_proj(vector1: list[float], vector2: list[float]):\n    \"\"\"Находит проекцию одного вектора на другой\"\"\"\n    return sum([vector1[i] * vector2[i] for i in range(len(vector1))])\n\n\ndef lower_norm(A: list[list[float]]) -> float:\n    \"\"\"Подсчитывает норму нижней области под главной диагональю\"\"\"\n    summ = 0\n\n    for i in range(1, len(A)):\n        for j in range(i):\n            summ += A[i][j] ** 2\n\n    return math.sqrt(summ)\n\ndef QR_decomposition(\n    A: list[list[float]],\n) -> tuple[list[list[float]], list[list[float]]]:\n    \"\"\"Метод QR-разложение с помощью процесса Грама-Шмидта\"\"\"\n    n = len(A)\n    m = len(A[0])\n\n    Q = [[0] * m for _ in range(n)]\n\n    # процесс Грама-Шмидта\n    for j in range(m):\n        b = [A[i][j] for i in range(n)]\n\n        for i in range(j):\n            a = [Q[k][i] for k in range(n)]\n            proj = get_proj(a, b)\n            b = [b[k] - proj * a[k] for k in range(n)]\n\n        normed = norm_vector(b)\n\n        e = [b[k] / normed for k in range(n)]\n\n        for i in range(n):\n            Q[i][j] = e[i]\n\n    R = matmul(transpose(Q), A)\n\n    return Q, R\n\ndef QR_algorithm_shifted(A: list[list[float]], eps: float = 1e-5):\n    \"\"\"QR-алгоритм со сдвигами\"\"\"\n    A_ = copy.deepcopy(A)\n\n    k = 0\n\n    Qs = []\n\n    while lower_norm(A_) > eps:\n        v = get_diag(A_)[-1]\n        A_shifted = [\n            [A_[i][j] - (v if i == j else 0) for j in range(len(A[0]))]\n            for i in range(len(A))\n        ]\n        Q, R = QR_decomposition(A_shifted)\n        Qs.append(Q)\n        A_ = matmul(R, Q)\n        A_ = [\n            [A_[i][j] + (v if i == j else 0) for j in range(len(A[0]))]\n            for i in range(len(A))\n        ]\n        k += 1\n\n    eingvals = get_diag(A_)\n\n    eingvectors = Qs[0]\n    for Q in Qs[1:]:\n        eingvectors = matmul(eingvectors, Q)\n\n    return eingvals, eingvectors, k\n\n# пример\n\nA = [\n    [-3, 4, 6],\n    [4, 2, 1],\n    [6, 1, 5],\n]\n\neingvals, eingvectors, k = QR_algorithm_shifted(A)"
        }
    },
    "Численное дифференцирование": {
        "Локальная глобальная ошибки Порядок метода": {
            "markdown": "**Порядок численного метода** характеризует скорость уменьшения погрешности приближённого решения при уменьшении шага вычислений.\n\nГоворят, что метод имеет порядок точности \\( p \\), если глобальная ошибка \\( E \\) этого метода пропорциональна \\( h^p \\), где \\( h \\) — шаг вычислений.\n\nЭто означает, что при уменьшении шага в \\( k \\) раз, ошибка уменьшается приблизительно в \\( k^p \\) раз.\n\n**Локальная ошибка** (или шаговая ошибка) — это погрешность, возникающая на каждом отдельном шаге численного метода. Она показывает, насколько приближенное решение на данном шаге отклоняется от точного решения. Локальная ошибка обычно зависит от шага и порядка метода.\n\n**Глобальная ошибка** — это накопленная погрешность на всём интервале решения, учитывающая влияние всех предыдущих шагов. Глобальная ошибка обычно имеет порядок на единицу ниже, чем локальная ошибка (при увеличении шага убывает меньше). \n\n**Ошибки округления** возникают из-за ограниченной точности представления чисел в компьютере. При выполнении арифметических операций с числами, представленными с конечной точностью, неизбежно появляются небольшие погрешности, которые могут накапливаться и существенно влиять на результат. \n\n**Ошибки сокращения** (или ошибки усечения) связаны с аппроксимацией непрерывных математических операций дискретными вычислительными процессами. Например, при численном интегрировании или дифференцировании использование конечного числа шагов приводит к появлению таких ошибок. \n\nОбщая ошибка численного метода является суммой глобальной ошибки сокращения и глобальной ошибки округления."
        },
        "Метод центральной разности": {
            "markdown": "Метод центральной разности - метод конечных разностей, применяемый для численного вычисления производных по следующей формуле:\n\n$f'(x) \\approx \\frac{f(x + h) - f(x - h)}{2h}$\n\nДанный метод более точный, чем метод левой или правой разности, однако не может вычислить значения производной, близких к ограничениям, из-за $\\pm h$",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\n\nf = np.exp\nh = np.logspace(-5, 1)\nx0 = 0\n\ntrue_value = 1\n\ndef central_difference_method(f, h, x0):\n    return (f(x0 + h) - f(x0 - h)) / h / 2\n\nestimate = central_difference_method(f, h, x0)\n\nerr = np.abs(true_value - estimate)\n\np = np.polyfit(np.log(h), np.log(err), 1)\n\nplt.figure(figsize=(10, 6))\nplt.grid(alpha=0.5, linestyle=\":\")\nplt.xlabel(\"h\")\nplt.ylabel(\"Ошибка\")\nplt.title(\"Сходимость оценки значения производной\")\n\nplt.loglog(h, err, label=\"Расчётные данные\", c=\"red\")\n\nplt.loglog(h, f(p[1]) * h ** p[0], label=\"Линейная аппроксимация\")\n\nplt.legend()\nplt.show()"
        },
        "Метод прямой разности": {
            "markdown": "Метод центральной разности - метод конечных разностей, применяемый для численного вычисления производных по следующей формуле:\n\n$f'(x) \\approx \\frac{f(x + h) - f(x)}{h}$\n\nДанный метод более точный, чем метод левой или правой разности, однако не может вычислить значения производной, близких к ограничениям, из-за $\\pm h$",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\n\nf = np.exp\nh = np.logspace(-7, 1)\nx0 = 0\ntrue = 1\n\ndef direct_difference_method(f, h, x0):\n    return (f(x0 + h) - f(x0)) / h\n    \nestimate = direct_difference_method(f, h, x0)\n\nerr = np.abs(true - estimate)\n\np = np.polyfit(np.log(h), np.log(err), 1)\n\nplt.figure(figsize=(10, 6))\nplt.grid(alpha=0.5, linestyle=\":\")\nplt.xlabel(\"h\")\nplt.ylabel(\"Ошибка\")\nplt.title(\"Сходимость оценки значения производной\")\n\nplt.loglog(h, err, label=\"Расчётные данные\", c=\"red\")\n\nplt.loglog(h, np.exp(p[1]) * h ** p[0], label=\"Линейная аппроксимация\")\n\nplt.legend()\nplt.show()"
        },
        "Метод обратной разности": {
            "markdown": "Метод обратной разности - метод конечных разностей, применяемый для численного вычисления производных по следующей формуле:\n\n$f'(x_0) \\approx g'(x_0) = \\frac{f(x_0) - f(x_0 - h)}{h}$",
            "code": "# latex\n# $$f'(x_0) \\approx g'(x_0) = \\frac{f(x_0) - f(x_0 - h)}{h}$$\n\nimport matplotlib.pyplot as plt\nimport numpy as np\n\n# Задаем значения h\nf = np.exp\nx0 = 0\nh = np.logspace(-5, 1, num=100)  # 100 значений от 10^(-5) до 10^(1)\n\n# Оценка производной методом обратной разности\ndef inverse_difference_method(f, h, x0):\n    return (f(x0) - f(x0 - h)) / h\nestimate = inverse_difference_method(f, h, x0)  # f(0) - f(0 - h)\n\n# Вычисление ошибки по сравнению с истинным значением производной\nerr = np.abs(estimate - 1)  # Истинное значение производной f'(0) = 1\n\n# Линейная аппроксимация в логарифмическом масштабе\np = np.polyfit(np.log(h), np.log(err), 1)\n\nplt.figure(figsize=(10, 6))\nplt.grid(alpha=0.5, linestyle=\":\")\nplt.xlabel(\"h\")\nplt.ylabel(\"Ошибка\")\nplt.title(\"Сходимость оценки значения производной\")\n\nplt.loglog(h, err, label=\"Расчётные данные\", c=\"red\")\n\nplt.loglog(h, np.exp(p[1]) * h ** p[0], label=\"Линейная аппроксимация\")\n\nplt.legend()\nplt.show()"
        }
    },
    "Методы решения задачи Коши": {
        "Обыкновенные дифференциальные уравнения задача Коши": {
            "markdown": "**Задача Коши** для обыкновенного дифференциального уравнения (ОДУ) заключается в нахождении решения дифференциального уравнения, которое удовлетворяет заданным начальным условиям.\n\n**Определение:**\n\nПусть дано ОДУ первого порядка:\n\n$ y' = f(x, y) $\n\nс начальным условием:\n\n$ y(x_0) = y_0 $\n\nТребуется найти функцию $ y(x) $, которая удовлетворяет этому уравнению и условию.\n\n**Теорема существования и единственности:**\n\nЕсли функция $ f(x, y) $ и её частная производная по $ y $ непрерывны в некоторой окрестности точки $ (x_0, y_0) $, то существует единственное решение $ y(x) $ задачи Коши в этой окрестности.\n\n## Пример\n\nРассмотрим уравнение:\n\n$ y' = x + y $\n\nс начальным условием:\n\n$ y(0) = 1 $\n\n**Решение:**\n\n1. **Метод разделения переменных:**\n\n   Перепишем уравнение в форме:\n\n   $ \\frac{dy}{dx} - y = x $\n\n   Это линейное ОДУ первого порядка. Найдём интегрирующий множитель:\n\n   $ \\mu(x) = e^{\\int -1 \\, dx} = e^{-x} $\n\n   Умножим обе части уравнения на $ \\mu(x) $:\n\n   $ e^{-x} \\frac{dy}{dx} - e^{-x} y = x e^{-x} $\n\n   Левая часть уравнения теперь является производной произведения:\n\n   $ \\frac{d}{dx} \\left( e^{-x} y \\right) = x e^{-x} $\n\n   Интегрируем обе части по $ x $:\n\n   $ e^{-x} y = \\int x e^{-x} \\, dx $\n\n2. **Интегрирование правой части:**\n\n   Используем метод интегрирования по частям:\n\n   $ \\int x e^{-x} \\, dx = -x e^{-x} - \\int -e^{-x} \\, dx = -x e^{-x} + e^{-x} + C $\n\n   Таким образом:\n\n   $ e^{-x} y = -x e^{-x} + e^{-x} + C $\n\n   Умножим обе части на $ e^{x} $:\n\n   $ y = -x + 1 + C e^{x} $\n\n3. **Определение константы $ C $:**\n\n   Используем начальное условие $ y(0) = 1 $:\n\n   $ 1 = -0 + 1 + C e^{0} $\n\n   $ 1 = 1 + C $\n\n   $ C = 0 $\n\n   Таким образом, решение задачи Коши:\n\n   $ y = -x + 1 $\n\n**Проверка:**\n\nПроверим, удовлетворяет ли найденное решение исходному уравнению и начальному условию.\n\nВычислим производную:\n\n$ y' = -1 $\n\nПодставим в уравнение:\n\n$ -1 = x + (-x + 1) $\n\n$ -1 = 1 $\n\nЭто верно при $ x = 0 $. Начальное условие также выполняется: $ y(0) = 1 $.\n\nТаким образом, решение $ y = -x + 1 $ является правильным. "
        },
        "Метод Эйлера": {
            "markdown": "Метод Эйлера позволяет численно решить систему дифференциальных уравнений. Данный метод является явным, одношаговым, первого порядка точности\n\nДля применения метода необходимо задать шаг $h$, тогда:\n\n$y_i^{k + 1} = y_i^k + h\\cdot f(x^k, y^k)$\n\nпричём $x^{k+1} = x^k + h$",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\n\n\ndef method_euler(f, x_0, x_n, y_0, N):\n    dx = (x_n - x_0) / N\n    x = np.linspace(x_0, x_n, N + 1)\n    y = np.zeros((N + 1, len(y_0)))\n    y[0, :] = y_0\n\n    for n in range(N):\n        y[n + 1, :] = y[n, :] + dx * f(x[n], y[n, :])\n\n    return x, y\n\n# Пример\n\ndef fun_sin(x, y):\n    return -np.sin(x)\n\n\nx_5, y_5 = method_euler(fun_sin, 0.0, 0.5, [1.0], 5)\n\nx_100, y_100 = method_euler(fun_sin, 0, 0.5, [1], 100)\n\ny_5[-1, 0], y_100[-1, 0], np.cos(0.5)\n\n# Сходимость\n\nN = np.array([2**i for i in range(3, 12)])\n\ndx = 0.5 / N\n\nerr = np.zeros_like(dx)\n\nfor i in range(len(N)):\n    n = N[i]\n    x, y = method_euler(fun_sin, 0, 0.5, [1], n)\n    err[i] = np.abs(np.cos(0.5) - y[-1, 0])\n\nplt.plot(err)\n\nplt.show()\n\np = np.polyfit(np.log(dx[:-1]), np.log(err[:-1]), 1)\n\nfig = plt.figure(figsize=(10, 6))\nax = fig.add_subplot(111)\nax.loglog(dx, err, \"kx\", label=\"Посчитанные невязки\")\nax.loglog(dx, np.exp(p[1]) * dx ** p[0], label=f\"Наклон линии {p[0]:.2f}\")\nax.set_xlabel(\"dx\")\nax.set_ylabel(\"Ошибка\")\nax.set_title(\"Сходимость метода Эйлера\")\nax.legend(loc=2)\n\nplt.show()\n\ndef motion(x, y):\n    dfdt = np.zeros_like(y)\n    dfdt[0] = -y[1]\n    dfdt[1] = y[0]\n\n    return dfdt\n\n\ny0 = np.array([1, 0])\n\nt_01, y_01 = method_euler(motion, 0, 50, y0, 500)\nt_001, y_001 = method_euler(motion, 0, 50, y0, 5000)\n\nfig = plt.figure(figsize=(10, 6))\nax1 = fig.add_subplot(121)\nax1.plot(y_01[:, 0], y_01[:, 1], label=\"$h = 0.1$\")\nax1.set_xlabel(\"$x$\")\nax1.set_ylabel(\"$y$\")\nax1.legend()\n\nax2 = fig.add_subplot(122)\nax2.plot(y_001[:, 0], y_001[:, 1], label=\"$h = 0.01$\")\nax2.set_xlabel(\"$x$\")\nax2.set_ylabel(\"$y$\")\nax2.legend()\n\nplt.show()\n\n# Сходимость 2\n\nN = np.array([500 * 2**i for i in range(0, 10)])\n\ndx = 0.5 / N\n\nerr = np.zeros_like(dx)\n\nfor i in range(len(N)):\n    n = N[i]\n    x, y = method_euler(motion, 0, 50, y0, n)\n    err[i] = np.abs(y[-1, 0] ** 2 + y[-1, 1] ** 2 - 1)\n\np = np.polyfit(np.log(dx), np.log(err), 1)\n\nfig = plt.figure(figsize=(10, 6))\nax = fig.add_subplot(111)\nax.loglog(dx, err, \"kx\", label=\"Посчитанные невязки\")\nax.loglog(dx, np.exp(p[1]) * dx ** p[0], label=f\"Наклон линии {p[0]:.2f}\")\nax.set_xlabel(\"dx\")\nax.set_ylabel(\"Ошибка\")\nax.set_title(\"Сходимость метода Эйлера\")\nax.legend(loc=2)\n\nplt.show()"
        },
        "Метод предиктора-корректора": {
            "markdown": "Схема предиктор-корректор семейство алгоритмов численного решения различных задач, которые состоят из двух шагов: на первом шаге (предиктор) вычисляется грубое приближение требуемой величины, на втором шаге при помощи иного метода приближение уточняется (корректируется).\n\nАлгоритм предиктора-корректора на основе метода Эйлера (2-ой порядок):\n\n1. Вычисление грубого приближения величины $y_{i, pred}$: $y_{i, pred} = y_i^k + h f(x^k, y^k)$.\n\n2. Вычисление уточненного приближения величины $y_i^{k+1}$: $y_i^{k+1} = y_i^k + \\frac{h}{2} (f(x^k, y_i^k) + f(x^{k+1}, y^{k+1}, y_{i, pred}))$.\n\nБлагодаря корректору, ошибка значительно снижается",
            "code": "def euler_pc(f, x_0, x_n, y_0, N):\n    dx = (x_n - x_0) / N\n    x = np.linspace(x_0, x_n, N + 1)\n    y = np.zeros((N + 1, len(y_0)))\n    y[0, :] = y_0\n\n    for n in range(N):\n        yp = y[n, :] + dx * f(x[n], y[n, :])\n        y[n + 1, :] = y[n, :] + dx / 2 * (f(x[n], y[n, :]) + f(x[n + 1], yp))\n\n    return x, y\n\n# Пример\n\ndef fun_sin(x, y):\n    return -np.sin(x)\n\n\nx_5, y_5 = euler_pc(fun_sin, 0.0, 0.5, [1.0], 5)\n\nx_100, y_100 = euler_pc(fun_sin, 0, 0.5, [1], 100)\n\ny_5[-1, 0], y_100[-1, 0]\n\n# Сходимость\n\nN = np.array([2**i for i in range(3, 12)])\n\ndx = 0.5 / N\n\nerr = np.zeros_like(dx)\n\nfor i in range(len(N)):\n    n = N[i]\n    x, y = euler_pc(fun_sin, 0, 0.5, [1], n)\n    err[i] = np.abs(np.cos(0.5) - y[-1, 0])\n\np = np.polyfit(np.log(dx[:-1]), np.log(err[:-1]), 1)\n\nfig = plt.figure(figsize=(10, 6))\nax = fig.add_subplot(111)\nax.loglog(dx, err, \"kx\", label=\"Посчитанные невязки\")\nax.loglog(dx, np.exp(p[1]) * dx ** p[0], label=f\"Наклон линии {p[0]:.2f}\")\nax.set_xlabel(\"dx\")\nax.set_ylabel(\"Ошибка\")\nax.set_title(\"Сходимость метода предиктора-корректора\")\nax.legend(loc=2)\n\nplt.show()\n\ndef motion(x, y):\n    dfdt = np.zeros_like(y)\n    dfdt[0] = -y[1]\n    dfdt[1] = y[0]\n\n    return dfdt\n\n\ny0 = np.array([1, 0])\n\nt_01, y_01 = euler_pc(motion, 0, 50, y0, 500)\nt_001, y_001 = euler_pc(motion, 0, 50, y0, 5000)\n\nfig = plt.figure(figsize=(10, 6))\nax1 = fig.add_subplot(121)\nax1.plot(y_01[:, 0], y_01[:, 1], label=\"$h = 0.1$\")\nax1.set_xlabel(\"$x$\")\nax1.set_ylabel(\"$y$\")\nax1.legend()\n\nax2 = fig.add_subplot(122)\nax2.plot(y_001[:, 0], y_001[:, 1], label=\"$h = 0.01$\")\nax2.set_xlabel(\"$x$\")\nax2.set_ylabel(\"$y$\")\nax2.legend()\n\nplt.show()"
        },
        "Метод Рунге-Кутты 1-4 порядков": {
            "markdown": "Главная идея методов Рунге-Кутты заключается в последовательном вычислении промежуточных значений производной функции на каждом шаге интегрирования. Эти промежуточные значения используются для построения более точного приближения решения ОДУ по сравнению с простыми методами, такими как метод Эйлера.\n\nМетоды различного порядка отличаются количеством вычисляемых промежуточных значений и, соответственно, точностью аппроксимации:\n\nМетод первого порядка является методом Эйлера\n\nС увеличением порядка метода Рунге-Кутты повышается точность решения, однако возрастает и вычислительная нагрузка из-за необходимости расчёта большего числа промежуточных значений на каждом шаге.\n\nМетод Рунге-Кутта 2\n\n$y_{n+1} = y_n + a k-1 + b k_2$, где\n\n$k_1 = h f(x_n, y_n)$\n\n$k_2 = h f(x_n + \\alpha h, y_n + \\beta * k_1)$\n\n\nС коэффициентами:\n\n$a + b = 1$\n\n$\\alpha b = \\frac 1 2$\n\n$\\beta b = \\frac 1 2$\n\nМетод Рунге-Кутта 4\n\n$y_{n+1} = y_n + \\frac 1 6 (k-1 + 2 (k_2 + k_3) + k_4)$, где\n\n$k_1 = h f(x_n, y_n)$\n\n$k_2 = h * f(x_n + \\frac h 2, y_n + \\frac{k_1}{2})$\n\n$k_3 = h * f(x_n + \\frac h 2, y_n + \\frac{k_2}{2})$\n\n$k_4= h f(x_n + h, y_n + k_3)$\n\n\nЛокальная ошибка: $O(h^5)$\n\nГлобальная ошибка: $O(h^4)$\n",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\n\n\ndef rk4_method(f, x_0, x_n, y_0, N):\n\n    dx = (x_n - x_0) / N\n\n    x = np.linspace(x_0, x_n, N + 1)\n\n    y = np.zeros((N + 1, len(y_0)))\n    y[0, :] = y_0\n\n    k1 = np.zeros_like(y_0)\n\n    k2 = np.zeros_like(y_0)\n\n    k3 = np.zeros_like(y_0)\n\n    k4 = np.zeros_like(y_0)\n\n    for n in range(N):\n\n        k1 = dx * f(x[n], y[n, :])\n\n        k2 = dx * f(x[n] + dx / 2, y[n, :] + k1 / 2)\n\n        k3 = dx * f(x[n] + dx / 2, y[n, :] + k2 / 2)\n\n        k4 = dx * f(x[n] + dx, y[n, :] + k3)\n\n        y[n + 1, :] = y[n, :] + 1 / 6 * (k1 + 2 * k2 + 2 * k3 + k4)\n\n    return x, y\n\n# Пример\n\ndef fun_sin(x, y):\n    return -np.sin(x)\n\n\nx_5, y_5 = rk4_method(fun_sin, 0.0, 0.5, [1.0], 5)\n\nx_100, y_100 = rk4_method(fun_sin, 0, 9.5, [1], 100)\n\ny_5[-1][0], y_100[-1][0]\n\n# Сходимость\n\ndef motion(x, y):\n    dfdt = np.zeros_like(y)\n    dfdt[0] = -y[1]\n    dfdt[1] = y[0]\n\n    return dfdt\n\n\ny0 = np.array([1, 0])\n\nt_01, y_01 = rk4_method(motion, 0, 50, y0, 500)\nt_001, y_001 = rk4_method(motion, 0, 50, y0, 50)\n\nfig = plt.figure(figsize=(10, 6))\nax1 = fig.add_subplot(121)\nax1.plot(y_01[:, 0], y_01[:, 1], label=\"$h = 0.1$\")\nax1.set_xlabel(\"$x$\")\nax1.set_ylabel(\"$y$\")\nax1.legend()\n\nax2 = fig.add_subplot(122)\nax2.plot(y_001[:, 0], y_001[:, 1], label=\"$h = 1$\")\nax2.set_xlabel(\"$x$\")\nax2.set_ylabel(\"$y$\")\nax2.legend()\n\nplt.show()\n\n# видно, что длина радиус вектора при шаге 1 не всегда 1 (так как не круг, а многоугольник)\n\nfig = plt.figure(figsize=(10, 6))\nax1 = fig.add_subplot(121)\nax1.plot(t_01, np.sqrt(y_01[:, 1] ** 2 + y_01[:, 0] ** 2), label=\"$h = 0.1$\")\nax1.set_xlabel(\"$x$\")\nax1.set_ylabel(\"$y$\")\nax1.legend()\n\nax2 = fig.add_subplot(122)\nax2.plot(t_001, np.sqrt(y_001[:, 1] ** 2 + y_001[:, 0] ** 2), label=\"$h = 1$\")\nax2.set_xlabel(\"$x$\")\nax2.set_ylabel(\"$y$\")\nax2.legend()\n\nplt.show()\n\nimport math\n\n\nN = np.array([50 * 2**i for i in range(0, 10)])\n\nh = np.zeros_like(N, float)\n\ndx = 50 / N\n\nerr_r = np.zeros_like(h)\nerr_theta = np.zeros_like(h)\nerr_x = np.zeros_like(h)\nerr_y = np.zeros_like(h)\nerr_all = np.zeros_like(h)\n\nfor i in range(len(N)):\n    n = N[i]\n    x, y = rk4_method(motion, 0, 50, y0, n)\n    err_r[i] = np.abs(y[-1, 0] ** 2 + y[-1, 1] ** 2 - 1)\n    err_theta[i] = np.abs(\n        (math.atan2(y[-1, 1], y[-1, 0]) % (2 * np.pi)) - np.mod(50, 2 * np.pi)\n    )\n    err_x[i] = np.abs(y[-1, 0] - np.cos(50))\n    err_y[i] = np.abs(y[-1, 1] - np.sin(50))\n    err_all[i] = np.linalg.norm([y[-1, 0] - np.cos(50), y[-1, 1] - np.sin(50)], 2)\n\np_r = np.polyfit(np.log(dx[:-1]), np.log(err_r[:-1]), 1)\np_theta = np.polyfit(np.log(dx[:-1]), np.log(err_theta[:-1]), 1)\np_x = np.polyfit(np.log(dx[:-1]), np.log(err_x[:-1]), 1)\np_y = np.polyfit(np.log(dx[:-1]), np.log(err_y[:-1]), 1)\np_all = np.polyfit(np.log(dx[:-1]), np.log(err_all[:-1]), 1)\n\nfig = plt.figure(figsize=(14, 6))\nax1 = fig.add_subplot(111)\n\n# ошибка в радиусе (невязка между длиной радиус-вектора и 1)\nax1.loglog(dx, err_r, \"kx\", label=\"Невязки в радиусе\")\nax1.loglog(dx, np.exp(p_r[1]) * dx ** p_r[0], label=f\"Наклон линии $r$ {p_r[0]:.2f}\")\n\n# ошибка в угле (разница между численным и теоретическим углом)\nax1.loglog(dx, err_theta, \"ro\", label=\"Невязки в углах\", color=\"green\")\nax1.loglog(\n    dx,\n    np.exp(p_theta[1]) * dx ** p_theta[0],\n    label=f\"Наклон линии $theta$ {p_theta[0]:.2f}\",\n)\n\n# ошибки по осям x и y\nax1.loglog(dx, err_x, \"kx\", label=\"Невязки в $x$\")\nax1.loglog(dx, np.exp(p_x[1]) * dx ** p_x[0], label=f\"Наклон линии $x$ {p_x[0]:.2f}\")\nax1.loglog(dx, err_y, \"ro\", label=\"Невязки в $y$\")\nax1.loglog(dx, np.exp(p_y[1]) * dx ** p_y[0], label=f\"Наклон линии $y$ {p_y[0]:.2f}\")\nax1.loglog(dx, err_all, \"c+\", label=\"Норма ошибки в $y$\")\n\n# норма ошибки в пространстве (x, y)\nax1.loglog(\n    dx, np.exp(p_all[1]) * dx ** p_all[0], label=f\"Наклон линии $all$ {p_all[0]:.2f}\"\n)\nax1.set_xlabel(\"dx\")\nax1.set_ylabel(\"Ошибка\")\nax1.set_title(\"Сходимость метода Рунге-Кутты 4 порядка\")\nax1.legend(loc=2)\n\nplt.show()"
        },
        "Методы Адамса-Мултона методы Адамса-Бэшфорта": {
            "markdown": "Методы Адамса-Бэшфорта и Адамса-Мултона являются многошаговыми методами численного интегрирования обыкновенных дифференциальных уравнений (ОДУ). Они используют информацию о предыдущих шагах для вычисления следующего значения решения, что отличает их от одношаговых методов, таких как методы Рунге-Кутты.\n\n**Методы Адамса-Бэшфорта (явные методы):**\n\nМетоды Адамса-Бэшфорта являются явными многошаговыми методами, где новое значение решения вычисляется напрямую на основе предыдущих значений функции. Они имеют следующий общий вид:\n\n$ y_{n+1} = y_n + h \\sum_{i=0}^{k} b_i f_{n-i} $\n\nгде $ h $ — шаг интегрирования, $ f_{n-i} = f(t_{n-i}, y_{n-i}) $ — значения производной в предыдущих точках, а $ b_i $ — коэффициенты, зависящие от порядка метода.\n\nДля метода Адамса-Бэшфорта второго порядка (двухшаговый) формула имеет вид:\n\n$ y_{n+1} = y_n + h \\left( \\frac{3}{2} f_n - \\frac{1}{2} f_{n-1} \\right) $\n\nМетоды Адамса-Бэшфорта требуют начальных значений, которые обычно вычисляются с помощью одношаговых методов, таких как метод Рунге-Кутты. Они эффективны, так как требуют вычисления только одного нового значения правой части уравнения на каждом шаге, но могут быть менее устойчивыми для жестких задач. \n\n**Методы Адамса-Мултона (неявные методы):**\n\nМетоды Адамса-Мултона являются неявными многошаговыми методами, где новое значение решения зависит как от предыдущих значений функции, так и от значения функции в новой точке. Они имеют следующий общий вид:\n\n$ y_{n+1} = y_n + h \\sum_{i=-1}^{k-1} v_i f_{n-i} $\n\nгде $ v_i $ — коэффициенты, зависящие от порядка метода.\n\nДля метода Адамса-Мултона второго порядка (двухшаговый) формула имеет вид:\n\n$ y_{n+1} = y_n + \\frac{h}{2} \\left( f_{n+1} + f_n \\right) $\n\nПоскольку $ f_{n+1} = f(t_{n+1}, y_{n+1}) $ зависит от неизвестного $ y_{n+1} $, такие методы являются неявными и требуют решения нелинейных уравнений на каждом шаге. Они более устойчивы и точны по сравнению с явными методами, особенно для жестких задач, но требуют дополнительных вычислительных затрат. \n\n**Комбинированные методы (предиктор-корректор):** (код ниже)\n\nЧасто методы Адамса-Бэшфорта и Адамса-Мултона используются совместно в схемах предиктор-корректор. Сначала явный метод Адамса-Бэшфорта (предиктор) используется для вычисления предварительного значения решения, затем неявный метод Адамса-Мултона (корректор) уточняет это значение. Такая комбинация обеспечивает баланс между вычислительной эффективностью и устойчивостью. ",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\n\n\ndef am5_method(f, x_0, x_n, y_0, N):\n    dx = (x_n - x_0) / N\n    x = np.linspace(x_0, x_n, N + 1)\n    y = np.zeros((N + 1, len(y_0)))\n    fn = np.zeros_like(y)\n    y[0, :] = y_0\n\n    k1 = np.zeros_like(y_0)\n    k2 = np.zeros_like(y_0)\n    k3 = np.zeros_like(y_0)\n    k4 = np.zeros_like(y_0)\n\n    # предиктор методом Адамса-Бэшфорта 4 порядка\n    for n in range(4):\n        fn[n, :] = f(x[n], y[n, :])\n        k1 = dx * fn[n, :]\n        k2 = dx * f(x[n] + dx / 2, y[n, :] + k1 / 2)\n        k3 = dx * f(x[n] + dx / 2, y[n, :] + k2 / 2)\n        k4 = dx * f(x[n] + dx, y[n, :] + k3)\n        y[n + 1, :] = y[n, :] + 1 / 6 * (k1 + 2 * k2 + 2 * k3 + k4)\n\n    coeff_A = np.array(\n        [\n            [1, 1, 1, 1, 1],\n            [0, -1, -2, -3, -4],\n            [0, 0, 2, 6, 12],\n            [0, 0, 0, -6, -24],\n            [0, 0, 0, 0, 24],\n        ]\n    )\n    coeff_b = np.array([1, 1 / 2, 5 / 6, 9 / 4, 251 / 30])\n    b_ab4 = np.linalg.solve(coeff_A, coeff_b)\n    b_am5 = np.array([251, 646, -264, 106, -19]) / 720\n\n    for n in range(4, N):\n        fn[n, :] = f(x[n], y[n, :])\n        yp = y[n, :] + dx * (\n            b_ab4[0] * fn[n, :]\n            + b_ab4[1] * fn[n - 1, :]\n            + b_ab4[2] * fn[n - 2, :]\n            + b_ab4[3] * fn[n - 3, :]\n            + b_ab4[4] * fn[n - 4, :]\n        )\n        y[n + 1, :] = y[n, :] + dx * (\n            b_am5[0] * f(x[n + 1], yp)\n            + b_am5[1] * fn[n, :]\n            + b_am5[2] * fn[n - 1, :]\n            + b_am5[3] * fn[n - 2, :]\n            + b_am5[4] * fn[n - 3, :]\n        )\n\n    return x, y\n\n# Пример\n\ndef fun_sin(x, y):\n    return -np.sin(x)\n\n\nx_5, y_5 = am5_method(fun_sin, 0, 0.5, [1], 5)\n\nx_50, y_50 = am5_method(fun_sin, 0, 0.5, [1], 50)\n\ny_5[-1][0], y_50[-1][0]\n\n# Сходимость\n\ndef motion(x, y):\n    dfdt = np.zeros_like(y)\n    dfdt[0] = -y[1]\n    dfdt[1] = y[0]\n\n    return dfdt\n\n\ny0 = np.array([1, 0])\n\nt_01, y_01 = am5_method(motion, 0, 50, y0, 500)\nt_001, y_001 = am5_method(motion, 0, 50, y0, 200)\n\nfig = plt.figure(figsize=(10, 6))\nax1 = fig.add_subplot(121)\nax1.plot(y_01[:, 0], y_01[:, 1], label=\"$h = 0.1$\")\nax1.set_xlabel(\"$x$\")\nax1.set_ylabel(\"$y$\")\nax1.legend()\n\nax2 = fig.add_subplot(122)\nax2.plot(y_001[:, 0], y_001[:, 1], label=\"$h = 0.25$\")\nax2.set_xlabel(\"$x$\")\nax2.set_ylabel(\"$y$\")\nax2.legend()\n\nplt.show()\n\nfig = plt.figure(figsize=(10, 6))\nax1 = fig.add_subplot(121)\nax1.plot(t_01, np.sqrt(y_01[:, 1] ** 2 + y_01[:, 0] ** 2) - 1, label=\"$h = 0.1$\")\nax1.set_xlabel(\"$x$\")\nax1.set_ylabel(\"$y$\")\nax1.legend()\n\nax2 = fig.add_subplot(122)\nax2.plot(t_001, np.sqrt(y_001[:, 1] ** 2 + y_001[:, 0] ** 2) - 1, label=\"$h = 0.25$\")\nax2.set_xlabel(\"$x$\")\nax2.set_ylabel(\"$y$\")\nax2.legend()\n\nplt.show()\n\nimport math\n\n\nN = np.array([50 * 2**i for i in range(3, 10)])\n\nh = np.zeros_like(N, float)\n\ndx = 50 / N\n\nerr = np.zeros_like(h)\n\nfor i in range(len(N)):\n    n = N[i]\n    x, y = am5_method(motion, 0, 50, y0, n)\n    err[i] = np.abs(y[-1, 0] ** 2 + y[-1, 1] ** 2 - 1)\n\np = np.polyfit(np.log(dx[:-1]), np.log(err[:-1]), 1)\n\nfig = plt.figure(figsize=(14, 6))\nax1 = fig.add_subplot(111)\nax1.loglog(dx, err, \"kx\", label=\"Невязки в радиусе\")\nax1.loglog(dx, np.exp(p[1]) * dx ** p[0], label=f\"Наклон линии $r$ {p[0]:.2f}\")\nax1.set_xlabel(\"dx\")\nax1.set_ylabel(\"Ошибка\")\nax1.set_title(\"Сходимость метода Адамса-Бэфшорта\")\nax1.legend(loc=2)\n\nplt.show()"
        },
        "Метод Милна": {
            "markdown": "Метод Милна используется для решения ОДУ и систем из них и базируется на схеме предиктора-корректора:\n\nИзначально необходимо посчитать первые 3 значения $y_1, y_2, y_3$ любым доступным методом - например, методом предиктора-корректора на основе метода Эйлера\n\nПредиктор: $y_m^p = y_{m - 4} + \\frac{4h}{3}(2f_{m-3} - f_{m-2} + 2f_{m-1})$, $m = 4, 5, ...$, $f_m = f(x_m, y_m)$\n\nКорректор: $y_{m+1} = y_{m-2} + \\frac h 3 (f_{m-2} + 4f_{m-1} + f_m)$, $m = 4, 5, ...$, $f_m = f(x_m, y_m^p)$",
            "code": "import numpy as np\n\n\ndef milnes_method(f, x_0, x_n, y_0, N):\n    dx = (x_n - x_0) / N\n    x = np.linspace(x_0, x_n, N + 1)\n    y = np.zeros((N + 1, len(y_0)))\n    y[0, :] = y_0\n\n    for n in range(3):\n        yp = y[n, :] + dx * f(x[n], y[n, :])\n        y[n + 1, :] = y[n, :] + dx / 2 * (f(x[n], y[n, :]) + f(x[n + 1], yp))\n\n    for i in range(4, N + 1):\n        y_pred = y[i - 4, :] + 4 * dx / 3 * (\n            2 * f(x[i - 3], y[i - 3, :])\n            - f(x[i - 2], y[i - 2, :])\n            + 2 * f(x[i - 1], y[i - 1, :])\n        )\n        y_corrected = y[i - 2, :] + dx / 3 * (\n            f(x[i - 2], y[i - 2, :]) + 4 * f(x[i - 1], y[i - 1, :]) + f(x[i], y_pred)\n        )\n        y[i, :] = y_corrected\n\n    return x, y\n\n# Пример\n\ndef fun_sin(x, y):\n    return -np.sin(x)\n\n\nx_5, y_5 = milnes_method(fun_sin, 0, 0.5, [1], 5)\n\nx_50, y_50 = milnes_method(fun_sin, 0, 0.5, [1], 50)\n\ny_5[-1][0], y_50[-1][0]\n\n# Сходимость\n\nN = np.array([2**i for i in range(3, 12)])\n\ndx = 0.5 / N\n\nerr = np.zeros_like(dx)\n\nfor i in range(len(N)):\n    n = N[i]\n    x, y = milnes_method(fun_sin, 0, 0.5, [1], n)\n    err[i] = np.abs(np.cos(0.5) - y[-1, 0])\n\np = np.polyfit(np.log(dx[:-1]), np.log(err[:-1]), 1)\n\nfig = plt.figure(figsize=(10, 6))\nax = fig.add_subplot(111)\nax.loglog(dx, err, \"kx\", label=\"Посчитанные невязки\")\nax.loglog(dx, np.exp(p[1]) * dx ** p[0], label=f\"Наклон линии {p[0]:.2f}\")\nax.set_xlabel(\"dx\")\nax.set_ylabel(\"Ошибка\")\nax.set_title(\"Сходимость метода Милна\")\nax.legend(loc=2)\n\nplt.show()\n\ndef motion(x, y):\n    dfdt = np.zeros_like(y)\n    dfdt[0] = -y[1]\n    dfdt[1] = y[0]\n\n    return dfdt\n\n\ny0 = np.array([1, 0])\n\nt_01, y_01 = milnes_method(motion, 0, 50, y0, 500)\nt_001, y_001 = milnes_method(motion, 0, 50, y0, 5000)\n\nfig = plt.figure(figsize=(10, 6))\nax1 = fig.add_subplot(121)\nax1.plot(y_01[:, 0], y_01[:, 1], label=\"$h = 0.1$\")\nax1.set_xlabel(\"$x$\")\nax1.set_ylabel(\"$y$\")\nax1.legend()\n\nax2 = fig.add_subplot(122)\nax2.plot(y_001[:, 0], y_001[:, 1], label=\"$h = 0.01$\")\nax2.set_xlabel(\"$x$\")\nax2.set_ylabel(\"$y$\")\nax2.legend()\n\nplt.show()"
        },
        "Согласованность устойчивость сходимость условия устойчивости": {
            "markdown": "Согласованность численного метода означает, что при уменьшении шага дискретизации $ h $ аппроксимация исходного дифференциального уравнения становится всё более точной. Иными словами, локальная погрешность метода стремится к нулю при $ h \\to 0 $.\n\n**Формально**, метод считается согласованным, если:\n\n$\n\\lim_{h \\to 0} \\tau(h) = 0,\n$\n\nгде $ \\tau(h) $ — локальная погрешность на шаге $ h $.\n\nУстойчивость численного метода — это его способность ограничивать рост погрешностей, возникающих из-за ошибок округления, начальных условий или возмущений правой части уравнения. Метод называется устойчивым, если эти ошибки не накапливаются в процессе выполнения разностной схемы, а остаются ограниченными или стремятся к нулю.\n\n### **Основные условия устойчивости**\n\n1. **Чувствительность к возмущениям:**\n   - Метод считается устойчивым, если малое возмущение правой части уравнения или начальных условий вызывает лишь незначительное изменение численного решения.\n   - Формально:\n     $\n     \\lim_{\\varepsilon \\to 0} x_\\varepsilon(t) = x(t),\n     $\n     где $ x_\\varepsilon(t) $ — решение возмущённого уравнения, а $ x(t) $ — точное решение.\n\n2. **Анализ погрешностей:**\n   - На каждом шаге вычислений погрешность $ \\varepsilon_i $ может быть вызвана округлениями, численным методом или неточностью данных. Метод устойчив, если на $ i+1 $-м шаге погрешность $ \\varepsilon_{i+1} $ не превышает $ \\varepsilon_i $ в геометрической прогрессии.\n\n3. **Корневое условие:**\n   - Условие устойчивости для линейных схем часто проверяется с использованием спектрального анализа:\n     - Корни характеристического уравнения разностной схемы $ \\lambda $ должны удовлетворять:\n       $\n       |\\lambda| \\leq 1.\n       $\n     - Если $ |\\lambda| > 1 $, погрешности будут увеличиваться, и метод станет неустойчивым.\n\n4. **Условие шага сетки (явные методы):**\n   - Для явных методов устойчивость часто требует ограничения на шаг времени $ h $. Например, для метода Эйлера:\n     $\n     h \\leq \\frac{2}{|\\lambda_{\\max}|},\n     $\n     где $ \\lambda_{\\max} $ — максимальное по модулю собственное значение матрицы коэффициентов.\n\n5. **Область устойчивости:**\n   - Для некоторых методов устойчивость зависит от диапазона параметра $ z = h \\lambda $, где $ h $ — шаг, а $ \\lambda $ — собственное значение задачи. Область устойчивости — это множество значений $ z $, для которых метод остаётся устойчивым.\n\n\n### **Пример анализа устойчивости**\n\nДля метода Эйлера при решении простого линейного уравнения:\n$\n\\frac{du}{dt} = -Au,\n$\nчисленный метод задаётся:\n$\nu_{n+1} = u_n + h(-A u_n) = u_n (1 - hA).\n$\n\n- Собственное значение: $ \\lambda = 1 - hA $.\n- Условие устойчивости:\n  $\n  |1 - hA| \\leq 1.\n  $\n\nИз этого следует ограничение на шаг:\n$\nh \\leq \\frac{2}{A}.\n$\n\n**Сходимость** численного метода — это его способность обеспечивать приближение численного решения к точному решению задачи при уменьшении шага дискретизации или при увеличении числа итераций.\n\nСходимость гарантирует, что численный метод при достаточном уменьшении шага или увеличении числа итераций будет давать решение, близкое к истинному. Без этого свойства метод может быть бесполезен, так как не обеспечит необходимой точности.\n\n\nСогласно теореме Лакса-Рихтмайера, для линейных разностных схем с постоянными коэффициентами сходимость эквивалентна сочетанию согласованности и устойчивости:\n\n$\n\\text{Сходимость} \\iff \\text{Согласованность} + \\text{Устойчивость}.\n$",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\nfrom scipy.integrate import solve_ivp\n\n\ndef motion(t, y):\n    dfdt = np.zeros_like(y)\n    dfdt[0] = -y[1]\n    dfdt[1] = y[0]\n    return dfdt\n\n\ndef method_euler(func, t0, t_end, y0, n_steps):\n    t = np.linspace(t0, t_end, n_steps + 1)\n    h = t[1] - t[0]\n    y = np.zeros((n_steps + 1, len(y0)))\n    y[0] = y0\n    for i in range(n_steps):\n        y[i + 1] = y[i] + h * func(t[i], y[i])\n    return t, y\n\n\n# Начальные условия\ny0 = np.array([1, 0])\n\nt_01, y_01 = method_euler(motion, 0, 50, y0, 500)\nt_001, y_001 = method_euler(motion, 0, 50, y0, 5000)\n\n# решение методом Рунге-Кутты (используем solve_ivp)\nsolution = solve_ivp(\n    motion, [0, 50], y0, method=\"RK45\", t_eval=np.linspace(0, 50, 5000)\n)\n\nfig = plt.figure(figsize=(12, 6))\nplt.suptitle(\"Сравнение сходимости\")\n\nax1 = fig.add_subplot(131)\nax1.plot(y_01[:, 0], y_01[:, 1], label=\"Эйлер h=0.1\")\nax1.set_title(\"Метод Эйлера (h=0.1)\")\nax1.set_xlabel(\"x\")\nax1.set_ylabel(\"y\")\nax1.legend()\n\nax2 = fig.add_subplot(132)\nax2.plot(y_001[:, 0], y_001[:, 1], label=\"Эйлер h=0.01\")\nax2.set_title(\"Метод Эйлера (h=0.01)\")\nax2.set_xlabel(\"x\")\nax2.set_ylabel(\"y\")\nax2.legend()\n\n# Точное решение (метод Рунге-Кутты)\nax3 = fig.add_subplot(133)\nax3.plot(\n    solution.y[0], solution.y[1], label=\"Точное (метод Рунге-Кутты 45)\", color=\"red\"\n)\nax3.set_title(\"Точное решение (метод Рунге-Кутты 45)\")\nax3.set_xlabel(\"x\")\nax3.set_ylabel(\"y\")\nax3.legend()\n\nplt.tight_layout()\nplt.show()"
        }
    },
    "Моделирование сигналов с помощью рядов Фурье": {
        "Моделирование волны с использованием математических инструментов амплитуда период длина волны частота Герц дискретизация частота дискретизации фаза угловая частота": {
            "markdown": "Формула волны: $y(t) = A\\sin(\\omega t + \\phi)$, где:\n\n$A$ - амплитуда волны, мера её силы (интенсивности)\n\n$\\phi$ - начальная фаза гармонического колебания\n\n$\\omega$ - угловая частота, мера частоты движения, сколько циклов проходит за единицу времени\n\n$\\omega = \\frac{2\\pi}{T} = 2\\pi f$, где:\n\n$T$ - период волны\n\n$f$ - частота волны, количество колебаний за секунду, измеряется в Герцах (Гц) / 1 Гц = 1 колебание в секунду, 10 Гц = 10 колебаний в секунду и т.п.\n\n$T = \\frac 1 f$\n\n$\\lambda = vT$ - длина волны, расстояние, которое проходит волна за один период\n\n$v$ - скорость волны\n\nДискретизация - процесс перевода непрерывного аналогового сигнала в цифровой сигнал. Для этого непрерывный сигнал разбивают на диксретные отрезки, которые затем преобразуют в цифровую форму\n\nЧастота дискретизации - количество измерений входного сигнала за одну секунду, измеряется в Гц",
            "code": "import numpy as np\nimport matplotlib.pyplot as plt\n\nx = np.linspace(0, 20, 201)\ny = np.sin(x)\n\nplt.figure(figsize=(8, 6))\nplt.plot(x, y)\nplt.ylabel(\"Амплитуда\")\nplt.xlabel(\"Положение (x)\")\nplt.show()\n\nx = np.linspace(0, 20, 201)\n\nplt.figure(figsize=(8, 16))\ntimes = np.arange(5)\nn = len(times)\n\nfor t in times:\n    plt.subplot(n, 1, t + 1)\n    y = np.sin(x + t)\n    plt.plot(x, y)\n    plt.plot(x[25], y[25], \"ro\")\n    plt.title(f\"t={t}\")\n    plt.ylabel(\"Амплитуда\")\n\nplt.xlabel(\"Положение (x)\")\n\nplt.show()\n\nsr = 100\nts = 1 / sr\nt = np.arange(0, 1, ts)\n\nfreq = 5\ny = np.sin(2 * np.pi * freq * t)\n\nplt.figure(figsize=(9, 9))\nplt.subplot(311)\nplt.plot(t, y, \"b\")\nplt.ylabel(\"Амплитуда\")\n\nfreq = 10\ny = np.sin(2 * np.pi * freq * t)\n\nplt.subplot(312)\nplt.plot(t, y, \"b\")\nplt.ylabel(\"Больше частота\")\n\nsr = 100\nts = 1 / sr\nt = np.arange(0, 1, ts)\n\nfreq = 5\ny = 5 * np.sin(2 * np.pi * freq * t)\n\nplt.subplot(313)\nplt.plot(t, y, \"b\")\nplt.ylabel(\"Больше амплитуда\")\n\nplt.xlabel(\"Время\")\n\nplt.show()"
        },
        "Дискретное преобразование Фурье обратное дискретное преобразование Фурье их ограничения симметрии в дискретном преобразовании Фурье": {
            "markdown": "Преобразование Фурье помогает разложить любой сигнал на сумму простых синусоидальных волн, что позволяет легко измерять их частоту, амплитуду, фазу.\n\nИспользуя DFT, можно представить сигнал как сумму синусоид с разными частотами.\n\nДля дискретного преобразования Фурье используется следующая формула:\n\n$X(n) = \\sum_{k=0}^{N - 1}{x(k)e^{-j\\frac{2\\pi nk}{N}}}$\n\nОбратное:\n\n$x(k) = \\frac{1}{N} \\sum_{n=0}^{N - 1}{X(n)e^{j\\frac{2\\pi nk}{N}}}$\n\nЕсть некоторые нюансы, возникающие вследствие дискретности, например:\n- при перемножении сигналов их длины должны быть одинаковыми (N);\n- суммирование элементов произведения производиться по одному периоду (полученный результат называется круговой сверткой спектров исходных сигналов)\n\nГлавная проблема данного алгоритма - сложность $O(N^2)$, что приводит к большим временным затратам при обработке сигнала\n\nДействительная часть преобразования Фурье является чётной функцией, а значит дискретное преобразование Фурье обладает свойством симметрии относительно 0.",
            "code": "def DFT(x):\n    N = len(x)\n    n = np.arange(N)\n    k = n.reshape((N, 1))\n    e = np.exp(-2j * np.pi * k * n / N)\n\n    X = np.dot(e, x)\n\n    return X\n\ndef IDFT(x):\n    N = len(x)\n    n = np.arange(N)\n    k = n.reshape((N, 1))\n    e = np.exp(2j * np.pi * k * n / N)\n\n    X = np.dot(e, x)\n\n    return (X / N).real\n\n# ускорение с использованием numba:\n\nfrom numba import njit, prange\n\n\n@njit(parallel=True)\ndef DFT(x):\n    N = len(x)\n    res = np.zeros_like(x, dtype=np.complex128)\n    for n in prange(N):\n        summ = 0\n        for k in prange(N):\n            summ += x[k] * np.exp(-2j * np.pi * k * n / N)\n        res[n] = summ\n\n    return res\n\n\n@njit(parallel=True)\ndef IDFT(x):\n    N = len(x)\n    res = np.zeros_like(x, dtype=np.complex128)\n    for n in prange(N):\n        summ = 0\n        for k in prange(N):\n            summ += x[k] * np.exp(2j * np.pi * k * n / N)\n        res[n] = summ\n\n    return (1 / N * res).real\n\n# Пример\n\nsr = 100\nts = 1 / sr\nt = np.arange(0, 1, ts)\nplt.figure(figsize=(9, 9))\n\nfreq = 1\nx = 3 * np.sin(2 * np.pi * freq * t)\n\nfreq = 4\nx += 1 * np.sin(2 * np.pi * freq * t)\n\nfreq = 7\nx += 0.5 * np.sin(2 * np.pi * freq * t)\n\nplt.plot(t, x)\nplt.ylabel(\"Амплитуда\")\n\nplt.show()\n\nX = DFT(x)\n\nN = len(x)\nn = np.arange(N)\nT = N / sr\nfreq = n / T\n\nplt.figure(figsize=(8, 6))\nplt.stem(freq, abs(X), \"b\", markerfmt=\" \", basefmt=\"-b\")\nplt.xlabel(\"Частота, Гц\")\nplt.ylabel(\"Амплитуда DFT\")\nplt.show()\n\n# Из-за симметрии можно избавиться от симметричной части:\n\nn_oneside = N // 2\nf_oneside = freq[:n_oneside]\n\nX_oneside = X[:n_oneside] / n_oneside\n\nplt.figure(figsize=(12, 6))\n\nplt.subplot(121)\n\nplt.stem(f_oneside, abs(X_oneside), \"b\", markerfmt=\" \", basefmt=\"-b\")\nplt.xlabel(\"Частота, Гц\")\nplt.ylabel(\"Амплитуда DFT\")\n\nplt.subplot(122)\nplt.stem(f_oneside, abs(X_oneside), \"b\", markerfmt=\" \", basefmt=\"-b\")\nplt.xlim(0, 10)\nplt.xlabel(\"Частота, Гц\")\nplt.ylabel(\"Амплитуда DFT\")\n\nplt.show()"
        },
        "Быстрое преобразование Фурье его принципы фильтрация сигнала с использованием быстрого преобразования Фурье": {
            "markdown": "Быстрое преобразование Фурье использует принцип \"разделяй и властвуй\" над DFT, чтобы ускорить работу алгоритма до $O(N\\log{N})$\n\nПоследовательность данных сигнала разделяется на две подпоследовательности на чётные и нечётные элементы и обрабатывается рекурсивно до тех пор, пока длина подпоследовательности не будет достаточной маленькой для вычисления преобразования с помощью DFT\n\nИз-за данного подхода размер исходного массива данных строго ограничен степенью двойки\n\nФормально, на каждом шаге массив данных разбивается на 2 (с чётными и нечётными по счёту элементами), применяется рекурсивно к ним алгоритм, а дальше идёт объединение с \"поворачивающим множителем\":\n\n$W(N) = e^{-2\\frac{j\\pi k}{N}}$, $k = 1...N$\n\nПервая часть:\n\n$X_{2n} + W_1^{N/2} \\cdot X_{2n+1}$\n\nВторая часть:\n\n$X_{2n} + W_{N/2}^N \\cdot X_{2n+1}$\n\nДля фильтрации сигнала необходимо получить спектограмму с помощью преобразования Фурье, выделить выходящие за некоторые пределы (threshold) частоты и удалить их, а зател преобразовать обратно",
            "code": "import matplotlib.pyplot as plt\nimport numpy as np\n\n\ndef FFT(x):\n    N = len(x)\n\n    if N == 1:\n        return x\n\n    else:\n        X_even = FFT(x[::2])\n        X_odd = FFT(x[1::2])\n        factor = np.exp(-2j * np.pi * np.arange(N) / N)\n        X = np.concatenate(\n            [X_even + factor[: N // 2] * X_odd, X_even + factor[N // 2 :] * X_odd]\n        )\n\n        return X\n\ndef IFFT(x):\n    N = len(x)\n\n    if N == 1:\n        return x\n\n    else:\n        X_even = IFFT(x[::2])\n        X_odd = IFFT(x[1::2])\n        factor = np.exp(2j * np.pi * np.arange(N) / N)\n\n        X = np.concatenate(\n            [X_even + factor[: N // 2] * X_odd, X_even + factor[N // 2 :] * X_odd]\n        )\n\n        return X / 2\n\n# Пример\n\nsr = 128\nts = 1 / sr\nt = np.arange(0, 1, ts)\nplt.figure(figsize=(10, 5))\n\nfreq = 1\nx = 3 * np.sin(2 * np.pi * freq * t)\n\nfreq = 4\nx += 1 * np.sin(2 * np.pi * freq * t)\n\nfreq = 7\nx += 0.5 * np.sin(2 * np.pi * freq * t)\n\nplt.plot(t, x, \"b\")\nplt.ylabel(\"Амплитуда\")\nplt.xlabel(\"Время\")\n\nplt.show()\n\nX = FFT(x)\n\nN = len(X)\nn = np.arange(N)\n\nT = N / sr\n\nfreq = n / T\n\nn_oneside = N // 2\nf_oneside = freq[:n_oneside]\nX_oneside = X[:n_oneside] / n_oneside\n\n\nplt.figure(figsize=(10, 5))\n\nplt.subplot(121)\n\nplt.stem(freq, abs(X), markerfmt=\" \", basefmt=\"b\")\nplt.ylabel(\"Амплитуда\")\nplt.xlabel(\"Частота, Гц\")\n\nplt.subplot(122)\n\nplt.stem(f_oneside, abs(X_oneside), markerfmt=\" \", basefmt=\"b\")\nplt.ylabel(\"Амплитуда\")\nplt.xlabel(\"Частота, Гц\")\n\nplt.show()"
        },
        "Операции свёртки связь с быстрым преобразованием Фурье операции дискретной свёртки": {
            "markdown": "#### **Дискретная свёртка**\nДискретная свёртка — это операция, объединяющая две последовательности (сигналы) в новую последовательность, отражающую влияние одного сигнала на другой. Она определяется как:\n\n$\ny[n] = \\sum_{k=-\\infty}^{\\infty} x[k] \\cdot h[n - k]\n$\n\nДискретная свёртка двух векторов $x$ и $h$ длины $n$ может быть эффективно вычислена с использованием дискретного преобразования Фурье (ДПФ). Процесс включает следующие шаги:\n\n1. **Преобразование входных векторов в частотную область**: вычисляются ДПФ для $x$ и $h$, обозначаемые как $\\text{DFT}(x)$ и $\\text{DFT}(h)$ соответственно.\n\n2. **Поэлементное умножение спектров**: выполняется поэлементное умножение полученных спектров, результатом которого является $\\text{DFT}(x) \\cdot \\text{DFT}(h)$.\n\n3. **Обратное преобразование Фурье**: применяется обратное дискретное преобразование Фурье (IDFT) к результату умножения, чтобы получить итоговый вектор свёртки $y$ в исходной области.\n\nИтоговая формула свёртки через ДПФ записывается как:\n\n$$\ny = \\text{IDFT}(\\text{DFT}(x) \\cdot \\text{DFT}(h))\n$$\n\nИтоговое применение свёртки делает за O(n logn)\n",
            "code": "# пример на python\n\n\ndef discrete_convolution(x, h):\n    n = len(x)\n    m = len(h)\n    y = [0] * (n + m - 1)\n\n    for i in range(len(y)):\n        for j in range(m):\n            if 0 <= i - j < n:\n                y[i] += x[i - j] * h[j]\n    return y\n\n\nx = [1, 2, 3]  # сигнал\nh = [0, 1, 0.5]  # импульсный отклик\n\nresult = discrete_convolution(x, h)  # будто h[::-1] и прошлись скользящим окном\n\nimport numpy as np\n\n\n# пример кода\ndef discrete_convolution_dft(x, h):\n    n = len(x) + len(h) - 1\n\n    x_padded = np.pad(x, (0, n - len(x)), mode=\"constant\")\n    h_padded = np.pad(h, (0, n - len(h)), mode=\"constant\")\n\n    # Вычисляем преобразование Фурье\n    X = np.fft.fft(x_padded)\n    H = np.fft.fft(h_padded)\n\n    # поэлементное умножение в частотной области\n    Y = X * H\n\n    # обратное преобразование Фурье\n    y = np.fft.ifft(Y).real\n\n    return y\n\n\nx = [1, 2, 3]  # сигнал\nh = [0, 1, 0.5]  # импульсный отклик\n\nresult = discrete_convolution_dft(x, h)\nresult"
        },
        "Дискретная свёртка и Тёплицевы матрицы Ганкелевы матрицы": {
            "markdown": "Матрица Тёплицевой, если её элементы определены как:\n\n$a_{ij} = t_{i-j}$\n\n- Все элементы на диагонали этой матрицы одинаковы, она определяется двумя векторами - верхней строкой и первым столбцом (2n-1 - параметр)\n- Плотная матрица\n- Основная операция для вычисления дискретной свёртки - произведение матрицы Тёплицевой на вектор\n\n### Дискретная свёртка и Тёплицевы матрицы (Ганкелевы матрицы)\n\n\n#### **Дискретная свёртка**\nДискретная свёртка — это операция, объединяющая две последовательности (сигналы) в новую последовательность, отражающую влияние одного сигнала на другой. Она определяется как:\n\n$\ny[n] = \\sum_{k=-\\infty}^{\\infty} x[k] \\cdot h[n - k]\n$\n\n#### **Тёплицевы матрицы**\nТёплицевы матрицы — это квадратные матрицы, у которых элементы постоянны вдоль диагоналей, параллельных главной. <br>\nВсе её элементы определены как:\n\n$a_{ij} = t_{i-j}$\n\n- Все элементы на диагонали этой матрицы одинаковы, она определяется двумя векторами - верхней строкой и первым столбцом (2n-1 - параметр)\n- Плотная матрица\n- Основная операция для вычисления дискретной свёртки - произведение матрицы Тёплицевой на вектор\nПример:\n\n$\nT = \n\\begin{pmatrix}\nt_0 & t_{-1} & t_{-2} & \\cdots & t_{-n+1} \\\\\nt_1 & t_0 & t_{-1} & \\cdots & t_{-n+2} \\\\\nt_2 & t_1 & t_0 & \\cdots & t_{-n+3} \\\\\n\\vdots & \\vdots & \\vdots & \\ddots & \\vdots \\\\\nt_{n-1} & t_{n-2} & t_{-n+3} & \\cdots & t_0\n\\end{pmatrix}\n$\n\n#### **Ганкелевы матрицы**\nГанкелевы матрицы — это матрицы, у которых элементы постоянны вдоль побочных диагоналей. Элемент $a_{i,j}$ зависит только от суммы индексов $i + j$. Пример:\n\n$\nH = \n\\begin{pmatrix}\nh_0 & h_1 & h_2 & \\cdots & h_{n-1} \\\\\nh_1 & h_2 & h_3 & \\cdots & h_n \\\\\nh_2 & h_3 & h_4 & \\cdots & h_{n+1} \\\\\n\\vdots & \\vdots & \\vdots & \\ddots & \\vdots \\\\\nh_{n-1} & h_n & h_{n+1} & \\cdots & h_{2n-2}\n\\end{pmatrix}\n$\n\n\n#### **Связь между дискретной свёрткой и Тёплицевыми матрицами**\nОперацию свёртки можно представить в виде матричного умножения, где одна из последовательностей формирует тёплицеву матрицу, а другая — вектор. Это позволяет эффективно вычислять свёртку с использованием матричных методов. Например:\n\nЕсли $x[n]$ — сигнал, а $h[n]$ — импульсный отклик, то свёртка может быть выражена как:\n\n$\ny = T \\cdot x\n$\n\nгде $T$ — Тёплицева матрица, сформированная из $h[n]$.",
            "code": "# пример на python\n\ndef discrete_convolution(x, h):\n    n = len(x)\n    m = len(h)\n    y = [0] * (n + m - 1)\n\n    for i in range(len(y)):\n        for j in range(m):\n            if 0 <= i - j < n:\n                y[i] += x[i - j] * h[j]\n    return y\n\n\nx = [1, 2, 3]  # сигнал\nh = [0, 1, 0.5]  # импульсный отклик\n\nresult = discrete_convolution(x, h)  # будто h[::-1] и прошлись скользящим окном\n\n# через матрицу Тёплицева\n\nimport numpy as np\nfrom scipy.linalg import toeplitz\n\nx = np.array([1, 2, 3])\nh = np.array([0, 1, 0.5])\n\ntoeplitz_matrix = toeplitz(\n    np.r_[h, np.zeros(len(x) - 1)],  # первый столбец\n    np.zeros(len(h) + len(x) - 1),  # первый ряд (нулевой)\n)\n\n# расширяем x нулями\nx_padded = np.r_[x, np.zeros(len(h) - 1)]\n\n# свёртка через умножение матрицы на вектор\nresult = np.dot(toeplitz_matrix, x_padded)\n\ntoeplitz_matrix, result"
        },
        "Циркулянтные матрицы Матрицы Фурье": {
            "markdown": "### Циркулянтные матрицы. Матрицы Фурье\n\n#### **Циркулянтные матрицы**\nЦиркулянтные матрицы — это особый класс квадратных матриц, каждая строка которых является циклическим сдвигом предыдущей. Они имеют следующий вид:\n\n$\nC = \\begin{pmatrix}\nc_0 & c_{n-1} & \\cdots & c_2 & c_1 \\\\\nc_1 & c_0 & \\cdots & c_3 & c_2 \\\\\n\\vdots & \\vdots & \\ddots & \\vdots & \\vdots \\\\\nc_{n-1} & c_{n-2} & \\cdots & c_1 & c_0\n\\end{pmatrix}\n$\n\nЭлемент $ c_{i,j} $ определяется как:\n$\nc_{i,j} = c_{(j-i) \\mod n}\n$\n\nСвёртка в пространственной области соответствует поэлементному умножению в частотной области. Матрицы Фурье позволяют быстро перейти в частотную область и обратно.\nБыстродействие:\n\nПрямое вычисление свёртки с помощью циркулянтной матрицы имеет сложность O($n^2$), но использование алгоритмов быстрого преобразования Фурье (БПФ) снижает сложность до O($n\\log_2n$).\n\n#### **Матрицы Фурье**\nМатрицы Фурье связаны с циркулянтными матрицами через дискретное преобразование Фурье (ДПФ). ДПФ диагонализует циркулянтные матрицы, что упрощает операции, такие как свёртка, сводя их к поэлементному умножению в частотной области.\n\n### Связь между циркулянтными матрицами и матрицами Фурье\n\nСвязь между циркулянтными матрицами и матрицами Фурье выражается через **диагонализацию циркулянтных матриц** с использованием дискретного преобразования Фурье (ДПФ). \n\n#### 1. **Диагонализация циркулянтной матрицы**\n\nПусть $C$ — циркулянтная матрица размером $n \\times n$, а $F$ — матрица дискретного преобразования Фурье:\n\n$\nF = \\frac{1}{\\sqrt{n}} \\begin{pmatrix}\n1 & 1 & 1 & \\cdots & 1 \\\\\n1 & \\omega & \\omega^2 & \\cdots & \\omega^{n-1} \\\\\n1 & \\omega^2 & \\omega^4 & \\cdots & \\omega^{2(n-1)} \\\\\n\\vdots & \\vdots & \\vdots & \\ddots & \\vdots \\\\\n1 & \\omega^{n-1} & \\omega^{2(n-1)} & \\cdots & \\omega^{(n-1)(n-1)}\n\\end{pmatrix}\n$\n\nгде $\\omega = e^{-2\\pi i / n}$ — корень из единицы.\n\nЦиркулянтная матрица $C$ диагонализуется следующим образом:\n\n$\nC = F^* \\Lambda F\n$\n\nгде:\n- $F^*$ — эрмитово-сопряжённая (транспонированная и комплексно сопряжённая) матрица $F$,\n- $\\Lambda$ — диагональная матрица собственных значений $C$, которая вычисляется как $\\Lambda = F \\cdot c$,  \n  а $c$ — первый столбец циркулянтной матрицы $C$.\n\n#### 2. **Свёртка через преобразование Фурье**\n\nОперация свёртки $y = x * h$, где $x$ и $h$ — векторы длины $n$, в матричном виде через циркулянтную матрицу $C$ выражается так:\n\n$\ny = C \\cdot x\n$\n\nС использованием диагонализации $C = F^* \\Lambda F$, это преобразуется в:\n\n$\ny = F^* \\Lambda F \\cdot x\n$\n\nИнтерпретация:\n1. Преобразование $x$ в частотную область:\n   $\n   X = F \\cdot x\n   $\n2. Поэлементное умножение в частотной области:\n   $\n   Y = \\Lambda \\cdot X\n   $\n3. Обратное преобразование Фурье для получения результата в исходной области:\n   $\n   y = F^* \\cdot Y\n   $\n\n#### Итоговая формула свёртки\n\nСвёртка через ДПФ в частотной области записывается так:\n\n$\ny = F^* \\cdot (\\text{diag}(F \\cdot h) \\cdot F \\cdot x)\n$\n\nгде $\\text{diag}(F \\cdot h)$ — диагональная матрица преобразования Фурье от импульсного отклика $h$.\n\nИли в упрощённой записи:\n\n$\ny = \\text{IDFT}(\\text{DFT}(x) \\cdot \\text{DFT}(h))\n$\n\nгде $\\text{DFT}$ и $\\text{IDFT}$ — прямое и обратное дискретное преобразование Фурье.\n",
            "code": "# пример вычисления циркулянтной матрицы вручную и её применения для свёртки\n\ndef circulant_matrix(first_column, from_row: bool = False):\n    if not from_row:\n        first_column = first_column[::-1]\n        first_column = [first_column[-1]] + first_column[:-1]\n    n = len(first_column)\n    circ_matrix = []\n    for i in range(n):\n        circ_matrix.append(first_column[i:] + first_column[:i])\n    return np.array(circ_matrix)\n\n\ndef circular_convolution(x, h):\n    C = circulant_matrix(h)  # циркулянтная матрица, построенная из h\n    return np.dot(C, x)  # умножение на сигнал x\n\n\nx = [1, 2, 3]  # сигнал\nh = [0, 1, 0.5]  # импульсный отклик\n\nresult = circular_convolution(x, h)\ncirculant_matrix(h), result\n\nimport numpy as np\n\ndef dft_circular_convolution(x, h):\n    n = len(x)\n\n    if len(h) != n:\n        raise ValueError(\"Длины сигналов должны совпадать для циклической свёртки.\")\n\n    # DFT\n    X = np.fft.fft(x)\n    H = np.fft.fft(h)\n\n    # поэлементное умножение в частотной области\n    Y = X * H\n\n    # IDFT\n    y = np.fft.ifft(Y).real\n\n    return y\n\n\nx = np.array([1, 2, 3])\nh = np.array([0, 1, 0.5])\n\n\n# циклическая свёртка через DFT\nresult_dft = dft_circular_convolution(x, h)\n\nresult_dft"
        },
        "Быстрый матвек с циркулянтом": {
            "markdown": "#### **Циркулянтная матрица**\nЦиркулянтная матрица — это особый класс квадратных матриц, каждая строка которых является циклическим сдвигом предыдущей. Вид циркулянтной матрицы:\n\n$\nC = \\begin{pmatrix}\nc_0 & c_{n-1} & \\cdots & c_2 & c_1 \\\\\nc_1 & c_0 & \\cdots & c_3 & c_2 \\\\\n\\vdots & \\vdots & \\ddots & \\vdots & \\vdots \\\\\nc_{n-1} & c_{n-2} & \\cdots & c_1 & c_0\n\\end{pmatrix}\n$\n\n#### **Матвек с циркулянтной матрицей**\nМатвек (умножение матрицы на вектор) для циркулянтной матрицы \\( C \\) и вектора \\( v \\) выполняется как:\n\n$\ny = C \\cdot v\n$\n\nГде \\( y \\) — результат умножения. <br>\nОднако циркулянтные матрицы диагонализуемы в базисе матриц Фурье, что позволяет заменить прямое матричное умножение более быстрым алгоритмом.\n\nС использованием диагонализации $C = F^* \\Lambda F$, это преобразуется в:\n\n$\ny = F^* \\Lambda F \\cdot v\n$\n\nИнтерпретация:\n1. Преобразование $v$ в частотную область:\n   $\n   V = F \\cdot v\n   $\n2. Поэлементное умножение в частотной области:\n   $\n   Y = \\Lambda \\cdot V\n   $\n3. Обратное преобразование Фурье для получения результата в исходной области:\n   $\n   y = F^* \\cdot Y\n   $\n\n#### Итоговая формула для матвека\n\nУмножение циркулянтной матрицы на вектор через дискретное преобразование Фурье (ДПФ) записывается так:\n\n$\ny = F^* \\cdot (\\text{diag}(F \\cdot c) \\cdot F \\cdot x)\n$\n\nгде $\\text{diag}(F \\cdot c)$ — диагональная матрица преобразования Фурье от первого столбца циркулянтной матрицы $c$.\n\nИли в упрощённой записи:\n\n$\ny = \\text{IDFT}(\\text{DFT}(c) \\cdot \\text{DFT}(x))\n$\n\nгде $\\text{DFT}$ и $\\text{IDFT}$ — прямое и обратное дискретное преобразование Фурье.\n\nЭта формула позволяет эффективно вычислять результат умножения вектора на циркулянтную матрицу, используя свойства преобразования Фурье.",
            "code": "def circulant_matrix(first_column, from_row: bool = False):\n    if not from_row:\n        first_column = first_column[::-1]\n        first_column = [first_column[-1]] + first_column[:-1]\n    n = len(first_column)\n    circ_matrix = []\n    for i in range(n):\n        circ_matrix.append(first_column[i:] + first_column[:i])\n    return np.array(circ_matrix)\n\n\nc = [1, 2, 3, 4]\nv = [0.5, 1.0, -1.0, 2.0]\n\nC = circulant_matrix(c)\n\nC @ v\n\nimport numpy as np\n\n\n# через DFT\ndef fast_circulant_matvec(c, v):\n    # вычисляем DFT первого столбца циркулянтной матрицы и вектора\n    C_dft = np.fft.fft(c)\n    V_dft = np.fft.fft(v)\n\n    # поэлементное умножение в частотной области\n    Y_dft = C_dft * V_dft\n\n    # обратное преобразование Фурье\n    result = np.fft.ifft(Y_dft).real\n\n    return result\n\n\nc = [1, 2, 3, 4]\nv = [0.5, 1.0, -1.0, 2.0]\n\nresult = fast_circulant_matvec(c, v)\nresult"
        }
    }
}

themes_numerated = {
    i: {
        j: {k: v for k, (k_name, v) in enumerate(q.items())}
        for j, (q_name, q) in enumerate(v.items())
    }
    for i, (theme_name, v) in enumerate(themes.items())
}

def get(i=None, j=None, k=None, silent: bool = False):
    if i is None or j is None or k is None:
        info_string = ''
        for i, (theme, questions) in enumerate(themes.items()):
            info_string += f"{i} {theme}\n"
            for j, (question, codes) in enumerate(questions.items()):
                info_string += f'{"-"*2} {i}.{j} {question}\n'
                for k, code in enumerate(codes.keys()):
                    info_string += f'{"-"*4} {i}.{j}.{k} {code}\n'
        if silent:
            pyperclip.copy(info_string)
        else:
            print(info_string)
    else:
        pyperclip.copy(themes_numerated[i][j][k].strip())
