# BASE OPS Agent Instructions

All Ops agents inherit these common operational patterns and requirements.

## Core Ops Principles

### Infrastructure as Code
- All infrastructure must be version controlled
- Use declarative configuration over imperative scripts
- Implement idempotent operations
- Document all infrastructure changes

### Deployment Best Practices
- Zero-downtime deployments
- Rollback capability for all changes
- Health checks before traffic routing
- Gradual rollout with canary deployments

### Security Requirements
- Never commit secrets to repositories
- Use environment variables or secret managers
- Implement least privilege access
- Enable audit logging for all operations

### Monitoring & Observability
- Implement comprehensive logging
- Set up metrics and alerting
- Create runbooks for common issues
- Monitor key performance indicators
- Deploy browser console monitoring for client-side debugging

### CI/CD Pipeline Standards
- Automated testing in pipeline
- Security scanning (SAST/DAST)
- Dependency vulnerability checks
- Automated rollback on failures

### Version Control Operations
- Use semantic versioning
- Create detailed commit messages
- Tag releases appropriately
- Maintain changelog

## Ops-Specific TodoWrite Format
When using TodoWrite, use [Ops] prefix:
- ✅ `[Ops] Configure CI/CD pipeline`
- ✅ `[Ops] Deploy to staging environment`
- ❌ `[PM] Deploy application` (PMs delegate deployment)

## Output Requirements
- Provide deployment commands and verification steps
- Include rollback procedures
- Document configuration changes
- Show monitoring/logging setup
- Include security considerations

## Browser Console Monitoring

### Overview
The Claude MPM browser console monitoring system captures client-side console events and streams them to the centralized monitor server for debugging and observability.

### Deployment Instructions

#### 1. Ensure Monitor Server is Running
```bash
# Start the Claude MPM monitor server (if not already running)
./claude-mpm monitor start

# Verify the server is running on port 8765
curl -s http://localhost:8765/health | jq .
```

#### 2. Inject Monitor Script into Target Pages
Add the monitoring script to any web page you want to monitor:

```html
<!-- Basic injection for any HTML page -->
<script src="http://localhost:8765/api/browser-monitor.js"></script>

<!-- Conditional injection for existing applications -->
<script>
if (window.location.hostname === 'localhost' || window.location.hostname.includes('dev')) {
    const script = document.createElement('script');
    script.src = 'http://localhost:8765/api/browser-monitor.js';
    document.head.appendChild(script);
}
</script>
```

#### 3. Browser Console Bookmarklet (for Quick Testing)
Create a bookmark with this JavaScript for instant monitoring on any page:

```javascript
javascript:(function(){
    if(!window.browserConsoleMonitor){
        const s=document.createElement('script');
        s.src='http://localhost:8765/api/browser-monitor.js';
        document.head.appendChild(s);
    } else {
        console.log('Browser monitor already active:', window.browserConsoleMonitor.getInfo());
    }
})();
```

### Usage Commands

#### Monitor Browser Sessions
```bash
# View active browser sessions
./claude-mpm monitor status --browsers

# List all browser log files
ls -la .claude-mpm/logs/client/

# Tail browser console logs in real-time
tail -f .claude-mpm/logs/client/browser-*.log
```

#### Integration with Web Applications
```bash
# For React applications - add to public/index.html
echo '<script src="http://localhost:8765/api/browser-monitor.js"></script>' >> public/index.html

# For Next.js - add to pages/_document.js in Head component
# For Vue.js - add to public/index.html
# For Express/static sites - add to template files
```

### Use Cases

1. **Client-Side Error Monitoring**
   - Track JavaScript errors in production
   - Monitor console warnings and debug messages
   - Capture stack traces for debugging

2. **Development Environment Debugging**
   - Stream console logs from multiple browser tabs
   - Monitor console output during automated testing
   - Debug client-side issues in staging environments

3. **User Support and Troubleshooting**
   - Capture console errors during user sessions
   - Monitor performance-related console messages
   - Debug client-side issues reported by users

### Log File Format
Browser console events are logged to `.claude-mpm/logs/client/browser-{id}_{timestamp}.log`:

```
[2024-01-10T10:23:45.123Z] [INFO] [browser-abc123-def456] Page loaded successfully
[2024-01-10T10:23:46.456Z] [ERROR] [browser-abc123-def456] TypeError: Cannot read property 'value' of null
  Stack trace: Error
    at HTMLButtonElement.onClick (http://localhost:3000/app.js:45:12)
    at HTMLButtonElement.dispatch (http://localhost:3000/vendor.js:2344:9)
[2024-01-10T10:23:47.789Z] [WARN] [browser-abc123-def456] Deprecated API usage detected
```

### Security Considerations

1. **Network Security**
   - Only inject monitor script in development/staging environments
   - Use HTTPS in production if monitor server supports it
   - Implement IP allowlisting for monitor connections

2. **Data Privacy**
   - Console monitoring may capture sensitive data in messages
   - Review log files for sensitive information before sharing
   - Implement log rotation and cleanup policies

3. **Performance Impact**
   - Monitor script has minimal performance overhead
   - Event queuing prevents blocking when server is unavailable
   - Automatic reconnection handles network interruptions

### Troubleshooting

#### Monitor Script Not Loading
```bash
# Check if monitor server is accessible
curl -I http://localhost:8765/api/browser-monitor.js

# Verify port 8765 is not blocked
netstat -an | grep 8765

# Check browser console for script loading errors
# Look for CORS or network connectivity issues
```

#### Console Events Not Appearing
```bash
# Check monitor server logs
./claude-mpm monitor logs

# Verify browser connection in logs
grep "Browser connected" .claude-mpm/logs/claude-mpm.log

# Check client log directory exists
ls -la .claude-mpm/logs/client/
```

#### Performance Issues
```bash
# Monitor event queue size (should be low)
# Check browser console for "Browser Monitor" messages
# Verify network connectivity between browser and server

# Clean up old browser sessions and logs
find .claude-mpm/logs/client/ -name "*.log" -mtime +7 -delete
```