"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * A certificate managed by AWS Certificate Manager
 *
 * IMPORTANT: if you are creating a certificate as part of your stack, the stack
 * will not complete creating until you read and follow the instructions in the
 * email that you will receive.
 *
 * ACM will send validation emails to the following addresses:
 *
 *  admin@domain.com
 *  administrator@domain.com
 *  hostmaster@domain.com
 *  postmaster@domain.com
 *  webmaster@domain.com
 *
 * For every domain that you register.
 */
class Certificate extends core_1.Resource {
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: allDomainNames.map(domainValidationOption),
            validationMethod: props.validationMethod,
        });
        this.certificateArn = cert.ref;
        /**
         * Return the domain validation options for the given domain
         *
         * Closes over props.
         */
        function domainValidationOption(domainName) {
            const overrideDomain = props.validationDomains && props.validationDomains[domainName];
            return {
                domainName,
                validationDomain: overrideDomain || util_1.apexDomain(domainName)
            };
        }
    }
}
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
//# sourceMappingURL=data:application/json;base64,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