"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
/**
 * A certificate managed by AWS Certificate Manager.  Will be automatically
 * validated using DNS validation against the specified Route 53 hosted zone.
 *
 * @resource AWS::CertificateManager::Certificate
 * @experimental
 */
class DnsValidatedCertificate extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.domainName = props.domainName;
        this.normalizedZoneName = props.hostedZone.zoneName;
        // Remove trailing `.` from zone name
        if (this.normalizedZoneName.endsWith('.')) {
            this.normalizedZoneName = this.normalizedZoneName.substring(0, this.normalizedZoneName.length - 1);
        }
        // Remove any `/hostedzone/` prefix from the Hosted Zone ID
        this.hostedZoneId = props.hostedZone.hostedZoneId.replace(/^\/hostedzone\//, '');
        const requestorFunction = new lambda.Function(this, 'CertificateRequestorFunction', {
            code: lambda.Code.asset(path.resolve(__dirname, '..', 'lambda-packages', 'dns_validated_certificate_handler', 'lib')),
            handler: 'index.certificateRequestHandler',
            runtime: lambda.Runtime.NODEJS_8_10,
            timeout: cdk.Duration.minutes(15)
        });
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['acm:RequestCertificate', 'acm:DescribeCertificate', 'acm:DeleteCertificate'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:GetChange'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:changeResourceRecordSets'],
            resources: [`arn:aws:route53:::hostedzone/${this.hostedZoneId}`],
        }));
        const certificate = new cfn.CustomResource(this, 'CertificateRequestorResource', {
            provider: cfn.CustomResourceProvider.lambda(requestorFunction),
            properties: {
                DomainName: props.domainName,
                SubjectAlternativeNames: props.subjectAlternativeNames,
                HostedZoneId: this.hostedZoneId,
                Region: props.region,
            }
        });
        this.certificateArn = certificate.getAtt('Arn').toString();
    }
    validate() {
        const errors = [];
        // Ensure the zone name is a parent zone of the certificate domain name
        if (this.domainName !== this.normalizedZoneName && !this.domainName.endsWith('.' + this.normalizedZoneName)) {
            errors.push(`DNS zone ${this.normalizedZoneName} is not authoritative for certificate domain name ${this.domainName}`);
        }
        return errors;
    }
}
exports.DnsValidatedCertificate = DnsValidatedCertificate;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG5zLXZhbGlkYXRlZC1jZXJ0aWZpY2F0ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRucy12YWxpZGF0ZWQtY2VydGlmaWNhdGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxtREFBb0Q7QUFDcEQsd0NBQXlDO0FBQ3pDLDhDQUErQztBQUUvQyxxQ0FBc0M7QUFDdEMsNkJBQThCO0FBc0I5Qjs7Ozs7O0dBTUc7QUFDSCxNQUFhLHVCQUF3QixTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBTXJELFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBbUM7UUFDN0UsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFDbkMsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDO1FBQ3BELHFDQUFxQztRQUNyQyxJQUFJLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDdkMsSUFBSSxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7U0FDdEc7UUFFRCwyREFBMkQ7UUFDM0QsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakYsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLDhCQUE4QixFQUFFO1lBQ2hGLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsaUJBQWlCLEVBQUUsbUNBQW1DLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDckgsT0FBTyxFQUFFLGlDQUFpQztZQUMxQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7U0FDcEMsQ0FBQyxDQUFDO1FBQ0gsaUJBQWlCLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0RCxPQUFPLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSx5QkFBeUIsRUFBRSx1QkFBdUIsQ0FBQztZQUN2RixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDbkIsQ0FBQyxDQUFDLENBQUM7UUFDSixpQkFBaUIsQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RELE9BQU8sRUFBRSxDQUFDLG1CQUFtQixDQUFDO1lBQzlCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNuQixDQUFDLENBQUMsQ0FBQztRQUNKLGlCQUFpQixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEQsT0FBTyxFQUFFLENBQUMsa0NBQWtDLENBQUM7WUFDN0MsU0FBUyxFQUFFLENBQUMsZ0NBQWdDLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztTQUNuRSxDQUFDLENBQUMsQ0FBQztRQUVKLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsOEJBQThCLEVBQUU7WUFDN0UsUUFBUSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLENBQUMsaUJBQWlCLENBQUM7WUFDOUQsVUFBVSxFQUFFO2dCQUNSLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtnQkFDNUIsdUJBQXVCLEVBQUUsS0FBSyxDQUFDLHVCQUF1QjtnQkFDdEQsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO2dCQUMvQixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU07YUFDdkI7U0FDSixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDL0QsQ0FBQztJQUVTLFFBQVE7UUFDZCxNQUFNLE1BQU0sR0FBYSxFQUFFLENBQUM7UUFDNUIsdUVBQXVFO1FBQ3ZFLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxJQUFJLENBQUMsa0JBQWtCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEVBQUU7WUFDekcsTUFBTSxDQUFDLElBQUksQ0FBQyxZQUFZLElBQUksQ0FBQyxrQkFBa0IscURBQXFELElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQzFIO1FBQ0QsT0FBTyxNQUFNLENBQUM7SUFDbEIsQ0FBQztDQUNKO0FBM0RELDBEQTJEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBjZm4gPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtY2xvdWRmb3JtYXRpb24nKTtcbmltcG9ydCBpYW0gPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtaWFtJyk7XG5pbXBvcnQgbGFtYmRhID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWxhbWJkYScpO1xuaW1wb3J0IHJvdXRlNTMgPSByZXF1aXJlKCdAYXdzLWNkay9hd3Mtcm91dGU1MycpO1xuaW1wb3J0IGNkayA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2NvcmUnKTtcbmltcG9ydCBwYXRoID0gcmVxdWlyZSgncGF0aCcpO1xuaW1wb3J0IHsgQ2VydGlmaWNhdGVQcm9wcywgSUNlcnRpZmljYXRlIH0gZnJvbSAnLi9jZXJ0aWZpY2F0ZSc7XG5cbi8qKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIERuc1ZhbGlkYXRlZENlcnRpZmljYXRlUHJvcHMgZXh0ZW5kcyBDZXJ0aWZpY2F0ZVByb3BzIHtcbiAgICAvKipcbiAgICAgKiBSb3V0ZSA1MyBIb3N0ZWQgWm9uZSB1c2VkIHRvIHBlcmZvcm0gRE5TIHZhbGlkYXRpb24gb2YgdGhlIHJlcXVlc3QuICBUaGUgem9uZVxuICAgICAqIG11c3QgYmUgYXV0aG9yaXRhdGl2ZSBmb3IgdGhlIGRvbWFpbiBuYW1lIHNwZWNpZmllZCBpbiB0aGUgQ2VydGlmaWNhdGUgUmVxdWVzdC5cbiAgICAgKi9cbiAgICByZWFkb25seSBob3N0ZWRab25lOiByb3V0ZTUzLklIb3N0ZWRab25lO1xuICAgIC8qKlxuICAgICAqIEFXUyByZWdpb24gdGhhdCB3aWxsIGhvc3QgdGhlIGNlcnRpZmljYXRlLiBUaGlzIGlzIG5lZWRlZCBlc3BlY2lhbGx5XG4gICAgICogZm9yIGNlcnRpZmljYXRlcyB1c2VkIGZvciBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbnMsIHdoaWNoIHJlcXVpcmUgdGhlIHJlZ2lvblxuICAgICAqIHRvIGJlIHVzLWVhc3QtMS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHRoZSByZWdpb24gdGhlIHN0YWNrIGlzIGRlcGxveWVkIGluLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBBIGNlcnRpZmljYXRlIG1hbmFnZWQgYnkgQVdTIENlcnRpZmljYXRlIE1hbmFnZXIuICBXaWxsIGJlIGF1dG9tYXRpY2FsbHlcbiAqIHZhbGlkYXRlZCB1c2luZyBETlMgdmFsaWRhdGlvbiBhZ2FpbnN0IHRoZSBzcGVjaWZpZWQgUm91dGUgNTMgaG9zdGVkIHpvbmUuXG4gKlxuICogQHJlc291cmNlIEFXUzo6Q2VydGlmaWNhdGVNYW5hZ2VyOjpDZXJ0aWZpY2F0ZVxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGUgZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJQ2VydGlmaWNhdGUge1xuICAgIHB1YmxpYyByZWFkb25seSBjZXJ0aWZpY2F0ZUFybjogc3RyaW5nO1xuICAgIHByaXZhdGUgbm9ybWFsaXplZFpvbmVOYW1lOiBzdHJpbmc7XG4gICAgcHJpdmF0ZSBob3N0ZWRab25lSWQ6IHN0cmluZztcbiAgICBwcml2YXRlIGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRG5zVmFsaWRhdGVkQ2VydGlmaWNhdGVQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgICAgIHRoaXMuZG9tYWluTmFtZSA9IHByb3BzLmRvbWFpbk5hbWU7XG4gICAgICAgIHRoaXMubm9ybWFsaXplZFpvbmVOYW1lID0gcHJvcHMuaG9zdGVkWm9uZS56b25lTmFtZTtcbiAgICAgICAgLy8gUmVtb3ZlIHRyYWlsaW5nIGAuYCBmcm9tIHpvbmUgbmFtZVxuICAgICAgICBpZiAodGhpcy5ub3JtYWxpemVkWm9uZU5hbWUuZW5kc1dpdGgoJy4nKSkge1xuICAgICAgICAgICAgdGhpcy5ub3JtYWxpemVkWm9uZU5hbWUgPSB0aGlzLm5vcm1hbGl6ZWRab25lTmFtZS5zdWJzdHJpbmcoMCwgdGhpcy5ub3JtYWxpemVkWm9uZU5hbWUubGVuZ3RoIC0gMSk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBSZW1vdmUgYW55IGAvaG9zdGVkem9uZS9gIHByZWZpeCBmcm9tIHRoZSBIb3N0ZWQgWm9uZSBJRFxuICAgICAgICB0aGlzLmhvc3RlZFpvbmVJZCA9IHByb3BzLmhvc3RlZFpvbmUuaG9zdGVkWm9uZUlkLnJlcGxhY2UoL15cXC9ob3N0ZWR6b25lXFwvLywgJycpO1xuXG4gICAgICAgIGNvbnN0IHJlcXVlc3RvckZ1bmN0aW9uID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnQ2VydGlmaWNhdGVSZXF1ZXN0b3JGdW5jdGlvbicsIHtcbiAgICAgICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmFzc2V0KHBhdGgucmVzb2x2ZShfX2Rpcm5hbWUsICcuLicsICdsYW1iZGEtcGFja2FnZXMnLCAnZG5zX3ZhbGlkYXRlZF9jZXJ0aWZpY2F0ZV9oYW5kbGVyJywgJ2xpYicpKSxcbiAgICAgICAgICAgIGhhbmRsZXI6ICdpbmRleC5jZXJ0aWZpY2F0ZVJlcXVlc3RIYW5kbGVyJyxcbiAgICAgICAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU184XzEwLFxuICAgICAgICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLm1pbnV0ZXMoMTUpXG4gICAgICAgIH0pO1xuICAgICAgICByZXF1ZXN0b3JGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWydhY206UmVxdWVzdENlcnRpZmljYXRlJywgJ2FjbTpEZXNjcmliZUNlcnRpZmljYXRlJywgJ2FjbTpEZWxldGVDZXJ0aWZpY2F0ZSddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgfSkpO1xuICAgICAgICByZXF1ZXN0b3JGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWydyb3V0ZTUzOkdldENoYW5nZSddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgfSkpO1xuICAgICAgICByZXF1ZXN0b3JGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWydyb3V0ZTUzOmNoYW5nZVJlc291cmNlUmVjb3JkU2V0cyddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbYGFybjphd3M6cm91dGU1Mzo6Omhvc3RlZHpvbmUvJHt0aGlzLmhvc3RlZFpvbmVJZH1gXSxcbiAgICAgICAgfSkpO1xuXG4gICAgICAgIGNvbnN0IGNlcnRpZmljYXRlID0gbmV3IGNmbi5DdXN0b21SZXNvdXJjZSh0aGlzLCAnQ2VydGlmaWNhdGVSZXF1ZXN0b3JSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIHByb3ZpZGVyOiBjZm4uQ3VzdG9tUmVzb3VyY2VQcm92aWRlci5sYW1iZGEocmVxdWVzdG9yRnVuY3Rpb24pLFxuICAgICAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgICAgICAgIERvbWFpbk5hbWU6IHByb3BzLmRvbWFpbk5hbWUsXG4gICAgICAgICAgICAgICAgU3ViamVjdEFsdGVybmF0aXZlTmFtZXM6IHByb3BzLnN1YmplY3RBbHRlcm5hdGl2ZU5hbWVzLFxuICAgICAgICAgICAgICAgIEhvc3RlZFpvbmVJZDogdGhpcy5ob3N0ZWRab25lSWQsXG4gICAgICAgICAgICAgICAgUmVnaW9uOiBwcm9wcy5yZWdpb24sXG4gICAgICAgICAgICB9XG4gICAgICAgIH0pO1xuXG4gICAgICAgIHRoaXMuY2VydGlmaWNhdGVBcm4gPSBjZXJ0aWZpY2F0ZS5nZXRBdHQoJ0FybicpLnRvU3RyaW5nKCk7XG4gICAgfVxuXG4gICAgcHJvdGVjdGVkIHZhbGlkYXRlKCk6IHN0cmluZ1tdIHtcbiAgICAgICAgY29uc3QgZXJyb3JzOiBzdHJpbmdbXSA9IFtdO1xuICAgICAgICAvLyBFbnN1cmUgdGhlIHpvbmUgbmFtZSBpcyBhIHBhcmVudCB6b25lIG9mIHRoZSBjZXJ0aWZpY2F0ZSBkb21haW4gbmFtZVxuICAgICAgICBpZiAodGhpcy5kb21haW5OYW1lICE9PSB0aGlzLm5vcm1hbGl6ZWRab25lTmFtZSAmJiAhdGhpcy5kb21haW5OYW1lLmVuZHNXaXRoKCcuJyArIHRoaXMubm9ybWFsaXplZFpvbmVOYW1lKSkge1xuICAgICAgICAgICAgZXJyb3JzLnB1c2goYEROUyB6b25lICR7dGhpcy5ub3JtYWxpemVkWm9uZU5hbWV9IGlzIG5vdCBhdXRob3JpdGF0aXZlIGZvciBjZXJ0aWZpY2F0ZSBkb21haW4gbmFtZSAke3RoaXMuZG9tYWluTmFtZX1gKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gZXJyb3JzO1xuICAgIH1cbn1cbiJdfQ==