"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const dns_validated_certificate_1 = require("../lib/dns-validated-certificate");
module.exports = {
    'creates CloudFormation Custom Resource'(test) {
        const stack = new core_1.Stack();
        const exampleDotComZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
            zoneName: 'example.com'
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            hostedZone: exampleDotComZone,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            DomainName: 'test.example.com',
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertificateCertificateRequestorFunction5E845413',
                    'Arn'
                ]
            },
            HostedZoneId: {
                Ref: 'ExampleDotCom4D1B83AA',
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Handler: 'index.certificateRequestHandler',
            Runtime: 'nodejs8.10',
            Timeout: 900,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyName: 'CertificateCertificateRequestorFunctionServiceRoleDefaultPolicy3C8845BC',
            Roles: [
                {
                    Ref: 'CertificateCertificateRequestorFunctionServiceRoleC04C13DA',
                }
            ],
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: [
                            'acm:RequestCertificate',
                            'acm:DescribeCertificate',
                            'acm:DeleteCertificate'
                        ],
                        Effect: 'Allow',
                        Resource: '*'
                    },
                    {
                        Action: 'route53:GetChange',
                        Effect: 'Allow',
                        Resource: '*'
                    },
                    {
                        Action: 'route53:changeResourceRecordSets',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:aws:route53:::hostedzone/',
                                    {
                                        Ref: 'ExampleDotCom4D1B83AA'
                                    }
                                ]
                            ]
                        }
                    },
                ],
            }
        }));
        test.done();
    },
    'adds validation error on domain mismatch'(test) {
        const stack = new core_1.Stack();
        const helloDotComZone = new aws_route53_1.PublicHostedZone(stack, 'HelloDotCom', {
            zoneName: 'hello.com'
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'example.com',
            hostedZone: helloDotComZone,
        });
        // a bit of a hack: expect(stack) will trigger validation.
        test.throws(() => assert_1.expect(stack), /DNS zone hello.com is not authoritative for certificate domain name example.com/);
        test.done();
    },
    'test root certificate'(test) {
        const stack = new core_1.Stack();
        const exampleDotComZone = new aws_route53_1.PublicHostedZone(stack, 'ExampleDotCom', {
            zoneName: 'example.com'
        });
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'example.com',
            hostedZone: exampleDotComZone,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertCertificateRequestorFunction98FDF273',
                    'Arn'
                ]
            },
            DomainName: 'example.com',
            HostedZoneId: {
                Ref: 'ExampleDotCom4D1B83AA'
            }
        }));
        test.done();
    },
    'works with imported zone'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { account: '12345678', region: 'us-blue-5' },
        });
        const imported = aws_route53_1.HostedZone.fromLookup(stack, 'ExampleDotCom', {
            domainName: 'mydomain.com',
        });
        // WHEN
        new dns_validated_certificate_1.DnsValidatedCertificate(stack, 'Cert', {
            domainName: 'mydomain.com',
            hostedZone: imported,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
            ServiceToken: {
                'Fn::GetAtt': [
                    'CertCertificateRequestorFunction98FDF273',
                    'Arn'
                ]
            },
            DomainName: 'mydomain.com',
            HostedZoneId: 'DUMMY'
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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