"""Startup script builder with clean separation of concerns.

This implementation provides:
- Independent, testable script sections
- Abstracted template rendering
- Separate compression handling
- Explicit validation
- Clear orchestration without implementation details
"""

import base64
import gzip
import logging
from dataclasses import dataclass
from pathlib import Path
from typing import Any, Protocol

from flow.adapters.providers.builtin.mithril.core.constants import STARTUP_SCRIPT_MAX_SIZE
from flow.adapters.providers.builtin.mithril.runtime.startup.sections import (
    CodeUploadSection,
    CodeWaitSection,
    CompletionSection,
    DevVMDockerSection,
    DockerSection,
    EnvironmentSection,
    GPUdHealthSection,
    HeaderSection,
    IScriptSection,
    MultinodeSection,
    PortForwardingSection,
    RuntimeMonitorSection,
    S3Section,
    ScriptContext,
    SlurmSetupSection,
    UserScriptSection,
    UvInstallSection,
    VolumeSection,
    WorkloadResumeSection,
)
from flow.adapters.providers.builtin.mithril.runtime.startup.templates import (
    ITemplateEngine,
    create_template_engine,
)
from flow.sdk.models import TaskConfig
from flow.utils.paths import WORKSPACE_DIR

logger = logging.getLogger(__name__)


@dataclass
class StartupScript:
    """Structured representation of a startup script."""

    content: str
    compressed: bool = False
    sections: list[str] = None
    validation_errors: list[str] = None
    metadata: dict[str, Any] = None

    def __post_init__(self):
        self.sections = self.sections or []
        self.validation_errors = self.validation_errors or []
        self.metadata = self.metadata or {}

    @property
    def is_valid(self) -> bool:
        """Check if the script is valid (no validation errors)."""
        return not self.validation_errors

    @property
    def size_bytes(self) -> int:
        """Get the size of the script content in bytes."""
        return len(self.content.encode("utf-8"))


class IScriptCompressor(Protocol):
    """Protocol for script compression strategies."""

    def should_compress(self, content: str, max_size: int) -> bool:
        """Determine if compression is needed."""
        ...

    def compress(self, content: str) -> str:
        """Compress the script content."""
        ...


class GzipCompressor:
    """Gzip-based script compression."""

    def should_compress(self, content: str, max_size: int) -> bool:
        """Check if content exceeds max size."""
        return len(content.encode("utf-8")) > max_size

    def compress(self, content: str) -> str:
        """Compress and create bootstrap script."""
        compressed = gzip.compress(content.encode("utf-8"))
        encoded = base64.b64encode(compressed).decode("ascii")

        # Create bootstrap script that decompresses and executes
        from flow.adapters.providers.builtin.mithril.runtime.startup.origin import (
            get_flow_origin_header,
        )

        return f"""#!/bin/bash
# Bootstrap script for compressed startup script
{get_flow_origin_header()}
# Original size: {len(content)} bytes
# Compressed size: {len(compressed)} bytes

set -euxo pipefail

echo "Decompressing and executing startup script..."
echo "{encoded}" | base64 -d | gunzip | bash
"""


class IScriptBuilder(Protocol):
    """Protocol for startup script builders."""

    def build(self, config: TaskConfig) -> StartupScript:
        """Build a startup script from configuration."""
        ...


class StartupScriptBuilder:
    """Orchestrates startup script generation with clean architecture."""

    def __init__(
        self,
        sections: list[IScriptSection] | None = None,
        template_engine: ITemplateEngine | None = None,
        compressor: IScriptCompressor | None = None,
        max_uncompressed_size: int = STARTUP_SCRIPT_MAX_SIZE,
    ):
        """Initialize builder with dependencies.

        Args:
            sections: List of script sections (uses defaults if None)
            template_engine: Template engine for rendering
            compressor: Compression strategy
            max_uncompressed_size: Maximum size before compression
        """
        # Initialize template engine first so sections can receive it
        self.template_engine = template_engine or create_template_engine("simple")
        self.sections = sections or self._default_sections(self.template_engine)
        self.compressor = compressor or GzipCompressor()
        self.max_uncompressed_size = max_uncompressed_size

    def _default_sections(self, template_engine: ITemplateEngine) -> list[IScriptSection]:
        """Get default script sections in priority order."""

        return [
            HeaderSection(template_engine),
            # Install uv early so it's available to any later sections/user scripts
            UvInstallSection(template_engine),
            EnvironmentSection(template_engine),
            PortForwardingSection(template_engine),
            VolumeSection(template_engine),
            S3Section(template_engine),
            DevVMDockerSection(template_engine),  # Docker for dev VMs (without containers)
            GPUdHealthSection(template_engine),  # Added GPUd health monitoring
            CodeUploadSection(template_engine),
            CodeWaitSection(template_engine),
            MultinodeSection(template_engine),
            # Provision Slurm when requested via env (_FLOW_WITH_SLURM=1)
            SlurmSetupSection(template_engine),
            DockerSection(template_engine),
            UserScriptSection(template_engine),
            WorkloadResumeSection(template_engine),
            # Optional terminate-on-exit watcher (active only when enabled)
            __import__(
                "flow.adapters.providers.builtin.mithril.runtime.startup.sections.terminate_on_exit",
                fromlist=["TerminateOnExitSection"],
            ).TerminateOnExitSection(template_engine),
            RuntimeMonitorSection(template_engine),  # Runtime limit monitoring
            CompletionSection(template_engine),
        ]

    def build(self, config: TaskConfig) -> StartupScript:
        """Build startup script from task configuration.

        This method orchestrates the script generation process:
        1. Creates context from config
        2. Validates all sections
        3. Generates content from each section
        4. Combines sections
        5. Compresses if needed

        Args:
            config: Task configuration

        Returns:
            StartupScript with content and metadata
        """
        # Create context from config
        context = self._create_context(config)

        # Validate all sections
        validation_errors = self._validate_sections(context)
        if validation_errors:
            return StartupScript(
                content="",
                validation_errors=validation_errors,
                metadata={"config": config.model_dump()},
            )

        # Generate sections
        section_contents = self._generate_sections(context)

        # Combine sections
        full_content = self._combine_sections(section_contents)

        # Add debug logging
        import logging

        logger = logging.getLogger(__name__)
        logger.debug("=" * 80)
        logger.debug("STARTUP SCRIPT CONTENT:")
        logger.debug("=" * 80)
        logger.debug(f"Sections included: {[s['name'] for s in section_contents]}")
        logger.debug(f"Script size: {len(full_content.encode('utf-8'))} bytes")
        logger.debug("--- Script Content ---")
        logger.debug(full_content)
        logger.debug("--- End Script Content ---")
        logger.debug("=" * 80)

        # Defer compression/splitting decisions to the ScriptSizeHandler
        # used by the provider context (ScriptPreparationService). Returning
        # raw content here avoids double-compression and keeps sizing logic in
        # one place.
        original_size = len(full_content.encode("utf-8"))
        logger.debug(f"StartupScriptBuilder produced raw script size: {original_size} bytes")

        return StartupScript(
            content=full_content,
            compressed=False,
            sections=[s["name"] for s in section_contents],
            metadata={
                "config": config.model_dump(),
                "original_size": original_size,
                "section_count": len(section_contents),
            },
        )

    def _create_context(self, config: TaskConfig) -> ScriptContext:
        """Create script context from task configuration."""
        # Extract code archive from environment if present
        env = config.env.copy() if config.env else {}
        code_archive = env.pop("_FLOW_CODE_ARCHIVE", None)

        # Determine command type and set appropriate context fields
        docker_command = None
        user_script = None

        # If a docker image is provided, we treat command as a docker command.
        # If no docker image is provided, we must run the command directly on the host
        # as a user script so that non-containerized workloads (e.g., Colab Jupyter) start.
        docker_image = config.image or None

        # For dev VMs, always run the startup commands on the host to keep the
        # script small and avoid launching a long-lived container. The CLI will
        # manage containers during interactive sessions.
        dev_vm_hint = getattr(config, "dev_vm", None)
        is_dev_vm = bool(dev_vm_hint) if dev_vm_hint is not None else False
        if docker_image and not is_dev_vm:
            # Containerized path: forward command to Docker
            if isinstance(config.command, list):
                docker_command = config.command
            elif isinstance(config.command, str) and config.command.strip():
                docker_command = [config.command]
        else:
            # Host path: convert command to a user script
            if isinstance(config.command, list):
                # Common pattern is ["bash", "-c", SCRIPT]
                if (
                    len(config.command) >= 3
                    and config.command[1] == "-c"
                    and config.command[0] in {"bash", "/bin/bash"}
                ):
                    user_script = config.command[2]
                else:
                    # Fallback: join argv into a single shell command
                    import shlex  # local import to avoid module-level overhead

                    joined = " ".join(shlex.quote(arg) for arg in config.command)
                    user_script = f"#!/bin/bash\n{joined}\n"
            elif isinstance(config.command, str) and config.command.strip():
                user_script = config.command

        # No implicit test hooks; only pass task_id if present in env or config
        task_id = env.get("FLOW_TASK_ID")

        # Propagate allow_docker_cache into environment for DockerSection handling
        try:
            if (
                bool(getattr(config, "allow_docker_cache", False))
                and int(getattr(config, "num_instances", 1) or 1) == 1
            ):
                env.setdefault("FLOW_ALLOW_DOCKER_CACHE", "true")
        except Exception:  # noqa: BLE001
            pass

        # Health configuration (centralized)
        health_cfg = getattr(self, "_health_config", None)
        health_enabled = False
        health_payload = None
        if isinstance(health_cfg, dict):
            try:
                # Default disabled unless explicitly enabled via runtime settings
                health_enabled = bool(health_cfg.get("enabled", False))
                # Only pass through selected keys that scripts need
                health_payload = {
                    "gpud_version": health_cfg.get("gpud_version", "v0.5.1"),
                    "gpud_port": int(health_cfg.get("gpud_port", 15132) or 15132),
                    "gpud_bind": health_cfg.get("gpud_bind", "127.0.0.1"),
                    "endpoints": list(
                        health_cfg.get(
                            "endpoints",
                            ["/healthz", "/readyz", "/livez", "/health"],
                        )
                    ),
                    "gpud_health_timeout": int(health_cfg.get("gpud_health_timeout", 2) or 2),
                    "gpud_http_timeout": int(health_cfg.get("gpud_http_timeout", 5) or 5),
                    "ssh_curl_timeout": int(health_cfg.get("ssh_curl_timeout", 5) or 5),
                    "tunnel_timeout": int(health_cfg.get("tunnel_timeout", 10) or 10),
                    "metrics_endpoint": health_cfg.get("metrics_endpoint") or "",
                    "metrics_batch_size": int(health_cfg.get("metrics_batch_size", 100) or 100),
                    "metrics_interval": int(health_cfg.get("metrics_interval", 60) or 60),
                }
            except Exception:  # noqa: BLE001
                # Defensive: fall back to disabled if unexpected shapes
                health_enabled = False
                health_payload = None

        return ScriptContext(
            num_instances=config.num_instances,
            distributed_mode=config.distributed_mode or "auto",
            ports=list(getattr(config, "ports", []) or []),
            volumes=[v.model_dump() for v in config.volumes] if config.volumes else [],
            docker_image=docker_image,  # Treat empty string as no docker
            docker_command=docker_command,
            user_script=user_script,
            environment=env,  # Environment without the code archive
            upload_code=config.upload_code,
            code_archive=code_archive,
            instance_type=config.instance_type,  # Pass instance type for GPU detection
            task_id=task_id,  # Optional, from env
            task_name=config.name,  # Pass task name for identification
            # Keep scripts small for minimal tasks: opt-in to workload resume
            enable_workload_resume=getattr(config, "enable_workload_resume", False),
            # Runtime limit fields
            max_run_time_hours=config.max_run_time_hours,
            min_run_time_hours=config.min_run_time_hours,
            deadline_hours=config.deadline_hours,
            # Centralized health
            health_enabled=health_enabled,
            health=health_payload,
            # Prefer typed hint for dev VM semantics
            dev_vm=getattr(config, "dev_vm", None),
            # Terminate on exit (optional, batch-friendly)
            terminate_on_exit=bool(getattr(config, "terminate_on_exit", False)),
            # Propagate working directory so sections can honor overrides
            working_directory=getattr(config, "working_dir", WORKSPACE_DIR),
        )

    def _validate_sections(self, context: ScriptContext) -> list[str]:
        """Validate all sections and collect errors."""
        all_errors = []

        for section in sorted(self.sections, key=lambda s: s.priority):
            if section.should_include(context):
                errors = section.validate(context)
                if errors:
                    all_errors.extend([f"{section.name}: {e}" for e in errors])

        return all_errors

    def _generate_sections(self, context: ScriptContext) -> list[dict[str, Any]]:
        """Generate content for all applicable sections."""
        section_contents = []

        for section in sorted(self.sections, key=lambda s: s.priority):
            if section.should_include(context):
                content = section.generate(context)
                if content.strip():  # Only include non-empty sections
                    section_contents.append(
                        {
                            "name": section.name,
                            "priority": section.priority,
                            "content": content,
                        }
                    )

        return section_contents

    def _combine_sections(self, sections: list[dict[str, Any]]) -> str:
        """Combine section contents into final script."""
        if not sections:
            return "#!/bin/bash\n# Empty startup script\n"

        # Combine sections with simple spacing
        combined = []
        for i, section in enumerate(sections):
            combined.append(section["content"])
            # Add spacing between sections (but not after the last one)
            if i < len(sections) - 1:
                combined.append("")

        return "\n".join(combined)


class MithrilStartupScriptBuilder(StartupScriptBuilder):
    """Mithril-specific startup script builder.

    This is a convenience class that configures the builder
    with Mithril-specific defaults and behaviors.
    """

    def __init__(self):
        """Initialize with Mithril-specific configuration."""
        # Prefer Jinja2 templates for complex script sections; fall back to simple if unavailable
        try:
            from importlib.resources import as_file, files

            # Resolve provider templates package to a real path for Jinja FileSystemLoader
            provider_pkg = "flow.resources.templates.runtime"
            with as_file(files(provider_pkg)) as tpl_dir:
                jinja_engine = create_template_engine("jinja2", template_dir=Path(tpl_dir))
        except Exception:
            logger.exception("MithrilStartupScriptBuilder: falling back to Simple engine.")
            jinja_engine = create_template_engine("simple")

        # Initialize parent first to set template_engine attribute
        super().__init__(
            sections=None,
            template_engine=jinja_engine,
            compressor=GzipCompressor(),
            max_uncompressed_size=STARTUP_SCRIPT_MAX_SIZE,
        )
        # Now that template_engine is initialized on self, compute sections
        self.sections = self._mithril_sections()

    def _mithril_sections(self) -> list[IScriptSection]:
        """Get Mithril-specific script sections."""
        # For now, use default sections
        # Could add Mithril-specific sections here
        return self._default_sections(self.template_engine)
