"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BastionHost = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * The database is located in an isolated subnet, meaning that it is not accessible from the public internet. As such, to interact with the database directly, a user must tunnel through a bastion host.
 *
 * ### Configuring
 *
 * This codebase controls _who_ is allowed to connect to the bastion host. This requires two steps:
 *
 * 1. Adding the IP address from which you are connecting to the `ipv4Allowlist` array
 * 1. Creating a bastion host system user by adding the user's configuration inform to `userdata.yaml`
 *
 * #### Adding an IP address to the `ipv4Allowlist` array
 *
 * The `BastionHost` construct takes in an `ipv4Allowlist` array as an argument. Find your IP address (eg `curl api.ipify.org`) and add that to the array along with the trailing CIDR block (likely `/32` to indicate that you are adding a single IP address).
 *
 * #### Creating a user via `userdata.yaml`
 *
 * Add an entry to the `users` array with a username (likely matching your local systems username, which you can get by running the `whoami` command in your terminal) and a public key (likely your default public key, which you can get by running `cat ~/.ssh/id_*.pub` in your terminal).
 *
 * #### Tips & Tricks when using the Bastion Host
 *
 * **Connecting to RDS Instance via SSM**
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID \
 * --document-name AWS-StartPortForwardingSessionToRemoteHost \
 * --parameters '{
 * "host": [
 * "example-db.c5abcdefghij.us-west-2.rds.amazonaws.com"
 * ],
 * "portNumber": [
 * "5432"
 * ],
 * "localPortNumber": [
 * "9999"
 * ]
 * }' \
 * --profile $AWS_PROFILE
 * ```
 *
 * ```sh
 * psql -h localhost -p 9999 # continue adding username (-U) and db (-d) here...
 * ```
 *
 * Connect directly to Bastion Host:
 *
 * ```sh
 * aws ssm start-session --target $INSTANCE_ID --profile $AWS_PROFILE
 * ```
 *
 * **Setting up an SSH tunnel**
 *
 * In your `~/.ssh/config` file, add an entry like:
 *
 * ```
 * Host db-tunnel
 * Hostname {the-bastion-host-address}
 * LocalForward 9999 {the-db-hostname}:5432
 * ```
 *
 * Then a tunnel can be opened via:
 *
 * ```
 * ssh -N db-tunnel
 * ```
 *
 * And a connection to the DB can be made via:
 *
 * ```
 * psql -h 127.0.0.1 -p 9999 -U {username} -d {database}
 * ```
 *
 * **Handling `REMOTE HOST IDENTIFICATION HAS CHANGED!` error**
 *
 * If you've redeployed a bastion host that you've previously connected to, you may see an error like:
 *
 * ```
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * @    WARNING: REMOTE HOST IDENTIFICATION HAS CHANGED!     @
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 * IT IS POSSIBLE THAT SOMEONE IS DOING SOMETHING NASTY!
 * Someone could be eavesdropping on you right now (man-in-the-middle attack)!
 * It is also possible that a host key has just been changed.
 * The fingerprint for the ECDSA key sent by the remote host is
 * SHA256:mPnxAOXTpb06PFgI1Qc8TMQ2e9b7goU8y2NdS5hzIr8.
 * Please contact your system administrator.
 * Add correct host key in /Users/username/.ssh/known_hosts to get rid of this message.
 * Offending ECDSA key in /Users/username/.ssh/known_hosts:28
 * ECDSA host key for ec2-12-34-56-789.us-west-2.compute.amazonaws.com has changed and you have requested strict checking.
 * Host key verification failed.
 * ```
 *
 * This is due to the server's fingerprint changing. We can scrub the fingerprint from our system with a command like:
 *
 * ```
 * ssh-keygen -R 12.34.56.789
 * ```
 *
 */
class BastionHost extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { stackName } = aws_cdk_lib_1.Stack.of(this);
        // Build ec2 instance
        this.instance = new aws_cdk_lib_1.aws_ec2.Instance(this, "bastion-host", {
            vpc: props.vpc,
            vpcSubnets: { subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC },
            instanceName: `${stackName} bastion host`,
            instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE4_GRAVITON, aws_cdk_lib_1.aws_ec2.InstanceSize.NANO),
            machineImage: aws_cdk_lib_1.aws_ec2.MachineImage.latestAmazonLinux({
                generation: aws_cdk_lib_1.aws_ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                cpuType: aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.ARM_64,
            }),
            userData: props.userData,
            userDataCausesReplacement: true,
        });
        // Assign elastic IP
        if (props.createElasticIp ?? true) {
            new aws_cdk_lib_1.aws_ec2.CfnEIP(this, "IP", {
                instanceId: this.instance.instanceId,
                tags: [{ key: "Name", value: stackName }],
            });
        }
        // Allow bastion host to connect to db
        this.instance.connections.allowTo(props.db.connections.securityGroups[0], aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow connection from bastion host");
        // Allow IP access to bastion host
        for (const ipv4 of props.ipv4Allowlist) {
            this.instance.connections.allowFrom(aws_cdk_lib_1.aws_ec2.Peer.ipv4(ipv4), aws_cdk_lib_1.aws_ec2.Port.tcp(props.sshPort || 22), "SSH Access");
        }
        // Integrate with SSM
        this.instance.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: [
                "ssmmessages:*",
                "ssm:UpdateInstanceInformation",
                "ec2messages:*",
            ],
            resources: ["*"],
        }));
        new aws_cdk_lib_1.CfnOutput(this, "instance-id-output", {
            value: this.instance.instanceId,
            exportName: `${stackName}-instance-id`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-ip-output", {
            value: this.instance.instancePublicIp,
            exportName: `${stackName}-instance-public-ip`,
        });
        new aws_cdk_lib_1.CfnOutput(this, "instance-public-dns-name-output", {
            value: this.instance.instancePublicDnsName,
            exportName: `${stackName}-public-dns-name`,
        });
    }
}
exports.BastionHost = BastionHost;
_a = JSII_RTTI_SYMBOL_1;
BastionHost[_a] = { fqn: "eoapi-cdk.BastionHost", version: "10.2.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZDQU1xQjtBQUNyQiwyQ0FBdUM7QUFFdkM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FpR0c7QUFDSCxNQUFhLFdBQVksU0FBUSxzQkFBUztJQUd4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCO1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxFQUFFLFNBQVMsRUFBRSxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXJDLHFCQUFxQjtRQUNyQixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUkscUJBQUcsQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNyRCxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7WUFDZCxVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUscUJBQUcsQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFO1lBQ2pELFlBQVksRUFBRSxHQUFHLFNBQVMsZUFBZTtZQUN6QyxZQUFZLEVBQUUscUJBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUMvQixxQkFBRyxDQUFDLGFBQWEsQ0FBQyxtQkFBbUIsRUFDckMscUJBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUN0QjtZQUNELFlBQVksRUFBRSxxQkFBRyxDQUFDLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQztnQkFDL0MsVUFBVSxFQUFFLHFCQUFHLENBQUMscUJBQXFCLENBQUMsY0FBYztnQkFDcEQsT0FBTyxFQUFFLHFCQUFHLENBQUMsa0JBQWtCLENBQUMsTUFBTTthQUN2QyxDQUFDO1lBQ0YsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLHlCQUF5QixFQUFFLElBQUk7U0FDaEMsQ0FBQyxDQUFDO1FBRUgsb0JBQW9CO1FBQ3BCLElBQUksS0FBSyxDQUFDLGVBQWUsSUFBSSxJQUFJLEVBQUUsQ0FBQztZQUNsQyxJQUFJLHFCQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUU7Z0JBQ3pCLFVBQVUsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVU7Z0JBQ3BDLElBQUksRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLENBQUM7YUFDMUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUVELHNDQUFzQztRQUN0QyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQy9CLEtBQUssQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsRUFDdEMscUJBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUNsQixvQ0FBb0MsQ0FDckMsQ0FBQztRQUVGLGtDQUFrQztRQUNsQyxLQUFLLE1BQU0sSUFBSSxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUN2QyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQ2pDLHFCQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFDbkIscUJBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksRUFBRSxDQUFDLEVBQ2pDLFlBQVksQ0FDYixDQUFDO1FBQ0osQ0FBQztRQUVELHFCQUFxQjtRQUNyQixJQUFJLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FDM0IsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixPQUFPLEVBQUU7Z0JBQ1AsZUFBZTtnQkFDZiwrQkFBK0I7Z0JBQy9CLGVBQWU7YUFDaEI7WUFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDakIsQ0FBQyxDQUNILENBQUM7UUFFRixJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO1lBQ3hDLEtBQUssRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVU7WUFDL0IsVUFBVSxFQUFFLEdBQUcsU0FBUyxjQUFjO1NBQ3ZDLENBQUMsQ0FBQztRQUNILElBQUksdUJBQVMsQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLEVBQUU7WUFDL0MsS0FBSyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsZ0JBQWdCO1lBQ3JDLFVBQVUsRUFBRSxHQUFHLFNBQVMscUJBQXFCO1NBQzlDLENBQUMsQ0FBQztRQUNILElBQUksdUJBQVMsQ0FBQyxJQUFJLEVBQUUsaUNBQWlDLEVBQUU7WUFDckQsS0FBSyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMscUJBQXFCO1lBQzFDLFVBQVUsRUFBRSxHQUFHLFNBQVMsa0JBQWtCO1NBQzNDLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBekVILGtDQTBFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7XG4gIFN0YWNrLFxuICBhd3NfZWMyIGFzIGVjMixcbiAgYXdzX2lhbSBhcyBpYW0sXG4gIGF3c19yZHMgYXMgcmRzLFxuICBDZm5PdXRwdXQsXG59IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSBcImNvbnN0cnVjdHNcIjtcblxuLyoqXG4gKiBUaGUgZGF0YWJhc2UgaXMgbG9jYXRlZCBpbiBhbiBpc29sYXRlZCBzdWJuZXQsIG1lYW5pbmcgdGhhdCBpdCBpcyBub3QgYWNjZXNzaWJsZSBmcm9tIHRoZSBwdWJsaWMgaW50ZXJuZXQuIEFzIHN1Y2gsIHRvIGludGVyYWN0IHdpdGggdGhlIGRhdGFiYXNlIGRpcmVjdGx5LCBhIHVzZXIgbXVzdCB0dW5uZWwgdGhyb3VnaCBhIGJhc3Rpb24gaG9zdC5cbiAqXG4gKiAjIyMgQ29uZmlndXJpbmdcbiAqXG4gKiBUaGlzIGNvZGViYXNlIGNvbnRyb2xzIF93aG9fIGlzIGFsbG93ZWQgdG8gY29ubmVjdCB0byB0aGUgYmFzdGlvbiBob3N0LiBUaGlzIHJlcXVpcmVzIHR3byBzdGVwczpcbiAqXG4gKiAxLiBBZGRpbmcgdGhlIElQIGFkZHJlc3MgZnJvbSB3aGljaCB5b3UgYXJlIGNvbm5lY3RpbmcgdG8gdGhlIGBpcHY0QWxsb3dsaXN0YCBhcnJheVxuICogMS4gQ3JlYXRpbmcgYSBiYXN0aW9uIGhvc3Qgc3lzdGVtIHVzZXIgYnkgYWRkaW5nIHRoZSB1c2VyJ3MgY29uZmlndXJhdGlvbiBpbmZvcm0gdG8gYHVzZXJkYXRhLnlhbWxgXG4gKlxuICogIyMjIyBBZGRpbmcgYW4gSVAgYWRkcmVzcyB0byB0aGUgYGlwdjRBbGxvd2xpc3RgIGFycmF5XG4gKlxuICogVGhlIGBCYXN0aW9uSG9zdGAgY29uc3RydWN0IHRha2VzIGluIGFuIGBpcHY0QWxsb3dsaXN0YCBhcnJheSBhcyBhbiBhcmd1bWVudC4gRmluZCB5b3VyIElQIGFkZHJlc3MgKGVnIGBjdXJsIGFwaS5pcGlmeS5vcmdgKSBhbmQgYWRkIHRoYXQgdG8gdGhlIGFycmF5IGFsb25nIHdpdGggdGhlIHRyYWlsaW5nIENJRFIgYmxvY2sgKGxpa2VseSBgLzMyYCB0byBpbmRpY2F0ZSB0aGF0IHlvdSBhcmUgYWRkaW5nIGEgc2luZ2xlIElQIGFkZHJlc3MpLlxuICpcbiAqICMjIyMgQ3JlYXRpbmcgYSB1c2VyIHZpYSBgdXNlcmRhdGEueWFtbGBcbiAqXG4gKiBBZGQgYW4gZW50cnkgdG8gdGhlIGB1c2Vyc2AgYXJyYXkgd2l0aCBhIHVzZXJuYW1lIChsaWtlbHkgbWF0Y2hpbmcgeW91ciBsb2NhbCBzeXN0ZW1zIHVzZXJuYW1lLCB3aGljaCB5b3UgY2FuIGdldCBieSBydW5uaW5nIHRoZSBgd2hvYW1pYCBjb21tYW5kIGluIHlvdXIgdGVybWluYWwpIGFuZCBhIHB1YmxpYyBrZXkgKGxpa2VseSB5b3VyIGRlZmF1bHQgcHVibGljIGtleSwgd2hpY2ggeW91IGNhbiBnZXQgYnkgcnVubmluZyBgY2F0IH4vLnNzaC9pZF8qLnB1YmAgaW4geW91ciB0ZXJtaW5hbCkuXG4gKlxuICogIyMjIyBUaXBzICYgVHJpY2tzIHdoZW4gdXNpbmcgdGhlIEJhc3Rpb24gSG9zdFxuICpcbiAqICoqQ29ubmVjdGluZyB0byBSRFMgSW5zdGFuY2UgdmlhIFNTTSoqXG4gKlxuICogYGBgc2hcbiAqIGF3cyBzc20gc3RhcnQtc2Vzc2lvbiAtLXRhcmdldCAkSU5TVEFOQ0VfSUQgXFxcbiAqIC0tZG9jdW1lbnQtbmFtZSBBV1MtU3RhcnRQb3J0Rm9yd2FyZGluZ1Nlc3Npb25Ub1JlbW90ZUhvc3QgXFxcbiAqIC0tcGFyYW1ldGVycyAne1xuICogXCJob3N0XCI6IFtcbiAqIFwiZXhhbXBsZS1kYi5jNWFiY2RlZmdoaWoudXMtd2VzdC0yLnJkcy5hbWF6b25hd3MuY29tXCJcbiAqIF0sXG4gKiBcInBvcnROdW1iZXJcIjogW1xuICogXCI1NDMyXCJcbiAqIF0sXG4gKiBcImxvY2FsUG9ydE51bWJlclwiOiBbXG4gKiBcIjk5OTlcIlxuICogXVxuICogfScgXFxcbiAqIC0tcHJvZmlsZSAkQVdTX1BST0ZJTEVcbiAqIGBgYFxuICpcbiAqIGBgYHNoXG4gKiBwc3FsIC1oIGxvY2FsaG9zdCAtcCA5OTk5ICMgY29udGludWUgYWRkaW5nIHVzZXJuYW1lICgtVSkgYW5kIGRiICgtZCkgaGVyZS4uLlxuICogYGBgXG4gKlxuICogQ29ubmVjdCBkaXJlY3RseSB0byBCYXN0aW9uIEhvc3Q6XG4gKlxuICogYGBgc2hcbiAqIGF3cyBzc20gc3RhcnQtc2Vzc2lvbiAtLXRhcmdldCAkSU5TVEFOQ0VfSUQgLS1wcm9maWxlICRBV1NfUFJPRklMRVxuICogYGBgXG4gKlxuICogKipTZXR0aW5nIHVwIGFuIFNTSCB0dW5uZWwqKlxuICpcbiAqIEluIHlvdXIgYH4vLnNzaC9jb25maWdgIGZpbGUsIGFkZCBhbiBlbnRyeSBsaWtlOlxuICpcbiAqIGBgYFxuICogSG9zdCBkYi10dW5uZWxcbiAqIEhvc3RuYW1lIHt0aGUtYmFzdGlvbi1ob3N0LWFkZHJlc3N9XG4gKiBMb2NhbEZvcndhcmQgOTk5OSB7dGhlLWRiLWhvc3RuYW1lfTo1NDMyXG4gKiBgYGBcbiAqXG4gKiBUaGVuIGEgdHVubmVsIGNhbiBiZSBvcGVuZWQgdmlhOlxuICpcbiAqIGBgYFxuICogc3NoIC1OIGRiLXR1bm5lbFxuICogYGBgXG4gKlxuICogQW5kIGEgY29ubmVjdGlvbiB0byB0aGUgREIgY2FuIGJlIG1hZGUgdmlhOlxuICpcbiAqIGBgYFxuICogcHNxbCAtaCAxMjcuMC4wLjEgLXAgOTk5OSAtVSB7dXNlcm5hbWV9IC1kIHtkYXRhYmFzZX1cbiAqIGBgYFxuICpcbiAqICoqSGFuZGxpbmcgYFJFTU9URSBIT1NUIElERU5USUZJQ0FUSU9OIEhBUyBDSEFOR0VEIWAgZXJyb3IqKlxuICpcbiAqIElmIHlvdSd2ZSByZWRlcGxveWVkIGEgYmFzdGlvbiBob3N0IHRoYXQgeW91J3ZlIHByZXZpb3VzbHkgY29ubmVjdGVkIHRvLCB5b3UgbWF5IHNlZSBhbiBlcnJvciBsaWtlOlxuICpcbiAqIGBgYFxuICogQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBcbiAqIEAgICAgV0FSTklORzogUkVNT1RFIEhPU1QgSURFTlRJRklDQVRJT04gSEFTIENIQU5HRUQhICAgICBAXG4gKiBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQFxuICogSVQgSVMgUE9TU0lCTEUgVEhBVCBTT01FT05FIElTIERPSU5HIFNPTUVUSElORyBOQVNUWSFcbiAqIFNvbWVvbmUgY291bGQgYmUgZWF2ZXNkcm9wcGluZyBvbiB5b3UgcmlnaHQgbm93IChtYW4taW4tdGhlLW1pZGRsZSBhdHRhY2spIVxuICogSXQgaXMgYWxzbyBwb3NzaWJsZSB0aGF0IGEgaG9zdCBrZXkgaGFzIGp1c3QgYmVlbiBjaGFuZ2VkLlxuICogVGhlIGZpbmdlcnByaW50IGZvciB0aGUgRUNEU0Ega2V5IHNlbnQgYnkgdGhlIHJlbW90ZSBob3N0IGlzXG4gKiBTSEEyNTY6bVBueEFPWFRwYjA2UEZnSTFRYzhUTVEyZTliN2dvVTh5Mk5kUzVoeklyOC5cbiAqIFBsZWFzZSBjb250YWN0IHlvdXIgc3lzdGVtIGFkbWluaXN0cmF0b3IuXG4gKiBBZGQgY29ycmVjdCBob3N0IGtleSBpbiAvVXNlcnMvdXNlcm5hbWUvLnNzaC9rbm93bl9ob3N0cyB0byBnZXQgcmlkIG9mIHRoaXMgbWVzc2FnZS5cbiAqIE9mZmVuZGluZyBFQ0RTQSBrZXkgaW4gL1VzZXJzL3VzZXJuYW1lLy5zc2gva25vd25faG9zdHM6MjhcbiAqIEVDRFNBIGhvc3Qga2V5IGZvciBlYzItMTItMzQtNTYtNzg5LnVzLXdlc3QtMi5jb21wdXRlLmFtYXpvbmF3cy5jb20gaGFzIGNoYW5nZWQgYW5kIHlvdSBoYXZlIHJlcXVlc3RlZCBzdHJpY3QgY2hlY2tpbmcuXG4gKiBIb3N0IGtleSB2ZXJpZmljYXRpb24gZmFpbGVkLlxuICogYGBgXG4gKlxuICogVGhpcyBpcyBkdWUgdG8gdGhlIHNlcnZlcidzIGZpbmdlcnByaW50IGNoYW5naW5nLiBXZSBjYW4gc2NydWIgdGhlIGZpbmdlcnByaW50IGZyb20gb3VyIHN5c3RlbSB3aXRoIGEgY29tbWFuZCBsaWtlOlxuICpcbiAqIGBgYFxuICogc3NoLWtleWdlbiAtUiAxMi4zNC41Ni43ODlcbiAqIGBgYFxuICpcbiAqL1xuZXhwb3J0IGNsYXNzIEJhc3Rpb25Ib3N0IGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgaW5zdGFuY2U6IGVjMi5JbnN0YW5jZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQmFzdGlvbkhvc3RQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCB7IHN0YWNrTmFtZSB9ID0gU3RhY2sub2YodGhpcyk7XG5cbiAgICAvLyBCdWlsZCBlYzIgaW5zdGFuY2VcbiAgICB0aGlzLmluc3RhbmNlID0gbmV3IGVjMi5JbnN0YW5jZSh0aGlzLCBcImJhc3Rpb24taG9zdFwiLCB7XG4gICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFVCTElDIH0sXG4gICAgICBpbnN0YW5jZU5hbWU6IGAke3N0YWNrTmFtZX0gYmFzdGlvbiBob3N0YCxcbiAgICAgIGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZS5vZihcbiAgICAgICAgZWMyLkluc3RhbmNlQ2xhc3MuQlVSU1RBQkxFNF9HUkFWSVRPTixcbiAgICAgICAgZWMyLkluc3RhbmNlU2l6ZS5OQU5PXG4gICAgICApLFxuICAgICAgbWFjaGluZUltYWdlOiBlYzIuTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHtcbiAgICAgICAgZ2VuZXJhdGlvbjogZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMixcbiAgICAgICAgY3B1VHlwZTogZWMyLkFtYXpvbkxpbnV4Q3B1VHlwZS5BUk1fNjQsXG4gICAgICB9KSxcbiAgICAgIHVzZXJEYXRhOiBwcm9wcy51c2VyRGF0YSxcbiAgICAgIHVzZXJEYXRhQ2F1c2VzUmVwbGFjZW1lbnQ6IHRydWUsXG4gICAgfSk7XG5cbiAgICAvLyBBc3NpZ24gZWxhc3RpYyBJUFxuICAgIGlmIChwcm9wcy5jcmVhdGVFbGFzdGljSXAgPz8gdHJ1ZSkge1xuICAgICAgbmV3IGVjMi5DZm5FSVAodGhpcywgXCJJUFwiLCB7XG4gICAgICAgIGluc3RhbmNlSWQ6IHRoaXMuaW5zdGFuY2UuaW5zdGFuY2VJZCxcbiAgICAgICAgdGFnczogW3sga2V5OiBcIk5hbWVcIiwgdmFsdWU6IHN0YWNrTmFtZSB9XSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIEFsbG93IGJhc3Rpb24gaG9zdCB0byBjb25uZWN0IHRvIGRiXG4gICAgdGhpcy5pbnN0YW5jZS5jb25uZWN0aW9ucy5hbGxvd1RvKFxuICAgICAgcHJvcHMuZGIuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHNbMF0sXG4gICAgICBlYzIuUG9ydC50Y3AoNTQzMiksXG4gICAgICBcIkFsbG93IGNvbm5lY3Rpb24gZnJvbSBiYXN0aW9uIGhvc3RcIlxuICAgICk7XG5cbiAgICAvLyBBbGxvdyBJUCBhY2Nlc3MgdG8gYmFzdGlvbiBob3N0XG4gICAgZm9yIChjb25zdCBpcHY0IG9mIHByb3BzLmlwdjRBbGxvd2xpc3QpIHtcbiAgICAgIHRoaXMuaW5zdGFuY2UuY29ubmVjdGlvbnMuYWxsb3dGcm9tKFxuICAgICAgICBlYzIuUGVlci5pcHY0KGlwdjQpLFxuICAgICAgICBlYzIuUG9ydC50Y3AocHJvcHMuc3NoUG9ydCB8fCAyMiksXG4gICAgICAgIFwiU1NIIEFjY2Vzc1wiXG4gICAgICApO1xuICAgIH1cblxuICAgIC8vIEludGVncmF0ZSB3aXRoIFNTTVxuICAgIHRoaXMuaW5zdGFuY2UuYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgXCJzc21tZXNzYWdlczoqXCIsXG4gICAgICAgICAgXCJzc206VXBkYXRlSW5zdGFuY2VJbmZvcm1hdGlvblwiLFxuICAgICAgICAgIFwiZWMybWVzc2FnZXM6KlwiLFxuICAgICAgICBdLFxuICAgICAgICByZXNvdXJjZXM6IFtcIipcIl0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBuZXcgQ2ZuT3V0cHV0KHRoaXMsIFwiaW5zdGFuY2UtaWQtb3V0cHV0XCIsIHtcbiAgICAgIHZhbHVlOiB0aGlzLmluc3RhbmNlLmluc3RhbmNlSWQsXG4gICAgICBleHBvcnROYW1lOiBgJHtzdGFja05hbWV9LWluc3RhbmNlLWlkYCxcbiAgICB9KTtcbiAgICBuZXcgQ2ZuT3V0cHV0KHRoaXMsIFwiaW5zdGFuY2UtcHVibGljLWlwLW91dHB1dFwiLCB7XG4gICAgICB2YWx1ZTogdGhpcy5pbnN0YW5jZS5pbnN0YW5jZVB1YmxpY0lwLFxuICAgICAgZXhwb3J0TmFtZTogYCR7c3RhY2tOYW1lfS1pbnN0YW5jZS1wdWJsaWMtaXBgLFxuICAgIH0pO1xuICAgIG5ldyBDZm5PdXRwdXQodGhpcywgXCJpbnN0YW5jZS1wdWJsaWMtZG5zLW5hbWUtb3V0cHV0XCIsIHtcbiAgICAgIHZhbHVlOiB0aGlzLmluc3RhbmNlLmluc3RhbmNlUHVibGljRG5zTmFtZSxcbiAgICAgIGV4cG9ydE5hbWU6IGAke3N0YWNrTmFtZX0tcHVibGljLWRucy1uYW1lYCxcbiAgICB9KTtcbiAgfVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJhc3Rpb25Ib3N0UHJvcHMge1xuICByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuICByZWFkb25seSBkYjogcmRzLklEYXRhYmFzZUluc3RhbmNlO1xuICByZWFkb25seSB1c2VyRGF0YTogZWMyLlVzZXJEYXRhO1xuICByZWFkb25seSBpcHY0QWxsb3dsaXN0OiBzdHJpbmdbXTtcbiAgcmVhZG9ubHkgc3NoUG9ydD86IG51bWJlcjtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgYW4gZWxhc3RpYyBJUCBzaG91bGQgYmUgY3JlYXRlZCBmb3IgdGhlIGJhc3Rpb24gaG9zdC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNyZWF0ZUVsYXN0aWNJcD86IGJvb2xlYW47XG59XG4iXX0=