"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PgBouncer = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const fs = require("fs");
const path = require("path");
class PgBouncer extends constructs_1.Construct {
    // The max_connections parameter in PgBouncer determines the maximum number of
    // connections to open on the actual database instance. We want that number to
    // be slightly smaller than the actual max_connections value on the RDS instance
    // so we perform this calculation.
    getDefaultPgbouncerConfig(dbMaxConnections) {
        // maxDbConnections (and maxUserConnections) are the only settings that need
        // to be responsive to the database size/max_connections setting
        return {
            poolMode: "transaction",
            maxClientConn: 1000,
            defaultPoolSize: 5,
            minPoolSize: 0,
            reservePoolSize: 5,
            reservePoolTimeout: 5,
            maxDbConnections: dbMaxConnections - 10,
            maxUserConnections: dbMaxConnections - 10,
        };
    }
    constructor(scope, id, props) {
        super(scope, id);
        // Set defaults for optional props
        const defaultPgbouncerConfig = this.getDefaultPgbouncerConfig(props.dbMaxConnections);
        // Merge provided config with defaults
        const pgBouncerConfig = {
            ...defaultPgbouncerConfig,
            ...props.pgBouncerConfig,
        };
        // Create role for PgBouncer instance to enable writing to CloudWatch
        const role = new aws_cdk_lib_1.aws_iam.Role(this, "InstanceRole", {
            description: "pgbouncer instance role with Systems Manager + CloudWatch permissions",
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("ec2.amazonaws.com"),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("AmazonSSMManagedInstanceCore"),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("CloudWatchAgentServerPolicy"),
            ],
        });
        // Add policy to allow reading RDS credentials from Secrets Manager
        role.addToPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ["secretsmanager:GetSecretValue"],
            resources: [props.database.secret.secretArn],
        }));
        // Create a security group and allow connections from the Lambda IP ranges for this region
        this.securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, "PgBouncerSecurityGroup", {
            vpc: props.vpc,
            description: "Security group for PgBouncer instance",
            allowAllOutbound: true,
        });
        // Create PgBouncer instance
        const defaultInstanceConfig = {
            instanceName: "pgbouncer",
            instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.T3, aws_cdk_lib_1.aws_ec2.InstanceSize.MICRO),
            vpcSubnets: {
                subnetType: props.usePublicSubnet
                    ? aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC
                    : aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_EGRESS,
            },
            machineImage: aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter("/aws/service/canonical/ubuntu/server/noble/stable/current/amd64/hvm/ebs-gp3/ami-id", { os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX }),
            blockDevices: [
                {
                    deviceName: "/dev/xvda",
                    volume: aws_cdk_lib_1.aws_ec2.BlockDeviceVolume.ebs(20, {
                        volumeType: aws_cdk_lib_1.aws_ec2.EbsDeviceVolumeType.GP3,
                        encrypted: true,
                        deleteOnTermination: true,
                    }),
                },
            ],
            role,
            securityGroup: this.securityGroup,
            userData: this.loadUserDataScript(pgBouncerConfig, props.database),
            userDataCausesReplacement: true,
            associatePublicIpAddress: props.usePublicSubnet,
        };
        this.instance = new aws_cdk_lib_1.aws_ec2.Instance(this, "Instance", {
            ...defaultInstanceConfig,
            ...props.instanceProps,
            vpc: props.vpc,
        });
        // Allow PgBouncer to connect to RDS
        props.database.connections.allowFrom(this.instance, aws_cdk_lib_1.aws_ec2.Port.tcp(5432), "Allow PgBouncer to connect to RDS");
        // Create a new secret for pgbouncer connection credentials
        this.pgbouncerSecret = new aws_cdk_lib_1.aws_secretsmanager.Secret(this, "PgBouncerSecret", {
            description: "Connection information for PgBouncer instance",
            generateSecretString: {
                generateStringKey: "dummy",
                secretStringTemplate: "{}",
            },
        });
        // Grant the role permission to read the new secret
        this.pgbouncerSecret.grantRead(role);
        // Update pgbouncerSecret to contain pgstacSecret values but with new value for host
        const secretUpdaterFn = new aws_cdk_lib_1.aws_lambda.Function(this, "SecretUpdaterFunction", {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_20_X,
            handler: "index.handler",
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(path.join(__dirname, "lambda/pgbouncer-secret-updater")),
            environment: {
                SOURCE_SECRET_ARN: props.database.secret.secretArn,
                TARGET_SECRET_ARN: this.pgbouncerSecret.secretArn,
            },
        });
        props.database.secret.grantRead(secretUpdaterFn);
        this.pgbouncerSecret.grantWrite(secretUpdaterFn);
        this.secretUpdateComplete = new aws_cdk_lib_1.CustomResource(this, "pgbouncerSecretBootstrapper", {
            serviceToken: secretUpdaterFn.functionArn,
            properties: {
                instanceIp: props.usePublicSubnet
                    ? this.instance.instancePublicIp
                    : this.instance.instancePrivateIp,
            },
        });
        // Add health check custom resource
        const healthCheckFunction = new aws_cdk_lib_1.aws_lambda.Function(this, "HealthCheckFunction", {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_20_X,
            handler: "index.handler",
            timeout: aws_cdk_lib_1.Duration.minutes(10),
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(path.join(__dirname, "lambda/pgbouncer-health-check")),
            description: "PgBouncer health check function",
        });
        // Grant SSM permissions for health check
        healthCheckFunction.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: [
                "ssm:SendCommand",
                "ssm:GetCommandInvocation",
                "ssm:DescribeInstanceInformation",
                "ssm:ListCommandInvocations",
            ],
            resources: ["*"],
        }));
        this.healthCheck = new aws_cdk_lib_1.CustomResource(this, "PgBouncerHealthCheck", {
            serviceToken: healthCheckFunction.functionArn,
            properties: {
                InstanceId: this.instance.instanceId,
                // Add timestamp to force re-execution on stack updates
                Timestamp: new Date().toISOString(),
            },
        });
        // Ensure health check runs after instance is created but before secret update
        this.healthCheck.node.addDependency(this.instance);
        this.secretUpdateComplete.node.addDependency(this.healthCheck);
    }
    loadUserDataScript(pgBouncerConfig, database) {
        const userDataScript = aws_cdk_lib_1.aws_ec2.UserData.forLinux();
        // Set environment variables with configuration parameters
        userDataScript.addCommands('export SECRET_ARN="' + database.secret.secretArn + '"', 'export REGION="' + aws_cdk_lib_1.Stack.of(this).region + '"', 'export POOL_MODE="' + pgBouncerConfig.poolMode + '"', 'export MAX_CLIENT_CONN="' + pgBouncerConfig.maxClientConn + '"', 'export DEFAULT_POOL_SIZE="' + pgBouncerConfig.defaultPoolSize + '"', 'export MIN_POOL_SIZE="' + pgBouncerConfig.minPoolSize + '"', 'export RESERVE_POOL_SIZE="' + pgBouncerConfig.reservePoolSize + '"', 'export RESERVE_POOL_TIMEOUT="' +
            pgBouncerConfig.reservePoolTimeout +
            '"', 'export MAX_DB_CONNECTIONS="' + pgBouncerConfig.maxDbConnections + '"', 'export MAX_USER_CONNECTIONS="' + pgBouncerConfig.maxUserConnections + '"');
        // Load the startup script
        const scriptPath = path.join(__dirname, "./pgbouncer-setup.sh");
        let script = fs.readFileSync(scriptPath, "utf8");
        userDataScript.addCommands(script);
        return userDataScript;
    }
}
exports.PgBouncer = PgBouncer;
//# sourceMappingURL=data:application/json;base64,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