import { CustomResource, aws_ec2 as ec2, aws_rds as rds, aws_secretsmanager as secretsmanager } from "aws-cdk-lib";
import { Construct } from "constructs";
import { CustomLambdaFunctionProps } from "../utils";
/**
 * An RDS instance with pgSTAC installed and PgBouncer connection pooling.
 *
 * This construct creates an optimized pgSTAC database setup that includes:
 * - RDS PostgreSQL instance with pgSTAC extension
 * - PgBouncer connection pooler (enabled by default)
 * - Automated health monitoring system
 * - Optimized database parameters for the selected instance type
 *
 * ## Connection Pooling with PgBouncer
 *
 * By default, this construct deploys PgBouncer as a connection pooler running on
 * a dedicated EC2 instance. PgBouncer provides several benefits:
 *
 * - **Connection Management**: Pools and reuses database connections to reduce overhead
 * - **Performance**: Optimizes connection handling for high-traffic applications
 * - **Scalability**: Allows more concurrent connections than the RDS instance alone
 * - **Health Monitoring**: Includes comprehensive health checks to ensure availability
 *
 * ### PgBouncer Configuration
 * - Pool mode: Transaction-level pooling (default)
 * - Maximum client connections: 1000
 * - Default pool size: 20 connections per database/user combination
 * - Instance type: t3.micro EC2 instance
 *
 * ### Health Check System
 * The construct includes an automated health check system that validates:
 * - PgBouncer service is running and listening on port 5432
 * - Connection tests to ensure accessibility
 * - Cloud-init setup completion before validation
 * - Detailed diagnostics for troubleshooting
 *
 * ### Connection Details
 * When PgBouncer is enabled, applications connect through the PgBouncer instance
 * rather than directly to RDS. The `pgstacSecret` contains connection information
 * pointing to PgBouncer, and the `connectionTarget` property refers to the
 * PgBouncer EC2 instance.
 *
 * To disable PgBouncer and connect directly to RDS, set `addPgbouncer: false`.
 *
 * This is a wrapper around the `rds.DatabaseInstance` higher-level construct
 * making use of the BootstrapPgStac construct.
 */
export declare class PgStacDatabase extends Construct {
    db: rds.DatabaseInstance;
    pgstacSecret: secretsmanager.ISecret;
    private _pgBouncerServer?;
    readonly pgstacVersion: string;
    readonly connectionTarget: rds.IDatabaseInstance | ec2.Instance;
    readonly securityGroup?: ec2.SecurityGroup;
    readonly secretBootstrapper?: CustomResource;
    readonly pgbouncerHealthCheck?: CustomResource;
    constructor(scope: Construct, id: string, props: PgStacDatabaseProps);
    getParameters(instanceType: string, parameters: PgStacDatabaseProps["parameters"]): DatabaseParameters;
}
export interface PgStacDatabaseProps extends rds.DatabaseInstanceProps {
    /**
     * Name of database that is to be created and onto which pgSTAC will be installed.
     *
     * @default pgstac
     */
    readonly pgstacDbName?: string;
    /**
     * Version of pgstac to install on the database
     *
     * @default 0.8.5
     */
    readonly pgstacVersion?: string;
    /**
     * Prefix to assign to the generated `secrets_manager.Secret`
     *
     * @default pgstac
     */
    readonly secretsPrefix?: string;
    /**
     * Name of user that will be generated for connecting to the pgSTAC database.
     *
     * @default pgstac_user
     */
    readonly pgstacUsername?: string;
    /**
     * Add pgbouncer instance for managing traffic to the pgSTAC database
     *
     * @default true
     */
    readonly addPgbouncer?: boolean;
    /**
     * Properties for the pgbouncer ec2 instance
     *
     * @default - defined in the construct
     */
    readonly pgbouncerInstanceProps?: ec2.InstanceProps | any;
    /**
     * Lambda function Custom Resource properties. A custom resource property is going to be created
     * to trigger the boostrapping lambda function. This parameter allows the user to specify additional properties
     * on top of the defaults ones.
     *
     */
    readonly customResourceProperties?: {
        [key: string]: any;
    };
    /**
     * Can be used to override the default lambda function properties.
     *
     * @default - defined in the construct.
     */
    readonly bootstrapperLambdaFunctionOptions?: CustomLambdaFunctionProps;
}
export interface DatabaseParameters {
    /**
     * @default - LEAST({DBInstanceClassMemory/9531392}, 5000)
     */
    readonly maxConnections: string;
    /**
     * Note: This value is measured in 8KB blocks.
     *
     * @default '{DBInstanceClassMemory/32768}' 25% of instance memory, ie `{(DBInstanceClassMemory/(1024*8)) * 0.25}`
     */
    readonly sharedBuffers: string;
    /**
     * @default - 75% of instance memory
     */
    readonly effectiveCacheSize: string;
    /**
     * @default - shared buffers divided by max connections
     */
    readonly workMem: string;
    /**
     * @default - 25% of shared buffers
     */
    readonly maintenanceWorkMem: string;
    /**
     * @default 1024
     */
    readonly maxLocksPerTransaction: string;
    /**
     * @default 131172 (128 * 1024)
     */
    readonly tempBuffers: string;
    /**
     * @default 1
     */
    readonly seqPageCost: string;
    /**
     * @default 1.1
     */
    readonly randomPageCost: string;
}
