"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PgStacDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const utils_1 = require("../utils");
const PgBouncer_1 = require("./PgBouncer");
const instanceSizes = require("./instance-memory.json");
let defaultPgSTACCustomOptions = {
    context: "FALSE",
    mosaic_index: "TRUE",
};
function hasVpc(instance) {
    return instance.vpc !== undefined;
}
/**
 * An RDS instance with pgSTAC installed and PgBouncer connection pooling.
 *
 * This construct creates an optimized pgSTAC database setup that includes:
 * - RDS PostgreSQL instance with pgSTAC extension
 * - PgBouncer connection pooler (enabled by default)
 * - Automated health monitoring system
 * - Optimized database parameters for the selected instance type
 *
 * ## Connection Pooling with PgBouncer
 *
 * By default, this construct deploys PgBouncer as a connection pooler running on
 * a dedicated EC2 instance. PgBouncer provides several benefits:
 *
 * - **Connection Management**: Pools and reuses database connections to reduce overhead
 * - **Performance**: Optimizes connection handling for high-traffic applications
 * - **Scalability**: Allows more concurrent connections than the RDS instance alone
 * - **Health Monitoring**: Includes comprehensive health checks to ensure availability
 *
 * ### PgBouncer Configuration
 * - Pool mode: Transaction-level pooling (default)
 * - Maximum client connections: 1000
 * - Default pool size: 20 connections per database/user combination
 * - Instance type: t3.micro EC2 instance
 *
 * ### Health Check System
 * The construct includes an automated health check system that validates:
 * - PgBouncer service is running and listening on port 5432
 * - Connection tests to ensure accessibility
 * - Cloud-init setup completion before validation
 * - Detailed diagnostics for troubleshooting
 *
 * ### Connection Details
 * When PgBouncer is enabled, applications connect through the PgBouncer instance
 * rather than directly to RDS. The `pgstacSecret` contains connection information
 * pointing to PgBouncer, and the `connectionTarget` property refers to the
 * PgBouncer EC2 instance.
 *
 * To disable PgBouncer and connect directly to RDS, set `addPgbouncer: false`.
 *
 * This is a wrapper around the `rds.DatabaseInstance` higher-level construct
 * making use of the BootstrapPgStac construct.
 */
class PgStacDatabase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const defaultParameters = this.getParameters(props.instanceType?.toString() || "m5.large", props.parameters);
        const parameterGroup = new aws_cdk_lib_1.aws_rds.ParameterGroup(this, "parameterGroup", {
            engine: props.engine,
            parameters: {
                max_connections: defaultParameters.maxConnections,
                shared_buffers: defaultParameters.sharedBuffers,
                effective_cache_size: defaultParameters.effectiveCacheSize,
                work_mem: defaultParameters.workMem,
                maintenance_work_mem: defaultParameters.maintenanceWorkMem,
                max_locks_per_transaction: defaultParameters.maxLocksPerTransaction,
                temp_buffers: defaultParameters.tempBuffers,
                seq_page_cost: defaultParameters.seqPageCost,
                random_page_cost: defaultParameters.randomPageCost,
                ...props.parameters,
            },
        });
        this.db = new aws_cdk_lib_1.aws_rds.DatabaseInstance(this, "db", {
            instanceIdentifier: aws_cdk_lib_1.Stack.of(this).stackName,
            parameterGroup,
            ...props,
        });
        this.pgstacVersion = props.pgstacVersion || utils_1.DEFAULT_PGSTAC_VERSION;
        const handler = new aws_cdk_lib_1.aws_lambda.Function(this, "lambda", {
            // defaults
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_12,
            handler: "handler.handler",
            memorySize: 128,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.minutes(2),
            code: aws_cdk_lib_1.aws_lambda.Code.fromDockerBuild(__dirname, {
                file: "bootstrapper_runtime/Dockerfile",
                buildArgs: {
                    PYTHON_VERSION: "3.12",
                    PGSTAC_VERSION: this.pgstacVersion,
                },
            }),
            vpc: hasVpc(this.db) ? this.db.vpc : props.vpc,
            allowPublicSubnet: true,
            // overwrites defaults with user-provided configurable properties,
            ...props.bootstrapperLambdaFunctionOptions,
        });
        this.pgstacSecret = new aws_cdk_lib_1.aws_secretsmanager.Secret(this, "bootstrappersecret", {
            secretName: [
                props.secretsPrefix || "pgstac",
                id,
                this.node.addr.slice(-8),
            ].join("/"),
            generateSecretString: {
                secretStringTemplate: JSON.stringify({
                    dbname: props.pgstacDbName || "pgstac",
                    engine: "postgres",
                    port: 5432,
                    host: this.db.instanceEndpoint.hostname,
                    username: props.pgstacUsername || "pgstac_user",
                }),
                generateStringKey: "password",
                excludePunctuation: true,
            },
            description: `PgSTAC database bootstrapped by ${aws_cdk_lib_1.Stack.of(this).stackName}`,
        });
        // Allow lambda to...
        // read new user secret
        this.pgstacSecret.grantRead(handler);
        // read database secret
        this.db.secret.grantRead(handler);
        // connect to database
        this.db.connections.allowFrom(handler, aws_cdk_lib_1.aws_ec2.Port.tcp(5432));
        let customResourceProperties = props.customResourceProperties
            ? { ...defaultPgSTACCustomOptions, ...props.customResourceProperties }
            : defaultPgSTACCustomOptions;
        // update properties
        customResourceProperties["conn_secret_arn"] = this.db.secret.secretArn;
        customResourceProperties["new_user_secret_arn"] =
            this.pgstacSecret.secretArn;
        // if props.lambdaFunctionOptions doesn't have 'code' defined, update pgstac_version (needed for default runtime)
        if (!props.bootstrapperLambdaFunctionOptions?.code) {
            customResourceProperties["pgstac_version"] = this.pgstacVersion;
        }
        // add timestamp to properties to ensure the Lambda gets re-executed on each deploy
        customResourceProperties["timestamp"] = new Date().toISOString();
        const bootstrapper = new aws_cdk_lib_1.CustomResource(this, "bootstrapper", {
            serviceToken: handler.functionArn,
            properties: customResourceProperties,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN, // This retains the custom resource (which doesn't really exist), not the database
        });
        // PgBouncer: connection poolercustomresource trigger on redeploy
        const defaultPgbouncerInstanceProps = {
            instanceName: `${aws_cdk_lib_1.Stack.of(this).stackName}-pgbouncer`,
            instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.T3, aws_cdk_lib_1.aws_ec2.InstanceSize.MICRO),
        };
        const addPgbouncer = props.addPgbouncer ?? true;
        if (addPgbouncer) {
            this._pgBouncerServer = new PgBouncer_1.PgBouncer(this, "pgbouncer", {
                instanceProps: {
                    ...defaultPgbouncerInstanceProps,
                    ...props.pgbouncerInstanceProps,
                },
                vpc: props.vpc,
                database: {
                    connections: this.db.connections,
                    secret: this.pgstacSecret,
                },
                dbMaxConnections: parseInt(defaultParameters.maxConnections),
                usePublicSubnet: !props.vpcSubnets ||
                    props.vpcSubnets.subnetType === aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC,
                pgBouncerConfig: {
                    poolMode: "transaction",
                    maxClientConn: 1000,
                    defaultPoolSize: 20,
                    minPoolSize: 10,
                    reservePoolSize: 5,
                    reservePoolTimeout: 5,
                },
            });
            this._pgBouncerServer.node.addDependency(bootstrapper);
            this.pgstacSecret = this._pgBouncerServer.pgbouncerSecret;
            this.connectionTarget = this._pgBouncerServer.instance;
            this.securityGroup = this._pgBouncerServer.securityGroup;
            this.secretBootstrapper = this._pgBouncerServer.secretUpdateComplete;
            this.pgbouncerHealthCheck = this._pgBouncerServer.healthCheck;
        }
        else {
            this.connectionTarget = this.db;
        }
    }
    getParameters(instanceType, parameters) {
        // https://github.com/aws/aws-cli/issues/1279#issuecomment-909318236
        const memory_in_kb = instanceSizes[instanceType] * 1024;
        // It's only necessary to consider passed in parameters for any value that used to
        // derive subsequent values. Values that don't have dependencies will be overriden
        // when we unpack the passed-in user parameters
        const maxConnections = parameters?.maxConnections
            ? Number.parseInt(parameters.maxConnections)
            : // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/CHAP_Limits.html#RDS_Limits.MaxConnections
                Math.min(Math.round((memory_in_kb * 1024) / 9531392), 5000);
        const sharedBuffers = parameters?.sharedBufers
            ? Number.parseInt(parameters.sharedBufers)
            : Math.round(0.25 * memory_in_kb);
        const effectiveCacheSize = Math.round(0.75 * memory_in_kb);
        const workMem = Math.floor(sharedBuffers / maxConnections);
        const maintenanceWorkMem = Math.round(0.25 * sharedBuffers);
        const tempBuffers = 128 * 1024;
        const seqPageCost = 1;
        const randomPageCost = 1.1;
        return {
            maxConnections: `${maxConnections}`,
            sharedBuffers: `${sharedBuffers / 8}`, // Represented in 8kb blocks
            effectiveCacheSize: `${effectiveCacheSize / 8}`, // Represented in 8kb blocks
            workMem: `${workMem}`,
            maintenanceWorkMem: `${maintenanceWorkMem}`,
            maxLocksPerTransaction: "1024",
            tempBuffers: `${tempBuffers / 8}`, // Represented in 8kb blocks
            seqPageCost: `${seqPageCost}`,
            randomPageCost: `${randomPageCost}`,
        };
    }
}
exports.PgStacDatabase = PgStacDatabase;
_a = JSII_RTTI_SYMBOL_1;
PgStacDatabase[_a] = { fqn: "eoapi-cdk.PgStacDatabase", version: "10.2.1" };
//# sourceMappingURL=data:application/json;base64,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