"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_ssm_1 = require("@aws-sdk/client-ssm");
const https = require("https");
const url = require("url");
const fs = require("fs");
const path = require("path");
const ssmClient = new client_ssm_1.SSMClient();
const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));
async function sendResponse(event, context, responseStatus, responseData, physicalResourceId) {
    const responseBody = JSON.stringify({
        Status: responseStatus,
        Reason: responseData.Reason ||
            `See CloudWatch Log Stream: ${context.logStreamName}`,
        PhysicalResourceId: physicalResourceId || context.logStreamName,
        StackId: event.StackId,
        RequestId: event.RequestId,
        LogicalResourceId: event.LogicalResourceId,
        Data: responseData,
    });
    const parsedUrl = url.parse(event.ResponseURL);
    return new Promise((resolve, reject) => {
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: "PUT",
            headers: {
                "Content-Type": "",
                "Content-Length": responseBody.length,
            },
        };
        const req = https.request(options, (res) => {
            console.log(`STATUS: ${res.statusCode}`);
            console.log(`HEADERS: ${JSON.stringify(res.headers)}`);
            resolve();
        });
        req.on("error", (err) => {
            console.log("sendResponse Error:", err);
            reject(err);
        });
        req.write(responseBody);
        req.end();
    });
}
async function waitForInstanceReady(instanceId, maxWaitMinutes = 8) {
    console.log(`Waiting for instance ${instanceId} to be ready for SSM commands...`);
    const maxAttempts = maxWaitMinutes * 4; // Check every 15 seconds
    for (let attempt = 1; attempt <= maxAttempts; attempt++) {
        try {
            const command = new client_ssm_1.DescribeInstanceInformationCommand({
                InstanceInformationFilterList: [
                    {
                        key: "InstanceIds",
                        valueSet: [instanceId],
                    },
                ],
            });
            const result = await ssmClient.send(command);
            if (result.InstanceInformationList &&
                result.InstanceInformationList.length > 0) {
                const instance = result.InstanceInformationList[0];
                if (instance.PingStatus === "Online") {
                    console.log(`Instance ${instanceId} is online and ready for SSM commands`);
                    return true;
                }
                console.log(`Instance ${instanceId} ping status: ${instance.PingStatus}, attempt ${attempt}/${maxAttempts}`);
            }
            else {
                console.log(`Instance ${instanceId} not found in SSM, attempt ${attempt}/${maxAttempts}`);
            }
        }
        catch (error) {
            const errorMessage = error instanceof Error ? error.message : "Unknown error";
            console.log(`Error checking instance readiness (attempt ${attempt}/${maxAttempts}):`, errorMessage);
        }
        if (attempt < maxAttempts) {
            await sleep(15000); // Wait 15 seconds between attempts
        }
    }
    throw new Error(`Instance ${instanceId} did not become ready for SSM commands within ${maxWaitMinutes} minutes`);
}
async function checkPgBouncerHealth(instanceId) {
    console.log(`Running health check on instance ${instanceId}`);
    // Load health check script from file
    const scriptPath = path.join(__dirname, "health-check.sh");
    let healthCheckScript;
    try {
        healthCheckScript = fs.readFileSync(scriptPath, "utf8");
    }
    catch (error) {
        console.error("Failed to load health check script:", error);
        return {
            success: false,
            error: `Failed to load health check script: ${error}`,
        };
    }
    // Use the entire script as a single command
    const healthCheckCommands = [healthCheckScript];
    const command = new client_ssm_1.SendCommandCommand({
        InstanceIds: [instanceId],
        DocumentName: "AWS-RunShellScript",
        Parameters: {
            commands: healthCheckCommands,
            executionTimeout: ["600"], // 10 minutes timeout
        },
        Comment: "PgBouncer comprehensive health check with setup validation",
        TimeoutSeconds: 600,
    });
    try {
        console.log("Sending SSM command...");
        const result = await ssmClient.send(command);
        const commandId = result.Command?.CommandId;
        if (!commandId) {
            return {
                success: false,
                error: "Failed to get command ID from SSM response",
            };
        }
        console.log(`Command sent with ID: ${commandId}`);
        // Wait for command to complete
        const maxAttempts = 60; // 10 minutes with 10-second intervals
        for (let attempt = 1; attempt <= maxAttempts; attempt++) {
            await sleep(10000); // Wait 10 seconds between status checks
            try {
                const invocationCommand = new client_ssm_1.GetCommandInvocationCommand({
                    CommandId: commandId,
                    InstanceId: instanceId,
                });
                const invocation = await ssmClient.send(invocationCommand);
                console.log(`Command status (attempt ${attempt}): ${invocation.Status}`);
                if (invocation.Status === "Success") {
                    console.log("Health check passed!");
                    console.log("STDOUT:", invocation.StandardOutputContent);
                    return {
                        success: true,
                        output: invocation.StandardOutputContent,
                    };
                }
                else if (invocation.Status === "Failed") {
                    console.log("Health check failed!");
                    console.log("STDOUT:", invocation.StandardOutputContent);
                    console.log("STDERR:", invocation.StandardErrorContent);
                    return {
                        success: false,
                        error: `Health check failed: ${invocation.StandardErrorContent}`,
                        output: invocation.StandardOutputContent,
                    };
                }
                else if (["Cancelled", "TimedOut", "Cancelling"].includes(invocation.Status || "")) {
                    return {
                        success: false,
                        error: `Health check ${invocation.Status?.toLowerCase()}`,
                    };
                }
                // Continue waiting for InProgress, Pending, Delayed states
            }
            catch (error) {
                if (error.name === "InvocationDoesNotExist") {
                    console.log(`Command invocation not yet available (attempt ${attempt})`);
                    continue;
                }
                throw error;
            }
        }
        return {
            success: false,
            error: "Health check timed out waiting for command completion",
        };
    }
    catch (error) {
        const errorMessage = error instanceof Error ? error.message : "Unknown error";
        console.log("Error running health check:", error);
        return {
            success: false,
            error: `Health check error: ${errorMessage}`,
        };
    }
}
const handler = async (event, context) => {
    console.log("Event:", JSON.stringify(event, null, 2));
    const instanceId = event.ResourceProperties?.InstanceId;
    const requestType = event.RequestType;
    try {
        if (requestType === "Delete") {
            console.log("Delete request - no action needed");
            await sendResponse(event, context, "SUCCESS", {});
            return;
        }
        if (!instanceId) {
            throw new Error("InstanceId is required");
        }
        console.log(`Processing ${requestType} request for instance ${instanceId}`);
        // Wait for instance to be ready for SSM
        await waitForInstanceReady(instanceId);
        // Run health check
        const healthResult = await checkPgBouncerHealth(instanceId);
        if (healthResult.success) {
            console.log("PgBouncer health check passed");
            await sendResponse(event, context, "SUCCESS", {
                Message: "PgBouncer is healthy",
            }, `pgbouncer-health-${instanceId}`);
        }
        else {
            console.log("PgBouncer health check failed:", healthResult.error);
            await sendResponse(event, context, "FAILED", {
                Reason: `PgBouncer health check failed: ${healthResult.error}`,
            }, `pgbouncer-health-${instanceId}`);
        }
    }
    catch (error) {
        const errorMessage = error instanceof Error ? error.message : "Unknown error occurred";
        console.log("Handler error:", error);
        await sendResponse(event, context, "FAILED", {
            Reason: errorMessage,
        }, `pgbouncer-health-${instanceId || "unknown"}`);
    }
};
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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