"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const https = require("https");
const url = require("url");
function sendResponse(event, context, responseStatus, responseData, physicalResourceId) {
    return new Promise((resolve, reject) => {
        const responseBody = JSON.stringify({
            Status: responseStatus,
            Reason: "See the details in CloudWatch Log Stream: " + context.logStreamName,
            PhysicalResourceId: physicalResourceId || context.logStreamName,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            Data: responseData,
        });
        console.log("Response body:", responseBody);
        const parsedUrl = url.parse(event.ResponseURL);
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: "PUT",
            headers: {
                "content-type": "",
                "content-length": responseBody.length,
            },
        };
        const request = https.request(options, (response) => {
            console.log("Status code:", response.statusCode);
            console.log("Status message:", response.statusMessage);
            resolve();
        });
        request.on("error", (error) => {
            console.log("sendResponse Error:", error);
            reject(error);
        });
        request.write(responseBody);
        request.end();
    });
}
const client = new client_secrets_manager_1.SecretsManagerClient();
function validateEnvironment() {
    const { SOURCE_SECRET_ARN, TARGET_SECRET_ARN } = process.env;
    if (!SOURCE_SECRET_ARN) {
        throw new Error("SOURCE_SECRET_ARN environment variable is required");
    }
    if (!TARGET_SECRET_ARN) {
        throw new Error("TARGET_SECRET_ARN environment variable is required");
    }
    return {
        SOURCE_SECRET_ARN,
        TARGET_SECRET_ARN,
    };
}
const handler = async (event, context) => {
    console.log("Event:", JSON.stringify(event, null, 2));
    try {
        const env = validateEnvironment();
        // Skip processing for DELETE requests, but still send response
        if (event.RequestType === "Delete") {
            await sendResponse(event, context, "SUCCESS", {}, event.PhysicalResourceId);
            return;
        }
        const instanceIp = event.ResourceProperties.instanceIp;
        // Get the original secret value
        const getSecretResponse = await client.send(new client_secrets_manager_1.GetSecretValueCommand({
            SecretId: env.SOURCE_SECRET_ARN,
        }));
        if (!getSecretResponse.SecretString) {
            throw new Error("Secret string is empty");
        }
        // Parse the secret string
        const secretData = JSON.parse(getSecretResponse.SecretString);
        // Update the host value with the PgBouncer instance IP
        secretData.host = instanceIp;
        // Put the modified secret value
        await client.send(new client_secrets_manager_1.PutSecretValueCommand({
            SecretId: env.TARGET_SECRET_ARN,
            SecretString: JSON.stringify(secretData),
        }));
        const physicalResourceId = env.TARGET_SECRET_ARN;
        const responseData = {
            SecretArn: env.TARGET_SECRET_ARN,
        };
        await sendResponse(event, context, "SUCCESS", responseData, physicalResourceId);
    }
    catch (error) {
        console.error("Error:", error);
        const errorMessage = error instanceof Error ? error.message : "An unknown error occurred";
        await sendResponse(event, context, "FAILED", { Error: errorMessage });
        throw error;
    }
};
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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